/*
 * PROPRIETARY INFORMATION.  This software is proprietary to POWDER
 * Development, and is not to be reproduced, transmitted, or disclosed
 * in any way without written permission.
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        ai.cpp ( Save Scummer Library, C++ )
 *
 * COMMENTS:
 */

#include "mob.h"

#include "map.h"
#include "msg.h"
#include "text.h"
#include "speed.h"
#include "item.h"
#include "avatar.h"

bool
MOB::aiForcedAction()
{
    // Check for the phase...
    PHASE_NAMES		phase;

    phase = spd_getphase();

    // On normal & slow phases, items get heartbeat.
    if (phase == PHASE_NORMAL || phase == PHASE_SLOW)
    {
	ITEM	*item, *next;
	for (item = myInventory; item; item = next)
	{
	    next = item->getNext();

	    if (item->runHeartbeat())
	    {
		removeItem(item);
		delete item;
	    }
	}

	// Regenerate if we are capable of it
	if (defn().isregenerate || hasItem(ITEM_REGENRING))
	{
	    gainHP(1);
	}

	// Apply damage if inside.
	if (isSwallowed())
	{
	    msg_format("%S <be> digested!", this);
	    if (applyDamage(0, 1, DPDF(1)))
		return true;
	}
    }

    switch (phase)
    {
	case PHASE_FAST:
	    if (hasItem(ITEM_SPEEDBOOTS))
		return false;
	    
	    // Not fast, no phase
	    if (defn().isfast)
		return false;
	    return true;

	case PHASE_QUICK:
	    // Not quick, no phase!
	    if (hasItem(ITEM_QUICKBOOST))
		return false;
	    return true;

	case PHASE_SLOW:
	    if (defn().isslow)
		return true;
	    if (hasItem(ITEM_SLOW))
		return true;
	    break;

	case PHASE_NORMAL:
	    break;
    }
    return false;
}

bool
MOB::aiDoAI()
{
    // Rebuild our home location.
    if (myHX < 0)
    {
	myHX = getX();
	myHY = getY();
    }

    if (myFleeCount > 0)
    {
	myFleeCount--;
	if (!isAvatar() && aiFleeFromAvatar())
	    return true;
    }

    myBoredom++;
    if (myBoredom > 1000)
	myBoredom = 1000;

    switch (getAI())
    {
	case AI_NONE:
	    // Just stay put!
	    return true;

	case AI_STAYHOME:
	    // If we are at home stay put.
	    if (getX() == myHX && getY() == myHY)
		return true;

	    // Otherwise, go home.
	    // FALL THROUGH

	case AI_HOME:
	    // Move towards our home square.
	    int		dist;

	    dist = MAX(ABS(getX()-myHX), ABS(getY() - myHY));

	    // If we can shoot the avatar or charge him, do so.
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(MOB::getAvatar(), AI_CHARGE))
		return true;

	    // A good chance to just stay put.
	    if (rand_chance(70))
	    {
		return actionBump(0, 0);
	    }

	    if (rand_choice(dist))
	    {
		// Try to home.
		if (aiMoveTo(myHX, myHY))
		    return true;
	    }
	    // Default to wandering
	    return aiRandomWalk();

	case AI_ORTHO:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(MOB::getAvatar(), AI_CHARGE, true))
		return true;
	    // Default to wandering
	    return aiRandomWalk(true);

	case AI_CHARGE:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(MOB::getAvatar(), AI_CHARGE))
		return true;
	    // Default to wandering
	    return aiRandomWalk();

	case AI_PATHTO:
	    if (aiKillPathTo(MOB::getAvatar()))
		return true;
		    
	    // Default to wandering
	    return aiRandomWalk();

	case AI_FLANK:
	    if (aiRangeAttack())
		return true;
	    if (aiCharge(MOB::getAvatar(), AI_FLANK))
		return true;
	    // Default to wandering
	    return aiRandomWalk();

	case AI_RANGECOWARD:
	    // If we can see avatar, shoot at them!
	    // We keep ourselves at range if possible, unless
	    // retreat is blocked, in which case we will melee.
	    if (getMap()->isFOV(getX(), getY()))
	    {
		int		 dist, dx, dy;

		if (aiAcquireAvatar())
		{
		    dx = SIGN(myTX - getX());
		    dy = SIGN(myTY - getY());
		    dist = MAX(ABS(getX()-myTX), ABS(getY() - myTY));

		    if (dist == 1)
		    {
			// We are in melee range.  Try to flee.
			if (aiFleeFrom(myTX, myTY))
			    return true;

			// Failed to flee.  Kill!
			return actionMelee(dx, dy);
		    }

		    // Try ranged attack.
		    if (aiRangeAttack())
			return true;

		    // Failed range attack.  If outside of range, charge.
		    if (dist > getRangedRange())
		    {
			if (aiCharge(MOB::getAvatar(), AI_FLANK))
			    return true;
		    }

		    // Otherwise, wander within the range hoping to line up.
		    // Trying to double think the human is likely pointless
		    // as they'll be trying to avoid lining up to.
		    return aiRandomWalk();
		}
	    }
		
	    if (aiCharge(MOB::getAvatar(), AI_FLANK))
		return true;

	    // Default to wandering
	    return aiRandomWalk();

	    
	case AI_COWARD:
	    if (aiRangeAttack())
		return true;

	    if (aiFleeFromAvatar())
		return true;
	    // Failed to flee, cornered, so charge!
	    if (aiCharge(MOB::getAvatar(), AI_CHARGE))
		return true;

	    return aiRandomWalk();

	case AI_MOUSE:
	    return aiDoMouse();

	case AI_STRAIGHTLINE:
	    return aiStraightLine();

	case AI_AVATAR:
	    return aiAvatar();
    }

    return false;
}

bool
MOB::aiAvatar()
{
    // Pop to hunt state if we see anything.
    if (hasVisibleEnemies())
    {
	myAIState = AI_STATE_HUNT;
    }

    // If boredom too high, reset and flee.
    // LIkely a perpetual check like condition.
    if (myBoredom > 200)
    {
	msg_format("%S <grow> bored.", this);
	clearBoredom();
	myFleeCount += 30;
    }
    
    // If running away, force next level.
    if (myFleeCount)
    {
	myAIState = AI_STATE_NEXTLEVEL;
    }

    // Determine our current state...
    switch (myAIState)
    {
	case AI_STATE_EXPLORE:
	    return aiAvatarExplore();
	case AI_STATE_NEXTLEVEL:
	    return aiAvatarNextLevel();
	case AI_STATE_HUNT:
	    return aiAvatarHunt();
	case AI_STATE_PACKRAT:
	    return aiAvatarPackrat();
    }

    return aiRandomWalk();
}

bool
MOB::aiAvatarExplore()
{
    // Check if there is a juicy item waiting to be pickedup.
    ITEM		*item;

    item = aiFindCloseMappedItem();
    if (item)
    {
	myAIState = AI_STATE_PACKRAT;
	myHX = item->getX();
	myHY = item->getY();
	return aiAvatarPackrat();
    }
    
    // If my home position is negative, it isn't yet discovered.
    // Attempt to path to the current home.
    if (myHX != -1 && aiPathFindTo(myHX, myHY))
	return true;

    // Find a new location to search for...
    if (getMap()->findCloseMoveableUnexplored(myHX, myHY, getX(), getY()))
    {
	return aiPathFindTo(myHX, myHY);
    }
	
    // Done exploring!  Let's dive.
    myAIState = AI_STATE_NEXTLEVEL;
    return aiAvatarNextLevel();
}

bool
MOB::aiAvatarPackrat()
{
    if (aiPathFindTo(myHX, myHY))
	return true;

    // Failed to go to item or already done so, pop back to explore mode.
    myAIState = AI_STATE_EXPLORE;
    return aiAvatarExplore();
}

bool
MOB::aiAvatarNextLevel()
{
    // Locate stairs.
    getMap()->findTile(TILE_DOWNSTAIRS, myHX, myHY, false);

    // See if we are alread there.
    if (getX() == myHX && getY() == myHY)
    {
	myAIState = AI_STATE_EXPLORE;
	myHX = myHY = -1;
	return actionClimb();
    }
    
    // Explore there..
    return aiPathFindTo(myHX, myHY);
}

ITEM *
MOB::aiFindCloseMappedItem() const
{
    ITEM		*item, *minitem = 0;
    int			 mindist = 1000, dist;

    for (item = getMap()->getItemHead(); item; item = item->getNext())
    {
	// Only count mapped items we know about.
	if (!getMap()->isMapped(item->getX(), item->getY()))
	    continue;

	dist = MAP::dist(this, item->getX(), item->getY());

	// Ignore the case we are standing on the item as we can
	// only pick up by walking onto.
	if (dist == 0)
	    continue;

	if (!minitem || dist < mindist)
	{
	    minitem = item;
	    mindist = dist;
	}
    }
    return minitem;
}

MOB *
MOB::aiFindEnemy() const
{
    PTRLIST<MOB *>	list;
    int			i;
    MOB			*biggestfoe, *foe;
    
    // The problem with this is if we have two stationary foes
    // a and b, where we hate b more, if b is out of sight and
    // we step to a and we then step to b, but in doing so hide
    // b, so we return our attention again to a.  An infinite loop
    // then develops.
    getVisibleEnemies(list);
    biggestfoe = 0;
    for (i = 0; i < list.entries(); i++)
    {
	foe = list(i);
	biggestfoe = aiHateMore(biggestfoe, foe);
    }
    return biggestfoe;
}

MOB *
MOB::aiHateMore(MOB *a, MOB *b) const
{
    // Check which is closer.
    int		bdx, bdy, adx, ady;
    int		adist, bdist;
    bool	alinedup, blinedup;

    // Trivial cases.
    if (!a)
	return b;
    if (!b)
	return a;

    adx = a->getX() - getX();
    ady = a->getY() - getY();
    bdx = b->getX() - getX();
    bdy = b->getY() - getY();

    adist = MAP::dist(a, this);
    bdist = MAP::dist(b, this);

    // If a is melee, take it.
    if (adist <= 1)
    {
	if (bdist <= 1)
	{
	    // Both are close.  We need to take out the one projected
	    // to do the most damage over the length of time it is
	    // expected we could kill it.
	    double		admg, bdmg, ourdmg;
	    double		acost, bcost;

	    ourdmg = getMeleeDPDF().expectedValue();
	    admg = a->getMeleeDPDF().expectedValue();
	    bdmg = b->getMeleeDPDF().expectedValue();
	    
	    // We compare the opportunity costs of killing either
	    // a or b first.
	    // It will take a->getHP() / ourdmg turns to kill a.
	    // In that time, b will do (bdmg * a->getHP() / ourdmg)
	    // damage to us.  Since we want to minimize damage, we pick
	    // the enemy for whom this value is highest.
	    // (Thanks to Thijs van Ommen who corrected my earlier
	    // calculation)
	    // TODO: The big exception is if the enemy can regenerate.
	    // Normal regenration should be modelled by lowering our damage
	    // output, vampirism modelled by subtracting admg from ourdmg.
	    acost = (admg * b->getHP()) / ourdmg;
	    bcost = (bdmg * a->getHP()) / ourdmg;

	    if (acost > bcost)
		return a;
	    return b;
	}
	return a;
    }
    // Ditto for b.
    if (bdist <= 1)
	return b;

    alinedup = MAP::isLinedUp(a, this);
    blinedup = MAP::isLinedUp(b, this);

    // Take the one that is lined up for attack
    if (alinedup)
    {
	if (blinedup)
	{
	    // Take closer..
	    if (adist < bdist)
		return a;
	    return b;
	}
	// Pick the lined up one.
	return a;
    }
    if (blinedup)
	return b;

    // Take closer
    if (adist < bdist)
    {
	return a;
    }
    return b;
}

ITEM *
MOB::aiLikeMore(ITEM *a, ITEM *b) const
{
    // Trivial case
    if (!a) return b;
    if (!b) return a;
    
    assert(a->getDefinition() == b->getDefinition());

    switch (a->getDefinition())
    {
	case ITEM_WEAPON:
	{
	    // Pick highest expected value.
	    if (avatar_weightweapon(a) > avatar_weightweapon(b))
		return a;
	    return b;
	    break;
	}

	case ITEM_SPELLBOOK:
	{
	    // Pick highest expected value.
	    if (avatar_weightbook(a) > avatar_weightbook(b))
		return a;
	    return b;
	}
    }

    // Default to a.
    return a;
}

bool
MOB::aiAvatarHunt()
{
    MOB		*foe;
    // Find most evil enemy
    foe = aiFindEnemy();
    if (foe)
    {
	myTX = foe->getX();
	myTY = foe->getY();

	// Try a ranged attack.
	if (aiRangeAttack(foe))
	    return true;

	// Do a path find, this will try to walk into which for avatars
	// does a melee attack.

	// If, however, we are swallowed, our worst enemy is
	// ourself.
	if (isSwallowed())
	{
	    return actionMelee(0, 0);
	}

	if (aiPathFindTo(myTX, myTY))
	{
	    return true;
	}
    }

    // No enemy, go to last known loc.
    if (aiPathFindTo(myTX, myTY))
    {
	return true;
    }

    // No enemy and we went to last known location of enemy, so 
    // drop down to exploration
    myAIState = AI_STATE_EXPLORE;
    myHX = myHY = -1;

    return aiAvatarExplore();
}


bool
MOB::aiDoMouse()
{
    int		lastdir, dir, newdir;
    int		dx, dy;
    bool	found = false;
    int		i;

    // Least three bits track our last direction.
    lastdir = myAIState & 7;

    // We want to try and track a wall.  We want this wall to
    // be on our right hand side.

    // Take a look at right hand square.
    rand_angletodir((lastdir + 2) & 7, dx, dy);
    
    if (!canMove(getX() + dx, getY() + dy, true) || 
	getMap()->getMob(getX() + dx, getY() + dy))
    {
	// Got a wall to the right.  Try first available direction.
	// Because we want to hug using diagonals, we try the forward
	// and right first.
	dir = (lastdir+1) & 7;
    }
    else
    {
	// No wall on right.  Try and go straight forward by default.
	dir = lastdir;
    }

    // If everyway is blocked, bump straight forward.
    newdir = lastdir;
    for (i = 0; i < 8; i++)
    {
	rand_angletodir(dir, dx, dy);
	if (canMove(getX() + dx, getY() + dy, true) &&
	    !getMap()->getMob(getX() + dx, getY() + dy))    
	{
	    newdir = dir;
	    break;
	}
	// Keeping wall on right means turning left first!
	dir--;
	dir &= 7;
    }

    if (newdir == -1)
	newdir = lastdir;
    else
	found = true;

    // Store our last direction.
    myAIState &= ~7;
    myAIState |= newdir;

    if (found)
    {
	return actionBump(dx, dy);
    }

    // Mouse is cornered!  Return to normal AI.
    return false;
}

bool
MOB::aiStraightLine()
{
    int		lastdir;
    int		dx, dy;
    MOB		*target;

    // Least three bits track our last direction.
    lastdir = myAIState & 7;

    // Bump & go.

    // Take a look at right hand square.
    rand_angletodir(lastdir, dx, dy);
    
    // If target is avatar, run into him!
    target = getMap()->getMob(getX() + dx, getY() + dy);
    if (target && !isFriends(target))
    {
	return actionBump(dx, dy);
    }
    
    if (!canMove(getX() + dx, getY() + dy, true))
    {
	// Can't go there!  Pick new direction.
	// Do not want same direction again.
	lastdir += rand_choice(7) + 1;
	lastdir &= 7;
    }
    myAIState = lastdir;
    rand_angletodir(lastdir, dx, dy);
    
    // If target is avatar, run into him!
    target = getMap()->getMob(getX() + dx, getY() + dy);
    if (target && !isFriends(target))
    {
	return actionBump(dx, dy);
    }

    // Bump failed.
    if (!canMove(getX() + dx, getY() + dy, true))
	return false;

    // Move forward!
    return actionBump(dx, dy);
}

bool
MOB::aiFleeFromAvatar()
{
    if (aiAcquireAvatar())
    {
	if (aiFleeFrom(myTX, myTY))
	    return true;
    }

    return false;
}

bool
MOB::aiFleeFrom(int x, int y)
{
    int		dx, dy;
    int		angle, resultangle = 0, i;
    int		choice = 0;

    dx = -SIGN(x - getX());
    dy = -SIGN(y - getY());
    
    angle = rand_dirtoangle(dx, dy);

    // 3 ones in same direction get equal chance.
    for (i = angle-1; i <= angle+1; i++)
    {
	rand_angletodir(i, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = i;
	}
    }

    if (!choice)
    {
	// Try a bit more desperately...
	for (i = angle-2; i <= angle+2; i += 4)
	{
	    rand_angletodir(i, dx, dy);
	    if (canMove(getX()+dx, getY()+dy))
	    {
		choice++;
		if (!rand_choice(choice))
		    resultangle = i;
	    }
	}
    }

    if (choice)
    {
	// Move in the direction
	rand_angletodir(resultangle, dx, dy);
	return actionBump(dx, dy);
    }

    // Failed
    return false;
}

bool
MOB::aiAcquireAvatar()
{
    MOB		*a;
    a = MOB::getAvatar();

    return aiAcquireTarget(a);
}

bool
MOB::aiAcquireTarget(MOB *a)
{
    if (a && a->isSwallowed())
    {
	myTX = -1;
	return false;
    }
    // If we can see avatar, charge at them!  Otherwise, move
    // to our target if set.  Otherwise random walk.
    if (a && 
	getMap()->isFOV(getX(), getY()) &&
	getMap()->isFOV(a->getX(), a->getY()))
    {
	int		dist;
	// Determine distance
	dist = MAX(ABS(a->getX() - getX()), ABS(a->getY() - getY()));
	if (dist <= defn().sightrange)
	{
	    myTX = a->getX();
	    myTY = a->getY();
	}
    }
    else
    {
	// Random chance of forgetting avatar location.
	// Also, if current location is avatar location, we forget it.
	if (getX() == myTX && getY() == myTY)
	    myTX = -1;
	if (!rand_choice(10))
	    myTX = -1;
    }

    if (myTX < 0)
	return false;
    return true;
}

bool
MOB::aiRandomWalk(bool orthoonly)
{
    int		dx, dy;
    int		angle, resultangle = 0;
    int		choice = 0;

    for (angle = 0; angle < 8; angle++)
    {
	rand_angletodir(angle, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle;
	}
	if (orthoonly)
	    angle++;
    }

    if (choice)
    {
	// Move in the direction
	rand_angletodir(resultangle, dx, dy);
	return actionWalk(dx, dy);
    }

    // Failed
    return false;
}

bool
MOB::aiCharge(MOB *foe, AI_NAMES aitype, bool orthoonly)
{
    if (aiAcquireTarget(foe))
    {
	if (aitype == AI_FLANK)
	{
	    if (aiFlankTo(myTX, myTY))
		return true;
	}
	else
	{
	    if (aiMoveTo(myTX, myTY, orthoonly))
		return true;
	}
    }

    return false;
}

bool
MOB::aiRangeAttack(MOB *target)
{
    // See if we have a ranged attack at all!
    if (!defn().range_valid && !hasItem(ITEM_SPELLBOOK))
	return false;

    // Check for sufficient mana.
    if (isAvatar())
    {
	ITEM		*r;

	r = lookupItem(ITEM_SPELLBOOK);
	if (r)
	{
	    if (r->getRangeMana() > avatar_mp())
	    {
		return false;
	    }
	}
    }

    int		 dx, dy;
    MOB		*a = MOB::getAvatar(), *v;

    // Override the target from the avatar...
    if (target)
	a = target;

    if (!a)
	return false;

    dx = SIGN(a->getX() - getX());
    dy = SIGN(a->getY() - getY());

    // If we are in melee range, don't use our ranged attack!
    if (MAX(ABS(a->getX()-getX()), ABS(a->getY()-getY())) <= 1)
    {
	return false;
    }

    // Try ranged attack.
    v = getMap()->traceBullet(getX(), getY(), getRangedRange(), dx, dy);
    if (v == a)
    {
	// Potential range attack.
	return actionFire(dx, dy);
    }

    return false;
}

bool
MOB::aiMoveTo(int x, int y, bool orthoonly)
{
    int		dx, dy, dist;
    int		angle, resultangle = 0, i;
    int		choice = 0;

    dx = SIGN(x - getX());
    dy = SIGN(y - getY());

    if (orthoonly && dx && dy)
    {
	// Ensure source is orthogonal.
	if (rand_choice(2))
	    dx = 0;
	else
	    dy = 0;
    }

    dist = MAX(ABS(x - getX()), ABS(y - getY()));

    if (dist == 1)
    {
	// Attack!
	if (canMove(x, y, false))
	    return actionBump(dx, dy);
	return false;
    }

    // Move in general direction, preferring a straight line.
    angle = rand_dirtoangle(dx, dy);

    for (i = 0; i <= 2; i++)
    {
	if (orthoonly && (i & 1))
	    continue;

	rand_angletodir(angle-i, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle-i;
	}
	
	rand_angletodir(angle+i, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle+i;
	}

	if (choice)
	{
	    rand_angletodir(resultangle, dx, dy);

	    // If wew can leap, maybe leap?
	    if (defn().canleap)
	    {
		if (dist > 2 &&
		    canMove(getX()+2*dx, getY()+2*dy))
		{
		    if (getMap()->isFOV(getX(), getY()))
		    {
			msg_format("%S <leap>.", this);
		    }
		    dx *= 2;
		    dy *= 2;
		}
	    }
	    
	    return actionWalk(dx, dy);
	}
    }

    return false;
}

bool
MOB::aiKillPathTo(MOB *target)
{
    if (!target)
	return false;
    
    return aiPathFindTo(target->getX(), target->getY());
}

bool
MOB::aiPathFindTo(int x, int y)
{
    int		dx, dy, ndx, ndy;
    int		curdist, dist;

    bool	found;

    // see if already there.
    if (getX() == x && getY() == y)
	return false;

    // Really inefficient.
    getMap()->buildDistMap(x, y);

    curdist = getMap()->getDistance(getX(), getY());
    
    // We don't care if this square is unreachable, as if it is the
    // square with the mob, almost by definition it is unreachable
    
    found = false;
    for (dy = -1; dy <= 1; dy++)
    {
	for (dx = -1; dx <= 1; dx++)
	{
	    if (dx || dy)
	    {
		dist = getMap()->getDistance(getX()+dx, getY()+dy);
		if (dist >= 0 && dist < curdist)
		{
		    found = true;
		    ndx = dx;
		    ndy = dy;
		    curdist = dist;
		}
	    }
	}
    }

    // If we didn't find a direction, abort
    if (!found)
    {
	msg_format("%S cannot find a path there.", this);
	return false;
    }

    // Otherwise, act.
    return actionWalk(ndx, ndy);
}

bool
MOB::aiFlankTo(int x, int y)
{
    int		dx, dy, dist;
    int		angle, resultangle = 0, i, j;
    int		choice = 0;

    dx = SIGN(x - getX());
    dy = SIGN(y - getY());

    dist = MAX(ABS(x - getX()), ABS(y - getY()));

    if (dist == 1)
    {
	// Attack!
	if (canMove(x, y, false))
	    return actionBump(dx, dy);
	return false;
    }

    // Move in general direction, preferring a straight line.
    angle = rand_dirtoangle(dx, dy);

    for (j = 0; j <= 2; j++)
    {
	// To flank, we prefer the non-straight approach.
	switch (j)
	{
	    case 0:
		i = 1;
		break;
	    case 1:
		i = 0;
		break;
	    case 2:
		i = 2;
		break;
	}
	rand_angletodir(angle-i, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle-i;
	}
	
	rand_angletodir(angle+i, dx, dy);
	if (canMove(getX()+dx, getY()+dy))
	{
	    choice++;
	    if (!rand_choice(choice))
		resultangle = angle+i;
	}

	if (choice)
	{
	    rand_angletodir(resultangle, dx, dy);
	    return actionWalk(dx, dy);
	}
    }

    return false;
}

