/*
 * Sandora, a Falling Sand Game for the Pandora
 * Copyright (C) 2011  WaveHack <email@wavehack.net>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <math.h>
#include <time.h>
#include "common.h"
#include "particle.h"

#define SANDORA_VERSION "0.1-alpha"

SDL_Surface *screen;

// Redirect output from stdout.txt to console
#undef main

int main(int argc, char *argv[]) {
	if (SDL_Init(SDL_INIT_FLAGS) < 0) {
		printf("SDL initialization failed: %s\n", SDL_GetError());
		return 1;
	}
	atexit(SDL_Quit);

	screen = SDL_SetVideoMode(SCREEN_WIDTH, SCREEN_HEIGHT, 16, SDL_VIDEOMODE_FLAGS);
	if (screen == NULL) {
		printf("Unable to set video mode: %s\n", SDL_GetError());
		return 1;
	}
SDL_EnableUNICODE(1);
#ifdef _PANDORA
	SDL_ShowCursor(SDL_DISABLE);
#else
#ifdef _WIN32
	SDL_WM_SetCaption("Sandora " SANDORA_VERSION " for Windows", NULL);
#endif // _WIN32
#endif // _PANDORA

	srand(time(NULL));

	initParticles();
	clearParticleScreen();

	bool done = false;
	SDL_Event event;

	Uint32 currentTick;
	Uint32 lastTick = SDL_GetTicks();
	int fps;
	int frames = 0;

	while (!done) {
		currentTick = SDL_GetTicks();
		if ((currentTick - lastTick) > 1000) {
			fps = ((frames * 1000) / (currentTick - lastTick));
			lastTick = currentTick;
			frames = 0;
			//printf("fps: %d\n", fps);
		}

		while (SDL_PollEvent(&event)) {
			if (event.type == SDL_QUIT)
				done = true;

			if (event.type == SDL_KEYDOWN) {
				switch (event.key.keysym.sym) {
				case SDLK_ESCAPE:
				case SDLK_BACKSPACE:
#ifdef _PANDORA
				case SDLK_LALT: // Start button
#endif // _PANDORA
					done = true;
					break;

				// None/Eraser
				case SDLK_SPACE: currentParticleType = PT_NONE; break;

				// Stillbound
				case SDLK_a: currentParticleType = PT_WALL; break;
				case SDLK_s: currentParticleType = PT_IRONWALL; break;
				case SDLK_d: currentParticleType = PT_STOVE; break;
				case SDLK_f: currentParticleType = PT_ICE; break;
				case SDLK_g: currentParticleType = PT_RUST; break;
				case SDLK_h: currentParticleType = PT_EMBER; break;
				case SDLK_j: currentParticleType = PT_PLANT; break;
				case SDLK_k: currentParticleType = PT_VOID; break;

				// Spouts
				case SDLK_z: currentParticleType = PT_WATERSPOUT; break;
				case SDLK_x: currentParticleType = PT_SANDSPOUT; break;
				case SDLK_c: currentParticleType = PT_SALTSPOUT; break;
				case SDLK_v: currentParticleType = PT_OILSPOUT; break;
				case SDLK_b: currentParticleType = PT_TORCH; break;
				
				// Floating
				case SDLK_o: currentParticleType = PT_STEAM; break;
				case SDLK_p: currentParticleType = PT_FIRE; break;

				// Fluids
				case SDLK_q: currentParticleType = PT_WATER; break;
				case SDLK_w: currentParticleType = PT_SAND; break;
				case SDLK_e: currentParticleType = PT_SALT; break;
				case SDLK_r: currentParticleType = PT_OIL; break;
				case SDLK_t: currentParticleType = PT_DIRT; break;

				// Combined
				case SDLK_y: currentParticleType = PT_ACID; break;
				case SDLK_u: currentParticleType = PT_SALTWATER; break;
				case SDLK_i: currentParticleType = PT_MUD; break;

				// Special
				case SDLK_COMMA: currentParticleType = PT_ELECTRIC; break;

				// Other
				case SDLK_KP_ENTER: clearParticleScreen(); break;

				default:
					break;
				}
			}

			if (event.type == SDL_MOUSEBUTTONDOWN) {
				SDL_MouseButtonEvent mbe = event.button;
#if (PARTICLE_ZOOM == 1)
				addParticles(mbe.x, mbe.y, 8);
#else
				addParticles(floor(mbe.x / PARTICLE_ZOOM), floor(mbe.y / PARTICLE_ZOOM), 8);
#endif
			}

			if (event.type == SDL_MOUSEMOTION) {
				SDL_MouseMotionEvent mme = event.motion;

				if (mme.state & SDL_BUTTON(1))
#if (PARTICLE_ZOOM == 1)
					addParticles(mme.x, mme.y, 8);
#else
					addParticles(floor(mme.x / PARTICLE_ZOOM), floor(mme.y / PARTICLE_ZOOM), 8);
#endif
			}
		}
	
		updateParticles();

		SDL_FillRect(screen, NULL, 0);
		drawParticles();
		SDL_Flip(screen);

		frames++;

		SDL_Delay(1);
	}

	deinitParticles();
	SDL_Quit();

	return 0;
}
