/*
  RulerJinni - A simple virtual ruler
  Copyright (C) 2009  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Configuration.h"

#include <iomanip>
#include <iostream>
#include <sstream>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

#include <Lum/Combo.h>
#include <Lum/Label.h>
#include <Lum/Multi.h>
#include <Lum/String.h>
#include <Lum/Text.h>

Lum::Model::StringRef unit=new Lum::Model::String();
Lum::Model::DoubleRef factor=new Lum::Model::Double();

std::wstring DoubleToWString(double value)
{
  std::ostringstream buffer;

  buffer.imbue(std::locale(""));

  buffer << std::setprecision(5) << value;

  return Lum::Base::StringToWString(buffer.str());
}

bool WStringToDouble(const std::wstring& string, double& value)
{
  std::istringstream buffer(Lum::Base::WStringToString(string));
  value=0.0;

  buffer.imbue(std::locale(""));

  buffer >> value;

  std::cout << "String: '" << Lum::Base::WStringToString(string) << "' Value: " << value << std::endl;

  return buffer;
}


Settings::Settings()
: okAction(new Lum::Model::Action()),
  choice(new Lum::Model::SizeT()),
  choices(new Lum::Model::StringTable()),
  setting(new Lum::Model::SizeT()),
  preset(new Lum::Model::SizeT()),
  presets(new Lum::Model::StringTable()),
  factor(new Lum::Model::String(L"")),
  unit(new Lum::Model::String(L"")),
  success(false)
{
  Observe(okAction);
  Observe(GetClosedAction());
  Observe(choice);
  Observe(preset);
  Observe(factor);
  Observe(unit);

  preset->Set(1);
  choice->Set(1);

  choices->Append(L"Presets");
  choices->Append(L"DPI (cm)");
  choices->Append(L"Manual definition");

  presets->Append(L"Nokia N810 (cm)");
}

Lum::Object* Settings::GetContent()
{
  return Lum::VPanel::Create(true,true)
           ->Add(Lum::Label::Create(true,false)
                 ->AddLabel(L"Choice:",
                            Lum::IndexCombo::Create(choice,choices,true,false)))
            ->AddSpace()
            ->Add(Lum::Multi::Create(setting,true,true)
                  ->Append(Lum::Label::Create(true,false)
                           ->AddLabel(L"Preset:",
                                      Lum::IndexCombo::Create(preset,presets,true,false)))
                  ->Append(Lum::Text::Create(L"Scale will be calculated based on display DPI settings.\n"
                                             L"\n"
                                             L"Depending of the quality of the DPI settings, scale\n"
                                             L"may thus be inprecise!"))
                  ->Append(Lum::Label::Create(true,false)
                           ->AddLabel(L"Display width in unit:",
                                      Lum::String::Create(factor,true,false))
                           ->AddLabel(L"Name of unit:",
                                      Lum::String::Create(unit,true,false))
                  )
          );
}

void Settings::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
}

void Settings::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==okAction && okAction->IsFinished()) {
    double value;

    if (!WStringToDouble(*factor,value)) {
      return;
    }

    success=true;
    ::unit->Set(unit->Get());
    ::factor->Set(Lum::OS::display->GetScreenWidth()/value/10);

    Exit();
  }
  else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==choice && !choice->IsNull()) {
    setting->Set(choice->Get()-1);

    switch (choice->Get()) {
    case 1:
      preset->Notify();
      break;
    case 2:
      factor->Set(DoubleToWString(Lum::OS::display->GetScreenWidth()/(Lum::OS::display->GetDPI()/25.4)/10));
      unit->Set(L"cm");
      break;
    case 3:
      // Nothing to to
      break;
    default:
      assert(false);
    }
  }
  else if (model==preset && !preset->IsNull()) {
    switch (preset->Get()) {
    case 1:
      factor->Set(DoubleToWString(8.95));
      unit->Set(L"cm");
      break;
    default:
      assert(false);
    }
  }
  else if (model==unit || model==factor) {
    double tmp;

    if (!unit->Empty() && !factor->Empty() && WStringToDouble(*factor,tmp) && tmp>0.0) {
      okAction->Enable();
    }
    else {
      okAction->Disable();
    }
  }

  Lum::Dlg::ActionDialog::Resync(model,msg);
}

bool Settings::HasSuccess() const
{
  return success;
}

bool Settings::GetSettings(Lum::OS::Window* parent)
{
  Settings *settings=new Settings();
  bool     success;

  settings->SetParent(parent);
  if (settings->Open()) {
    settings->EventLoop();
    settings->Close();
  }

  success=settings->HasSuccess();

  delete settings;

  return success;
}

bool LoadConfig()
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(Lum::Base::Path::GetApplicationConfigPath());

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"RuleJinni") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  std::wstring factorString;
  std::wstring unit;

  if (top->GetAttribute(L"factor",factorString)) {
    double factor;

    if (WStringToDouble(factorString,factor)) {
      ::factor->Set(factor);
    }
  }
  if (top->GetAttribute(L"unit",unit)) {
    ::unit->Set(unit);
  }

  delete top;

  return true;
}

bool SaveConfig()
{
  Lum::Config::Node                   *top;
  Lum::Base::Path                     path(Lum::Base::Path::GetApplicationConfigPath());
  std::wstring                        config;
  bool                                res;

  top=new Lum::Config::Node();
  top->SetName(L"RuleJinni");
  if (!factor->IsNull()) {
    top->SetAttribute(L"factor",DoubleToWString(factor->Get()));
  }
  if (!unit->IsNull()) {
    top->SetAttribute(L"unit",unit->Get());
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}


