/*cc -shared rs-utils.c rs-registry.c rs-ui.c  -o librs.so $(pkg-config  glib-2.0 gconf-2.0 dbus-1 libosso hildon-1 --cflags --libs) -lcrypt */

/*
    This file belongs to rs (Remote security) package for Nokia N900
    Copyright (C) 2009  Pavan Patale <pavan.patale@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <gconf/gconf-client.h>
#include <crypt.h>
#include <string.h>
#include <unistd.h>

#include "rs-registry.h"


#define GCONF_PASSPHRASE 	"/system/remotesecurity/passphrase"
#define GCONF_REMOTELOCK 	"/system/remotesecurity/remotelock"
#define GCONF_LOCK 		"/system/remotesecurity/lock"
#define GCONF_UNLOCK 		"/system/remotesecurity/unlock"
#define GCONF_REMOTESECDIR 	"/system/remotesecurity"
#define GCONF_LAST_IMSI 	"/system/remotesecurity/last-imsi"
#define GCONF_RSCHOICE          "/system/remotesecurity/choice"
#define KEY_DEVICELOCK_ENABLED 	"/system/osso/dsm/locks/devicelock_autolock_enabled"
#define KEY_DEVICELOCK_TIMEOUT 	"/system/osso/dsm/locks/devicelock_autolock_timeout"
#define LOCK_TIMEOUT 		0
#define UNLOCK_TIMEOUT 		10

#define CONFIRM 		-5

static GConfClient *gclient = NULL;

static gboolean simple_decrypt(const char *const pass, gchar *orig)
{
  char *result;
  int ok;
  
  result = crypt(orig, pass);

  ok = strcmp (result, pass) == 0;

  return ok ? TRUE : FALSE;
}

static gchar *simple_encrypt(const gchar *str) 
{
  gchar *encrypted = NULL;
  
  /* Read in the user's password and encrypt it. */
  if(str)
  {
      encrypted = (gchar *)crypt(str, "hash");
  }
  
  /* Print the results. */
  return encrypted;
}


static gboolean
read_bool_gconf(const gchar *key)
{
    return gconf_client_get_bool(gclient, key, NULL);	
}

static gchar *
read_string_gconf(const gchar *key)
{
    return gconf_client_get_string(gclient, key, NULL);	
}

static gboolean 
write_string_gconf(const gchar *key, const gchar *value)
{
    gboolean ret = FALSE;	
    ret = gconf_client_set_string(gclient, key , value, NULL);
    return ret;
}

static gboolean 
write_bool_gconf(const gchar *key, gboolean value)
{
    gboolean ret = FALSE;	
    ret = gconf_client_set_bool(gclient, key , value, NULL);
    return ret;
}

static gboolean
compare_lock(gchar *str)
{
    gchar *actual = NULL;
    gboolean ret = FALSE;
    actual = read_string_gconf(GCONF_LOCK);
    if(actual){
        ret = simple_decrypt(actual, str);
	g_free(actual);
    }
    return ret;
}

static gboolean
compare_unlock(gchar *str)
{
    gchar *actual = NULL;
    gboolean ret = FALSE;
    actual = read_string_gconf(GCONF_UNLOCK);
    if(actual){
        ret = simple_decrypt(actual, str);
	g_free(actual);
    }
    return ret;
}


gboolean 
rs_registry_set_passphrase(const gchar *passphrase)
{
    return write_string_gconf(GCONF_PASSPHRASE, simple_encrypt(passphrase));
}

gboolean 
rs_registry_set_lockcmd(const gchar *lock)
{
    return write_string_gconf(GCONF_LOCK, simple_encrypt(lock));
}

gboolean 
rs_registry_set_unlockcmd(const gchar *unlock)
{
    return write_string_gconf(GCONF_UNLOCK, simple_encrypt(unlock));
}


gboolean
rs_registry_set_remote_enabled(gboolean val)
{
    return write_bool_gconf(GCONF_REMOTELOCK, val);
}


gboolean
rs_registry_compare_passphrase(gchar *str)
{
    gchar *actual = NULL;
    gboolean ret = FALSE;
    actual = read_string_gconf(GCONF_PASSPHRASE);
    if(actual){
        ret = simple_decrypt(actual, str);
	g_free(actual);
    }
    return ret;
}


gint
rs_registry_compare_cmd(char *str)
{
    if(compare_lock(str))
	return LOCK;
    if(compare_unlock(str))
	return UNLOCK;
    else
	return UNKNOWN;
}

gboolean
rs_registry_remote_enabled()
{
    if(gconf_client_dir_exists(gclient, GCONF_REMOTESECDIR, NULL) && 
		    read_bool_gconf(GCONF_REMOTELOCK))
        return TRUE;
    else
        return FALSE;
}

gchar*
rs_registry_get_last_imsi()
{
    return read_string_gconf(GCONF_LAST_IMSI);
}

void rs_registry_set_current_imsi(const gchar *str)
{
    write_string_gconf(GCONF_LAST_IMSI, str);
}


gboolean rs_registry_init()
{
    if(!gclient) {
        gclient = gconf_client_get_default();
        if(!gclient){
            return FALSE;
	}
    }
    return TRUE;
}

gboolean rs_registry_set_rs_choice(int choice)
{
    gboolean ret = FALSE;
    ret = gconf_client_set_int(gclient, GCONF_RSCHOICE , choice, NULL);
    return ret;
}

gint rs_registry_get_rs_choice()
{
    return gconf_client_get_int(gclient, GCONF_RSCHOICE , NULL);
}

gboolean
rs_registry_lock_device()
{
    gboolean ret = FALSE;
    ret = gconf_client_set_bool(gclient, KEY_DEVICELOCK_ENABLED , TRUE, NULL);
    ret = gconf_client_set_int(gclient, KEY_DEVICELOCK_TIMEOUT , LOCK_TIMEOUT, NULL);
    sleep(3);  //TODO: This is crude.  come up with some decent solution.
    ret = gconf_client_set_int(gclient, KEY_DEVICELOCK_TIMEOUT , UNLOCK_TIMEOUT, NULL);
    if(!ret) {
        g_print("oh! unable to lock\n");	    
    }
    return ret;
}

gboolean
rs_registry_unlock_device()
{
   gboolean ret = FALSE;
   ret = gconf_client_set_int(gclient, KEY_DEVICELOCK_TIMEOUT , UNLOCK_TIMEOUT, NULL);
    if(!ret) {
        g_print("Hehehe screwed up\n");	    
    }
    return ret;
}

void
rs_registry_watch_remote_lock(gchar *func_name)
{

	gconf_client_add_dir (gclient,
			      "/system/remotesecurity",
			      GCONF_CLIENT_PRELOAD_NONE,
			      NULL);
 
	gconf_client_notify_add (gclient,
				 GCONF_REMOTELOCK,
				 (GConfClientNotifyFunc)func_name,
				 NULL,
				 NULL,
				 NULL);
	gconf_client_notify_add (gclient,
				 GCONF_RSCHOICE,
				 (GConfClientNotifyFunc)func_name,
				 NULL,
				 NULL,
				 NULL);
}
