/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "special-ringtones-settings.h"
#include <libringtoned/ringtoned.h>
#include "i18n.h"
#include "ringtone-chooser.h"
#include "settings.h"
#include "settings-dialog.h"

#define NEW_RINGTONE_IS_SET_KEY "new-ringtone-is-set"
#define NEW_RINGTONE_VALUE_KEY "new-ringtone-value"
#define NEW_RINGTONE_NAME_KEY "new-ringtone-name"

typedef struct
{
    PCRSpecialContact special_contact;
    GtkWidget *settings_dialog;
    GtkWidget *button;
} SpecialContactSetRingtoneClosure;

static void
special_contact_dialog_response_cb (GtkDialog *dialog,
                                    gint       response_id,
                                    gpointer   user_data)
{
    SpecialContactSetRingtoneClosure *closure = user_data;

    if (response_id == GTK_RESPONSE_OK)
    {
        gchar *ringtone;
        gchar *display_name;

        g_object_get (dialog,
                "ringtone", &ringtone,
                "ringtone-name", &display_name,
                NULL);

        hildon_button_set_value (HILDON_BUTTON (closure->button), display_name);

        g_object_set_data (G_OBJECT (closure->button),
                NEW_RINGTONE_IS_SET_KEY, GUINT_TO_POINTER (TRUE));
        /* Leave ownership of ringtone and display_name */
        g_object_set_data_full (G_OBJECT (closure->button),
                NEW_RINGTONE_VALUE_KEY, ringtone, g_free);
        g_object_set_data_full (G_OBJECT (closure->button),
                NEW_RINGTONE_NAME_KEY, display_name, g_free);
    }

    gtk_widget_destroy (GTK_WIDGET (dialog));
    gtk_widget_show (closure->settings_dialog);

    g_free (closure);
}

static void
special_contact_clicked_cb (GtkWidget *button,
                            gpointer   user_data)
{
    PCRSpecialContact special_contact = GPOINTER_TO_UINT (user_data);
    SpecialContactSetRingtoneClosure *closure;
    GtkWindow *main_window;
    gchar *ringtone;
    GtkWidget *dialog;

    closure = g_new0 (SpecialContactSetRingtoneClosure, 1);
    closure->settings_dialog = gtk_widget_get_toplevel (button);
    closure->special_contact = special_contact;
    closure->button = button;

    /* If our dialog pops up when the settings dialog is visible then
     * the Fremantle window manager blocks all the input... */
    gtk_widget_hide (closure->settings_dialog);
    main_window = gtk_window_get_transient_for (
            GTK_WINDOW (closure->settings_dialog));

    if (g_object_get_data (G_OBJECT (closure->button),
                NEW_RINGTONE_IS_SET_KEY))
        ringtone = g_strdup  (g_object_get_data (G_OBJECT (closure->button),
                    NEW_RINGTONE_VALUE_KEY));
    else
        pcr_settings_get_special_contact_ringtone (
                special_contact, &ringtone, NULL);

    dialog = pcr_ringtone_chooser_new (main_window, ringtone);
    g_signal_connect (dialog, "response",
            G_CALLBACK (special_contact_dialog_response_cb), closure);
    gtk_widget_show (dialog);

    g_free (ringtone);
}

typedef struct
{
    PCRSpecialContact special_contact;
    GtkWidget *button;
} SettingsDialogResponseClosure;

static void
settings_dialog_response_cb (GtkWidget *dialog,
                             gint       response_id,
                             gpointer   user_data)
{
    SettingsDialogResponseClosure *closure = user_data;

    if (response_id == GTK_RESPONSE_OK)
    {
        if (g_object_get_data (G_OBJECT (closure->button),
                    NEW_RINGTONE_IS_SET_KEY))
        {
            pcr_settings_set_special_contact_ringtone (
                    closure->special_contact,
                    g_object_get_data (G_OBJECT (closure->button),
                        NEW_RINGTONE_VALUE_KEY),
                    g_object_get_data (G_OBJECT (closure->button),
                        NEW_RINGTONE_NAME_KEY));
        }
    }

    g_free (closure);
}

static void
create_chooser_button (OssoABookSettingsDialog *dialog,
                       GtkContainer            *box,
                       PCRSpecialContact        special_contact)
{
    GtkWidget *button;
    const gchar *text;
    gchar *value;
    SettingsDialogResponseClosure *response_closure;

    switch (special_contact)
    {
        case PCR_SPECIAL_CONTACT_UNKNOWN:
            /* Note for translators: there is not a lot of horizontal space
             * for this button, so don't use a much wider strings.
             * Don't use the "unknown" word to translate this as it would
             * not be clear. When you receive a phone call from a hidden
             * number the call UI shows "Unknown number", so in our case
             * if you use "unknown" it would not be clear if you mean a
             * caller not in your address book or a contact with hidden
             * number.
             * Also, possibly don't use any word referring to phone numbers
             * as the caller could be using SIP or any voip protocol.
             * In the worst case try translating "Ringtone for unknown
             * contact" if it's clear that is the contact that is unknown
             * and not the calling phone number / ID.
             * Sorry, I know that it's a pain... */
            text = _("Ringtone for callers not in your contacts");
            break;
        case PCR_SPECIAL_CONTACT_ANONYMOUS:
            text = _("Ringtone for anonymous callers");
            break;
    }

    pcr_settings_get_special_contact_ringtone (special_contact, NULL, &value);
    button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
            HILDON_BUTTON_ARRANGEMENT_VERTICAL, text, value);
    g_signal_connect (button, "clicked",
            G_CALLBACK (special_contact_clicked_cb),
            GUINT_TO_POINTER (special_contact));
    hildon_button_set_style (HILDON_BUTTON (button),
            HILDON_BUTTON_STYLE_PICKER);
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0, 0.5);
    gtk_container_add (box, button);
    gtk_widget_show (button);

    response_closure = g_new0 (SettingsDialogResponseClosure, 1);
    response_closure->special_contact = special_contact;
    response_closure->button = button;
    g_signal_connect (dialog, "response",
            G_CALLBACK (settings_dialog_response_cb), response_closure);

    g_free (value);
}

void
special_ringtones_settings_widget_hook (OssoABookSettingsDialog *dialog,
                                        GtkContainer            *box)
{
    create_chooser_button (dialog, box, PCR_SPECIAL_CONTACT_UNKNOWN);
    create_chooser_button (dialog, box, PCR_SPECIAL_CONTACT_ANONYMOUS);
}
