/*
 * connection-watcher.c - Source for ConnectionWatcher
 * Copyright (C) 2010 Guillaume Desmottes
 * Copyright (C) 2010 Collabora
 * @author Guillaume Desmottes <gdesmott@gnome.org>
 * @author Alban Crequy <alban.crequy@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* This file was copied from Map Buddy[1] (and was originally copied from
 * Azimuth[2]) and modified to suit the needs of ringtoned.
 * Please keep the original indentation and clearly mark local changes
 * to make updates from upstream easier.
 *
 * [1] http://www.gitorious.org/map-buddy
 * [2] http://vcs.maemo.org/git/azimuth/?p=azimuth
 */


#include <stdio.h>
#include <stdlib.h>

#include <telepathy-glib/account-manager.h>
#include <telepathy-glib/account.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/gtypes.h>
#include <telepathy-glib/util.h>

/* NOTE: Changed from upstream */
/*#include "mapbuddy-marshallers.h"*/
#include "ringtoned-marshals.h"
#define mapbuddy_marshal_VOID__OBJECT_OBJECT \
  ringtoned_marshal_VOID__OBJECT_OBJECT

#undef g_debug
#define g_debug(...)
/* End changes */

#include "connection-watcher.h"

G_DEFINE_TYPE(ConnectionWatcher, connection_watcher, G_TYPE_OBJECT)

/* signal enum */
enum
{
    CONNECTION_ADDED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* private structure */
typedef struct _ConnectionWatcherPrivate ConnectionWatcherPrivate;

struct _ConnectionWatcherPrivate
{
  TpAccountManager *account_mgr;
  TpDBusDaemon *bus_daemon;
  GList *accounts;

  gboolean dispose_has_run;
};

typedef struct
{
  TpAccount *account;
  ConnectionWatcher *connection_watcher;
} ConnReadyData;

#define CONNECTION_WATCHER_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE ((o), CONNECTION_WATCHER_TYPE, ConnectionWatcherPrivate))

static void
connection_watcher_init (ConnectionWatcher *obj)
{
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (obj);

  priv->bus_daemon = tp_dbus_daemon_dup (NULL);
  g_assert (priv->bus_daemon != NULL);

  priv->account_mgr = tp_account_manager_new (priv->bus_daemon);
  priv->accounts = NULL;
}

static void connection_watcher_dispose (GObject *object);
static void connection_watcher_finalize (GObject *object);

static void account_invalidated_cb (TpProxy *proxy,
    guint domain,
    gint code,
    gchar *message,
    gpointer user_data);

static void
connection_watcher_class_init (ConnectionWatcherClass *connection_watcher_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (connection_watcher_class);

  g_type_class_add_private (connection_watcher_class, sizeof (ConnectionWatcherPrivate));

  object_class->dispose = connection_watcher_dispose;
  object_class->finalize = connection_watcher_finalize;

  signals[CONNECTION_ADDED] = g_signal_new ("connection-added",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    mapbuddy_marshal_VOID__OBJECT_OBJECT,
    G_TYPE_NONE, 2, TP_TYPE_ACCOUNT, TP_TYPE_CONNECTION);
}

void
connection_watcher_dispose (GObject *object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (object);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);
  GList *l;

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;

  g_object_unref (priv->account_mgr);

  for (l = priv->accounts; l != NULL; l = g_list_next (l))
    {
      g_signal_handlers_disconnect_by_func (l->data, account_invalidated_cb,
          self);
      g_object_unref (l->data);
    }

  g_object_unref (priv->bus_daemon);

  if (G_OBJECT_CLASS (connection_watcher_parent_class)->dispose)
    G_OBJECT_CLASS (connection_watcher_parent_class)->dispose (object);
}

void
connection_watcher_finalize (GObject *object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (object);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);

  g_list_free (priv->accounts);

  G_OBJECT_CLASS (connection_watcher_parent_class)->finalize (object);
}

ConnectionWatcher *
connection_watcher_new (void)
{
  return g_object_new (CONNECTION_WATCHER_TYPE,
      NULL);
}

static void
conn_ready_cb (TpConnection *conn,
    const GError *error,
    gpointer user_data)
{
  ConnReadyData *conn_ready_data = (ConnReadyData *) user_data;
  ConnectionWatcher *self =
    CONNECTION_WATCHER (conn_ready_data->connection_watcher);
  TpAccount *account = conn_ready_data->account;

  g_slice_free (ConnReadyData, conn_ready_data);

  if (error != NULL)
    {
      g_printerr ("connection is not ready: %s\n", error->message);
      goto out;
    }

  g_signal_emit (self, signals[CONNECTION_ADDED], 0, account, conn);

out:
  g_object_unref (conn);
}

static void
account_invalidated_cb (TpProxy *proxy,
    guint domain,
    gint code,
    gchar *message,
    gpointer user_data)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (user_data);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);

  g_debug ("remove invalidated account: %s\n",
      tp_proxy_get_object_path (proxy));

  priv->accounts = g_list_remove (priv->accounts, proxy);
  g_object_unref (proxy);
}

static void
create_connection (ConnectionWatcher *self, TpAccount *account,
    const gchar *path)
{
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);
  TpConnection *conn;
  GError *err = NULL;
  ConnReadyData *conn_ready_data;

  if (path == NULL || !tp_strdiff (path, "/"))
    return;

  conn = tp_connection_new (priv->bus_daemon, NULL, path, &err);
  if (conn == NULL)
    {
      g_printerr ("Failed to create TpConnection: %s\n", err->message);
      g_error_free (err);
      return;
    }

  conn_ready_data = g_slice_new0 (ConnReadyData);
  conn_ready_data->connection_watcher = self;
  conn_ready_data->account = account;
  tp_connection_call_when_ready (conn, conn_ready_cb, conn_ready_data);
}

static void
get_connection_cb (TpProxy *account,
    const GValue *out_Value,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (weak_object);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);

  if (error != NULL)
    {
      g_printerr ("Failed to fetch Connection property: %s\n", error->message);
      priv->accounts = g_list_remove (priv->accounts, account);
      g_object_unref (account);
      return;
    }

  create_connection (self, TP_ACCOUNT (account), g_value_get_boxed (out_Value));
}

static void
account_property_changed_cb (TpAccount *account,
    GHashTable *properties,
    gpointer user_data,
    GObject *weak_object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (weak_object);

  create_connection (self, account,
      tp_asv_get_object_path (properties, "Connection"));
}

static void
add_account (ConnectionWatcher *self,
    TpAccount *account)
{
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);

  priv->accounts = g_list_prepend (priv->accounts, account);

  tp_cli_dbus_properties_call_get (account, -1,
      TP_IFACE_ACCOUNT, "Connection", get_connection_cb,
      self, NULL, G_OBJECT (self));

  tp_cli_account_connect_to_account_property_changed (account,
      account_property_changed_cb, self, NULL, G_OBJECT (self), NULL);

  g_signal_connect (account, "invalidated",
      G_CALLBACK (account_invalidated_cb), self);
}

static void
account_validity_changed_cb (TpAccountManager *account_mgr,
    const gchar *account_path,
    gboolean valid,
    gpointer user_data,
    GObject *weak_object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (weak_object);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);
  GError *err = NULL;
  TpAccount *account;

  account = tp_account_new (priv->bus_daemon, account_path, &err);
  if (account == NULL)
    {
      g_printerr ("Failed to create TpAccount: %s\n", err->message);
      g_error_free (err);
      return;
    }

  if (g_list_find (priv->accounts, account) != NULL)
    return;

  add_account (self, account);
}

static void
get_valid_accounts_cb (TpProxy *proxy,
    const GValue *out_Value,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  ConnectionWatcher *self = CONNECTION_WATCHER (weak_object);
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);
  GPtrArray *valid_accounts;
  guint i;

  if (error != NULL)
    {
      g_printerr ("Failed to fetch ValidAccounts property: %s\n", error->message);
      return;
    }

  valid_accounts = g_value_get_boxed (out_Value);
  if (valid_accounts == NULL)
    return;

  for (i = 0; i < valid_accounts->len; i++)
    {
      const gchar *name;
      TpAccount *account;
      GError *err = NULL;

      name = g_ptr_array_index (valid_accounts, i);
      account = tp_account_new (priv->bus_daemon, name, &err);
      if (account == NULL)
        {
          g_printerr ("Failed to create TpAccount: %s\n", err->message);
          g_error_free (err);
          continue;
        }

      add_account (self, account);
    }
}

void
connection_watcher_start (ConnectionWatcher *self)
{
  ConnectionWatcherPrivate *priv = CONNECTION_WATCHER_GET_PRIVATE (self);

  tp_cli_dbus_properties_call_get (priv->account_mgr, -1,
      TP_IFACE_ACCOUNT_MANAGER, "ValidAccounts", get_valid_accounts_cb,
      self, NULL, G_OBJECT (self));

  tp_cli_account_manager_connect_to_account_validity_changed (priv->account_mgr,
      account_validity_changed_cb, NULL, NULL, G_OBJECT (self), NULL);
}
