/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "settings.h"
#include <libringtoned/ringtoned.h>
#include "i18n.h"


#define SPECIAL_RINGTONES_SECTION "special-ringtones"

static gchar *
get_config_file_path (void)
{
    return g_build_filename (g_get_user_config_dir (), "ringtoned",
            "ringtoned.conf", NULL);
}

static GKeyFile *
load_config (void)
{
    GKeyFile *kf;
    gchar *config_file;

    kf = g_key_file_new ();

    config_file = get_config_file_path ();
    /* It doesn't matter if this fails and the key file remains empty */
    g_key_file_load_from_file (kf, config_file, G_KEY_FILE_NONE, NULL);
    g_free (config_file);

    return kf;
}

static const gchar *
get_key_for_special_contact (PCRSpecialContact special_contact)
{
    const gchar *key = NULL;

    switch (special_contact)
    {
        case PCR_SPECIAL_CONTACT_UNKNOWN:
            key = "unknown-ringtone";
            break;
        case PCR_SPECIAL_CONTACT_ANONYMOUS:
            key = "anonymous-ringtone";
            break;
    }

    return key;
}

static const gchar *
get_key_for_special_contact_display (PCRSpecialContact special_contact)
{
    const gchar *key = NULL;

    switch (special_contact)
    {
        case PCR_SPECIAL_CONTACT_UNKNOWN:
            key = "unknown-display-name";
            break;
        case PCR_SPECIAL_CONTACT_ANONYMOUS:
            key = "anonymous-display-name";
            break;
    }

    return key;
}

void
pcr_settings_get_special_contact_ringtone (PCRSpecialContact   special_contact,
                                           gchar             **ringtone,
                                           gchar             **display_name)
{
    GKeyFile *kf;
    gchar *tmp;

    kf = load_config ();

    tmp = g_key_file_get_string (kf, SPECIAL_RINGTONES_SECTION,
            get_key_for_special_contact (special_contact), NULL);

    if (display_name)
    {
        if (pcr_ringtone_is_default (tmp))
            *display_name = g_strdup (_(PCR_RINGTONE_DEFAULT_DISPLAY_NAME));
        else if (pcr_ringtone_is_silent (tmp))
            *display_name = g_strdup (_(PCR_RINGTONE_SILENT_DISPLAY_NAME));
        else
        {
            *display_name = g_key_file_get_string (kf,
                    SPECIAL_RINGTONES_SECTION,
                    get_key_for_special_contact_display (special_contact),
                    NULL);
            if (*display_name == NULL || *display_name[0] == '\0')
            {
                g_free (*display_name);
                *display_name = g_path_get_basename (tmp);
            }
        }
    }

    if (ringtone)
        *ringtone = tmp;
    else
        g_free (tmp);

    g_key_file_free (kf);
}

void
pcr_settings_set_special_contact_ringtone (PCRSpecialContact  special_contact,
                                           const gchar       *new_ringtone,
                                           const gchar       *display_name)
{
    GKeyFile *kf;
    gchar *config_file;
    gchar *config_dir;
    gchar *data;
    gsize data_length;
    GError *error = NULL;

    DEBUG ("Setting ringtone for %s calls to %s",
        get_key_for_special_contact (special_contact), new_ringtone);

    config_file = get_config_file_path ();
    config_dir = g_path_get_dirname (config_file);
    g_mkdir_with_parents (config_dir, 0755);

    kf = load_config ();

    if (new_ringtone != NULL)
    {
        g_key_file_set_string (kf, SPECIAL_RINGTONES_SECTION,
                get_key_for_special_contact (special_contact),
                new_ringtone);
        g_key_file_set_string (kf, SPECIAL_RINGTONES_SECTION,
                get_key_for_special_contact_display (special_contact),
                display_name);
    }
    else
    {
        g_key_file_remove_key (kf, SPECIAL_RINGTONES_SECTION,
                get_key_for_special_contact (special_contact), NULL);
        g_key_file_remove_key (kf, SPECIAL_RINGTONES_SECTION,
                get_key_for_special_contact_display (special_contact), NULL);
    }

    data = g_key_file_to_data (kf, &data_length, NULL);
    if (!g_file_set_contents (config_file, data, data_length, &error))
    {
        g_critical ("Cannot save configuration: %s",
                error ? error->message : "unknown error");
        g_clear_error (&error);
    }

    g_key_file_free (kf);
    g_free (data);
    g_free (config_dir);
    g_free (config_file);
}
