/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "fake-tp-observer.h"
#include <libringtoned/ringtoned.h>
#include <telepathy-glib/account.h>
#include <telepathy-glib/channel.h>
#include <telepathy-glib/interfaces.h>
#include "connection-watcher.h"
#include "ringtoned-marshals.h"

#ifndef TP_PROP_CHANNEL_CHANNEL_TYPE
#define TP_PROP_CHANNEL_CHANNEL_TYPE \
    "org.freedesktop.Telepathy.Channel.ChannelType"
#endif

#define RINGTONED_FAKE_TP_OBSERVER_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RINGTONED_TYPE_FAKE_TP_OBSERVER, \
                                  RingtonedFakeTpObserverPrivate))

enum
{
    PROP_0,
};

enum
{
    NEW_CHANNEL_SIGNAL,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL];

struct _RingtonedFakeTpObserverPrivate
{
    ConnectionWatcher *conn_watcher;
    GHashTable *connections;
};

G_DEFINE_TYPE (RingtonedFakeTpObserver,
        ringtoned_fake_tp_observer, G_TYPE_OBJECT)

RingtonedFakeTpObserver *
ringtoned_fake_tp_observer_new (void)
{
    return g_object_new (RINGTONED_TYPE_FAKE_TP_OBSERVER, NULL);
}

static void
new_channels_cb (TpConnection    *connection,
                 const GPtrArray *channels,
                 gpointer         user_data,
                 GObject         *weak_object)
{
    RingtonedFakeTpObserver *self = RINGTONED_FAKE_TP_OBSERVER (weak_object);
    TpAccount *account = user_data;
    guint i;
    GValueArray *values;
    GValue *value;
    const gchar *channel_path;
    GHashTable *properties;
    TpChannel *channel;

    for (i = 0; i < channels->len; i++)
    {
        values = g_ptr_array_index (channels, i);

        value = g_value_array_get_nth (values, 0);
        channel_path = g_value_get_boxed (value);

        value = g_value_array_get_nth (values, 1);
        properties = g_value_get_boxed (value);

        value = g_hash_table_lookup (properties, TP_PROP_CHANNEL_CHANNEL_TYPE);
        if (!value ||
            g_strcmp0 (g_value_get_string (value),
                TP_IFACE_CHANNEL_TYPE_STREAMED_MEDIA) != 0)
        {
            continue;
        }

        channel = tp_channel_new_from_properties (connection,
                channel_path, properties, NULL);
        if (channel)
        {
            g_signal_emit (self, signals [NEW_CHANNEL_SIGNAL], 0,
                    tp_proxy_get_object_path (TP_PROXY (account)),
                    channel);
            g_object_unref (channel);
        }

    }
}

static void
connection_invalidated_cb (TpProxy  *connection,
                           guint     domain,
                           gint      code,
                           gchar    *message,
                           gpointer  user_data)
{
    RingtonedFakeTpObserver *self = user_data;

    g_hash_table_remove (self->priv->connections, connection);
}

static void
connection_added_cb (ConnectionWatcher *watcher,
                     TpAccount         *account,
                     TpConnection      *connection,
                     gpointer           user_data)
{
    RingtonedFakeTpObserver *self = user_data;

    g_signal_connect (connection, "invalidated",
            G_CALLBACK (connection_invalidated_cb), self);
    g_hash_table_insert (self->priv->connections,
            g_object_ref (connection), GINT_TO_POINTER (TRUE));

    tp_cli_connection_interface_requests_connect_to_new_channels (
            connection, new_channels_cb, g_object_ref (account),
            g_object_unref, G_OBJECT (self), NULL);
}

static void
ringtoned_fake_tp_observer_get_property (GObject    *object,
                                         guint       param_id,
                                         GValue     *value,
                                         GParamSpec *pspec)
{
    switch (param_id)
    {
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_fake_tp_observer_set_property (GObject      *object,
                                         guint         param_id,
                                         const GValue *value,
                                         GParamSpec   *pspec)
{
    switch (param_id)
    {
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_fake_tp_observer_init (RingtonedFakeTpObserver *self)
{
    self->priv = RINGTONED_FAKE_TP_OBSERVER_GET_PRIVATE (self);

    self->priv->connections = g_hash_table_new_full (g_direct_hash,
            g_direct_equal, g_object_unref, NULL);

    self->priv->conn_watcher = connection_watcher_new ();
    g_signal_connect (self->priv->conn_watcher, "connection-added",
            G_CALLBACK (connection_added_cb), self);
}

static void
ringtoned_fake_tp_observer_constructed (GObject *object)
{
    RingtonedFakeTpObserver *self = RINGTONED_FAKE_TP_OBSERVER (object);

    connection_watcher_start (self->priv->conn_watcher);

    if (G_OBJECT_CLASS (ringtoned_fake_tp_observer_parent_class)->constructed)
        G_OBJECT_CLASS (ringtoned_fake_tp_observer_parent_class)->constructed (
               object);
}

static void
ringtoned_fake_tp_observer_finalize (GObject *object)
{
    RingtonedFakeTpObserver *self = RINGTONED_FAKE_TP_OBSERVER (object);

    g_hash_table_destroy (self->priv->connections);

    if (self->priv->conn_watcher)
        g_object_unref (self->priv->conn_watcher);

    G_OBJECT_CLASS (ringtoned_fake_tp_observer_parent_class)->finalize (object);
}

static void
ringtoned_fake_tp_observer_class_init (RingtonedFakeTpObserverClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    object_class->constructed = ringtoned_fake_tp_observer_constructed;
    object_class->finalize = ringtoned_fake_tp_observer_finalize;
    object_class->get_property = ringtoned_fake_tp_observer_get_property;
    object_class->set_property = ringtoned_fake_tp_observer_set_property;

    g_type_class_add_private (object_class,
            sizeof (RingtonedFakeTpObserverPrivate));

    signals[NEW_CHANNEL_SIGNAL] = g_signal_new (
            "new-channel", RINGTONED_TYPE_FAKE_TP_OBSERVER,
            G_SIGNAL_RUN_LAST, 0, NULL, NULL,
            ringtoned_marshal_VOID__STRING_OBJECT,
            G_TYPE_NONE, 2, G_TYPE_STRING, TP_TYPE_CHANNEL);
}
