/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "decoder.h"
#include <glib/gstdio.h>
#include <dbus/dbus-glib.h>
#include <sys/types.h>
#include <unistd.h>
#include "libringtoned-marshals.h"
#include "libringtoned/ringtoned.h"
#include "libringtoned/utils.h"

#define DECODING_TIMEOUT (60 * 1000)

static void decode_next (void);

static GList *pending_operations = NULL;
gchar *current_file = NULL;
DBusGProxy *nsv_decoder = NULL;
static gint timeout_id = 0;
static gchar *decoder_category = NULL;

static gchar *
get_decoded_name_for_file (const gchar *fn)
{
    GChecksum *checksum;
    const gchar *tmp;
    gchar *wav;
    gchar *ret;

    g_return_val_if_fail (!IS_EMPTY (fn), g_strdup (""));

    checksum = g_checksum_new (G_CHECKSUM_MD5);
    g_checksum_update (checksum, (guchar *) fn, -1); /* guchar*? srsly? */
    tmp = g_checksum_get_string (checksum);

    wav = g_strconcat (tmp, ".wav", NULL);
    ret = g_build_filename (g_get_user_data_dir (), "sounds",
            "ringtoned", wav, NULL);

    g_free (wav);
    g_checksum_free (checksum);

    return ret;
}

gchar *
ringtoned_decoder_get_playable_path (const gchar *path)
{
    gchar *cache_path;
    gchar *ret = NULL;

    cache_path = g_build_filename (g_get_user_data_dir (), "sounds", NULL);

#define RETURN_IF_EXISTS(p) \
        if (g_str_has_prefix (p, cache_path)) \
            ret = g_strdup (p); \
        else \
            ret = get_decoded_name_for_file (p); \
        \
        if (!IS_EMPTY (ret) && g_file_test (ret, G_FILE_TEST_EXISTS)) \
            goto done; \
        else \
        { \
            g_free (ret); \
            ret = NULL; \
        }

    /* The file that the user wanted */
    RETURN_IF_EXISTS (path);

    /* The requested file doesn't have an uncompressed version, let's
     * recreate it */
    ringtoned_decoder_schedule (path);

    /* Fall back to the default ringtone */
    RETURN_IF_EXISTS (ringtoned_profile_get_default_ringtone ());

    /* Fall back to the default one, but in the version uncompressed by
     * the default ringtone program */
    RETURN_IF_EXISTS (ringtoned_profile_get_cached_default_ringtone ());

    /* Maybe we happen to have the original default ringtone
     * uncompressed? */
    RETURN_IF_EXISTS ("/usr/share/sounds/NokiaTune.aac");

#undef RETURN_IF_EXISTS

    /* Seriously, WTF? This should really not be reached, in any case
     * let's return something that is not NULL. */
    ret = g_strdup ("/dev/null");

done:
    g_free (cache_path);

    return ret;
}

void
ringtoned_decoder_schedule (const gchar *file_to_convert)
{
    g_return_if_fail (!IS_EMPTY (file_to_convert));

    DEBUG ("Adding decoding operation for %s", file_to_convert);

    pending_operations = g_list_prepend (pending_operations,
            g_strdup (file_to_convert));

    if (!decoder_category)
        decoder_category = g_strdup_printf ("ringtoned-%d", getpid ());
}

void
ringtoned_decoder_schedule_if_needed (const gchar *file_to_convert)
{
    gchar *decoded;

    g_return_if_fail (!IS_EMPTY (file_to_convert));

    decoded = get_decoded_name_for_file (file_to_convert);
    if (!g_file_test (decoded, G_FILE_TEST_EXISTS))
        ringtoned_decoder_schedule (file_to_convert);
    else
        DEBUG ("Decoded file for %s already exists", file_to_convert);
    g_free (decoded);
}

static void
file_processed (void)
{
    g_source_remove (timeout_id);
    timeout_id = 0;

    g_free (current_file);
    current_file = NULL;

    decode_next ();
}

static void
decoded_cb (DBusGProxy  *proxy,
            const gchar *category,
            const gchar *file,
            gpointer     user_data)
{
    gchar *decoded;
    gchar *argv[4];;

    if (g_strcmp0 (category, decoder_category) != 0 ||
        g_strcmp0 (file, current_file) != 0)
    {
        /* Not our file */
        return;
    }

    DEBUG ("Decoded %s", current_file);

    decoded = get_decoded_name_for_file (current_file);
    argv[0] = "ringtoned-fixwav";
    if (ringtoned_debug_is_enabled ())
    {
        argv[1] = "-d";
        argv[2] = decoded;
        argv[3] = NULL;
    }
    else
    {
        argv[1] = decoded;
        argv[2] = NULL;
    }

    g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL,
            NULL, NULL);

    file_processed ();

    g_free (decoded);
}

static void
error_decoding_cb (DBusGProxy  *proxy,
                   const gchar *category,
                   gpointer     user_data)
{
    gchar *out;

    if (g_strcmp0 (category, decoder_category) != 0)
    {
        /* Not our file */
        return;
    }

    DEBUG ("Failed to decode %s", current_file);

    out = get_decoded_name_for_file (current_file);
    g_unlink (out);
    g_free (out);

    file_processed ();
}

static void
initialize_service (void)
{
    DBusGConnection *connection;

    connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
    if (!connection)
        return;

    nsv_decoder = dbus_g_proxy_new_for_name (connection,
            "com.nokia.NsvDecoder",
            "/com/nokia/NsvDecoder",
            "com.nokia.NsvDecoder");

    dbus_g_object_register_marshaller (
            libringtoned_marshal_VOID__STRING_STRING, G_TYPE_NONE,
            G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal (nsv_decoder, "Decoded", G_TYPE_STRING,
            G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (nsv_decoder, "Decoded",
            G_CALLBACK (decoded_cb), NULL, NULL);

    dbus_g_proxy_add_signal (nsv_decoder, "ErrorDecoding", G_TYPE_STRING,
            G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (nsv_decoder, "ErrorDecoding",
            G_CALLBACK (error_decoding_cb), NULL, NULL);
}

static void
decoding_finished (void)
{
    g_return_if_fail (pending_operations == NULL);

    if (nsv_decoder)
    {
        g_object_unref (nsv_decoder);
        nsv_decoder = NULL;
    }
}

static gboolean
timeout_cb (gpointer user_data)
{
    gchar *out;

    DEBUG ("Decoding timeout for %s expired", current_file);

    timeout_id = 0;

    out = get_decoded_name_for_file (current_file);
    g_unlink (out);
    g_free (out);

    g_free (current_file);
    current_file = NULL;

    decode_next ();

    return FALSE;
}

static void
decode_next (void)
{
    gchar *out;
    gchar *out_dir;

    g_return_if_fail (nsv_decoder);
    g_return_if_fail (timeout_id == 0);

    if (!pending_operations)
    {
        DEBUG ("No more files to decode");
        decoding_finished ();
        return;
    }

    current_file = pending_operations->data; /* Transfer ownership */
    pending_operations = g_list_delete_link (pending_operations,
            pending_operations);

    out = get_decoded_name_for_file (current_file);

    DEBUG ("Decoding:\n"
            "    Input: %s\n"
            "    Output: %s",
            current_file, out);

    out_dir = g_path_get_dirname (out);
    g_mkdir_with_parents (out_dir, 0755);

    dbus_g_proxy_call_no_reply (nsv_decoder, "Decode",
            G_TYPE_STRING, decoder_category,
            G_TYPE_STRING, current_file,
            G_TYPE_STRING, out,
            G_TYPE_INVALID);

    /* We don't want to block the decoding forever if for some reason
     * something goes wrong, so we add a timeout to cancel the current
     * decoding */
    timeout_id = g_timeout_add (DECODING_TIMEOUT, timeout_cb, NULL);

    g_free (out_dir);
    g_free (out);
}

void
ringtoned_decoder_decode_pending (void)
{
    DEBUG ("Decoding requested");

    if (!pending_operations)
    {
        DEBUG ("    Nothing to decode");
        return;
    }

    if (timeout_id)
    {
        DEBUG ("    Decoding already in progress");
        return;
    }

    initialize_service ();
    decode_next ();
}
