/*************************************************************************}
{ qwidgethelper.cpp - some convenience widget functions                   }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This library is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This library is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 04 Sep 2012                                              }
{*************************************************************************/

#include "qwidgethelper.h"

void QWidgetHelper::setupDefaults(QWidget *widget)
{
    QWidget* parent = qobject_cast<QWidget*>(widget->parent());
    QFont f;
    if(parent)
    {
        f = parent->font();
        widget->setLocale(parent->locale());
    }
    else
    {
        f = widget->font();
        f.setPointSize(8);
        f.setFamily("Arial");
    }
    widget->setFont(f);
    widget->setContextMenuPolicy(Qt::NoContextMenu);
    widget->setMouseTracking(true);
}

void QWidgetHelper::fullyActivate(QWidget *widget)
{
    widget->show();
    widget->raise();
    widget->activateWindow();
    widget->setFocus();
}

void QWidgetHelper::setScreenOrientation(QWidget *widget, ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;

#if QT_VERSION < 0x040702
        const Qt::WidgetAttribute attrLockPortrtait = static_cast<Qt::WidgetAttribute>(128);
        const Qt::WidgetAttribute attrLockLandscape = static_cast<Qt::WidgetAttribute>(129);
        const Qt::WidgetAttribute attrAuto = static_cast<Qt::WidgetAttribute>(130);
#else // QT_VERSION < 0x040702
        const Qt::WidgetAttribute attrLockPortrtait = Qt::WA_LockPortraitOrientation;
        const Qt::WidgetAttribute attrLockLandscape = Qt::WA_LockLandscapeOrientation;
        const Qt::WidgetAttribute attrAuto = Qt::WA_AutoOrientation;
#endif // QT_VERSION < 0x040702

    switch (orientation)
    {
        case ScreenOrientationLockPortrait:
            attribute = attrLockPortrtait;
            break;

        case ScreenOrientationLockLandscape:
            attribute = attrLockLandscape;
            break;

        default:
            attribute = attrAuto;
    }

    widget->setAttribute(attrLockPortrtait, false);
    widget->setAttribute(attrLockLandscape, false);
    widget->setAttribute(attrAuto, false);
    widget->setAttribute(attribute, true);
}

void QWidgetHelper::showExpanded(QWidget *widget)
{
#ifdef Q_OS_SYMBIAN
    widget->showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    widget->showMaximized();
#else
    widget->show();
#endif
}

bool QWidgetHelper::simpleResize(QResizeEvent *event, int &width, int &height)
{
    height=event->size().height();
    width=event->size().width();
    bool isPortrait = (height>width);
    return isPortrait;
}

bool QWidgetHelper::maximizedAutoResize(QWidget* widget, QResizeEvent* event)
{
    int width, height;
    bool isPortrait = simpleResize(event, width, height);
    if((height <= 0) || (widget->windowState() == Qt::WindowFullScreen))
        return false;

    char* propName = (char*)(QString("isNowPortrait").constData());
    QVariant v = widget->property(propName);
    if(v.isValid())
    {
        bool isNowPortrait = v.toBool();
        if(isNowPortrait == isPortrait)
            return false;
    }
    v.setValue(isPortrait);
    widget->setProperty(propName, v);

    widget->move(0, 0);
    widget->resize(width, height);
    widget->setWindowState(Qt::WindowFullScreen);
    widget->setWindowState(Qt::WindowMaximized);

    if(qobject_cast<QDialog*>(widget) && widget->isVisible())
        fullyActivate(widget);

    return true;
}
