/*************************************************************************}
{ searchwindow.cpp - search window                                        }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 21 May 2012                                              }
{*************************************************************************/

#include "searchwindow.h"
#include "mainwindow.h"

SearchWindow::SearchWindow(QWidget *parent) :
    ContainerDialog(parent)
{
    initGui();
    fs = MAINWINDOW->getFSMan();
    connect(fs, SIGNAL(onSearchFinished()), SLOT(onSearchFinished()));
    connect(fs, SIGNAL(onSearchResult(bool,QString,QString)), SLOT(onSearchResult(bool,QString,QString)));
    if(fs->isUsingCache() && false)
    {
        isRealSearch = false;
        fs->startSearch("*"); // invoke fake search to precache files list
    }
    else
    {
        isRealSearch = true;
    }
    lastCursorPos = 0;
}

SearchWindow::~SearchWindow()
{
    fs->stopSearch();
    deinitGui();
}

void SearchWindow::showSearchWindow(const QString& txt)
{
    SearchWindow* win = new SearchWindow(MAINWINDOW);
    win->editExp->setText(txt.trimmed());
    win->setFocusToEdit();
    win->showDialog();
    delete win;
}

void SearchWindow::initGui()
{
    ContainerDialog::initGui();
    setWindowTitle(tr("Search for files and folders"));
    btnGo = new QPushButton(this);
    btnGo->setAutoDefault(false);
    btnGo->setDefault(false);
    editExp = new QLineEdit(this);
    editExp->move(0, 0);
    btnGo->resize(200, CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING);
    connect(btnGo, SIGNAL(clicked()), SLOT(onBtnGoClick()));
    connect(editExp, SIGNAL(returnPressed()), SLOT(onReturnPressed()));    
    lblNotFound = new QLabel(this);
    lblNotFound->setText(QString("(")+tr("nothing's found")+QString(")"));
    lblNotFound->setAlignment(Qt::AlignCenter);

    onSearchFinished();
}

void SearchWindow::deinitGui()
{
    delete btnGo;
    delete editExp;
    ContainerDialog::deinitGui();
}

void SearchWindow::reposition(const QSize &sz, int rows)
{
    Q_UNUSED(rows);

    editExp->resize(sz.width() - btnGo->width(), btnGo->height());
    btnGo->move(editExp->width(), 0);
    itemsContainer->move(0, btnGo->height());
    itemsContainer->resize(sz.width(), sz.height() - btnGo->height());
    int h = lblNotFound->fontMetrics().height();
    lblNotFound->move(0, (itemsContainer->height() - h)/2 + btnGo->height());
    lblNotFound->resize(sz.width(), h);
    itemsContainer->repositionChildrenV(ITEM_HEIGHT, rows);
}

void SearchWindow::setFocusToEdit(int cursorPos)
{
    editExp->setFocus();
    if(cursorPos < 0)
        cursorPos = editExp->text().size();
    editExp->setCursorPosition(cursorPos);
    QTimer::singleShot(0, this, SLOT(onDeselect()));
}

void SearchWindow::onSearchResult(bool isDir, const QString &basename, const QString &parentDir)
{
    if(!isRealSearch)
        return;

    QString path = parentDir;
    if(!path.endsWith("/"))
        path.append("/");
    KineticListItem* item;
    QString itemPath;
    int a = itemsContainer->children().size();
    int w = itemsContainer->width();

    item = new KineticListItem(itemsContainer);
    itemPath = path + basename;
    item->setString(itemPath);
    item->resize(w, ITEM_HEIGHT);
    item->move(0, a*ITEM_HEIGHT);
    item->setText(basename);
    if(isDir)
        item->setIcon(SKIN.icoFolder);
    else
        item->setIcon(SKIN.icoFile);
    item->setBool(isDir);
    item->show();

    itemsContainer->repositionChildrenV(ITEM_HEIGHT);
}

void SearchWindow::onSearchFinished()
{
    btnGo->setText(tr("search"));
    editExp->setEnabled(true);
    setFocusToEdit(lastCursorPos);
    lblNotFound->setVisible(itemsContainer->children().isEmpty() && !editExp->text().isEmpty());
}

void SearchWindow::onItemClick(QWidget *item)
{
    KineticListItem* theItem = qobject_cast<KineticListItem*>(item);
    if(theItem)
    {
        MAINWINDOW->onItemClick(item);
        if(theItem->getBool())
            done(1);
    }
}

void SearchWindow::onContextMenuShow(QWidget *item, QContextMenuEvent *event)
{
    MAINWINDOW->onContextMenuShow(item, event);
}

void SearchWindow::onBtnGoClick()
{
    if(editExp->isEnabled())
    {
        QString exp = editExp->text().trimmed();
        if(exp.isEmpty())
            return;
        exp = QString("*") + exp + QString("*");
        isRealSearch = true;
        editExp->setEnabled(false);
        btnGo->setText(tr("stop"));
        itemsContainer->deleteAllChildren();
        itemsContainer->setInnerHeight(1);
        itemsContainer->setY(0);
        itemsContainer->setSpeedY(0);
        lastCursorPos = editExp->cursorPosition();
        lblNotFound->setVisible(false);
        fs->startSearch(exp, fs->getCurrentDirectory());
    }
    else
    {
        fs->stopSearch();
        onSearchFinished();
    }
}

void SearchWindow::onDeselect()
{
    editExp->deselect();
}

void SearchWindow::onReturnPressed()
{
    if(editExp->isEnabled())
        onBtnGoClick();
}
