/*************************************************************************}
{ phononsound.cpp - single audio file for PhononPlayer                    }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 25 May 2012                                              }
{*************************************************************************/

#include "phononsound.h"
#include "phononplayer.h"
#include "mainwindow.h"

PhononSound::PhononSound(PhononPlayer *parent, bool enableEq) :
    QObject(parent)
{
    player = parent;

    dev = NULL;
    state = pp_state_Idle;
    sourceType = pp_stype_None;
    tmpFilename = "";
    titleTimer = 0;
    sentPlayEnd = false;
    equalizerEnabled = false;

    createMediaObject(enableEq);
}

PhononSound::~PhononSound()
{
    close();
    //delete equalizer;
    delete mediaObject;
    delete audioOutput;    
    if(!tmpFilename.isEmpty())
        QFile::remove(tmpFilename);
}

bool PhononSound::open(const QString &filename)
{
    close();
    mediaObject->setCurrentSource(Phonon::MediaSource(filename));
    currentFilename = filename;
    sourceType = pp_stype_File;
    startTitleTimer();
    //mediaObject->play();
    return true;
}

void PhononSound::createMediaObject(bool enableEq)
{
    mediaObject = new Phonon::MediaObject(this);
    audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    mediaPath = Phonon::createPath(mediaObject, audioOutput);
    equalizer = player->getEqualizer();
    //equalizer = new Equalizer(this);
    //equalizer->setPresetsIni(MAINWINDOW->getIniSettings());
    setEqualizerEnabled(enableEq);
    mediaObject->setPrefinishMark(PREFINISH_MARK);
    connect(mediaObject, SIGNAL(prefinishMarkReached(qint32)), SLOT(prefinishMarkReached(qint32)));
    // sometimes prefinishMarkReached never fires, so we need a backup
    connect(mediaObject, SIGNAL(finished()), SLOT(onCustomPlayEnd()));
    connect(mediaObject, SIGNAL(stateChanged(Phonon::State,Phonon::State)), SLOT(stateChanged(Phonon::State,Phonon::State)));
    connect(mediaObject, SIGNAL(metaDataChanged()), SLOT(onMetaDataChanged()));
    title = "";
}

void PhononSound::timerEvent(QTimerEvent *event)
{
    int id = event->timerId();
    if(id == titleTimer)
        onMetaDataChanged();
}

bool PhononSound::play()
{
    mediaObject->play();
    sentPlayEnd = false;
    return true;
}

void PhononSound::pause()
{
    mediaObject->pause();
}

void PhononSound::unpause()
{
    mediaObject->play();
}

void PhononSound::stop()
{
    close();
}

void PhononSound::prepareToDie(bool doClose)
{
    if(doClose || equalizerEnabled)
        doCloseOnPrepareToDie();
    disconnect();
    mediaObject->disconnect();
}

bool PhononSound::setEqualizerEnabled(bool doEnable)
{
    if(equalizerEnabled == doEnable)
        return true;
    if(doEnable)
        mediaPath.insertEffect(equalizer->getEffect());
    else
        mediaPath.removeEffect(equalizer->getEffect());
    equalizerEnabled = doEnable;
    return true;
}

void PhononSound::close(bool emitStop)
{
    state = pp_state_Idle;
    killTitleTimer();
    mediaObject->stop();
    mediaObject->clearQueue();
    if(dev)
    {
        dev->close();
        dev = NULL;
    }
    sourceType = pp_stype_None;
    if(emitStop)
        stateChanged(Phonon::StoppedState, mediaObject->state());
}

void PhononSound::startTitleTimer()
{
    killTitleTimer();
    titleTimer = startTimer(2000);
}

void PhononSound::killTitleTimer()
{
    if(!titleTimer)
        return;
    killTimer(titleTimer);
    titleTimer = 0;
}

void PhononSound::onMetaDataChanged()
{
    killTitleTimer();
    QString _title;
    QStringList titles = mediaObject->metaData(Phonon::TitleMetaData);
    if(!titles.isEmpty())
        _title = titles.at(0);
    if(_title.isEmpty())
    {
        QFileInfo info(currentFilename);
        _title = info.completeBaseName();
    }
    QString artist;
    QStringList artists = mediaObject->metaData(Phonon::ArtistMetaData);
    if(!artists.isEmpty())
        artist = artists.at(0);
    QString fullTitle;
    if(artist.isEmpty())
        fullTitle = _title;
    else
        fullTitle = artist+" - "+_title;
    if(title == fullTitle)
        return; // to prevent multiple events with a same title
    title = fullTitle;
    emit onTitleChange();
}

void PhononSound::doCloseOnPrepareToDie()
{
    // mute clicks at the end
    audioOutput->setMuted(true);
    // do not with equalizer in new tracks
    setEqualizerEnabled(false);
    close();
}

void PhononSound::prefinishMarkReached(qint32 msecs)
{
    // if equalizer is enabled, running multiple tracks will cause glitches,
    // so we must not use this event
    if(equalizerEnabled)
        return;

    // avoiding another phonon bug
    if(msecs <= 0)
        return;

    emit onPlayEnd();
}

void PhononSound::stateChanged(Phonon::State newstate, Phonon::State oldstate)
{
    switch(newstate)
    {
        case Phonon::ErrorState:
            qDebug() << currentFilename << mediaObject->errorString();
            if(mediaObject->errorType() == Phonon::FatalError)
            {
                if(sourceType == pp_stype_File)
                {
                    if(mediaObject->currentSource().fileName() != tmpFilename)
                    {
                        if(player->unzip(currentFilename, tmpFilename))
                        {
                            close(false);
                            mediaObject->setCurrentSource(Phonon::MediaSource(tmpFilename));
                            mediaObject->play();
                            return;
                        }
                    }
                }
                close();
                state = pp_state_Error;
                emit onError();
            }
            break;

        case Phonon::StoppedState:
            if(
                (oldstate != Phonon::PlayingState) &&
                (oldstate != Phonon::PausedState)
            )
            {
                break;
            }
            state = pp_state_Idle;            
            emit onStateChanged();
            break;

        case Phonon::PlayingState:
            state = pp_state_Playing;
            emit onStateChanged();
            break;

        case Phonon::PausedState:
            if(
                (oldstate != Phonon::PlayingState) ||
                (state != pp_state_Playing)
            )
            {
                break;
            }
            state = pp_state_Paused;
            emit onStateChanged();
            break;

        default:
            break;
    }
}

void PhononSound::onCustomPlayEnd()
{
    if(sentPlayEnd)
        return;
    sentPlayEnd = true;
    state = pp_state_Idle;
    emit onStateChanged();
    emit onPlayEnd();
}
