/*************************************************************************}
{ aboutwindow.cpp - information window                                    }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 19 May 2012                                              }
{*************************************************************************/

#include "aboutwindow.h"
#include "mainwindow.h"

AboutWindow::AboutWindow(QWidget *parent) :
    CommonDialog(parent)
{
    QString s;

    hash["app_title"] = APP_TITLE;
    hash["label_version"] = tr("Version");
    hash["app_version"] = APP_VERSION;
    hash["label_projsite"] = tr("Project site");
    hash["label_developer"] = tr("Developer");
    hash["label_contactemail"] = tr("Contact email");
    hash["label_contactname"] = tr("Contact name");
    hash["label_devsite"] = tr("Developer's site");
    hash["label_license"] = tr("License");
    s = tr("This project was inspired by %1.");
    hash["label_inspired_by"] =  s.arg("<a href=\"http://thecodecentral.com/symfonie\">symfonie</a>");
    hash["compressed_support_label"] = tr("Compressed modules support");
    if(PLAYER->getUnzipPath().isEmpty())
        hash["compressed_support"] = tr("NO (install 'unzip' package for this to work)");
    else
        hash["compressed_support"] = tr("YES");
    hash["equalizer_support_label"] = tr("Equalizer support");
    if(PLAYER->getUnzipPath().isEmpty())
    {
        s = tr("NO (install '%1' gstreamer plugin to enable equalizer)");
        s = s.arg(PLAYER->getEqualizer()->getModuleName());
        hash["equalizer_support"] = s;
    }
    else
    {
        hash["equalizer_support"] = tr("YES");
    }
    hash["supported_mimes_label"] = tr("Supported MIME types");
    hash["support_more_mimes_label"] = tr("To make more MIME types available, install appropriate gstreamer plugins");
    hash["supported_mimes"] = PLAYER->getSupportedMime().join("<br>");
    hash["cur_skin_label"] = tr("Current skin");
    hash["skin_name"] = SKIN.currentSkinName;

    s = "skins/";
    s.append(SKIN.currentSkinName);
    s = loadTextFromFile(s, "README");
    if(s.isEmpty())
    {
        hash["skin_readme_label"] = tr("The README file for the current skin is either empty or not found.");
    }
    else
    {
        hash["skin_readme_label"] = tr("The following is a contents of README file for a current skin:");
        s = Qt::escape(s);
        s.replace("\n", "<br>");
        // make links and emails clickable
        s.replace(QRegExp("(\\w+\\:\\/\\/[\\w\\.\\-\\d\\%\\_\\?\\&\\=\\/]+)"), "<a href=\"\\1\">\\1</a>");
        s.replace(QRegExp(
            "("
                "[\\w\\!\\#\\$\\%\\*\\/\\?\\|\\^\\{\\}\\`\\~\\&\\'\\+\\=]+"
                "(?:\\-[\\w\\!\\#\\$\\%\\*\\/\\?\\|\\^\\{\\}\\`\\~\\&\\'\\+\\=]+)*"
                "(?:\\."
                    "[\\w\\!\\#\\$\\%\\*\\/\\?\\|\\^\\{\\}\\`\\~\\&\\'\\+\\=]+"
                    "(?:\\-[\\w\\!\\#\\$\\%\\*\\/\\?\\|\\^\\{\\}\\`\\~\\&\\'\\+\\=]+)*"
                ")*"
                "@\\.?(?:[a-z0-9]+(?:\\-[a-z0-9]+)*\\.)+[a-z]{2,4}"
            ")"),
            "<a href=\"mailto:\\1\">\\1</a>");
    }
    hash["skin_readme"] = s+"<br><br>";

    hash["label_tips"] = tr("\"Hidden\" features");
    hash["tip_context_menus"] = tr(
                "Folders and files have context menus. "
                "To invoke a context menu use a long press.");
    hash["tip_search"] = tr(
                "You can search for a file by its name. "
                "Start typing to show a search window. "
                "A search will take place in a current folder and its subfolders.");
    hash["tip_stop_after_current"] = tr(
                "To stop a playback after a current track, do the following: "
                "check \"Let a current track to end before a stop by the timer\" option, "
                "then start a timer with a zero interval.");
    hash["tip_stop_after_queue"] = tr(
                "To stop a playback after the queue is empty, "
                "set \"Play mode\" to \"One track once\".");
    hash["tip_zipped"] = tr(
                "In order to play compressed tracker music (*.xmz, *.s3z, ...) you must install \"unzip\". "
                "This package can be found in \"extras-testing\" repository.");
    hash["tip_filter_update"] = tr(
                "After you (un)install a new gstreamer plugin, "
                "you may click \"Select all supported\" button in \"File types\" window "
                "to update filters list.");

    initGui();
}

AboutWindow::~AboutWindow()
{
    deinitGui();
}

void AboutWindow::showAboutDialog()
{
    AboutWindow* win = new AboutWindow(MAINWINDOW);
    win->showDialog();
    delete win;
}

QString AboutWindow::loadTextFromFile(const QString &foldername, const QString &filename)
{
    QFile f;
    QString fullFilename = APP_DIR + foldername + "/" + filename;
    f.setFileName(fullFilename);
    if(!f.open(QIODevice::ReadOnly))
        return "";
    QString text = QString::fromUtf8(f.readAll().constData()).trimmed();
    f.close();
    return text;
}

QString AboutWindow::getDisplayText(const QString &filename)
{
    return hash.replaceInString(loadTextFromFile("html", filename));
}

void AboutWindow::updateText()
{
    QPushButton* btn = (QPushButton*)(buttons->checkedButton());
    if(!btn)
        return;
    txt->setHtml(btn->property("txt").toString());
}

void AboutWindow::initGui()
{
    CommonDialog::initGui();

    setWindowTitle(tr("About"));

    buttons = new QButtonGroup(this);
    connect(buttons, SIGNAL(buttonClicked(QAbstractButton*)), SLOT(buttonClicked(QAbstractButton*)));

    buttons->addButton(btnApp = new QPushButton(this));
    buttons->addButton(btnSystem = new QPushButton(this));
    buttons->addButton(btnTips = new QPushButton(this));
    buttons->addButton(btnSkin = new QPushButton(this));

    btnApp->setText(tr("App"));
    btnSystem->setText(tr("System"));
    btnSkin->setText(tr("Skin"));
    btnTips->setText(tr("Tips"));

    btnApp->setProperty("txt", getDisplayText("app.html"));
    btnSystem->setProperty("txt", getDisplayText("system.html"));
    btnSkin->setProperty("txt", getDisplayText("skin.html"));
    btnTips->setProperty("txt", getDisplayText("tips.html"));

    btnApp->setCheckable(true);
    btnSystem->setCheckable(true);
    btnSkin->setCheckable(true);
    btnTips->setCheckable(true);

    btnApp->setChecked(true);

    txt = new QTextBrowser(this);
    txt->setStyleSheet("background:white;color:black;");
    txt->setOpenExternalLinks(true);
    txt->setReadOnly(true);
    txt->setWordWrapMode(QTextOption::WordWrap);
    txt->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    updateText();
}

void AboutWindow::deinitGui()
{
    delete btnApp;
    delete btnSystem;
    delete btnSkin;
    delete btnTips;
    delete buttons;

    delete txt;

    CommonDialog::deinitGui();
}

void AboutWindow::reposition(const QSize &sz)
{
    CommonDialog::reposition(sz);

    int w = sz.width();
    int h = sz.height();
    int ww;
    int x=0, y=0;

    if(w > h)
    {
        ww = w/4-5;
        btnApp->move(x, y); btnApp->resize(ww, CONTROLBUTTON_SIZE); x=x+ww+5;
        btnSystem->move(x, y); btnSystem->resize(ww, CONTROLBUTTON_SIZE); x=x+ww+5;
        btnSkin->move(x, y); btnSkin->resize(ww, CONTROLBUTTON_SIZE); x=x+ww+5;
        btnTips->move(x, y); btnTips->resize(ww+5, CONTROLBUTTON_SIZE);
    }
    else
    {
        ww = w/2-5;
        btnApp->move(x, y); btnApp->resize(ww, CONTROLBUTTON_SIZE); x=x+ww+5;
        btnSystem->move(x, y); btnSystem->resize(ww+5, CONTROLBUTTON_SIZE); x=0; y=y+btnSystem->height()+5;
        btnSkin->move(x, y); btnSkin->resize(ww, CONTROLBUTTON_SIZE); x=x+ww+5;
        btnTips->move(x, y); btnTips->resize(ww, CONTROLBUTTON_SIZE);
    }

    y=y+btnSystem->height()+5;

    txt->move(0, y);
    txt->resize(w, h-y);
}

void AboutWindow::buttonClicked(QAbstractButton *button)
{
    Q_UNUSED(button);
    updateText();
}
