/*************************************************************************}
{ mainwindow.cpp - main window with a folder view                         }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 11 Apr 2012                                              }
{*************************************************************************/

#include "mainwindow.h"

MainWindow* MainWindow::instance = NULL;

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent)
{
    instance = this;

    currentFileItem = NULL;
    currentDirItem = NULL;
    currentFileIndex = -1;
    isMovingBackwards = false;
    errCount = 0;
    timerId = 0;
    timerMinsLeft = -1;

    appDir = qApp->applicationDirPath()+"/../";
#ifndef MAEMO5
    appDir.append(APP_NAME).append("/");
#endif
    QFileInfo info(appDir);
    appDir = info.canonicalFilePath()+"/";

    openInis();

    skin = new Skin();

    optionsWindow = new OptionsWindow(this);

    updateScreenOrientation();
    initPlayer();
    initFSMan();
    initLocations();
    initGUI();

    cancelResumeCheck();

    QString lastDir = iniState->value("paths/lastdir", locMusic).toString();
    cd(lastDir);
}

MainWindow::~MainWindow()
{
    iniState->setValue("paths/lastdir", fs->getCurrentDirectory());

    delete optionsWindow;
    delete player;

    deinitGUI();
    closeInis();

    delete fs;
    delete skin;
}

void MainWindow::initGUI()
{
    setWindowTitle(APP_TITLE);

    btnPrev = new Button(this);
    btnStop = new Button(this);
    btnPlay = new Button(this);
    btnNext = new Button(this);
    btnParent = new Button(this);

    connect(btnPrev, SIGNAL(clicked()), SLOT(checkTimerCancel()));
    connect(btnStop, SIGNAL(clicked()), SLOT(checkTimerCancel()));
    connect(btnPlay, SIGNAL(clicked()), SLOT(checkTimerCancel()));
    connect(btnNext, SIGNAL(clicked()), SLOT(checkTimerCancel()));
    connect(btnParent, SIGNAL(clicked()), SLOT(checkTimerCancel()));

    connect(btnPrev, SIGNAL(clicked()), SLOT(onBtnPrevClick()));
    connect(btnStop, SIGNAL(clicked()), SLOT(onBtnStopClick()));
    connect(btnPlay, SIGNAL(clicked()), SLOT(onBtnPlayClick()));
    connect(btnNext, SIGNAL(clicked()), SLOT(onBtnNextClick()));
    connect(btnParent, SIGNAL(clicked()), SLOT(onBtnParentClick()));

    labelDir = new QLabel(this);
    labelDir->move(0, 0);
    QFont f = labelDir->font();
    f.setPixelSize(12);
    labelDir->setFont(f);
    labelDir->setTextInteractionFlags(Qt::NoTextInteraction);
    labelDir->setTextFormat(Qt::PlainText);
    labelDir->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    itemsContainer = new QKineticWidget(this);
    itemsContainer->setMoveByX(false);
    itemsContainer->setBackSpeedX(50);
    itemsContainer->setThresholdY(10);
    connect(itemsContainer, SIGNAL(onItemClick(QWidget*)), SLOT(onItemClick(QWidget*)));
    connect(itemsContainer, SIGNAL(onContextMenu(QWidget*,QContextMenuEvent*)), SLOT(onContextMenuShow(QWidget*,QContextMenuEvent*)));

    menuBar()->addAction(tr("Clear cache"), this, SLOT(onMenuClearClick()));
    menuBar()->addAction(tr("Jump to a current file"), this, SLOT(onMenuCurrentClick()));
    menuBar()->addAction(tr("File types"), this, SLOT(onMenuTypesClick()));
    actionEqualizer = menuBar()->addAction("", this, SLOT(onMenuEqualizerClick()));
    actionTimer = menuBar()->addAction("", this, SLOT(onMenuTimerClick()));
    menuBar()->addAction(tr("Options"), this, SLOT(onMenuOptionsClick()));
    menuBar()->addAction(tr("About"), this, SLOT(onMenuAboutClick()));

    menuFolder = new QMenu(this);
    menuFolder->addAction(tr("Play all files in that folder"), this, SLOT(onMenuFolderAllClick()));

    updateEqualizerActionLabel();
    updateTimerActionLabel();

    changeSkin(OPTIONS.skinName);

    reposition(size());
}

void MainWindow::initLocations()
{
    locDesktop = QDesktopServices::storageLocation(QDesktopServices::DesktopLocation);
    locMusic = getLocation(QDesktopServices::MusicLocation);
    locMovies = QDesktopServices::storageLocation(QDesktopServices::MoviesLocation);
    locPictures = QDesktopServices::storageLocation(QDesktopServices::PicturesLocation);
    locHome = getLocation(QDesktopServices::HomeLocation);
}

void MainWindow::initPlayer()
{
    player = new PhononPlayer();
    connect(player, SIGNAL(onPlayEnd()), SLOT(onPlayEnd()));
    connect(player, SIGNAL(onStateChanged()), SLOT(onStateChanged()));
    connect(player, SIGNAL(onError()), SLOT(onError()));
    connect(player, SIGNAL(onTitleChange(QString)), SLOT(onTitleChange(QString)));
    connect(player, SIGNAL(onHeadsetStatusChange(bool)), SLOT(onHeadsetStatusChange(bool)));
    connect(player, SIGNAL(onCallStart()), SLOT(onCallStart()));
    connect(player, SIGNAL(onCallEnd()), SLOT(onCallEnd()));

    player->getEqualizer()->setPresetsIni(iniSettings);
    currentPreset = iniSettings->value("equalizer/currentPreset", "").toString();
    if(player->getEqualizer()->getEffect())
    {
        if(currentPreset.isEmpty())
        {
            player->setEqualizerEnabled(false);
        }
        else
        {
            if(!player->getEqualizer()->loadFromPreset(currentPreset))
            {
                currentPreset = "";
                player->setEqualizerEnabled(false);
            }
            else
            {
                player->setEqualizerEnabled(true);
            }
        }
    }
    else
    {
        player->setEqualizerEnabled(false);
    }
}

void MainWindow::initFSMan()
{
    fs = new FSMan(this);
    fs->setUsingCache(true);
    fs->setListFetch(true);
    fs->setUsingNativeMethods(false);
    fs->setUpOnFail(true);

    connect(fs, SIGNAL(onDirFilter(bool*,QString,QString)), SLOT(onDirFilter(bool*,QString,QString)));
    connect(fs, SIGNAL(onFileFilter(bool*,QString,QString)), SLOT(onFileFilter(bool*,QString,QString)));

    updateNameFilters(TypesWindow::getSupportedTypes());
}

QString MainWindow::getLocation(QDesktopServices::StandardLocation loc)
{
    QString s = QDesktopServices::storageLocation(loc);
    if(s.isEmpty())
        return QDir::homePath();
    QDir dir;
    if(!dir.exists(s))
        return QDir::homePath();
    return s;
}

void MainWindow::changeSkin(const QString &skinName)
{
    if(!skin->load(skinName))
        skin->load();

    QString selWindows("MainWindow, OptionsWindow, SelectWindow, TypesWindow, EqualizerWindow, TimerWindow, AboutWindow");
    QString selControls("QLabel, Button, QSlider, QButton");

    qApp->setStyleSheet(
                selWindows+", "+selControls+" {"+skin->cssCommon+"} "+
                selWindows+" {"+skin->cssCommonWindows+"} "+
                selControls+" {"+skin->cssCommonControls+"} ");

    btnPrev->setIcon(skin->icoPrev);
    btnStop->setIcon(skin->icoStop);
    if(player->getState() != pp_state_Playing)
        btnPlay->setIcon(skin->icoPlay);
    else
        btnPlay->setIcon(skin->icoPause);
    btnNext->setIcon(skin->icoNext);
    btnParent->setIcon(skin->icoParent);

    skin->initKineticWidget(itemsContainer);
}

void MainWindow::updateScreenOrientation()
{
#ifdef MAEMO5
    switch(OPTIONS.screenOrientation)
    {
        case soo_Portrait:
            setScreenOrientation(ScreenOrientationLockPortrait);
            break;

        case soo_Landscape:
            setScreenOrientation(ScreenOrientationLockLandscape);
            break;

        default:
            setScreenOrientation(ScreenOrientationAuto);
            break;
    }
#endif
}

void MainWindow::deinitGUI()
{
    delete btnPrev;
    delete btnStop;
    delete btnPlay;
    delete btnNext;
    delete btnParent;

    delete labelDir;
    itemsContainer->deleteAllChildren();
    delete itemsContainer;
}

void MainWindow::updateNameFilters(const QStringList &extsList)
{
    nameFilters.clear();
    if(OPTIONS.useFileFilters)
    {
        foreach(QString ext, extsList)
            nameFilters.append("*."+ext);
    }
    else
    {
        nameFilters.append("*");
    }
    fs->setNameFilters(nameFilters);
}

void MainWindow::reposition(const QSize& sz)
{
    int h = sz.height();
    int w = sz.width();
    bool isPortrait = (h > w);

    int labelPathHeight = labelDir->fontMetrics().height();
    labelDir->resize(w, labelPathHeight);
    int x, y;

    itemsContainer->move(0, labelPathHeight);
    if(isPortrait)
    {
        y = h - (CONTROLBUTTON_SIZE + CONTROLBUTTON_PUSHOFFSET + CONTROLBUTTON_PADDING);
        itemsContainer->resize(w, y - labelPathHeight);
        x = 0;
        btnPrev->move(x, y); x = x + CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING + CONTROLBUTTON_PADDING;
        btnStop->move(x, y); x = x + CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING + CONTROLBUTTON_PADDING;
        btnPlay->move(x, y); x = x + CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING + CONTROLBUTTON_PADDING;
        btnNext->move(x, y);
        x = w - (CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING);
        btnParent->move(x, y);
    }
    else
    {
        x = w - CONTROLBUTTON_SIZE - CONTROLBUTTON_PADDING;
        itemsContainer->resize(x, h - labelPathHeight);
        y = labelPathHeight;
        btnPrev->move(x, y); y = y + CONTROLBUTTON_SIZE + CONTROLBUTTON_PUSHOFFSET+ CONTROLBUTTON_PADDING;
        btnStop->move(x, y); y = y + CONTROLBUTTON_SIZE + CONTROLBUTTON_PUSHOFFSET+ CONTROLBUTTON_PADDING;
        btnPlay->move(x, y); y = y + CONTROLBUTTON_SIZE + CONTROLBUTTON_PUSHOFFSET+ CONTROLBUTTON_PADDING;
        btnNext->move(x, y);
        y = h - (CONTROLBUTTON_SIZE + CONTROLBUTTON_PUSHOFFSET*3);
        btnParent->move(x,y);
    }
    KineticListItem* item;
    foreach(QObject* child, itemsContainer->children())
    {
        item = (KineticListItem*)child;
        item->resize(itemsContainer->width(), item->height());
    }
}

bool MainWindow::cd(const QString &dir)
{
    saveY();
    bool result = fs->cd(dir);
    labelDir->setText(fs->getCurrentDirectory());
    buildList();
    restoreY();
    return result;
}

void MainWindow::buildList()
{
    foreach(QObject* item, itemsContainer->children())
        delete item;

    QString path = fs->getCurrentDirectory();
    if(!path.endsWith("/"))
        path.append("/");
    KineticListItem* item;
    QString itemPath;
    int a = 0;
    int w = itemsContainer->width();

    currentFileItem = NULL;
    currentDirItem = NULL;

    itemsContainer->setY(0);
    itemsContainer->setSpeedY(0);
    QString fullDirName;
    foreach(QString dirName, fs->getCurrentDirs())
    {
        fullDirName = path + dirName;
        item = new KineticListItem(itemsContainer);
        item->setString(fullDirName);
        item->resize(w, ITEM_HEIGHT);
        item->move(0, a*ITEM_HEIGHT);
        item->setText(dirName);
        item->setIcon(skin->icoFolder);
        item->setBool(true);
        item->show();
        a++;
    }

    foreach(QString fileName, fs->getCurrentFiles())
    {
        item = new KineticListItem(itemsContainer);
        itemPath = path + fileName;
        item->setString(itemPath);
        item->resize(w, ITEM_HEIGHT);
        item->move(0, a*ITEM_HEIGHT);
        item->setText(fileName);
        item->setIcon(skin->icoFile);
        item->setBool(false);
        item->show();
        a++;
    }

    itemsContainer->setInnerHeight(a*ITEM_HEIGHT);
    labelDir->setText(path);

    updateListIcons();
}

void MainWindow::updateListIcons(bool playMode)
{
    KineticListItem* item;

    const QString& filename = player->getCurrentFilename();

    if(currentFileItem)
    {
        currentFileItem->setIcon(skin->icoFile);
        currentFileItem = NULL;
    }

    if(currentDirItem)
    {
        currentDirItem->setIcon(skin->icoFolder);
        currentDirItem = NULL;
    }

    if(filename.startsWith(fs->getCurrentDirectory()))
    {
        foreach(QObject* child, itemsContainer->children())
        {
            item = qobject_cast<KineticListItem*>(child);
            if(item)
            {
                if(item->getBool())
                {
                    if(filename.startsWith(item->getString()))
                    {
                        currentDirItem = item;
                        item->setIcon(skin->icoFolderPlay);
                        break;
                    }
                }
            }
        }
    }

    foreach(QObject* child, itemsContainer->children())
    {
        item = qobject_cast<KineticListItem*>(child);
        if(item)
        {
            if(!item->getBool())
            {
                if(item->getString() == filename)
                {
                    currentFileItem = item;
                    if(playMode)
                    {
                        item->setIcon(skin->icoFilePlay);
                    }
                    else
                    {
                        switch(player->getState())
                        {
                            case pp_state_Playing:
                                item->setIcon(skin->icoFilePlay);
                                break;

                            case pp_state_Paused:
                                item->setIcon(skin->icoFilePause);
                                break;

                            default:
                                item->setIcon(skin->icoFileStop);
                        }
                    }
                    break;
                }
            }
        }
    }
}

void MainWindow::play(const QString &filename)
{
    fs->getCurrentFilesFull(playlist);
    errCount = 0;
    playIndex(playlist.indexOf(filename));
}

void MainWindow::playIndex(int index)
{
    if((index >= playlist.size()) || (index < 0))
    {
        if(currentFileItem)
            currentFileItem->setIcon(skin->icoFileStop);
        return;
    }

    QString filename = playlist.at(index);
    currentFileIndex = index;
    player->play(filename);

    updateListIcons(true);
}

void MainWindow::playRandom()
{
    playIndex(rnd(playlist.size()));
}

void MainWindow::stop()
{
    player->stop();
}

void MainWindow::saveY()
{
    yCache[fs->getCurrentDirectory()] = itemsContainer->getY();
}

void MainWindow::restoreY()
{
    const QString& dir = fs->getCurrentDirectory();
    if(yCache.contains(dir))
        itemsContainer->setY(yCache[dir]);
}

void MainWindow::openInis()
{
    settingsDir = QDir::homePath();
    settingsDir.append("/.").append(QApplication::applicationName()).append("/");
    QDir dir;
    if(!dir.exists(settingsDir))
        dir.mkpath(settingsDir); // do not check for errors here

    openIni(iniState, "state");
    openIni(iniSettings, "settings");
}

void MainWindow::openIni(QSettings *&ini, const QString &filename)
{
    QString iniFilename = settingsDir;
    iniFilename.append(filename).append(".ini");
    ini = new QSettings(iniFilename, QSettings::IniFormat);
    ini->setIniCodec("UTF-8");
}

void MainWindow::closeInis()
{
    delete iniState;
    delete iniSettings;
}

void MainWindow::resizeEvent(QResizeEvent *event)
{
    reposition(event->size());
    emit onResize(event->size());
}

void MainWindow::timerEvent(QTimerEvent *event)
{
    Q_UNUSED(event);
    if(timerMinsLeft < 0)
        return;
    if(timerMinsLeft)
        timerMinsLeft--;
    else
        endTimer();
    updateTimerActionLabel();
}

void MainWindow::onItemClick(QWidget *item)
{
    KineticListItem* theItem = qobject_cast<KineticListItem*>(item);
    if(theItem)
    {
        if(theItem->getBool())
        {
            cd(theItem->getString()); // clicked on a folder
        }
        else
        {
            checkTimerCancel();
            if(currentFileItem == theItem)
                onBtnPlayClick(); // clicked on the current file
            else
                play(theItem->getString()); // clicked on an other file
        }
    }
}

void MainWindow::onContextMenuShow(QWidget *item, QContextMenuEvent *event)
{
    KineticListItem* i = qobject_cast<KineticListItem*>(item);
    if(i)
        if(i->getBool())
            menuFolder->exec(event->globalPos());
}

void MainWindow::checkTimerCancel()
{
    if(OPTIONS.buttonsCancelTimer && (timerMinsLeft >= 0))
    {
        killTheTimer();
        timerMinsLeft = -1;
        updateTimerActionLabel();
        showStopTimerNotification();
    }
}

void MainWindow::onBtnPrevClick()
{
    runPrev();
}

void MainWindow::onBtnStopClick()
{
    stop();
    cancelResumeCheck();
}

void MainWindow::onBtnPlayClick()
{
    switch(player->getState())
    {
        case pp_state_Idle:
            playIndex(currentFileIndex);
            break;

        case pp_state_Playing:
            player->pause();
            break;

        case pp_state_Paused:
            player->unpause();
            break;
    }
    cancelResumeCheck();
}

void MainWindow::onBtnNextClick()
{
    runNext();
}

void MainWindow::onBtnParentClick()
{
    cd(FSMan::getParentDir(fs->getCurrentDirectory()));
}

void MainWindow::onPlayEnd()
{
    if(OPTIONS.playMode == pmo_trackOnce)
    {
        if(currentFileItem)
            currentFileItem->setIcon(skin->icoFileStop);
        return;
    }
    if(!timerMinsLeft && OPTIONS.letTrackEnd)
    {
        stopByTimer();
        updateTimerActionLabel();
        return;
    }
    if(!runNext())
        showStopNotification();
}

void MainWindow::onError()
{
    if(currentFileItem)
        currentFileItem->setIcon(skin->icoFileStop);

    if((OPTIONS.playMode == pmo_trackOnce) || (OPTIONS.playMode == pmo_trackLoop))
    {
        // if only playing one track, then stop here
        showStopNotification();
        return;
    }
    else
    {
        // if in playlist mode, then stop, when all files are invalid
        errCount++;
        if(errCount == playlist.size())
        {
            showStopNotification();
            return;
        }
    }

    bool result;
    if(isMovingBackwards)
        result = runPrev();
    else
        result = runNext();
    if(result)
        showStopNotification();
}

void MainWindow::onStateChanged()
{
    bool changeCurrentItem = (currentFileItem != NULL);
    if(changeCurrentItem)
    {
        if((currentFileIndex < 0) || (currentFileIndex >= playlist.size()))
            changeCurrentItem = false;
        else
            changeCurrentItem = (currentFileItem->getString() == playlist[currentFileIndex]);
    }

    switch(player->getState())
    {
        case pp_state_Idle:
            if(changeCurrentItem)
                currentFileItem->setIcon(skin->icoFileStop);
            btnPlay->setIcon(skin->icoPlay);
            break;

        case pp_state_Playing:
            errCount = 0; // reset error counter on successful start
            isMovingBackwards = false;
            if(changeCurrentItem)
                currentFileItem->setIcon(skin->icoFilePlay);
            btnPlay->setIcon(skin->icoPause);
            break;

        case pp_state_Paused:
            isMovingBackwards = false;
            if(changeCurrentItem)
                currentFileItem->setIcon(skin->icoFilePause);
            btnPlay->setIcon(skin->icoPlay);
            break;
    }
}

void MainWindow::onTitleChange(const QString &title)
{
    if(OPTIONS.popupTrackTitles)
        showNotification(title);
}

void MainWindow::onMenuClearClick()
{
    fs->clearCache();
    showNotification(tr("Cache has been cleared"));
    cd(fs->getCurrentDirectory());
}

void MainWindow::onMenuCurrentClick()
{
    const QString& filename = player->getCurrentFilename();
    if(filename.isEmpty())
        return;

    if(!cd(FSMan::getParentDir(filename)))
        return;

    if(currentFileItem)
        itemsContainer->centerWidgetV(currentFileItem);
}

void MainWindow::onMenuTypesClick()
{
    QStringList list = TypesWindow::showTypesDialog();
    updateNameFilters(list);
    onMenuClearClick();
}

void MainWindow::onMenuOptionsClick()
{
    if(optionsWindow->showDialog())
    {
        updateNameFilters(TypesWindow::getSupportedTypes());
        onMenuClearClick();
    }
    updateScreenOrientation();
    cd(fs->getCurrentDirectory());
}

void MainWindow::onMenuEqualizerClick()
{
    if(!player->getEqualizer()->getEffect())
    {
        showNotification(tr("Equalizer module not found!") + " (" + player->getEqualizer()->getModuleName() + ")");
        return;
    }

    currentPreset = EqualizerWindow::showEqualizerWindow(currentPreset);
    iniSettings->setValue("equalizer/currentPreset", currentPreset);
    updateEqualizerActionLabel();
}

void MainWindow::onMenuTimerClick()
{
    killTheTimer();
    if(timerMinsLeft >= 0)
        showStopTimerNotification();
    timerMinsLeft = TimerWindow::showTimerDialog(timerMinsLeft);
    if(timerMinsLeft >= 0)
    {
        showNotification(tr("The timer has been started"));
        timerId = startTimer(60*1000);
        if(!timerMinsLeft)
            endTimer();
    }
    updateTimerActionLabel();
}

void MainWindow::onMenuAboutClick()
{
    AboutWindow::showAboutDialog();
}

void MainWindow::onMenuFolderAllClick()
{
    showNotification(tr("searching..."));
    stop();
    playlist.clear();
    QString curdir = fs->getCurrentDirectory();
    KineticListItem* item = (KineticListItem*)(itemsContainer->getActiveWidget());
    addFromDir(item->getString());
    fs->cd(curdir);
    if(OPTIONS.playMode == pmo_random)
        playRandom();
    else
        playIndex(0);
    showNotification(tr("found %n file(s)", "", playlist.size()));
}

void MainWindow::onHeadsetStatusChange(bool isConnected)
{
    if(checkResumeOnHangup)
    {
        checkResumeOnPlugin = (OPTIONS.headsetUnplugAction != bao_None) && !isConnected;
        return;
    }

    if(player->getState() == pp_state_Playing)
    {
        if(!isConnected)
        {
            switch(OPTIONS.headsetUnplugAction)
            {
                case bao_Pause:
                    player->pause();
                    checkResumeOnPlugin = true;
                    break;

                case bao_Stop:
                    player->stop();
                    checkResumeOnPlugin = true;
                    break;

                default:
                    break;
            }
        }
    }
    else
    {
        if(!checkResumeOnPlugin)
            return;
        checkResumeOnPlugin = false;
        if(PLAYER->isCalling())
            return; // the call is in progress
        if(!OPTIONS.resumeOnPlugin)
            return;
        if(!isConnected)
            return; // headset is not connected
        if(player->getState() == pp_state_Idle)
            playIndex(currentFileIndex);
        else // state == pp_state_Paused
            player->unpause();
    }
}

void MainWindow::onCallStart()
{
    if(player->getState() != pp_state_Playing)
        return;
    switch(OPTIONS.incomingCallAction)
    {
        case bao_Pause:
            player->pause();
            checkResumeOnHangup = true;
            break;

        case bao_Stop:
            player->stop();
            checkResumeOnHangup = true;
            break;

        default:
            break;
    }
}

void MainWindow::onCallEnd()
{
    if(!checkResumeOnHangup)
        return;
    checkResumeOnHangup = false;
    if(checkResumeOnPlugin)
        return; // waiting for headset to be plugged in
    if(!OPTIONS.resumeOnHangup)
        return;
    if(player->getState() == pp_state_Playing)
        return; // already playing
    if(player->getState() == pp_state_Idle)
        playIndex(currentFileIndex);
    else // state == pp_state_Paused
        player->unpause();
}

void MainWindow::onFileFilter(bool *pass, const QString &basename, const QString &parentDir)
{
    Q_UNUSED(parentDir);
    *pass = passFile(basename);
}

void MainWindow::onDirFilter(bool *pass, const QString &basename, const QString &parentDir)
{
    *pass = passDir(basename, parentDir+"/"+basename);
}

void MainWindow::addFromDir(const QString &dirname)
{
    if(!fs->cd(dirname))
        return;

    QString root;
    if(dirname.endsWith("/"))
        root = dirname;
    else
        root = dirname + "/";

    foreach(QString filename, fs->getCurrentFiles())
        playlist.append(root + filename);

    QString fullDir;
    foreach(QString subdirname, fs->getCurrentDirs())
    {
        fullDir = root + subdirname;
        addFromDir(fullDir);
    }
}

bool MainWindow::passFile(const QString &filename) // only filename part
{
    if(OPTIONS.hideHiddenFiles)
        return !filename.startsWith(".");
    return true;
}

bool MainWindow::passDir(const QString &dirname, const QString& fullDirName)
{
    if(OPTIONS.hideHiddenFolders)
        if(dirname.startsWith("."))
        {
            if(OPTIONS.showUserFolders)
            {
                return
                    (fullDirName == locDesktop)||
                    (fullDirName == locDocuments)||
                    (fullDirName == locMusic)||
                    (fullDirName == locMovies)||
                    (fullDirName == locPictures)||
                    (fullDirName == locHome);
            }
            else
            {
                return false;
            }
        }
    return true;
}

void MainWindow::updateEqualizerActionLabel()
{
    QString presetName;
    if(currentPreset.isEmpty())
        presetName = QString("<")+tr("off", "eqActionLabel")+QString(">");
    else
        presetName = currentPreset;
    actionEqualizer->setText(tr("Equalizer")+": "+presetName);
}

bool MainWindow::runNext()
{
    isMovingBackwards = false;
    cancelResumeCheck();

    int i;
    switch(OPTIONS.playMode)
    {
        case pmo_trackOnce:
            return false;

        case pmo_random:
            playRandom();
            return true;

        case pmo_trackLoop:
            i = currentFileIndex;
            break;

        case pmo_playlistOnce:
            if(currentFileIndex >= (playlist.size()-1))
                return false;
            i = currentFileIndex + 1;
            break;

        case pmo_playlistLoop:
            if(currentFileIndex >= (playlist.size()-1))
                i = 0;
            else
                i = currentFileIndex + 1;
            break;

        default:
            return false;
    }

    playIndex(i);
    return true;
}

bool MainWindow::runPrev()
{
    int i;
    cancelResumeCheck();
    switch(OPTIONS.playMode)
    {
        case pmo_trackOnce:
            isMovingBackwards = false;
            return false;

        case pmo_random:
            isMovingBackwards = false;
            playRandom();
            return true;

        case pmo_trackLoop:
            isMovingBackwards = false;
            i = currentFileIndex;
            break;

        case pmo_playlistOnce:
            if(currentFileIndex <= 0)
            {
                isMovingBackwards = false;
                return false;
            }
            i = currentFileIndex - 1;
            break;

        case pmo_playlistLoop:
            if(currentFileIndex <= 0)
                i = playlist.size() - 1;
            else
                i = currentFileIndex - 1;
            break;

        default:
            return false;
    }

    isMovingBackwards = true;
    playIndex(i);
    return true;
}

void MainWindow::showStopNotification()
{
    showNotification(tr("Playback has been stopped"));
}

void MainWindow::cancelResumeCheck()
{
    checkResumeOnPlugin = false;
    checkResumeOnHangup = false;
}

void MainWindow::killTheTimer()
{
    if(!timerId)
        return;
    killTimer(timerId);
    timerId = 0;
}

void MainWindow::endTimer()
{
    if(!OPTIONS.letTrackEnd || (player->getState() != pp_state_Playing))
        stopByTimer();
    killTheTimer();
}

void MainWindow::showStopTimerNotification()
{
    showNotification(tr("Timer has been stopped"));
}

void MainWindow::stopByTimer()
{
    timerMinsLeft = -1;
    stop();
    cancelResumeCheck();
}

void MainWindow::updateTimerActionLabel()
{
    QString s(tr("Timer"));
    s.append(": ");
    if(timerMinsLeft < 0)
    {
        s.append("<").append(tr("off")).append(">");
    }
    else
    {
        int digits[4];
        TimerWindow::minsToDigits(timerMinsLeft, &digits[0]);
        QString tmp;
        tmp.setNum(digits[3]);s.append(tmp);
        tmp.setNum(digits[2]);s.append(tmp);
        s.append(":");
        tmp.setNum(digits[1]);s.append(tmp);
        tmp.setNum(digits[0]);s.append(tmp);
    }
    actionTimer->setText(s);
}

void MainWindow::showNotification(const QString &text)
{
#ifdef MAEMO5
    QDBusMessage message = QDBusMessage::createMethodCall(
                "org.freedesktop.Notifications",
                "/org/freedesktop/Notifications",
                "org.freedesktop.Notifications",
                "SystemNoteInfoprint");
    message << text;
    QDBusConnection::sessionBus().call(message);
#else
    Q_UNUSED(text);
#endif
}
