/*************************************************************************}
{ fsman.cpp - filesystem functions using shell                            }
{                                                                         }
{ (c) Alexey Parfenov, 2011                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This library is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This library is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 14 Jan 2012                                              }
{*************************************************************************/

#include "fsman.h"
#include <QProcess>
#include <QRegExp>

FSMan::FSMan(QObject *parent) :
    QObject(parent)
{
    doFetchLists = true;
    doUseCache = true;
    doUseNativeMethods = false;
    doUpOnFail = true;
    doUseCallbackFilters = true;
    curDir = "";
}

bool caseInsensitiveLessThan(const QString &s1, const QString &s2)
{
    return s1.toLower() < s2.toLower();
}

QString FSMan::which(const QString& program)
{
#ifdef WIN32
    return FSMan::getShellCommandOutput("for %i in ("+program+".exe) do @echo.%~$PATH:i");
#else
    QStringList args;
    args.append(program);
    return FSMan::getProcessOutput("which", args).trimmed();
#endif
}

void FSMan::_fetchForce()
{
    FSMan::ls(curDir, dirs, files, doUseNativeMethods);
    QRegExp rx;
    rx.setPatternSyntax(QRegExp::WildcardUnix);
    rx.setCaseSensitivity(Qt::CaseInsensitive);
    QStringList filesFiltered;
    foreach(QString name, files)
    {
        foreach(QString filter, nameFilters)
        {
            rx.setPattern(filter);
            if(rx.exactMatch(name))
            {
                filesFiltered.append(name);
                break;
            }
        }
    }
    files = filesFiltered;

    if(doUseCallbackFilters)
    {
        QStringList dirsFiltered;
        filesFiltered.clear();
        bool passed;
        foreach(QString dirName, dirs)
        {
            passed = true;
            emit onDirFilter(&passed, dirName, curDir);
            if(passed)
                dirsFiltered.append(dirName);
        }
        foreach(QString fileName, files)
        {
            passed = true;
            emit onFileFilter(&passed, fileName, curDir);
            if(passed)
                filesFiltered.append(fileName);
        }
        files = filesFiltered;
        dirs = dirsFiltered;
    }
}

void FSMan::_fetch()
{
    if(doFetchLists)
    {
        if(doUseCache)
        {
            if(cache.contains(curDir))
            {
                FSManCacheRecord crec = cache.value(curDir);
                files = crec.files;
                dirs = crec.dirs;
            }
            else
            {
                _fetchForce();
                FSManCacheRecord rec;
                rec.files = files;
                rec.dirs = dirs;
                cache.insert(curDir, rec);
            }
        }
        else
        {
            _fetchForce();
        }
    }
}

bool FSMan::_cd()
{
    QString lastDir = curDir;
    bool result = true;
    while(!FSMan::isDirAccessible(curDir, doUseNativeMethods))
    {
        result = false;
        curDir = getParentDir(curDir);
        if(lastDir == curDir)
            break;
    }
    _fetch();
    return result;
}

void FSMan::getCurrentFilesFull(QStringList& list)
{
    list.clear();
    if(curDir.endsWith("/"))
        foreach(QString item, files)
            list.append(curDir+item);
    else
        foreach(QString item, files)
            list.append(curDir+"/"+item);
}

void FSMan::getCurrentDirsFull(QStringList& list)
{
    list.clear();
    if(curDir.endsWith("/"))
        foreach(QString item, dirs)
            list.append(curDir+item);
    else
        foreach(QString item, dirs)
            list.append(curDir+"/"+item);
}

bool FSMan::cd(const QString& dirPath)
{
    curDir = dirPath;
    bool result = true;

    if(doUseCache)
    {
        if(cache.contains(curDir))
        {
            _fetch();
        }
        else
        {
            result = _cd();
        }
    }
    else
    {
        result = _cd();
    }

    return result;
}

QString FSMan::getParentDir(const QString& dirPath)
{
    bool charMet = false;
    bool isSep;
    QChar c;
    int n = dirPath.size()-1;
    QString result;
    int a;
    for(a=n; a>=0; a--)
    {
        c = dirPath.at(a);
        isSep = (c == '/') || (c == '\\');
        if(isSep)
        {
            if(charMet)
                break;
        }
        else
        {
            charMet = true;
        }
    }
    result = dirPath.left(a);
    if(result.isEmpty())
    {
#ifdef WIN32
        return dirPath.at(0)+':';
#else
        return "/";
#endif
    }
    else
    {
        return result;
    }
}

void FSMan::setUsingNativeMethods(bool doUse)
{
#ifndef WIN32
    doUseNativeMethods = doUse;
#else
    Q_UNUSED(doUse);
#endif
}

bool FSMan::isDirAccessible(const QString& dirPath, bool useNativeMethod)
{
#ifndef WIN32
    if(useNativeMethod)
    {
        QString command = "\
ESCAPEDPATH=`echo -e \""+shellEscape(dirPath)+"\"`/\n\
if [ -d \"${ESCAPEDPATH}\" ]; then\n\
    echo 1\n\
fi\n";
        QString result = FSMan::getShellCommandOutput(command);
        return (result.trimmed() == "1");
    }
    else
    {
#endif
        QDir dir;
        bool result = dir.cd(dirPath);
        return result;
#ifndef WIN32
    }
#endif
}

QString FSMan::getShellCommandOutput(const QString& cmd)
{
    QStringList args;
    QString shell;
#ifdef WIN32
    shell = "cmd";
    args.append("/c");
#else
    shell = "sh";
    args.append("-c");
#endif
    args.append(cmd);
    return FSMan::getProcessOutput(shell, args);
}

QString FSMan::shellEscape(const QString& line)
{
#ifdef WIN32
    return line;
#endif
    QByteArray bytes = line.toUtf8();
    QString result;
    QString byteCode;
    int n = bytes.size();
    int realCode;
    for(int a=0; a<n; a++)
    {
        realCode = bytes.at(a);
        if(realCode<0)
            realCode = realCode + 256;
        byteCode.setNum(realCode, 16);
#ifndef WIN32
        result += "\\x"+byteCode;
#else
        //result += "\\x"+byteCode;
#endif
    }
    return result;
}

void FSMan::ls(const QString& path, QStringList& dirsList, QStringList& filesList, bool useNativeMethod)
{
    dirsList.clear();
    filesList.clear();
#ifndef WIN32
    if(useNativeMethod)
    {
        QString command = "\
ESCAPEDPATH=`echo -e \""+shellEscape(path)+"\"`/\n\
ls -a1 \"${ESCAPEDPATH}\" 2>/dev/null | while read line\n\
do\n\
if [ -d \"${ESCAPEDPATH}$line\" ]; then\n\
    echo d $line\n\
else\n\
    echo f $line\n\
fi\n\
done\n";
        QString output = FSMan::getShellCommandOutput(command);
        QStringList lines = output.split('\n', QString::SkipEmptyParts);
        QString trLine;
        foreach(QString line, lines)
        {
            trLine = line.trimmed();
            if(trLine.startsWith("d "))
            {
                trLine = trLine.right(trLine.size()-2);
                if((trLine != ".") && (trLine != ".."))
                    dirsList.append(trLine);
            }
            else
            {
                if(trLine.startsWith("f "))
                    filesList.append(trLine.right(trLine.size()-2));
            }
        }
    }
    else
    {
#endif
        QDir dir;
        dir.cd(path);
        QDir::Filters filters = QDir::AllDirs | QDir::Files | QDir::NoDotAndDotDot | QDir::Readable | QDir::Hidden | QDir::System;
        QFileInfoList list = dir.entryInfoList(filters);
        foreach(QFileInfo entry, list)
        {
            if(entry.isDir())
                dirsList.append(entry.fileName());
            else
                filesList.append(entry.fileName());
        }
#ifndef WIN32
    }
#endif
    qSort(dirsList.begin(), dirsList.end(), caseInsensitiveLessThan);
    qSort(filesList.begin(), filesList.end(), caseInsensitiveLessThan);
}

bool FSMan::getRawProcessOutput(QByteArray& dest, const QString& cmd, const QStringList& args)
{
    QProcess proc;
    proc.start(cmd, args, QIODevice::ReadOnly);
    if(!proc.waitForFinished())
        return false;
    dest.append(proc.readAllStandardOutput());
    proc.close();
    return true;
}

QString FSMan::getProcessOutput(const QString& cmd, const QStringList& args)
{
    QByteArray resultRaw;
    if(getRawProcessOutput(resultRaw, cmd, args))
        return QString::fromUtf8(resultRaw.constData());
    else
        return "";
}

QString FSMan::getProcessOutput(const QString& cmd)
{
     return getProcessOutput(cmd, QStringList());
}




