#!/usr/bin/env python
# -*- coding: utf-8 -*-

#########################################################################
#    Copyright (C) 2010, 2011 Sergio Villar Senin <svillar@igalia.com>
#
#    This file is part of ReSiStance
#
#    ReSiStance is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    ReSiStance is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with ReSiStance.  If not, see <http://www.gnu.org/licenses/>.
#########################################################################

from feedmanager import ReSiStanceFeedSummary
from pango import ELLIPSIZE_END
from portrait import FremantleRotation
from renderers import FeedCellRenderer, CellRendererRoundedRectangleText
from threading import Thread
from utils import *
from webkit import WebView
import calendar
import constants
import dbus
import feedparser
import gettext
import glib
import gobject
import gtk
import hildon
import os
import osso
import pygtk
pygtk.require('2.0')
import time
import urllib2
import urllib

_ = gettext.gettext

class FeedsView(hildon.GtkTreeView):

    DUMMY_FEED_STATUS = -1

    def __init__(self, ui_mode):
        super(FeedsView, self).__init__(ui_mode)

        # Feed icon column
        pix_renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = FeedsWindow.FEEDS_MODEL_PIXBUF_COLUMN)
        column.set_expand(False)
        self.append_column(column);

        # Feed title column
        google_reader_pixbuf = gtk.gdk.pixbuf_new_from_file(constants.GOOGLE_READER_ICON_FILE)
        text_renderer = FeedCellRenderer(google_reader_pixbuf)
        text_renderer.set_property('xalign', 0)
        text_renderer.set_property('xpad', 8)

        column = gtk.TreeViewColumn('Name', text_renderer, title=FeedsWindow.FEEDS_MODEL_TITLE_COLUMN, subtitle=FeedsWindow.FEEDS_MODEL_SUBTITLE_COLUMN, sync=FeedsWindow.FEEDS_MODEL_SYNC_COLUMN, id=FeedsWindow.FEEDS_MODEL_ID_COLUMN)
        column.set_expand(True)
        column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        self.append_column(column);

        # Unread entries column
        unread_renderer = CellRendererRoundedRectangleText()
        unread_renderer.set_property('xpad', 8)
        unread_renderer.set_property('xalign', 0.5)
        column = gtk.TreeViewColumn('Unread', unread_renderer, text = FeedsWindow.FEEDS_MODEL_READ_COLUMN)
        column.set_expand(False)
        self.append_column(column);

        # TODO: Fixed size to improve panning ?

class SettingsDialog(gtk.Dialog):
    RESPONSE_SYNC_GOOGLE = 1

    def __init__(self, parent, settings):
        super(SettingsDialog, self).__init__(_('Settings'), parent,
                                                gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                                (gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))

        self.settings = settings

        self.set_size_request(parent.get_size()[0], parent.get_size()[1])
        self.panel = hildon.PannableArea()
        list_panel = gtk.VBox(2)
        self.panel.add_with_viewport(list_panel)
        self.vbox.pack_start(self.panel, True)

        visual_label = gtk.Label(_('Visual Settings'))
        visual_label.set_alignment(0.5, 0.75)
        list_panel.pack_start(visual_label)

        # Orientation picker
        self.selector_orientation = hildon.TouchSelector(text=True)
        for caption in FremantleRotation.MODE_CAPTIONS:
            self.selector_orientation.append_text(caption)
        self.picker_orientation = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_orientation.set_selector(self.selector_orientation)
        self.picker_orientation.set_alignment(0, 0.5, 0, 0)
        self.picker_orientation.set_title(_("Screen orientation"))
        list_panel.pack_start(self.picker_orientation)
        self.picker_orientation.show()

        # Font size picker
        self.selector_font_size = hildon.TouchSelector(text=True)
        for i in constants.font_size_range:
            self.selector_font_size.append_text(str(i))
        self.picker_font_size = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_font_size.set_selector(self.selector_font_size)
        self.picker_font_size.set_alignment(0, 0.5, 0, 0)
        self.picker_font_size.set_title(_("Default font size"))
        list_panel.pack_start(self.picker_font_size)
        self.picker_font_size.show()

        # Load images check button
        self.auto_load_images = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.auto_load_images.set_label(_('Auto load images'))
        list_panel.pack_start(self.auto_load_images)
        self.auto_load_images.show()

        update_label = gtk.Label(_('Update Settings'))
        update_label.set_alignment(0.5, 0.75)
        list_panel.pack_start(update_label)

        # Auto update at startup
        self.auto_update_startup = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.auto_update_startup.set_label(_('Auto-update at start-up'))
        list_panel.pack_start(self.auto_update_startup)
        self.auto_update_startup.show()

        # Delete old check button
        self.delete_old = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.delete_old.set_label(_('Delete old items when updating'))
        list_panel.pack_start(self.delete_old)
        self.delete_old.show()

        # Automatic download check button
        self.automatic_download = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.automatic_download.set_label(_('Automatically download Podcasts'))
        self.automatic_download.connect('clicked', self._auto_download_cb)
        list_panel.pack_start(self.automatic_download)
        self.automatic_download.show()

        self.picker_download_folder = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_download_folder.set_title(_('Auto Download Folder'))
        self.picker_download_folder.set_value(self.settings.auto_download_folder)
        self.picker_download_folder.set_alignment(0, 0.5, 0, 0)
        self.picker_download_folder.connect('clicked', self._download_folder_cb)
        self.picker_download_folder.set_style(hildon.BUTTON_STYLE_PICKER)
        self.picker_download_folder.set_sensitive(self.settings.auto_download)
        list_panel.pack_start(self.picker_download_folder)
        self.picker_download_folder.show()

        # Load settings
        self.auto_load_images.set_active(self.settings.auto_load_images)
        self.automatic_download.set_active(self.settings.auto_download)
        self.selector_orientation.set_active(0, self.settings.rotation_mode)
        self.delete_old.set_active(self.settings.delete_old)
        self.auto_update_startup.set_active(self.settings.auto_update_startup)
        try:
            font_size_index = constants.font_size_range.index(self.settings.default_font_size)
        except ValueError:
            # defaults to 16pt
            font_size_index = constants.font_size_range.index(16)
        self.selector_font_size.set_active(0, font_size_index)

        google_label = gtk.Label(_('Google Reader Settings'))
        google_label.set_alignment(0.5, 0.75)
        list_panel.pack_start(google_label)

        self.entry_user = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry_user.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL)
        self.entry_user.set_text(self.settings.user)
        self.entry_user.set_sensitive(self.settings.sync_global)

        self.entry_password = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry_password.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL | gtk.HILDON_GTK_INPUT_MODE_INVISIBLE)
        self.entry_password.set_text(self.settings.password)
        self.entry_password.set_sensitive(settings.sync_global)

        # Synchronize feed check button
        self.synchronize_feeds = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.synchronize_feeds.set_label(_('Synchronize with Google Reader'))
        list_panel.pack_start(self.synchronize_feeds)
        self.synchronize_feeds.set_active(self.settings.sync_global)
        self.synchronize_feeds.show()

        caption = gtk.HBox(False, 16)
        label_user = gtk.Label(_('User') + ':')
        label_user.set_alignment(xalign=0, yalign=0.5)
        label_password = gtk.Label(_('Password') + ':')
        label_password.set_alignment(xalign=0, yalign=0.5)

        labels_group = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)
        labels_group.add_widget(label_user)
        labels_group.add_widget(label_password)

        caption = gtk.HBox(False, 16)
        caption.pack_start(label_user, False, False)
        caption.pack_start(self.entry_user, True, True)
        list_panel.pack_start(caption)

        caption = gtk.HBox(False, 16)
        caption.pack_start(label_password, False, False)
        caption.pack_start(self.entry_password, True, True)
        list_panel.pack_start(caption)

        self._force_sync_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._force_sync_button.set_label(_('Synchronize Now'))
        self._force_sync_button.set_sensitive(self.settings.sync_global)
        list_panel.pack_start(self._force_sync_button)

        # Connect signals
        self._force_sync_button.connect('clicked', self._settings_sync_google_reader_cb)
        self.synchronize_feeds.connect('clicked', self._synchronize_feeds_cb)

    def _settings_sync_google_reader_cb(self, button):
        self.response(self.RESPONSE_SYNC_GOOGLE)

    def _auto_download_cb(self, button):
        self.picker_download_folder.set_sensitive(button.get_active())

    def _download_folder_cb(self, button):
        stack = hildon.WindowStack.get_default()
        window = stack.peek()

        chooser = hildon.FileChooserDialog(window, gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)
        chooser.set_transient_for(window)
        chooser.set_title(_('Select a folder for automatic downloads'))
        chooser.set_current_folder(constants.MYDOCS_DIR)

        chooser.set_default_response(gtk.RESPONSE_OK)

        response = chooser.run()
        folder = chooser.get_filename() + '/'
        chooser.destroy()

        self.settings.auto_download_folder = folder
        button.set_value(folder)

    def _synchronize_feeds_cb(self, button):
        enabled = button.get_active()
        self.entry_user.set_sensitive(enabled)
        self.entry_password.set_sensitive(enabled)
        self._force_sync_button.set_sensitive(enabled)

    def jump_to_google_reader_auth(self):
        self.panel.jump_to_child(self.entry_user)

class NewFeedDialog(gtk.Dialog):

    def __init__(self, parent):
        super(NewFeedDialog, self).__init__(_('New Feed'), parent,
                                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            (gtk.STOCK_ADD, gtk.RESPONSE_ACCEPT))

        self.entry = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL)
        self.entry.connect('changed', self._entry_changed_cb)

        caption = gtk.HBox(False, 16)
        caption.pack_start(gtk.Label(_('Address') + ':'), False, False)
        caption.pack_start(self.entry)
        self.vbox.add(caption)

        self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

    def _entry_changed_cb(self, entry):
        if len(entry.get_text()) > 0:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, True)
        else:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

class FindFeedsDialog(gtk.Dialog):
    def __init__(self, parent):
        super(FindFeedsDialog, self).__init__(_('Find Feeds'), parent,
                                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            (_('Find'), gtk.RESPONSE_ACCEPT))


        self.entry = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL)
        self.entry.connect('changed', self._entry_changed_cb)

        caption = gtk.HBox(False, 16)
        caption.pack_start(gtk.Label(_('Keywords') + ':'), False, False)
        caption.pack_start(self.entry)
        self.vbox.add(caption)

        self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

    def _entry_changed_cb(self, entry):
        if len(entry.get_text()) > 0:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, True)
        else:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

    def _delete_event_cb(self, widget, event):
        return True #stop the signal propagation

class FeedsWindow(hildon.StackableWindow):

    FEEDS_MODEL_PIXBUF_COLUMN, FEEDS_MODEL_TITLE_COLUMN, FEEDS_MODEL_SUBTITLE_COLUMN, \
        FEEDS_MODEL_READ_COLUMN, FEEDS_MODEL_VISITS_COLUMN, FEEDS_MODEL_SYNC_COLUMN, \
        FEEDS_MODEL_ID_COLUMN, FEEDS_MODEL_HREF_COLUMN = range(8)

    def __init__(self, manager, settings, conn_manager):
        super(FeedsWindow, self).__init__()

        self.manager = manager
        self.settings = settings
        self._conn_manager = conn_manager
        self._conn_manager.connect('connection-changed', self._on_connection_changed)

        # Feeds
        self.view = FeedsView(gtk.HILDON_UI_MODE_NORMAL)
        self.view.set_model (gtk.ListStore(gtk.gdk.Pixbuf, str, str, str, int, 'gboolean', int, str))
        self.view.connect ("row-activated", self._on_feed_activated)
        self.view.show()

        # Used for osso context in rotation object
        app_name = constants.RSS_NAME
        app_version = constants.RSS_VERSION

        # Auto rotate
        self.rotation_object = FremantleRotation(app_name, self, app_version,
                                                 FremantleRotation.AUTOMATIC)

        # Edit toolbar
        self.edit_toolbar_button_handler = 0
        self.edit_toolbar = hildon.EditToolbar()
        self.edit_toolbar.connect('arrow-clicked', self._restore_normal_mode)

        self.align = gtk.Alignment(0, 0, 1, 1)
        self.align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_MINIMUM)
        pannable.show()

        self.align.add(pannable)
        self.add(self.align)
        self.align.show()

        self._create_menu()
        self.set_title(app_name)
        self.connect('configure-event', self._on_configuration_changed)
        self.connect('delete-event', self._on_exit)

        # Apply settings
        self.settings.load()
        self._sort(self.settings.feeds_order)
        if self.settings.feeds_order == constants.DESCENDING_ORDER:
            self.descending_filter_button.set_active(True)
        elif self.settings.feeds_order == constants.VISITS_ORDER:
            self.visits_filter_button.set_active(True)

        # Force online/offline status evaluation
        self._on_connection_changed(self._conn_manager)

        # Load feed summaries (could be none)
        try:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)
            self._load_tag = glib.timeout_add(1000, self._show_loading_banner)
            self.manager.load_feeds_summary(self._feeds_summary_loaded)
        except Exception:
            hildon.hildon_gtk_window_set_progress_indicator(self, False)
            glib.source_remove(self._load_tag)
            self._show_initial_dialog()

    def _show_loading_banner(self):
            hildon.hildon_banner_show_information(self, '', _('Loading feed data'))
            return False

    def _on_connection_changed(self, conn_manager):
        if self._conn_manager.is_online():
            self._new_feed_button.show()
            self._update_all_button.show()
            self._find_feeds_button.show()
            self._import_feeds_button.show()
        else:
            self._new_feed_button.hide()
            self._update_all_button.hide()
            self._find_feeds_button.hide()
            self._import_feeds_button.hide()

    def _sort(self, order):
        if (order == constants.ASCENDING_ORDER):
            self.view.get_model().set_sort_column_id(self.FEEDS_MODEL_TITLE_COLUMN,
                                                     gtk.SORT_ASCENDING)
        elif (order == constants.DESCENDING_ORDER):
            self.view.get_model().set_sort_column_id(self.FEEDS_MODEL_TITLE_COLUMN,
                                                     gtk.SORT_DESCENDING)
        else:
            self.view.get_model().set_sort_column_id(self.FEEDS_MODEL_VISITS_COLUMN,
                                                     gtk.SORT_DESCENDING)

    def _add_dummy_feed(self, url, sync=False):

        store = self.view.get_model()
        feed_iter = store.append()
        store.set(feed_iter,
                  self.FEEDS_MODEL_PIXBUF_COLUMN, None,
                  self.FEEDS_MODEL_TITLE_COLUMN, get_feed_title_markup(_('Retrieving info...')),
                  self.FEEDS_MODEL_SUBTITLE_COLUMN, get_feed_subtitle_markup(url),
                  self.FEEDS_MODEL_READ_COLUMN, '',
                  self.FEEDS_MODEL_VISITS_COLUMN, 0,
                  self.FEEDS_MODEL_SYNC_COLUMN, sync,
                  self.FEEDS_MODEL_ID_COLUMN, self.view.DUMMY_FEED_STATUS,
                  self.FEEDS_MODEL_HREF_COLUMN, url)

        return feed_iter

    def _show_entries_window(self, path):
        feed_iter = self.view.get_model().get_iter(path)
        feed_id = self.view.get_model().get_value(feed_iter, self.FEEDS_MODEL_ID_COLUMN)
        summary = self.manager.get_feed_summary(feed_id)
        feed_data = self.manager.get_feed_data(feed_id)

        news_window = EntriesWindow(feed_data, self.manager, self.settings, self._conn_manager)
        news_window.show()

        # When the news window is closed (in Maemo5 it's really
        # hidden) reset the unread cache. Get the row reference BEFORE
        # setting the visits count. Otherwise we will get the wrong
        # one
        news_window.connect('hide', self._on_news_window_closed,
                            gtk.TreeRowReference(self.view.get_model(), path))

        # Update the visits count
        summary.visits += 1
        self.view.get_model().set_value(feed_iter, self.FEEDS_MODEL_VISITS_COLUMN, summary.visits)

    def _on_feed_activated(self, treeview, path, column):
        feed_iter = self.view.get_model().get_iter(path)
        feed_id = self.view.get_model().get_value(feed_iter, self.FEEDS_MODEL_ID_COLUMN)

        # Check that we're not trying to open a dummy feed
        if feed_id == self.view.DUMMY_FEED_STATUS:
            hildon.hildon_banner_show_information(self, '', _('Wait until feed is refreshed'))
            return

        feed_data = self.manager.get_feed_data(feed_id)
        # Load feed data on demand
        if not feed_data:
            row_reference = gtk.TreeRowReference(self.view.get_model(), path)
            self.manager.load(feed_id, self._feed_data_loaded_cb, row_reference)
            return

        self._show_entries_window(path)

    def _feed_data_loaded_cb(self, feed_data, row_reference, error):
        if not feed_data:
            hildon.hildon_banner_show_information(self, '', _('Error loading feed data'))
            return

        self._show_entries_window(row_reference.get_path())

    def _on_news_window_closed(self, news_window, row_reference):
        feed_iter = self.view.get_model().get_iter(row_reference.get_path())
        feed_id = self.view.get_model().get_value(feed_iter, self.FEEDS_MODEL_ID_COLUMN)
        feed_data = self.manager.get_feed_data(feed_id)

        # Reset unread cache (the value have most likely changed).
        unread_count = len([entry for entry in feed_data.entries if entry.read == False])
        self.view.get_model().set(feed_iter, self.FEEDS_MODEL_READ_COLUMN,
                                  get_visual_unread_text(unread_count))

        # Update sync status (could have changed)
        summary = self.manager.get_feed_summary(feed_id)
        self.view.get_model().set(feed_iter, self.FEEDS_MODEL_SYNC_COLUMN, summary.sync)

    def _on_configuration_changed(self, window, event):
        # Change alignment padding. We don't want padding in portrait
        # Need to save space
        if event.width > event.height:
            self.align.set_padding(4, 0 , 16, 16)
        else:
            self.align.set_padding(4, 0 , 4, 4)

    def _on_exit_idle(self, data):
        # Explicitly remove the idle as this callback could last too
        # much and thus it could be triggered more than once
        glib.source_remove(self._exit_idle_handler)

        # Wait until all tasks are canceled
        self.manager.stop()
        self.manager.halt()

        self.settings.save()

        # Save to disk without using workers
        self.manager.save_sync()
        gobject.idle_add(gtk.main_quit, None)
        return True

    def _on_exit(self, window, event):
        # Hide window
        self.hide()

        # We do the final operations in an idle in order to let the window close
        self._exit_idle_handler = gobject.idle_add(self._on_exit_idle, None)
        return True

    def _create_menu(self):
        menu = hildon.AppMenu()

        # Sorting filter
        self.ascending_filter_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.ascending_filter_button.set_mode(False)
        self.ascending_filter_button.set_label(_('A-Z'))
        self.ascending_filter_button.connect('toggled', self._sort_ascending_cb)
        menu.add_filter(self.ascending_filter_button)
        self.descending_filter_button = \
            hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                  group = self.ascending_filter_button)
        self.descending_filter_button.set_mode(False)
        self.descending_filter_button.set_label(_('Z-A'))
        self.descending_filter_button.connect('toggled', self._sort_descending_cb)
        menu.add_filter(self.descending_filter_button)
        self.visits_filter_button = \
            hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                  group = self.ascending_filter_button)
        self.visits_filter_button.set_mode(False)
        self.visits_filter_button.set_label(_('Favorites'))
        self.visits_filter_button.connect('toggled', self._sort_visits_cb)
        menu.add_filter(self.visits_filter_button)

        self._new_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._new_feed_button.set_label(_('New Feed'))
        self._new_feed_button.connect('clicked', self._new_feed_cb)
        menu.append(self._new_feed_button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Remove Feed'))
        button.connect('clicked', self._remove_feed_cb)
        menu.append(button)

        self._update_all_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._update_all_button.set_label(_('Update all'))
        self._update_all_button.connect('clicked', self._update_all_cb)
        menu.append(self._update_all_button)

        self._import_feeds_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._import_feeds_button.set_label(_('Import Feeds'))
        self._import_feeds_button.connect('clicked', self._import_feed_cb)
        menu.append(self._import_feeds_button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Export Feeds'))
        button.connect('clicked', self._export_feed_cb)
        menu.append(button)
        self.export_opml_button = button

        self._find_feeds_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._find_feeds_button.set_label(_('Find Feeds'))
        self._find_feeds_button.connect('clicked', self._find_feed_cb)
        menu.append(self._find_feeds_button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Settings'))
        button.connect('clicked', self._preferences_cb)
        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _new_feed_cb(self, button):
        dialog = NewFeedDialog(self)
        dialog.connect('response', self._new_feed_response_cb)
        dialog.show_all()

    def _remove_feed_cb(self, button):

        # tree view edit mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_EDIT)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.set_label(_('Select Feeds to remove'))
        self.edit_toolbar.set_button_label(_('Remove'))
        if self.edit_toolbar.handler_is_connected (self.edit_toolbar_button_handler):
            self.edit_toolbar.disconnect(self.edit_toolbar_button_handler)
        self.edit_toolbar_button_handler = \
            self.edit_toolbar.connect('button-clicked', self._remove_button_clicked_cb)

        self.set_edit_toolbar(self.edit_toolbar)
        self.edit_toolbar.show()
        self.fullscreen()

    def _remove_button_clicked_cb(self, button):
        selection = self.view.get_selection()
        selected_rows = selection.get_selected_rows()
        model, paths = selected_rows
        if not paths:
            hildon.hildon_banner_show_information(self, '', _('Please select at least one Feed'))
            return

        removed = []
        unsubscribe = []
        for path in paths:
            try:
                self.manager.remove_feed(model[path][FeedsWindow.FEEDS_MODEL_ID_COLUMN])
            except IOError:
                print 'Could not remove', constants.RSS_DB_FILE
                hildon.hildon_banner_show_information(self, '', _('File error while removing Feed'))
            else:
                removed.append(gtk.TreeRowReference(model, path))
                # Remove from Google Reader if needed
                if model[path][FeedsWindow.FEEDS_MODEL_SYNC_COLUMN]:
                    unsubscribe.append(model[path][FeedsWindow.FEEDS_MODEL_HREF_COLUMN])

        if not len(model):
            self.export_opml_button.hide()

        if len(unsubscribe):
            message = gettext.ngettext('Do you want to remove the selected feed from Google Reader?',
                                       'Do you want to remove the selected feeds from Google Reader?',
                                       len(unsubscribe))
            note = hildon.hildon_note_new_confirmation(self, message)
            response = gtk.Dialog.run(note)
            note.destroy()
            if response == gtk.RESPONSE_OK:
                for feed_url in unsubscribe:
                    self.manager.subscribe_feed_google(feed_url, False)

        for reference in removed:
            model.remove(model.get_iter(reference.get_path()))

        # restore normal mode
        self._restore_normal_mode(button)

    def _restore_normal_mode(self, button):

        # tree view normal mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_NORMAL)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.hide()
        self.unfullscreen()

    def _preferences_cb(self, button):
        dialog = SettingsDialog(self, self.settings)
        dialog.connect('response', self._preferences_response_cb)
        dialog.show_all()

    def _multiple_update_cb(self, retval, user_data, error, static = {"count" : 0}):
        ''' Simulating a static variable with a default argument hack '''
        num_operations, callback, data, row_reference = user_data

        if row_reference:
            store = self.view.get_model()
            feed_iter = store.get_iter(row_reference.get_path())
            feed_id = store.get_value(feed_iter, self.FEEDS_MODEL_ID_COLUMN)
            summary = self.manager.get_feed_summary(feed_id)
            feed_data = self.manager.get_feed_data(feed_id)
            if feed_data:
                unread_count = len([entry for entry in feed_data.entries if entry.read == False])
                store.set_value(feed_iter, self.FEEDS_MODEL_READ_COLUMN, \
                                    get_visual_unread_text(unread_count))
            hildon.hildon_banner_show_information(self, '', 'Updated ' + feed_data.feed.title)

        static['count'] += 1
        # If all add operations have finished then call the all-done-callback
        if static['count'] == num_operations:
            hildon.hildon_gtk_window_set_progress_indicator(self, False)
            static['count'] = 0
            if callback:
                callback(data)
        else:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)

    def _update_multiple_feeds(self, callback=None, data=None):
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        store = self.view.get_model()
        num_rows = len(store)
        feed_iter = store.get_iter_first()
        while feed_iter:
            # Create a row reference
            row_reference = gtk.TreeRowReference(store, store.get_path(feed_iter))
            # Update feed
            self.manager.update_feed(store.get_value(feed_iter, self.FEEDS_MODEL_ID_COLUMN),\
                                         self._multiple_update_cb,\
                                         (num_rows, callback, data, row_reference))
            feed_iter = store.iter_next(feed_iter)

    def _update_all_cb(self, button):
        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Ask manager to update feed
        self._update_multiple_feeds(self._all_feeds_updated_cb)

    def _all_feeds_updated_cb(self, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        # Do not update read/unread status if all the feeds failed to sync
        synced_feeds = [row[self.FEEDS_MODEL_ID_COLUMN] for row in self.view.get_model() \
                            if row[self.FEEDS_MODEL_SYNC_COLUMN]]
        if len(synced_feeds) == 0:
            return

        # Update read/unread status
        hildon.hildon_banner_show_information(self, '', _('Synchronizing read/unread status with Google Reader'))
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        self.manager.sync_google_reader_read_status(self._sync_google_reader_read_status_cb)

    def _new_feed_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_ACCEPT:

            url = dialog.entry.get_text()

            # Quickly show feedback to user
            hildon.hildon_gtk_window_set_progress_indicator(self, True)

            # Insert a dummy row while information is retrieved
            feed_iter = self._add_dummy_feed(url)
            path = self.view.get_model().get_path(feed_iter)
            row_reference = gtk.TreeRowReference(self.view.get_model(), path)

            # Add feed to manager. Do not sync with Google
            self.manager.add_feed(url, False, self._feed_added_cb, row_reference)

            dialog.destroy()

    def _feed_added_cb(self, pixbuf_and_data, row_reference=None, error=None, stop_progress=True):

        # Remove progress information
        if stop_progress:
            hildon.hildon_gtk_window_set_progress_indicator(self, False)

        store = self.view.get_model()
        if row_reference != None:
            feed_iter = store.get_iter(row_reference.get_path())
        else:
            feed_iter = store.append()

        # Check that the feed data was retrieved
        if error:
            message = _('Could not retrieve feed')
            href = store.get_value (feed_iter, self.FEEDS_MODEL_HREF_COLUMN)
            if href:
                message += ' from ' + href
            store.remove(feed_iter)
            hildon.hildon_banner_show_information(self, '', message)
            return

        pixbuf, new_feed_data = pixbuf_and_data
        self.export_opml_button.show()

        # Update model
        feed_id = get_feed_id(new_feed_data)
        summary = self.manager.get_feed_summary(feed_id)
        subtitle = get_feed_subtitle(new_feed_data)
        store.set(feed_iter,
                  self.FEEDS_MODEL_PIXBUF_COLUMN, pixbuf,
                  self.FEEDS_MODEL_TITLE_COLUMN, get_feed_title_markup(new_feed_data.feed.title),
                  self.FEEDS_MODEL_SUBTITLE_COLUMN, get_feed_subtitle_markup(subtitle),
                  self.FEEDS_MODEL_READ_COLUMN, get_visual_unread_text(len(new_feed_data.entries)),
                  self.FEEDS_MODEL_VISITS_COLUMN, summary.visits,
                  self.FEEDS_MODEL_SYNC_COLUMN, summary.sync,
                  self.FEEDS_MODEL_ID_COLUMN, feed_id,
                  self.FEEDS_MODEL_HREF_COLUMN, new_feed_data.href)

    def _multiple_add_cb(self, pixbuf_and_data, user_data, error, static = {"count" : 0}):
        ''' Simulating a static variable with a default argument hack '''
        num_operations, callback, data, row_reference = user_data

        self._feed_added_cb(pixbuf_and_data, row_reference, error, stop_progress=False)
        static['count'] += 1
        # If all add operations have finished then call the all-done-callback
        if static['count'] == num_operations:
            hildon.hildon_gtk_window_set_progress_indicator(self, False)
            static['count'] = 0
            if callback:
                callback(data)
        else:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)

    def _add_multiple_feeds(self, urls, sync=False, callback=None, data=None):
        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        for url in urls:
            # Insert a dummy row while information is retrieved
            feed_iter = self._add_dummy_feed(url, sync)
            path = self.view.get_model().get_path(feed_iter)
            row_reference = gtk.TreeRowReference(self.view.get_model(), path)

            # Add feed to manager
            self.manager.add_feed(url, sync, self._multiple_add_cb,
                                  (len(urls), callback, data, row_reference))

    def _sync_google_reader_read_status_cb(self, synced, user_data, error):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        if not synced:
            hildon.hildon_banner_show_information(self, '', _('Error syncing read status with Google Reader'))
            return

        # TODO: Update counters
        store = self.view.get_model()
        for row in store:
            if row[self.FEEDS_MODEL_SYNC_COLUMN]:
                feed_data = self.manager.get_feed_data(row[self.FEEDS_MODEL_ID_COLUMN])
                unread_count = len([entry for entry in feed_data.entries if entry.read == False])
                row[self.FEEDS_MODEL_READ_COLUMN] = get_visual_unread_text(unread_count)

    def _all_synchronized_feeds_added_cb(self, user_data):
        ''' Called after adding all the feeds retrieved from Google Reader '''

        # Do not update read/unread status if all the feeds failed to sync
        synced_feeds = 0
        for row in self.view.get_model():
            if row[self.FEEDS_MODEL_ID_COLUMN]:
                synced_feeds += 1

        if not synced_feeds:
            return

        # Update read/unread status
        hildon.hildon_banner_show_information(self, '', _('Synchronizing read/unread status with Google Reader'))
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        self.manager.sync_google_reader_read_status(self._sync_google_reader_read_status_cb)

    def _feeds_synchronized(self, urls, user_data, error):

        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        if urls == None:
            hildon.hildon_banner_show_information(self, '', _('Error synchronizing feeds'))
            return
        elif urls == []:
            hildon.hildon_banner_show_information(self, '', _('No feeds in Google Reader to synchronize'))
            return

        # Update sync status of current feeds. We use the info from
        # the model in order to properly update the UI
        store = self.view.get_model()
        for row in store:
            summary = self.manager.get_feed_summary(row[self.FEEDS_MODEL_ID_COLUMN])
            if summary.href in urls:
                if not summary.sync:
                    summary.sync = True
                    row[self.FEEDS_MODEL_SYNC_COLUMN] = True
                urls.remove(summary.href)
            else:
                if summary.sync:
                    summary.sync = False
                    row[self.FEEDS_MODEL_SYNC_COLUMN] = False

        if urls:
            # Add feeds to UI asynchronously
            self._add_multiple_feeds(urls, sync=True, callback=self._all_synchronized_feeds_added_cb)

    def _preferences_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_DELETE_EVENT:
            dialog.destroy()
            return

        if response == SettingsDialog.RESPONSE_SYNC_GOOGLE:
            if self.settings.user=='' or self.settings.password=='':
                hildon.hildon_banner_show_information(self, '', _('Missing user name and/or password'))
                return

        # Save settings. We do this if the users click Save or Synchronize Now
        self.settings.auto_load_images = dialog.auto_load_images.get_active()
        self.settings.rotation_mode = dialog.selector_orientation.get_active(0)
        self.settings.default_font_size = int(dialog.selector_font_size.get_current_text())
        self.settings.auto_download = dialog.automatic_download.get_active()
        self.settings.user = dialog.entry_user.get_text()
        self.settings.password = dialog.entry_password.get_text()
        sync_global_before = self.settings.sync_global
        self.settings.sync_global = dialog.synchronize_feeds.get_active()
        self.settings.delete_old = dialog.delete_old.get_active()
        self.settings.auto_update_startup = dialog.auto_update_startup.get_active()

        dialog.destroy()

        # Update rotation
        self.rotation_object.set_mode(self.settings.rotation_mode)

        if (self.settings.sync_global and not sync_global_before) or \
                response == SettingsDialog.RESPONSE_SYNC_GOOGLE:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)
            self.manager.sync_with_google_reader(self._feeds_synchronized)

    def _feed_exported_cb(self, retval, user_data, error):
        if not error:
            message = _('Feeds exported')
        else:
            message = _('Error exporting feeds')
        hildon.hildon_banner_show_information(self, '', message)
        # Remove progress information
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def _initial_dialog_new_cb(self, button, dialog):
        dialog.hide()
        self._new_feed_cb(button)
        dialog.destroy()

    def _initial_dialog_import_cb(self, button, dialog):
        dialog.hide()
        self._import_feed_cb(button)
        dialog.destroy()

    def _initial_dialog_find_cb(self, button, dialog):
        dialog.hide()
        self._find_feed_cb(button)
        dialog.destroy()

    def _initial_dialog_sync_google_cb(self, button, dialog):
        dialog.destroy()

        # Let's assume that the user wants to enable Google Reader synchronization
        self.settings.sync_global = True

        if self.settings.user=='' or self.settings.password=='':
            hildon.hildon_banner_show_information(self, '', _('Missing user name and/or password'))
            dialog = SettingsDialog(self, self.settings)
            dialog.connect('response', self._preferences_response_cb)
            dialog.show_all()
            dialog.jump_to_google_reader_auth()
        else:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)
            hildon.hildon_banner_show_information(self, '', _('Connecting to Google Reader'))
            self.manager.sync_with_google_reader(self._feeds_synchronized)

    def _show_initial_dialog(self):
        dialog = gtk.Dialog()
        dialog.set_title(_('Start adding feeds'))

        # Create buttons
        add_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        add_feed_button.set_label(_('New feed'))
        add_feed_button.connect('clicked', self._initial_dialog_new_cb, dialog)

        import_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        import_feed_button.set_label(_('Import Feeds'))
        import_feed_button.connect('clicked', self._initial_dialog_import_cb, dialog)

        find_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        find_feed_button.set_label(_('Find Feeds'))
        find_feed_button.connect('clicked', self._initial_dialog_find_cb, dialog)

        sync_google_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        sync_google_button.set_label(_('Get from Google Reader'))
        sync_google_button.connect('clicked', self._initial_dialog_sync_google_cb, dialog)

        dialog.vbox.add(add_feed_button)
        dialog.vbox.add(import_feed_button)
        dialog.vbox.add(find_feed_button)
        dialog.vbox.add(sync_google_button)

        add_feed_button.show()
        import_feed_button.show()
        find_feed_button.show()
        sync_google_button.show()
        dialog.show()

    def _feeds_summary_loaded(self, feeds_summary, user_data, error):
        glib.source_remove(self._load_tag)
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        # If there is no data then show a dialog
        if not feeds_summary:
            self._show_initial_dialog()
            return

        # Iterate over summaries and fill the model
        store = self.view.get_model()
        for summary in feeds_summary:
            feed_iter = store.append()
            store.set(feed_iter,
                      self.FEEDS_MODEL_TITLE_COLUMN, get_feed_title_markup(summary.title),
                      self.FEEDS_MODEL_SUBTITLE_COLUMN, get_feed_subtitle_markup(summary.subtitle),
                      self.FEEDS_MODEL_READ_COLUMN, get_visual_unread_text(summary.unread),
                      self.FEEDS_MODEL_VISITS_COLUMN, summary.visits,
                      self.FEEDS_MODEL_SYNC_COLUMN, summary.sync,
                      self.FEEDS_MODEL_ID_COLUMN, summary.feed_id,
                      self.FEEDS_MODEL_HREF_COLUMN, summary.href)
            # Favicon might require a network download
            path = self.view.get_model().get_path(feed_iter)
            row_reference = gtk.TreeRowReference(store, path)
            pixbuf = self.manager.get_favicon(summary.favicon, self._get_favicon_cb, row_reference)

        # Auto-update feeds
        if self.settings.auto_update_startup:
            self._update_multiple_feeds(self._all_feeds_updated_cb)

    def _get_favicon_cb(self, pixbuf, row_reference, error):
        if pixbuf == None:
            return

        store = self.view.get_model()
        feed_iter = store.get_iter(row_reference.get_path())
        store.set(feed_iter, self.FEEDS_MODEL_PIXBUF_COLUMN, pixbuf)

    def _find_feed_response_cb(self, dialog, response):
        if response == gtk.RESPONSE_ACCEPT:
            keywords = dialog.entry.get_text()

            # Quickly show feedback to user
            hildon.hildon_gtk_window_set_progress_indicator(dialog, True)

            self.manager.find_feed(keywords, self._found_cb, dialog)
            dialog.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

            # Disable canceling
            dialog.connect('delete-event', lambda widget,event : True)

    def _save_feeds_after_multiple_add(self, data=None):
        self.manager.save()

    def _find_window_urls_found_cb(self, find_window, urls):
        if len(urls):
            self._add_multiple_feeds(urls, callback=self._save_feeds_after_multiple_add)

    def _found_cb(self, feeds_info, dialog, error):
        if feeds_info == None:
            # Something went wrong
            hildon.hildon_banner_show_information(self, '', _('Error finding feeds. Server error.'))
            dialog.destroy()
            return

        news_window = FindWindow(feeds_info, self)
        news_window.show()

        news_window.connect('urls-found', self._find_window_urls_found_cb)

        dialog.set_response_sensitive(gtk.RESPONSE_ACCEPT, True)
        hildon.hildon_gtk_window_set_progress_indicator(dialog, False)

        dialog.destroy()

    def _sort_ascending_cb(self, button):
        if not button.get_active():
            return
        self.settings.feeds_order = constants.ASCENDING_ORDER
        self._sort(constants.ASCENDING_ORDER)

    def _sort_descending_cb(self, button):
        if not button.get_active():
            return
        self.settings.feeds_order = constants.DESCENDING_ORDER
        self._sort(constants.DESCENDING_ORDER)

    def _sort_visits_cb(self, button):
        if not button.get_active():
            return
        self.settings.feeds_order = constants.VISITS_ORDER
        self._sort(constants.VISITS_ORDER)

    def _export_feed_cb(self, button):
        stack = hildon.WindowStack.get_default()
        window = stack.peek()

        chooser = hildon.FileChooserDialog(window, gtk.FILE_CHOOSER_ACTION_SAVE)
        chooser.set_transient_for(window)
        # Don't like this at all. Isn't there any define in the platform ?
        chooser.set_current_folder(constants.MYDOCS_DIR)
        chooser.set_current_name('resistance-feeds')

        #Overwrite the file if already exists
        chooser.set_do_overwrite_confirmation(True)
        chooser.set_default_response(gtk.RESPONSE_OK)

        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)
            self.manager.export_opml(chooser.get_filename() + '.opml', self._feed_exported_cb)
        elif response == gtk.RESPONSE_CANCEL:
            print 'Closed, no files selected'
        chooser.destroy()

    def _import_feed_cb(self, button):
        stack = hildon.WindowStack.get_default()
        window = stack.peek()

        #Calling a file chooser
        chooser = hildon.FileChooserDialog(window ,gtk.FILE_CHOOSER_ACTION_OPEN)
        chooser.set_transient_for(window)
        # Don't like this at all. Isn't there any define in the platform ?
        chooser.set_current_folder(constants.MYDOCS_DIR)
        chooser.set_default_response(gtk.RESPONSE_OK)

        #Filter for opml files
        filter = gtk.FileFilter()
        filter.set_name("OPML")
        filter.add_pattern("*.opml")
        chooser.add_filter(filter)

        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            hildon.hildon_gtk_window_set_progress_indicator(self, True)
            self.manager.import_opml(chooser.get_filename(), self._import_opml_cb)
        elif response == gtk.RESPONSE_CANCEL:
            print 'Closed, no files selected'

        chooser.destroy()

    def _import_opml_cb(self, feed_url_list, data, error):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        if error:
            hildon.hildon_banner_show_information(self, '', _('Error importing feeds OPML file'))
            return

        if len(feed_url_list) == 0:
            hildon.hildon_banner_show_information(self, '', _('No feeds to import in OPML file'))
            return

        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        self._add_multiple_feeds(feed_url_list, callback=self._save_feeds_after_multiple_add)

    def _find_feed_cb(self, button):
        dialog = FindFeedsDialog(self)
        dialog.connect('response', self._find_feed_response_cb)
        dialog.show_all()

class EntriesView(hildon.GtkTreeView):

    def __init__(self, feed_title, ui_mode):
        super(EntriesView, self).__init__(ui_mode)

        self.fallback_author = feed_title

        # Add columns
        pix_renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = 0)
        self.append_column(column);

        text_renderer = gtk.CellRendererText()
        text_renderer.set_property('ellipsize', ELLIPSIZE_END)
        column = gtk.TreeViewColumn('Name', text_renderer, markup = EntriesWindow.ENTRIES_MODEL_TEXT_COLUMN)
        column.set_expand(True)
        self.append_column(column);

        date_renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Date', date_renderer, markup = EntriesWindow.ENTRIES_MODEL_DATE_COLUMN)
        column.set_expand(False)
        self.append_column(column);

    def get_visual_entry_text(self, entry):

        # TODO: show entry.summary ?
        author = get_author_from_item(entry)
        if author == '':
            author = self.fallback_author

        words = entry.title.rsplit(":")
        if len(words) > 1 and author == words[0].lstrip():
            title = words[1].lstrip()
        else:
            title = entry.title

        if 'read' in entry and entry['read'] == True:
            color = SECONDARY_TEXT_COLOR
        else:
            color = ACTIVE_TEXT_COLOR

        return '<span foreground="' + color + '">' \
            + gobject.markup_escape_text(unescape(author)) \
            + '</span>\n<span size="small">' \
            + gobject.markup_escape_text(unescape(title)) + '</span>'

    def get_visual_entry_date(self, entry):
        if entry.has_key('updated_parsed') == False:
            if entry.has_key('updated'):
                return '<span size="xx-small">%s</span>' % \
                    entry.updated
            else:
                return ''

        now = time.localtime()

        if now.tm_year == entry.updated_parsed.tm_year:
            # Today
            if now.tm_yday == entry.updated_parsed.tm_yday:
                return '<span size="xx-small">%s</span>' % \
                    time.strftime('%H:%M', entry.updated_parsed)
            # This week
            elif now.tm_yday - entry.updated_parsed.tm_yday < 7 and \
                    entry.updated_parsed.tm_wday < now.tm_wday:
                return '<span size="xx-small">%s</span>' % \
                    time.strftime('%a', entry.updated_parsed)
            # This year
            else:
                return '<span size="xx-small">%s</span>' % \
                    time.strftime('%d %b', entry.updated_parsed)
        # Anything else
        else:
            return '<span size="xx-small">%s</span>' % \
                time.strftime('%x', entry.updated_parsed)

class EntriesWindow(hildon.StackableWindow):

    ENTRIES_MODEL_TEXT_COLUMN = 1
    ENTRIES_MODEL_DATA_COLUMN = 2
    ENTRIES_MODEL_DATE_COLUMN = 3
    ENTRIES_MODEL_DATE_PARSED_COLUMN = 4

    def __init__(self, feed_data, manager, settings, conn_manager):
        super(EntriesWindow, self).__init__()

        self.manager = manager
        self.settings = settings
        self.feed_data = feed_data
        self._conn_manager = conn_manager

        self.set_title(unescape(self.feed_data.feed.title))
        self.toolbar = gtk.Toolbar()

        # Edit toolbar
        self.edit_toolbar_button_handler = 0
        self.edit_toolbar = hildon.EditToolbar()
        self.edit_toolbar.connect('arrow-clicked', self._restore_normal_mode)

        # Feeds
        self.view = EntriesView(feed_data.feed.title, gtk.HILDON_UI_MODE_NORMAL)
        entries_model = gtk.ListStore(gtk.gdk.Pixbuf, str, gobject.TYPE_PYOBJECT, str, int)
        entries_model.set_sort_column_id(self.ENTRIES_MODEL_DATE_PARSED_COLUMN, gtk.SORT_DESCENDING)
        filter_model = entries_model.filter_new()
        filter_model.set_visible_func(self._row_visible_cb)
        self.view.set_model (filter_model)
        self.view.connect ("row-activated", self._on_entry_activated)
        self.view.show()

        # Draw entries
        self._add_entries()

        self.align = gtk.Alignment(0, 0, 1, 1)
        self.align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        self.align.add(pannable)
        self.add(self.align)
        self.align.show()

        # Update feed button
        action_box = hildon.GtkTreeView.get_action_area_box(self.view)
        self._update_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._update_feed_button.set_label(_('Update feed'))
        self._update_feed_button.connect('clicked', self._update_feed_cb)

        action_box.pack_start(self._update_feed_button, True, True)

        self._create_menu()

        if 'itunes' not in self.feed_data.namespaces:
            self._download_all_button.hide()

        self.connect('configure-event', self._on_configuration_changed)
        self.connect('hide', self._on_hide)

        # Force online/offline status evaluation
        self._connection_changed_id = self._conn_manager.connect('connection-changed', self._on_connection_changed)
        self._on_connection_changed(self._conn_manager)

    def _on_hide(self, window):
        # Do not listen for conn-changed events anymore. Causes
        # crashes when changing the connection after closing the
        # window
        if self._conn_manager.handler_is_connected(self._connection_changed_id):
            self._conn_manager.disconnect(self._connection_changed_id)

    def _row_visible_cb(self, model, iter):
        if self.settings.entries_filter == constants.SHOW_ALL_FILTER:
            return True

        entry = model.get_value(iter, self.ENTRIES_MODEL_DATA_COLUMN)
        if not entry:
            return False

        return not entry.read

    def _add_entries(self):
        model = self.view.get_model().get_model()
        for entry in self.feed_data.entries:
            entry_iter = model.append()
            if 'updated_parsed' in entry:
                date = entry.updated_parsed
            elif 'published_parsed' in entry:
                date = entry.published_parsed
            elif 'created_parsed' in entry:
                date = entry.created_parsed
            else:
                date = 0
            model.set(entry_iter,
                      self.ENTRIES_MODEL_TEXT_COLUMN, self.view.get_visual_entry_text(entry),
                      self.ENTRIES_MODEL_DATA_COLUMN, entry,
                      self.ENTRIES_MODEL_DATE_COLUMN, self.view.get_visual_entry_date(entry),
                      self.ENTRIES_MODEL_DATE_PARSED_COLUMN, calendar.timegm(date) if date else 0)

    def _on_connection_changed(self, conn_manager):
        if conn_manager.is_online():
            self._update_feed_button.show()
            # Check that we should not show it
            if 'itunes' in self.feed_data.namespaces:
                self._download_all_button.show()
            self._sync_feed_button.show()
            self.view.set_action_area_visible(True)
        else:
            self._update_feed_button.hide()
            self._download_all_button.hide()
            self._sync_feed_button.hide()
            self.view.set_action_area_visible(False)

    def _create_menu(self):
        menu = hildon.AppMenu()

        # Sorting filter
        self._all_filter_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._all_filter_button.set_mode(False)
        self._all_filter_button.set_label(_('Show All'))
        self._all_filter_button.connect('toggled', self._show_all_cb)
        menu.add_filter(self._all_filter_button)

        self._unread_filter_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                           group = self._all_filter_button)
        self._unread_filter_button.set_mode(False)
        self._unread_filter_button.set_label(_('Show Unread'))
        self._unread_filter_button.connect('toggled', self._show_unread_cb)
        menu.add_filter(self._unread_filter_button)

        if self.settings.entries_filter == constants.SHOW_UNREAD_FILTER:
            self._unread_filter_button.set_active(True)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Mark Entries as Read'))
        button.connect('clicked', self._mark_read_cb, True)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Mark Entries as Not Read'))
        button.connect('clicked', self._mark_read_cb, False)
        menu.append(button)

        self._download_all_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._download_all_button.set_label(_('Download all'))
        self._download_all_button.connect('clicked', self._download_all_items_cb)
        menu.append(self._download_all_button)

        self._sync_feed_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))
        label = _('Remove from Google Reader') if summary.sync else _('Add to Google Reader')
        self._sync_feed_button.set_label(label)
        self._sync_feed_button.connect('clicked', self._subscribe_feed_cb)
        menu.append(self._sync_feed_button)

        menu.show_all()
        self.set_app_menu(menu)

    def _show_unread_cb(self, button):
        if not button.get_active():
            return
        self.settings.entries_filter = constants.SHOW_UNREAD_FILTER
        self.view.get_model().refilter()

    def _show_all_cb(self, button):
        if not button.get_active():
            return
        self.settings.entries_filter = constants.SHOW_ALL_FILTER
        self.view.get_model().refilter()

    def _subscribe_feed_cb(self, button):
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Disable buttons while subscription takes place
        self._update_feed_button.set_sensitive(False)
        self._sync_feed_button.set_sensitive(False)

        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))
        self.manager.subscribe_feed_google(summary.href, not summary.sync, self._feed_subscribed_cb)

    def _feed_subscribed_cb(self, synced, user_data, error):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))

        if not synced:
            if summary.sync:
                message = _('Error removing from Google Reader')
            else:
                message = _('Error adding to Google Reader')
        else:
            summary.sync = not summary.sync
            if summary.sync:
                message = _('Added to Google Reader')
            else:
                message = _('Removed from Google Reader')

        hildon.hildon_banner_show_information(self, '', message)

        # Update the menu
        label = _('Remove from Google Reader') if summary.sync else _('Add to Google Reader')
        self._sync_feed_button.set_label(label)

        # Restore sensitiviness
        self._sync_feed_button.set_sensitive(True)
        self._update_feed_button.set_sensitive(True)

    def _sync_google_reader_read_status_cb(self, synced, user_data, error):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        self._update_feed_button.set_sensitive(True)

        if not synced:
            hildon.hildon_banner_show_information(self, '', _('Error syncing read status with Google Reader'))
            return

    def _mark_read_cb(self, button, read):
        # tree view edit mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_EDIT)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().unselect_all()

        if read:
            self.edit_toolbar.set_label(_('Select Feeds to mark as Read'))
        else:
            self.edit_toolbar.set_label(_('Select Feeds to mark as Not Read'))
        self.edit_toolbar.set_button_label(_('Mark'))
        if self.edit_toolbar.handler_is_connected (self.edit_toolbar_button_handler):
            self.edit_toolbar.disconnect(self.edit_toolbar_button_handler)
        self.edit_toolbar_button_handler = \
            self.edit_toolbar.connect('button-clicked', self._mark_read_button_clicked_cb, read)

        self.set_edit_toolbar(self.edit_toolbar)
        self.edit_toolbar.show()
        self.fullscreen()

    def _mark_read_button_clicked_cb(self, button, read):
        selection = self.view.get_selection()
        selected_rows = selection.get_selected_rows()
        model, paths = selected_rows
        if not paths:
            hildon.hildon_banner_show_information(self, '', _('Please select at least one Entry'))
            return

        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))
        online = self._conn_manager.is_online()
        for path in paths:
            entry = model[path][self.ENTRIES_MODEL_DATA_COLUMN]
            entry.read = read
            child_path = model.convert_path_to_child_path(path)
            model.get_model()[child_path][self.ENTRIES_MODEL_TEXT_COLUMN] = self.view.get_visual_entry_text(entry)
            if summary.sync and online:
                self.manager.mark_as_read_synchronize(self.feed_data.href, model[path][2].link,
                                                      read, self._mark_as_read_sync_cb)

        # restore normal mode
        self._restore_normal_mode(button)

    def _mark_as_read_sync_cb(self, synced, user_data, error):
        if not synced:
            hildon.hildon_banner_show_information(self, '', _('Error syncing read status with Google Reader'))

    def _restore_normal_mode(self, button):

        # tree view normal mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_NORMAL)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.hide()
        self.unfullscreen()

    def _on_entry_activated(self, treeview, path, column):
        item_window = ItemWindow(self.feed_data, self.manager, self.settings, self._conn_manager)

        item_window.connect('item-read', self._item_read_cb)
        item_window.set_item_from_model(treeview.get_model(), path)
        item_window.show()

    def _item_read_cb(self, item_window, path):
        filter_model = self.view.get_model()
        entry = filter_model[path][self.ENTRIES_MODEL_DATA_COLUMN]

        # Set item as read and update the UI
        entry['read'] = True
        child_path = filter_model.convert_path_to_child_path(path)
        filter_model.get_model()[child_path][self.ENTRIES_MODEL_TEXT_COLUMN] = self.view.get_visual_entry_text(entry)

        # Mark item as read in Google Reader. Do it in an idle as we
        # don't want to delay the item rendering
        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))
        if summary.sync and self._conn_manager.is_online():
            glib.idle_add(self._mark_as_read_google_idle_cb, entry)

    def _mark_as_read_google_idle_cb(self, entry):
        if not self._conn_manager.is_online():
            return

        self.manager.mark_as_read_synchronize(self.feed_data.href, entry.link, True,
                                              self._mark_as_read_google_cb)
        return False

    def _mark_as_read_google_cb(self, synced, user_data, error):
        if not synced:
            hildon.hildon_banner_show_information(self, '', _('Error syncing read status with Google Reader'))

    def _update_feed_cb(self, button):
        url = self.feed_data.href

        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        button.set_sensitive(False)

        # Ask manager to update feed
        self.manager.update_feed(get_feed_id(self.feed_data), self._feed_updated_cb, None)

    def _feed_updated_cb(self, retval, data, error):

        summary = self.manager.get_feed_summary(get_feed_id(self.feed_data))
        # Update read/unread status if the feed was subscribed
        if summary.sync:
            hildon.hildon_banner_show_information(self, '', _('Synchronizing read/unread status with Google Reader'))
            self.manager.sync_google_reader_read_status(self._sync_google_reader_read_status_cb)
        else:
            hildon.hildon_gtk_window_set_progress_indicator(self, False)
            self._update_feed_button.set_sensitive(True)

        self.view.get_model().get_model().clear()
        self._add_entries()

    def _download_all_items_cb(self, button):
        folder = self.settings.auto_download_folder

        urls = []
        paths_files = []
        for entry in self.feed_data.entries:
            try:
                url = entry['enclosures'][0]['href']
                urls.append(url)
                paths_files.append(folder + os.path.basename(urllib.url2pathname(url)))
            except:
                pass

        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        self.manager.download_all_items(urls, paths_files, self._all_items_downloaded_cb)

    def _all_items_downloaded_cb(self, downloaded, user_data, error):
        if downloaded:
            message = _('Items downloaded')
        else:
            message = _('Error downloading items')
        hildon.hildon_banner_show_information(self, '', message)
        # Remove progress information
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def _on_configuration_changed(self, window, event):
        # Change alignment padding. We don't want padding in portrait
        # Need to save space
        if event.width > event.height:
            self.align.set_padding(4, 0 , 16, 16)
        else:
            self.align.set_padding(4, 0 , 4, 4)

class ItemWindowHeader(gtk.HBox):

    def __init__(self, homogeneous, spacing):
        super(ItemWindowHeader, self).__init__(homogeneous, spacing)

        # Header
        header_vbox = gtk.VBox()
        navigation_hbox = gtk.HBox(False, 0)

        # Author and URL
        self.title = gtk.Label('')
        self.title.set_alignment(0,0.5)
        header_vbox.pack_start(self.title, True, False)
        self.title.show()
        header_vbox.show()

        # Navigation buttons. Sizes are important. We set the size of the icon but
        # we let the button grow as needed
        self.button_up = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_up_img = gtk.Image()
        button_up_img.set_from_icon_name(constants.ICON_UP_NAME,
                                              gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_up.set_image(button_up_img)
        self.button_down = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_down_img = gtk.Image()
        button_down_img.set_from_icon_name(constants.ICON_DOWN_NAME,
                                                gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_down.set_image(button_down_img)

        navigation_hbox.pack_start(self.button_up, False, False, 0)
        navigation_hbox.pack_start(self.button_down, False, False, 0)
        navigation_hbox.show_all()

        self.pack_start(header_vbox, True, True)
        self.pack_start(navigation_hbox, False, False)

        # Signals
        #self.url.connect('clicked', self._link_button_clicked)

    def _link_button_clicked(self, link_button):
        print 'Open ' + link_button.get_uri()

    def set_item(self, item):

        self.item = item

        self.title.set_markup('<span size="small">' + glib.markup_escape_text(unescape(item.title)) + '</span>')
        #self.url.set_uri(item.link)

class ItemWindow(hildon.StackableWindow):
    __gsignals__ = {
        "item-read": (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT, )),
        }

    def __init__(self, feed_data, manager, settings, conn_manager):
        super(ItemWindow, self).__init__()

        self.manager = manager
        self.settings = settings
        self.feed_data = feed_data
        self._conn_manager = conn_manager
        self._conn_manager.connect('connection-changed', self._on_connection_changed)

        self._osso_context = osso.Context(constants.RSS_COMPACT_NAME, constants.RSS_VERSION, False)
        self._screen_device = osso.DeviceState(self._osso_context)

        # For opening links in browser
        self._browser_dbus_iface = None

        self._prev_item_reference = None
        self._next_item_reference = None

        # Header
        self.item_window_header = ItemWindowHeader(False, 16)
        self.item_window_header.button_up.connect("clicked", self._up_button_clicked)
        self.item_window_header.button_down.connect("clicked", self._down_button_clicked)

        # HTML renderer
        self.view = WebView()
        self.view.set_full_content_zoom(True)

        # Disable text selection
        self.view.connect("motion-notify-event", lambda w, ev: True)

        # Set some settings
        wbk_settings = self.view.get_settings()
        wbk_settings.set_property('default_font_size', self.settings.default_font_size)
        wbk_settings.set_property('auto-load-images', self.settings.auto_load_images)
        wbk_settings.set_property('auto-shrink-images', True)

        # Pannable for showing content
        self.pannable = hildon.PannableArea()
        self.pannable.add(self.view)
        self.pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        self.pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
        self.view.show()

        # Header Alignment
        header_align = gtk.Alignment(0, 0, 1, 1)
        header_align.set_padding(4, 0 , 16, 16)
        header_align.add(self.item_window_header)
        header_align.show()

        vbox = gtk.VBox(False, 8)
        vbox.pack_start(header_align, False, False)
        vbox.pack_start(self.pannable, True, True)
        self.item_window_header.show()
        self.pannable.show()

        self.add(vbox)
        vbox.show()

        self._create_menu()
        self._create_toolbar()

        # For portrait/landscape modes
        self.connect('configure-event', self._on_configuration_changed)

    def _on_connection_changed(self, conn_manager):
        if conn_manager.is_online():
            if 'enclosures' in self.item_window_header.item:
                self._download_button.show()
        else:
            self._download_button.hide()

    def _on_configuration_changed(self, window, event):

        # Show toolbar in portrait mode. Show header in landscape mode
        if event.width > event.height:
            self.pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
            self.item_window_header.show_all()
            self.toolbar.hide_all()
        else:
            self.pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_VERT)
            self.item_window_header.button_up.hide()
            self.item_window_header.button_down.hide()
            self.toolbar.show_all()

    def _create_toolbar(self):

        self.toolbar = gtk.Toolbar()
        button_prev_img = gtk.Image()
        button_prev_img.set_from_icon_name(constants.ICON_UP_NAME,
                                           gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_prev = gtk.ToolButton(button_prev_img, None)

        button_next_img = gtk.Image()
        button_next_img.set_from_icon_name(constants.ICON_DOWN_NAME,
                                           gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_next = gtk.ToolButton(button_next_img, None)

        button_prev.set_expand(True)
        button_next.set_expand(True)
        button_prev.set_homogeneous(True)
        button_next.set_homogeneous(True)

        self.toolbar.insert (button_prev, -1)
        self.toolbar.insert (button_next, -1)

        button_prev.connect("clicked", self._up_button_clicked)
        button_next.connect("clicked", self._down_button_clicked)

        self.add_toolbar(self.toolbar)

    def _on_request_url(self, object, url, stream):
        if (url.lower().startswith("http")):
            request_thread = Thread(target=self._request_url_in_thread,
                                    args=(url, stream))
            request_thread.start()

    def _request_url_in_thread(self, url, stream):
        f = urllib2.urlopen(url)

        gtk.gdk.threads_enter()
        stream.write(f.read())
        stream.close()
        gtk.gdk.threads_leave()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Item Details'))
        button.connect('clicked', self._item_details_cb)
        button.show()
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('View in Browser'))
        button.connect('clicked', self._view_in_browser_cb)
        button.show()
        menu.append(button)

        self._download_button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self._download_button.set_label(_('Download'))
        self._download_button.connect('clicked', self._download_item_cb)
        self._download_button.show()
        menu.append(self._download_button)

        menu.show_all()
        self.set_app_menu(menu)

    def _add_conditional_to_table (self, table, item, attr, label, is_time=False):
        if attr in item and item[attr] != '':
            value = time.strftime('%x', item[attr]) if is_time else item[attr]
            self._add_to_table (table, label, value)

    def _add_to_table (self, table, label_text, value):
        # Two cells per row. So rows = cells/2
        y_coord = len(table)/2

        label = gtk.Label('')
        label.set_markup('<span foreground="%s">%s:</span>' % \
                             (SECONDARY_TEXT_COLOR, label_text))
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, y_coord, y_coord + 1, gtk.FILL, 0)

        value_label = gtk.Label('')
        value_label.set_markup(value)
        value_label.set_alignment(0.0, 0.5)
        table.attach(value_label, 1, 2, y_coord, y_coord + 1, gtk.FILL, 0)

    def _view_in_browser_cb(self, button):
        item = self.item_window_header.item
        if 'link' in item:
            # Lazy load. We do not do this at window creation time for
            # performance reasons
            if not self._browser_dbus_iface:
                bus = dbus.SystemBus()
                proxy_browser = bus.get_object('com.nokia.osso_browser','/com/nokia/osso_browser/request')
                self._browser_dbus_iface = dbus.Interface(proxy_browser, 'com.nokia.osso_browser')
            self._browser_dbus_iface.load_url(item.link)

    def _item_details_cb(self, button):
        dialog = gtk.Dialog()
        dialog.set_title(_('Item Details'))

        pannable = hildon.PannableArea()
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_VERT)

        item = self.item_window_header.item

        table = gtk.Table()
        table.set_row_spacings(6)
        table.set_col_spacings(12)
        self._add_to_table (table, _('Title'), item.title)
        self._add_conditional_to_table (table, item, 'author', _('Author'))
#        self._add_conditional_to_table (table, item, 'link', _('Home Page'))
        self._add_conditional_to_table (table, item, 'updated_parsed', _('Updated'), True)
        self._add_conditional_to_table (table, item, 'published_parsed', _('Published'), True)
        self._add_conditional_to_table (table, item, 'language', _('Language'))
        self._add_conditional_to_table (table, item, 'license', _('License'))
        self._add_conditional_to_table (table, item, 'rights', _('Rigths'))
        self._add_conditional_to_table (table, item, 'generator', _('Generated by'))

        pannable.add_with_viewport(table)

        table.show_all()
        pannable.show()
        dialog.vbox.add(pannable)

        dialog.show()

    def _up_button_clicked(self, button):
        if not self._prev_item_reference or not self._prev_item_reference.valid():
            return

        prev_path = self._prev_item_reference.get_path()
        prev_iter = self.model.get_iter(prev_path)
        if prev_iter:
            item = self.model.get_value(prev_iter, EntriesWindow.ENTRIES_MODEL_DATA_COLUMN)
            self._set_item(item, prev_path)

    def _down_button_clicked(self, button):
        if not self._next_item_reference or not self._next_item_reference.valid():
            return

        next_path = self._next_item_reference.get_path()
        next_iter = self.model.get_iter(next_path)
        if next_iter:
            item = self.model.get_value(next_iter, EntriesWindow.ENTRIES_MODEL_DATA_COLUMN)
            self._set_item(item, next_path)

    def _update_next_prev_references(self, path):
        # Work out previous item path
        # http://faq.pygtk.org/index.py?req=show&file=faq13.051.htp
        position = path[-1]
        if position:
            prev_path = list(path)[:-1]
            prev_path.append(position - 1)
            self._prev_item_reference = gtk.TreeRowReference(self.model, tuple(prev_path))
        else:
            self._prev_item_reference = None

        # Next item path
        next_iter = self.model.iter_next(self.model.get_iter(path))
        if next_iter:
            self._next_item_reference = gtk.TreeRowReference(self.model, self.model.get_path(next_iter))
        else:
            self._next_item_reference = None

    def set_item_from_model(self, model, path):
        self.model = model
        item = model.get_value(model.get_iter(path), EntriesWindow.ENTRIES_MODEL_DATA_COLUMN)
        self._set_item(item, path)

    def _set_item(self, item, path):

        # Update next&prev references
        self._update_next_prev_references(path)

        author = get_author_from_item(item)
        if author == '':
            author = self.feed_data.feed.title
        self.set_title(unescape(author))

        # Set item in header
        self.item_window_header.set_item(item)

        # Get content type and body
        content_type = 'text/html'
        if 'content' in item:
            # We are only considering the first content. TODO check more?
            if 'type' in item.content[0]:
                content_type = item.content[0].type
            body = item.content[0].value
        elif 'summary' in item:
            if 'summary_detail' in item and 'type' in item.summary_detail:
                content_type = item.summary_detail.type
            body = item.summary
        elif 'title' in item:
            # Yeah I know..., but some weird feeds provide their
            # content in the title. I want to support them anyway
            body = item.title
        else:
            # Should never happen
            body = _('No text')

        self.view.load_string(body.encode('utf-8'), content_type, 'utf-8', '')

        # Write HTML
        try:
            self.view.load_string(body.encode(self.feed_data.encoding), content_type, self.feed_data.encoding, '')
        except UnicodeEncodeError:
            # In case of decoding error then try with UTF-8
            try:
                self.view.load_string(body.encode('utf-8'), content_type, 'utf-8', '')
            except UnicodeEncodeError:
                # Ok, let's fallback to ASCII then...
                try:
                    self.view.load_string(body.encode('us-ascii'), content_type, 'us-ascii', '')
                except UnicodeEncodeError:
                    # If everything else fails, then clear the view and show an error
                    self.view.load_string('', content_type, 'ascii', '')
                    hildon.hildon_banner_show_information(self, '', _('Some text could not be shown'))

        # Update button sensitiviness
        toolbar_button_list = self.toolbar.get_children()
        prev_item_missing = not self._prev_item_reference or not self._prev_item_reference.valid()
        next_item_missing = not self._next_item_reference or not self._next_item_reference.valid()

        self.item_window_header.button_up.set_sensitive(not prev_item_missing)
        toolbar_button_list[0].set_sensitive(not prev_item_missing)
        self.item_window_header.button_down.set_sensitive(not next_item_missing)
        toolbar_button_list[1].set_sensitive(not next_item_missing)

        # Mark item as read
        if not item.read:
            self.emit('item-read', path)

        # Force online/offline status evaluation
        self._on_connection_changed(self._conn_manager)

        # Keep the display on for 60" without user input.
        self._screen_device.display_blanking_pause()

    def _download_item_cb(self, button):
        item = self.item_window_header.item
        url_file = item.enclosures[0]['href']
        file = os.path.basename(urllib.url2pathname(url_file))
        folder = self.settings.auto_download_folder

        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        self.manager.download_item(url_file, folder+file, self._item_downloaded_cb)

    def _item_downloaded_cb(self, downloaded, user_data, error):
        if downloaded:
            message = _('Item downloaded')
        else:
            message = _('Error downloading item')
        hildon.hildon_banner_show_information(self, '', message)
        # Remove progress information
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

class FindView(hildon.GtkTreeView):

    FIND_SITE_VIEW = 0

    def __init__(self, ui_mode):
        super(FindView, self).__init__(ui_mode)

        # Add columns
        text_renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Name', text_renderer, markup=self.FIND_SITE_VIEW)
        column.set_expand(True)
        self.append_column(column)


class FindWindow(hildon.StackableWindow):
    __gsignals__ = {
        "urls-found": (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT, )),
        }

    FIND_MODEL_SITE_COLUMN = 0
    FIND_MODEL_URL_COLUMN = 1

    def __init__(self, feedsinfo, feedsWindow):
        super(FindWindow, self).__init__()

        self.feeds_info = feedsinfo

        self.view = FindView(gtk.HILDON_UI_MODE_EDIT)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().unselect_all()

        self.edit_toolbar = hildon.EditToolbar()

        self.edit_toolbar_button_handler = 0
        self.edit_toolbar.set_label(_('Select Feeds to add'))
        self.edit_toolbar.set_button_label(_('Add'))
        self.edit_toolbar.connect('arrow-clicked', self._restore_normal_mode)

        if self.edit_toolbar.handler_is_connected (self.edit_toolbar_button_handler):
            self.edit_toolbar.disconnect(self.edit_toolbar_button_handler)
        self.edit_toolbar_button_handler = \
            self.edit_toolbar.connect('button-clicked', self._add_button_clicked_cb, feedsWindow)

        self.set_edit_toolbar(self.edit_toolbar)
        self.edit_toolbar.show()
        self.fullscreen()

        self.view.set_model (gtk.ListStore(str, str))

        store = self.view.get_model()

        for feed_info in self.feeds_info:
            feed_iter = store.append()

            store.set(feed_iter,
                      self.FIND_MODEL_SITE_COLUMN, gobject.markup_escape_text(unescape(feed_info['sitename'])),
                      self.FIND_MODEL_URL_COLUMN, feed_info['dataurl'])

        self.view.show()

        self.align = gtk.Alignment(0, 0, 1, 1)
        self.align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        self.align.add(pannable)
        self.add(self.align)
        self.align.show()


    def _add_button_clicked_cb(self, button, feedsWindow):
        selection = self.view.get_selection()
        selected_rows = selection.get_selected_rows()
        model, paths = selected_rows
        if not paths:
            hildon.hildon_banner_show_information(self, '', _('Please select at least one Feed'))
            return

        urls = [model[path][self.FIND_MODEL_URL_COLUMN] for path in paths]

        self.emit('urls-found', urls)

        # restore normal mode
        self._restore_normal_mode(button)

    def _restore_normal_mode(self, button):
        self.destroy()
