/* vim: set sts=4 sw=4 et: */
/*
 * settings.c
 * GConf settings
 * 
 * Copyright (C) 2006 Nokia Corporation
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <locale.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-caption.h>
#include <stdlib.h>
#include <string.h>
#include <osso-log.h>
#include <hildon-fm-2/hildon/hildon-file-chooser-dialog.h>

#include "maemo-recorder.h"
#include "settings.h"

#define GCONF_PATH          "/apps/maemo-recorder/settings"
#define KEY_DEFAULT_DIR     GCONF_PATH "/default_dir"
#define KEY_DEFAULT_FILTER  GCONF_PATH "/default_filter"

#define DEFAULT_SAVE_DIR "/home/user/MyDocs/.sounds"

typedef struct _SettingsPriv SettingsPriv;

struct _SettingsPriv
{
    GConfClient *gconf;
    gchar *default_dir;
    gint default_filter;
};

static SettingsPriv *settings_priv;

typedef struct _SettingsData SettingsData;

struct _SettingsData
{
    AppData *appdata;
    GtkWidget *dir_entry;
    GtkWidget *window;
    GtkWidget *format_combobox;
};

static SettingsData *settings_data;

static gboolean load_settings(void);

static void
reset_defaults()
{
    gchar *str;

    str = g_strdup_printf(DEFAULT_SAVE_DIR);
    gconf_client_set_string(settings_priv->gconf, KEY_DEFAULT_DIR, str, NULL);
    g_free(settings_priv->default_dir);
    settings_priv->default_dir = str;

    gconf_client_set_int(settings_priv->gconf,
                         KEY_DEFAULT_FILTER, FORMAT_WAV, NULL);
    settings_priv->default_filter = FORMAT_WAV;
}

gint
get_default_filter()
{
    load_settings();
    return settings_priv->default_filter;
}

void
set_default_filter(gint filter)
{
    gconf_client_set_int(settings_priv->gconf,
                         KEY_DEFAULT_FILTER, filter, NULL);
    settings_priv->default_filter = filter;

}

gchar *
get_default_dir()
{
    load_settings();
    return settings_priv->default_dir;
}

void
set_default_dir(gchar * selected)
{
    gconf_client_set_string(settings_priv->gconf,
                            KEY_DEFAULT_DIR, selected, NULL);
}

static void
cb_format_changed(gpointer data)
{
    gint active;

    active =
        gtk_combo_box_get_active(GTK_COMBO_BOX
                                 (settings_data->format_combobox));
    switch (active)
    {
        case 0:
            settings_priv->default_filter = FORMAT_WAV;
            break;
        case 1:
            settings_priv->default_filter = FORMAT_PCMA;
            break;
        case 2:
            settings_priv->default_filter = FORMAT_ILBC;
            break;
        default:
            break;
    }
}

static void
cb_chdir(gpointer data)
{
    GtkWidget *dialog = NULL;

    g_assert(settings_data);

    /* create dialog */
    dialog =
        GTK_WIDGET(hildon_file_chooser_dialog_new
                   (GTK_WINDOW(settings_data->window),
                    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER));

    /* show it */
    gtk_widget_show_all(dialog);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    {
        g_free(settings_priv->default_dir);
        settings_priv->default_dir =
            g_strdup(gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog)));
        gtk_label_set_text(GTK_LABEL(settings_data->dir_entry),
                           settings_priv->default_dir);
    }

    gtk_widget_destroy(dialog);

}

static gboolean
load_settings()
{
    g_free(settings_priv->default_dir);
    settings_priv->default_dir = gconf_client_get_string(settings_priv->gconf,
                                                         KEY_DEFAULT_DIR,
                                                         NULL);

    settings_priv->default_filter = gconf_client_get_int(settings_priv->gconf,
                                                         KEY_DEFAULT_FILTER,
                                                         NULL);

    if (settings_priv->default_dir && settings_priv->default_filter)
        return TRUE;
    else
        return FALSE;

}

GConfClient *
settings_init(gint argc, gchar * argv[])
{
    GError *err = NULL;

    settings_priv = g_new0(SettingsPriv, 1);
    settings_data = g_new0(SettingsData, 1);

    gconf_init(argc, argv, &err);

    if (err)
    {
        ULOG_ERR("Unable to initialize gconf");
        reset_defaults();
        return NULL;
    }

    settings_priv->gconf = gconf_client_get_default();

    if (!settings_priv->gconf)
    {
        ULOG_ERR("Unable to initialize gconf");
        reset_defaults();
        return NULL;
    }

    gconf_client_add_dir(settings_priv->gconf, GCONF_PATH,
                         GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

    if (!load_settings())
    {
        ULOG_ERR("Unable to load all settings");
        reset_defaults();
        return settings_priv->gconf;
    }

    return settings_priv->gconf;
}

void
settings_edit(GtkWidget * widget, AppData * data)
{
    GtkWidget *vbox;
    GtkWidget *btn_chdir;
    GtkWidget *alignment;
    GtkWidget *caption;
    GtkSizeGroup *group;
    gint res;

    load_settings();

    settings_data->appdata = data;

    settings_data->window = gtk_dialog_new_with_buttons(_("Settings"),
                                                        GTK_WINDOW(data->
                                                                   mainView),
                                                        GTK_DIALOG_MODAL |
                                                        GTK_DIALOG_DESTROY_WITH_PARENT
                                                        |
                                                        GTK_DIALOG_NO_SEPARATOR,
                                                        _("OK"),
                                                        GTK_RESPONSE_OK,
                                                        _("Cancel"),
                                                        GTK_RESPONSE_CANCEL,
                                                        NULL, NULL);

    g_signal_connect(G_OBJECT(settings_data->window),
                     "delete_event", G_CALLBACK(gtk_widget_destroy), NULL);

    group = GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));
    vbox = gtk_vbox_new(FALSE, HILDON_MARGIN_DEFAULT);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),
                                   HILDON_MARGIN_DEFAULT);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(settings_data->window)->vbox),
                      vbox);

    /* with the format combo on top and directory thing at the bottom it looks cleaner */

    /* format combo */
    settings_data->format_combobox = gtk_combo_box_new_text();

    gtk_combo_box_append_text(GTK_COMBO_BOX(settings_data->format_combobox),
                              FORMAT_NAME_WAV);
    gtk_combo_box_append_text(GTK_COMBO_BOX(settings_data->format_combobox),
                              FORMAT_NAME_PCMA);
    gtk_combo_box_append_text(GTK_COMBO_BOX(settings_data->format_combobox),
                              FORMAT_NAME_ILBC);

    switch (settings_priv->default_filter)
    {
        case FORMAT_WAV:
            gtk_combo_box_set_active(GTK_COMBO_BOX
                                     (settings_data->format_combobox), 0);
            break;

        case FORMAT_PCMA:
            gtk_combo_box_set_active(GTK_COMBO_BOX
                                     (settings_data->format_combobox), 1);
            break;

        case FORMAT_ILBC:
        default:
            gtk_combo_box_set_active(GTK_COMBO_BOX
                                     (settings_data->format_combobox), 2);
            break;
    }

    g_signal_connect(G_OBJECT(settings_data->format_combobox), "changed",
                     G_CALLBACK(cb_format_changed), settings_data);

    caption = hildon_caption_new(group, _("Default recording format:"),
                                 settings_data->format_combobox, NULL,
                                 HILDON_CAPTION_OPTIONAL);

    gtk_box_pack_start(GTK_BOX(vbox), caption, FALSE, FALSE, 0);

    /* save directory */
    /* GtkEntry looks worse because the text get ellipsized */
    settings_data->dir_entry = gtk_label_new(get_default_dir());

    /*
    gtk_editable_set_editable(GTK_EDITABLE(settings_data->dir_entry), FALSE);

    gtk_entry_set_text(GTK_ENTRY(settings_data->dir_entry), get_default_dir());
    */

    /* In maemo world it seems to "folder", not "directory" =( */
    caption = hildon_caption_new(group, _("Default save folder:"),
                                 settings_data->dir_entry, NULL,
                                 HILDON_CAPTION_OPTIONAL);

    gtk_box_pack_start(GTK_BOX(vbox), caption, FALSE, FALSE, 0);

    /* browse button */
    btn_chdir = gtk_button_new_with_label(_("Browse"));
    alignment = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);

    gtk_container_add(GTK_CONTAINER(alignment), btn_chdir);
    gtk_box_pack_start(GTK_BOX(vbox), alignment, FALSE, FALSE, 0);

    g_signal_connect(G_OBJECT(btn_chdir), "clicked",
                     G_CALLBACK(cb_chdir), settings_data);

    /*
    gtk_widget_set_size_request(btn_chdir, 40, 40);
    */

    gtk_widget_show_all(settings_data->window);
    res = gtk_dialog_run(GTK_DIALOG(settings_data->window));

    if (res == GTK_RESPONSE_OK)
    {
        set_default_dir(settings_priv->default_dir);
        set_default_filter(settings_priv->default_filter);
        ULOG_DEBUG("ok");
    }
    else
    {
        ULOG_DEBUG("cancel");
    }

    gtk_widget_destroy(settings_data->window);
}
