/* vim: set sts=4 sw=4 et: */
/*
 * maemo-recorder-ui.c
 *
 * Copyright (C) 2006 Nokia Corporation
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <gst/gst.h>
#include <glib/gi18n-lib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <locale.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-file-system-model.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <string.h>
#include <sys/time.h>
#include <libmodest-dbus-client/libmodest-dbus-client.h>
#include <glib/gstdio.h>
#include "maemo-recorder.h"
#include "maemo-recorder-ui.h"
#include "maemo-recorder-file.h"
#include "settings.h"

#define DEFAULT_REC_BLOCKSIZE "160"

#define STOP_DELAY 500
#define REC_UPDATE_INTERVAL 500
#define PLAY_UPDATE_INTERVAL 200

/* MACROs */

#define GST_TIME_MINS(t) \
        (guint) ((((GstClockTime)(t)) / (GST_SECOND * 60)) % 60)
#define GST_TIME_SECS(t) \
        (guint) ((((GstClockTime)(t)) / GST_SECOND) % 60)
#define GST_TIME_TO_SECS(t) \
        (gdouble) (((gdouble)(t)) / (gdouble) GST_SECOND) /* GST_SECOND should be 1e9 */
#define GST_TIME_MSECS(t) \
        (guint) (((GstClockTime)(t)) % GST_SECOND)

#define RECORDER_APP_TITLE "Recorder"
#define RECORDER_MSG_READY _("Ready")
#define RECORDER_MSG_STOPPED _("Stopped")
#define RECORDER_MSG_PAUSED _("Paused")
#define RECORDER_MSG_PLAYING  _("Playing")
#define RECORDER_MSG_RECORDING _("Recording")
#define RECORDER_FILE_UNTITLED _("Untitled")

#define RECORDER_FMT_STRING_NONE _("N/A")

/* general enumerations */
    
typedef enum
{
    DTX_OFF = 0,
    DTX_ON = 1
} DTX;

typedef enum 
{
    PIPELINE_PLAY = 1,
    PIPELINE_PLAY_MP3,
    PIPELINE_REC
} PipeLineType;

/* function prototypes */

static gboolean cbBus (GstBus *bus, 
               GstMessage *message, 
               gpointer data);

static void pipelineStateChanged (GstElement *element,
                  GstState old,
                  GstState new,
                  GstState pending,
                  AppData *data);

static void seekToTime(GstElement *pipeline, gdouble secs);
static gboolean seekToZero(AppData *data, GstElement *pipeline);
static void setLength(AppData *data, gdouble secs);
static void setFormatString(AppData *data, AudioFormat afmt);
static gboolean cbStopPlayback(AppData *data);
static void cbStop(GtkWidget* widget, GdkEventButton *event, AppData *data);
static void cbPlay(GtkWidget* widget, GdkEventButton *event, AppData *data);
static void cbRec(GtkWidget* widget, GdkEventButton *event, AppData *data);
static void cbNew(GtkWidget* widget, AppData *data);
static void cbOpen(GtkWidget* widget, AppData *data);
/*static void cbSave(GtkWidget* widget, AppData *data);*/
static void cbSaveAs(GtkWidget* widget, AppData *data);
static void createMenu( AppData *data );
static gboolean createPipeline(AppData *app, PipeLineType type);
static void openPlayPipeline( AppData *data );
static gboolean destroyPipeline(AppData *data, PipeLineType type);
static gboolean destroyPipelines(AppData *data);
static void cbItemGroupChanged(gpointer data);
static gboolean cbUpdateRecLength(AppData *data);
static void cbDestroy(GtkWidget* widget, GdkEvent *event, gpointer data);
static gboolean openURI(gpointer user_data);
static gboolean closeFile(AppData *data);
static const gchar *getFileName(AppData *data);
static gdouble guessMediaLength(AppData *data);
static GstCaps *createCapsFilter(AudioFormat format);
static gboolean evKeypress(GtkWidget *widget, GdkEventKey *ev, AppData *appdata);
static gboolean cbScaleRelease(GtkWidget *widget, GdkEventButton *ev, gpointer data);

static gboolean lengthSet = FALSE;

static gboolean createPipeline(AppData *app, PipeLineType type)
{
    GstElement *src = NULL;
    GstElement *sink = NULL;
    GstElement *filter = NULL;
    GstElement *pipeline = NULL;
    GstElement *parse = NULL;
    GstElement *bin = NULL;
    GstCaps *caps = NULL;

    g_assert(NULL != app);

    /* create elements */
    switch (type)
    {
        case PIPELINE_PLAY_MP3:
            ULOG_INFO("mp3 playback - queue");
            bin = gst_element_factory_make ("playbin2", "bin");
            gchar* uri = g_strdup_printf("file://%s", app->openFileName);
            g_object_set(G_OBJECT(bin), 
                "uri", uri, 
                NULL);
            g_free(uri);
            gst_bus_add_watch(gst_pipeline_get_bus (GST_PIPELINE (bin)),
                   cbBus, app);

            app->playPipeline = bin;
            app->playPipelineType = type;
                     
            return TRUE;
            break;

        case PIPELINE_PLAY:
            src = gst_element_factory_make ("filesrc", "source");
            /* we need also a filter to modify caps */
            filter = gst_element_factory_make("capsfilter", "filter");
            switch (app->filter)
            {
                case FORMAT_ILBC:
                    ULOG_INFO("using ilbc sink");
                    sink = gst_element_factory_make ("dspilbcsink", "sink");
                    break;

                case FORMAT_PCMA:
                case FORMAT_PCMU:
                case FORMAT_PCM:
                    ULOG_INFO("using pcm sink");
                    sink = gst_element_factory_make ("pulsesink", "sink");
                    break;

                case FORMAT_WAV:
                    ULOG_INFO("using wavparse & pcm sink");
                    bin = gst_element_factory_make ("playbin2", "bin");
                    gchar* uri = g_strdup_printf("file://%s", app->openFileName);
                    g_object_set(G_OBJECT(bin),
                        "uri", uri,
                        NULL);
                    g_free(uri);
                    
                    gst_bus_add_watch(gst_pipeline_get_bus (GST_PIPELINE (bin)),
                        cbBus, app);

                    app->playPipeline = bin;
                    app->playPipelineType = type;
                              
                    return TRUE;
                    break;
                    
                default:
                    break;  
            }

            g_object_set(G_OBJECT(src), 
                    "location", app->openFileName, 
                    NULL);

            caps = createCapsFilter(app->filter);
            g_object_set(G_OBJECT(filter), 
                  "caps", caps,
                  NULL);
            break;

        case PIPELINE_REC:
            switch (app->filter)  
            {    
                case FORMAT_ILBC:
                    ULOG_INFO("using ilbc source");
                    src = gst_element_factory_make("dspilbcsrc", "source");
                    g_object_set(G_OBJECT(src),
                        "dtx", DTX_OFF,
                        "mode", 1,
                        NULL);
                    break;
        
                case FORMAT_PCMA:
                case FORMAT_PCMU:
                case FORMAT_PCM:
                    ULOG_INFO("using pcm source");
                    src = gst_element_factory_make("pulsesrc", "source");
                    /*g_object_set(G_OBJECT (src), 
                          "blocksize", DEFAULT_REC_BLOCKSIZE, 
                          "dtx", DTX_OFF,
                        NULL);*/
                    break;
            
                case FORMAT_WAV:
                    ULOG_INFO("using pcm source & wavenc");
                    src = gst_element_factory_make("pulsesrc", "source");
                    parse = gst_element_factory_make("wavenc", "enc");
                    break;   
            
                default:
                    ULOG_WARN("Unknown filter type!");
                    break; 
            }

            filter = gst_element_factory_make("capsfilter", "filter");
            caps = createCapsFilter(app->filter);
            g_object_set(G_OBJECT(filter), 
                  "caps", caps,
                  NULL);

            sink = gst_element_factory_make("filesink", "sink");

            g_object_set(G_OBJECT(sink), 
                "location", app->saveFileName,
                NULL);
            break;

        default:
            ULOG_ERR("Invalid pipeline type!");
            return FALSE;
    }

    pipeline = gst_pipeline_new("pipeline");

    gst_bus_add_watch(gst_pipeline_get_bus (GST_PIPELINE (pipeline)),
                   cbBus, app);
    
    if (!src || !pipeline)
    {
        ULOG_ERR("Could not create GstElement!");
        return FALSE;
    }

    if (!sink && app->filter != FORMAT_WAV )
    {
        ULOG_ERR("Could not create GstElement!");
        return FALSE;        
    }
        
    ULOG_INFO("Create pipeline");
     
    /* add to pipeline and link */
    switch (type)
    {
        case PIPELINE_REC:
            switch (app->filter)
            {
                case FORMAT_ILBC:
                case FORMAT_PCM:
                case FORMAT_PCMA:
                    if (!filter)
                    {
                       ULOG_ERR("Could not create filter GstElement!");
                       return FALSE;
                    }
                    gst_bin_add_many(GST_BIN(pipeline), src, filter, sink, NULL);

                    if (!gst_element_link_many (src, filter, sink, NULL))
                    {
                        ULOG_ERR("gst_element_link failed for src, filter and sink!");
                        return FALSE;
                    }
                    break;

                case FORMAT_WAV:
                    gst_bin_add_many(GST_BIN(pipeline), src, filter, parse, sink, NULL);
                    if (!gst_element_link_many (src, filter, parse, sink, NULL))
                    {
                        ULOG_ERR("gst_element_link failed for src, parse and sink!");
                    }
                    break;

                default:
                    break;
          
            }
            break; 
           
        case PIPELINE_PLAY:
            switch (app->filter)
            {
                case FORMAT_ILBC:
                case FORMAT_PCM:
                case FORMAT_PCMA:
                    if (!filter)
                    {
                       ULOG_ERR("Could not create filter GstElement!");
                       return FALSE;
                    }
                    gst_bin_add_many(GST_BIN(pipeline), src, filter, sink, NULL);

                    if (!gst_element_link_many (src, filter, sink, NULL))
                    {
                        ULOG_ERR("gst_element_link failed for src, filter and sink!");
                        return FALSE;
                    }

                    break;

                default:
                    break;
            }

            break;
     
        default:
            gst_bin_add_many(GST_BIN(pipeline), src, sink, NULL);
 
            if(!gst_element_link_many(src, sink, NULL))
            {
                 ULOG_ERR("gst_element_link failed for src and sink!");
                 return FALSE;
            }

            break;
    }

    /* set application data */
    if (type == PIPELINE_REC)
    {
        app->recPipeline = pipeline;
    }
    else
    {
        app->playPipeline = pipeline;
        app->playPipelineType = type;
    }
     
    if (caps)
    {
        gst_caps_unref(caps);
        caps = NULL;
    }

    return TRUE;
}

static gboolean destroyPipelines(AppData *data)
{
    gboolean ret = FALSE;

    /* ugly hack with pipeline types, but works though */
    ret != destroyPipeline(data, PIPELINE_REC);
    ret != destroyPipeline(data, PIPELINE_PLAY);
    return ret;
}

static gboolean destroyPipeline(AppData *data, PipeLineType type)
{
    GstState state;
    GstState pending;
    GstElement *pipeline = NULL;

    ULOG_INFO("%s() - Stopping playback/recording", G_STRFUNC);

    //hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, RECORDER_MSG_STOPPED);         
    gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), "");
    gtk_widget_set_sensitive(data->buttonRec, TRUE);

    switch (type)
    {
        case PIPELINE_REC:
            pipeline = data->recPipeline;
            /*
            data->recPipeline = NULL;
            */
            break;

        default:
            pipeline = data->playPipeline;
            /*
            data->playPipeline = NULL;
            */
            break;
    }
    
    if (!GST_IS_ELEMENT(pipeline))
        return TRUE;

    /* this unallocates everything */
    gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_NULL);

    /* for some reason the state does not update manually */
    gst_element_get_state(pipeline, &state, 
            &pending, GST_CLOCK_TIME_NONE);
    pipelineStateChanged(pipeline,
           state,
           state,
           pending,
           data);

    /*
    gst_object_unref(pipeline);
    */

    return TRUE;
}


/* callbacks */

static void pipelineStateChanged (GstElement *element,
                  GstState old,
                  GstState new,
                  GstState pending,
                  AppData * data)
{
    g_assert(NULL != data);
     
    switch (new)
    {
        case GST_STATE_PLAYING:
          if(APPSTATE_RECORDING == getAppState(data))
          {
               gchar *tmp = g_strdup_printf("%s...", RECORDER_MSG_RECORDING);
               hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, tmp);
               g_free(tmp);
               tmp = NULL;
               ULOG_INFO("%s() - Recording", G_STRFUNC);
               gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), 
                      RECORDER_MSG_RECORDING);

               gtk_widget_set_state(data->buttonRec, GTK_STATE_ACTIVE);

               if (data->recUpdateId == 0)
               {
                   data->recUpdateId = g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, REC_UPDATE_INTERVAL, (GSourceFunc) cbUpdateRecLength, data, NULL);
               }
          }
          else
          {
               gchar *tmp = g_strdup_printf("%s...", RECORDER_MSG_PLAYING);
               hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, tmp);
               g_free(tmp);
               tmp = NULL;
               ULOG_INFO("%s() - Playing", G_STRFUNC);
               gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), 
                      RECORDER_MSG_PLAYING);  
               gtk_widget_set_state(data->buttonPlay, GTK_STATE_ACTIVE);
          }

          break;

        case GST_STATE_READY:
            /* hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, "Ready..."); */
            ULOG_INFO("%s() - Ready", G_STRFUNC);
            gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), ""); 
            break;

        case GST_STATE_PAUSED:
          {
              gint64 pos = 0;
              GstFormat fmt = GST_FORMAT_TIME;
              ULOG_INFO("%s() - Paused", G_STRFUNC);

              /* if pipeline pos == 0 => stopped, else => paused */
              if (GST_IS_ELEMENT(data->playPipeline) && gst_element_query_position(data->playPipeline, &fmt, &pos) && pos != 0)
              {
                  hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, RECORDER_MSG_PAUSED);       
                  gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), 
                         RECORDER_MSG_PAUSED);
              }
              gtk_widget_set_state(data->buttonRec, GTK_STATE_NORMAL);
          }
          break;
          
        case GST_STATE_NULL:
          ULOG_INFO("%s() - Null", G_STRFUNC);
          gtk_widget_set_state(data->buttonPlay, GTK_STATE_NORMAL);
          gtk_widget_set_state(data->buttonRec, GTK_STATE_NORMAL);
          
          break;

        default:
          ULOG_WARN("%s() - default case", G_STRFUNC);
          break;
    }
}


static gboolean cbBus(GstBus *bus,
               GstMessage *message,
               gpointer   data)
{
    AppData *app = (AppData*)data;

    switch (GST_MESSAGE_TYPE(message)) 
    {
        case GST_MESSAGE_WARNING:
        {
            GError *err;
            gchar *debug;

            gst_message_parse_error (message, &err, &debug);
            ULOG_WARN("%s() - Warning: %s", G_STRFUNC, err->message);
            g_error_free (err);
            g_free (debug);
            break;
        }

        case GST_MESSAGE_ERROR: 
        {
            GError *err;
            gchar *debug;

            gst_message_parse_error (message, &err, &debug);
            ULOG_ERR("%s() - Error: %s", G_STRFUNC, err->message);
            g_error_free (err);
            g_free (debug);
            /* break; */
            /* flow through to eos */
        }

        case GST_MESSAGE_EOS:
        {
            ULOG_INFO("%s() - eos", G_STRFUNC);

             switch(getAppState(app))
             {
                 case APPSTATE_PLAYING:
                    /* stop playback after a short break*/
                    g_timeout_add(STOP_DELAY, (GSourceFunc)cbStopPlayback, data);
                    //hildon_banner_show_information(GTK_WIDGET(app->mainView), NULL, RECORDER_MSG_STOPPED);       
                    gtk_label_set_text(GTK_LABEL(app->mainViewData.stateEntry), 
                         "");  

                    break;

                 case APPSTATE_RECORDING:
                     gst_element_set_state(GST_ELEMENT(app->recPipeline), 
                           GST_STATE_PAUSED);
                     destroyPipeline(app, PIPELINE_REC);
                     app->saved = FALSE;
                     setAppState(data, APPSTATE_READY);
                     break;

                 case APPSTATE_READY:
                 default:
                    /* destroyPipelines(app); */
                     break;
             }
             break;
        }

        case GST_MESSAGE_ASYNC_DONE:
        case GST_MESSAGE_STATE_CHANGED: 
        {
            GstState old;
            GstState new;
            GstState pending;

            gst_message_parse_state_changed(message, &old, &new, &pending);

            pipelineStateChanged(NULL, old, new, pending, app);

            break;
        }

        default:
            /* unhandled message */
            ULOG_WARN("%s() - Unhandled message, type = %d", G_STRFUNC, message->type);
            break;
    }

    /* remove message from the queue */
    return TRUE;
}

static void cbDestroy(GtkWidget* widget, GdkEvent *event, gpointer data)
{     
    AppData* app;

    g_assert(data);

    app = (AppData *) data;

    ULOG_DEBUG("delete_event");
    closeFile(app);

    destroyPipelines(app);
    if (app->playPipeline)
        gst_object_unref(GST_OBJECT(app->playPipeline));

    if (app->recPipeline)
        gst_object_unref(GST_OBJECT(app->recPipeline));

    gtk_main_quit();
}

static gboolean cbCheckPosition (AppData *data)
{
    GstFormat fmt = GST_FORMAT_TIME;
    gint64 pos = 0, len = 0;

    g_assert(NULL != data);

    /* get length */
    if(!lengthSet && gst_element_query_duration(data->playPipeline, &fmt, &len))
    {
        if (len > 0)
        {
            gdouble size = 0;
            size = GST_TIME_TO_SECS(len);
            setLength(data, size); /* sets lengthEntry and adjustment */
            lengthSet = TRUE;
        }
    }

    /* calculate position */
    if (gst_element_query_position(data->playPipeline, &fmt, &pos))
    {
        gdouble time = GST_TIME_TO_SECS(pos);
        guint mins = 0;
        gchar* tmp;

        ULOG_DEBUG("pos = %lld, time = %f", 
             pos,
           time 
           );

        gtk_adjustment_set_value( 
           GTK_ADJUSTMENT(data->mainViewData.adjustment),
           time);
        gtk_adjustment_value_changed(GTK_ADJUSTMENT(data->mainViewData.adjustment));
        if (time >= 60.0)
        {
            mins = time / 60;
            time -= mins * 60.0;
        }

        tmp = g_strdup_printf("%02u:%02d", mins, (int)time);

        gtk_label_set_text(GTK_LABEL(data->mainViewData.ctime),
                 tmp);
        g_free(tmp);

    }

    if (APPSTATE_PLAYING == getAppState(data))
    {
        return TRUE;
    }

    return FALSE;
}

static void cbNew(GtkWidget* widget, AppData *data) 
{
    g_assert(NULL != data);

    if (!closeFile(data))
        return;

    /* remove pipelines if existing */
    if (APPSTATE_PLAYING == getAppState(data) || APPSTATE_RECORDING == getAppState(data)) {
        cbStop(widget, NULL, data);
        destroyPipelines(data);
    }

    setAppState(data, APPSTATE_READY);
    ULOG_DEBUG_F("cbNew");
    /* destroy tmp file */

    /* clear filenames */
    g_free(data->openFileName);
    data->openFileName = NULL;
    g_free(data->saveFileName);
    data->saveFileName = NULL;
/*    data->filter = FORMAT_NONE;*/
    data->file_format = FORMAT_NONE;

    /* update display */
    gtk_label_set_text(GTK_LABEL(data->mainViewData.fileNameEntry), 
        RECORDER_FILE_UNTITLED);
    setLength(data, 0.0);
    /* update the display + scale */
    gtk_adjustment_set_value(GTK_ADJUSTMENT(data->mainViewData.adjustment),
              0);
    gtk_label_set_text(GTK_LABEL(data->mainViewData.ctime), "00:00");
    gtk_widget_set_sensitive(data->buttonSaveAs, FALSE);
    data->saved = TRUE;

    ULOG_DEBUG_F("cbNew end");
}

static void cbOpen(GtkWidget* widget, AppData *data) 
{
    GtkWidget* dialog = NULL;
    gchar *tmpfile = NULL;
    gchar *selected = NULL;
    AudioFormat format;
    gchar *basename;
    gdouble len = -1.0;
#if 0
    GtkFileFilter *filter;
#endif

    ULOG_DEBUG_F("begin");
    g_assert(NULL != data);

    if (!closeFile(data))
        return;
    if (APPSTATE_PLAYING == getAppState(data) || APPSTATE_RECORDING == getAppState(data)) {
        cbStop(widget, NULL, data);
        destroyPipelines(data);
    }

#if 0
    /* create filter */
    filter = gtk_file_filter_new();
    gtk_file_filter_add_mime_type(filter, "audio/x-mp3");
#endif

    g_assert(GTK_IS_WINDOW(data->mainView));
    /* create dialog */
    dialog = hildon_file_chooser_dialog_new_with_properties(
              GTK_WINDOW(data->mainView), 
              "action", GTK_FILE_CHOOSER_ACTION_OPEN,
              "file-system-model", NULL,
              "local-only", TRUE,
              NULL
              );

    /* show it */
    gtk_widget_show_all(dialog);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) 
    {
        selected = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    }

    ULOG_DEBUG("%s() - dialog finished", G_STRFUNC);

    if (dialog != NULL && GTK_IS_WIDGET(dialog))
    {
        gtk_widget_destroy(dialog);
        /*
        ULOG_DEBUG("%s() - dialog destroyed", G_STRFUNC);
        */
    }

    if (NULL == selected) 
    { 
        /* no file selected */
        if (!g_file_test(getFileName(data), G_FILE_TEST_EXISTS))
        {
                cbNew(widget, data);
        }
        return;
    }

    ULOG_INFO("%s() - selected filename = '%s'", G_STRFUNC, selected);

    if (openFile(selected, &format, &tmpfile))
    {
        ULOG_INFO("%s() - openFile() succeeded, format: %d, tmpfile %s", G_STRFUNC, format, tmpfile);

        g_assert(tmpfile);

        /* update filenames */
        basename = g_path_get_basename(selected);

        gtk_label_set_text(GTK_LABEL(data->mainViewData.fileNameEntry), basename);
        g_free(basename);

        g_free(data->openFileName);
        data->openFileName = tmpfile;
        data->file_format = format;
        //data->filter = format;
        g_free(data->saveFileName);
        data->saveFileName = NULL;
        gtk_widget_set_sensitive(data->buttonSaveAs, TRUE);

        len = guessMediaLength(data);
        if (len > 0.0)
            setLength(data, len);
        else
            setLength(data, 0.0);

        setFormatString(data, data->file_format);
        data->saved = TRUE;
    }
    else
    {
        ULOG_WARN("%s() - openFile() failed", G_STRFUNC);
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Could not open file!"));
    }

    g_free(selected);

    ULOG_DEBUG_F("end");
}

static gboolean
openURI(gpointer user_data)
{
    AppData *data;
    GnomeVFSURI *uri;
    gchar *selected = NULL;
    AudioFormat format;
    gchar *tmpfile = NULL;
    gchar *basename = NULL;
    gdouble len = -1.0;

    g_assert(user_data);
    data = (AppData *) user_data;

    if (NULL == data->mimeURI)
        return FALSE;

    uri = gnome_vfs_uri_new(data->mimeURI);
    selected = g_strdup(gnome_vfs_uri_get_path(uri));

    gnome_vfs_uri_unref(uri);
    uri = NULL;

    g_free(data->mimeURI);
    data->mimeURI = NULL;

    /* TODO: the following is duplicated in cbOpen(), move to a tryOpen() function ? */

    if (NULL == selected)
        return FALSE;

    ULOG_INFO("%s() - selected filename = '%s'", G_STRFUNC, selected);

    if (openFile(selected, &format, &tmpfile))
    {
        ULOG_INFO("%s: openFile() succeeded, format: %d, tmpfile %s", G_STRFUNC, format, tmpfile);

        g_assert(tmpfile);

        /* update filenames */
        basename = g_path_get_basename(selected);
        
        gtk_label_set_text(GTK_LABEL(data->mainViewData.fileNameEntry), basename);
        g_free(basename);

        g_free(data->openFileName);
        data->openFileName = tmpfile;
        data->file_format = format;
        //data->filter = format;
        g_free(data->saveFileName);
        data->saveFileName = NULL;
        gtk_widget_set_sensitive(data->buttonSaveAs, TRUE);
        
        len = guessMediaLength(data);
        if (len > 0.0)
            setLength(data, len);
        else
            setLength(data, 0.0);

        setFormatString(data, data->file_format);
        data->saved = TRUE;
    }
    else
    {
        ULOG_WARN("%s() - openFile() failed", G_STRFUNC);
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Could not open file!"));
    }

    g_free(selected);

    return FALSE;
}

static void openPlayPipeline( AppData *data )
{

    GstFormat fmt = GST_FORMAT_TIME;
    gint64 len;
    gdouble size = 0;
    lengthSet = FALSE;
    /* create pipelines */
    /* check file type */
    switch (data->file_format)
    {
        case FORMAT_PCMA:
        case FORMAT_PCMU:
        case FORMAT_PCM:
        case FORMAT_ILBC:
        case FORMAT_WAV:
            //destroyPipelines(data);
            //data->filter = data->file_format;
            createPipeline(data, PIPELINE_PLAY);
            break;

        case FORMAT_MP3:
            destroyPipelines(data);
            //data->filter = data->file_format;
            createPipeline(data, PIPELINE_PLAY_MP3);
            break;

        case FORMAT_NONE:
        default:
            ULOG_WARN("%s() - unknown file_format", G_STRFUNC);
            hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Unknown filetype!"));
            return;
            break;
    }

    if (!GST_IS_ELEMENT(data->playPipeline))
    {
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Could not create pipeline!"));
        return;
    }

    gst_element_set_state(GST_ELEMENT(data->playPipeline), GST_STATE_READY);
    gst_element_get_state(GST_ELEMENT(data->playPipeline), NULL, NULL, GST_CLOCK_TIME_NONE /* or ns */);

    /* calculate length */
    if (gst_element_query_duration (data->playPipeline, &fmt, &len))
    {
        if (len > 0)
        {
            size = GST_TIME_TO_SECS(len);
            ULOG_INFO("playSize: len:%lld size:%f", len, size);
            setLength(data, size);
        }
    }
    else 
    {
        setLength(data, 0.0);
    }
    
}

/* returns whether the action can proceed or should be aborted */
static gboolean 
closeFile(AppData *data)
{
    GtkWidget *note;
    gint i;

    if (data->saved)
        return TRUE;

    note = hildon_note_new_confirmation_add_buttons(GTK_WINDOW(data->mainView), _("Save recording?"),
            _("Yes"), GTK_RESPONSE_YES,
            _("No"), GTK_RESPONSE_NO,
            NULL);

    i = gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(note);

    switch (i)
    {
        case GTK_RESPONSE_CANCEL:
            return FALSE;

        case GTK_RESPONSE_NO:
            if (data->saveFileName)
                g_unlink(data->saveFileName);
            data->saved = TRUE;
            return TRUE;

        case GTK_RESPONSE_YES:
        {
            cbSaveAs(NULL, data);
            return data->saved;
        }
        default:
            ULOG_WARN("%s(): unknown response from dialog: %d", G_STRFUNC, i);
    }
    return FALSE;
}

static const gchar *
getFileName(AppData *data)
{
    g_assert(data);
    return gtk_label_get_text(GTK_LABEL(data->mainViewData.fileNameEntry));

}

static void cbSettings(GtkWidget* widget, AppData *data)
{
    settings_edit( widget, data );
}

static void cbEmailing(GtkWidget* widget, AppData *data)
{

    gboolean result;
    GSList *list = NULL;    
    gchar *file = NULL;

    g_assert(NULL != data);
    
    if (g_file_test(getFileName(data), G_FILE_TEST_EXISTS))
    {
        file = file2uri(getFileName(data));
        ULOG_INFO("Emailing: %s", file);
        list = g_slist_append(list, file);
        result = libmodest_dbus_client_compose_mail(data->osso,
                NULL, /*to*/
                NULL, /*cc*/
                NULL, /*bcc*/
                NULL, /*body*/
                NULL, /*subj*/
                list);
        if (!result)
            hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Emailing failed"));

        g_slist_free(list);
        g_free(file);
    }
}

static void cbSaveAs(GtkWidget* widget, AppData *data) 
{
    GtkWidget* dialog = NULL;
    const gchar *current;
    gchar *selected = NULL;

    g_assert(NULL != data);

    ULOG_DEBUG("%s() - begin", G_STRFUNC);

    current = getFileName(data);
    if (NULL == current || strcmp(current, RECORDER_FILE_UNTITLED) == 0) 
    {
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Nothing to save"));
        return;
    }

    /* create dialog */
    dialog = GTK_WIDGET(hildon_file_chooser_dialog_new(
              GTK_WINDOW(data->mainView), 
              GTK_FILE_CHOOSER_ACTION_SAVE));

    gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER(dialog), 
            get_default_dir() );

    /* show it */
    gtk_widget_show_all(dialog);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) 
    {
        selected = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    }

    ULOG_DEBUG("%s() - dialog finished", G_STRFUNC);
    gtk_widget_destroy(dialog);

    if (NULL == selected)
        return;

    ULOG_INFO("%s() - selected filename = '%s'", G_STRFUNC, selected);

    g_free(data->saveFileName);
    data->saveFileName = NULL;

    if (saveFile(selected, data->openFileName, data->file_format, &(data->saveFileName)))
    {
        gchar *basename;
        const gchar *ext;

        g_assert(data->saveFileName);

        /* set new title that has the file name */
        basename = g_path_get_basename(data->saveFileName);
        ULOG_DEBUG("%s() - file '%s' succesfully saved!", G_STRFUNC, data->saveFileName);

        /* Houston, we have a kludge:
         * for AU files we need to keep the old tmpfile for playback
         * for RAW/iLBC files, we can remove the tmpfile and point openFileName to the saved file
         */
        ext = getExtension(data->file_format);
        if (strcmp(ext, EXTENSION_AU) != 0)
        {
            g_free(data->openFileName);
            data->openFileName = g_strdup(data->saveFileName);
        }

        gtk_label_set_text(GTK_LABEL(data->mainViewData.fileNameEntry), 
                 basename);
        
        g_free(basename);
        data->saved = TRUE;
    }
    else
    {
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Saving file failed!"));
    }

    g_free(selected);
    selected = NULL;

    ULOG_DEBUG("%s() - end", G_STRFUNC);
}

static void cbRec(GtkWidget* widget, GdkEventButton *event, AppData *data) 
{
    g_assert(NULL != data);

    ULOG_DEBUG("%s() - begin", G_STRFUNC);     

    if (APPSTATE_RECORDING == getAppState(data))
    {
         if (GST_IS_ELEMENT(data->recPipeline))
         {
             gst_element_set_state(GST_ELEMENT(data->recPipeline), GST_STATE_PAUSED);
             gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry),
                                              RECORDER_MSG_PAUSED);
             hildon_banner_show_information(GTK_WIDGET(data->mainView), NULL, RECORDER_MSG_PAUSED);
             setAppState(data, APPSTATE_PAUSED);
         }
         return;
    }

    if (APPSTATE_PAUSED == getAppState(data)) 
    {
         if (GST_IS_ELEMENT(data->recPipeline))
         {
             gst_element_set_state(GST_ELEMENT(data->recPipeline),
                           GST_STATE_PLAYING);
             setAppState(data, APPSTATE_RECORDING);
             return;
         }
    }

    if (APPSTATE_READY != getAppState(data))
    {
        ULOG_WARN("%s() - state different than READY -> return", G_STRFUNC);
        if (APPSTATE_RECORDING == getAppState(data))
            cbStop(widget, NULL,data);
        return;
    }

    if (!closeFile(data))
     return;

    /* clear filenames, use tmp file */
    g_free(data->openFileName);
    data->openFileName = NULL;

    g_free(data->saveFileName);
    data->saveFileName = NULL;     

    switch (data->filter)
    {
        case FORMAT_PCM:
             data->saveFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_FILE);
             data->openFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_FILE);
             break;

         case FORMAT_PCMA:
             data->saveFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_PCMA_FILE);
             data->openFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_PCMA_FILE);
             break;
             
         case FORMAT_PCMU:
             data->saveFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_PCMU_FILE);
             data->openFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_PCMU_FILE);
             break;

         case FORMAT_WAV:
             data->saveFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_WAV_FILE);
             data->openFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_WAV_FILE);
             break;

         case FORMAT_ILBC:
         default:
             data->saveFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_ILBC_FILE);
             data->openFileName = g_strdup_printf("%s/%s", DEFAULT_TMP_DIR, DEFAULT_TMP_ILBC_FILE);
            break;
    }

    g_mkdir(DEFAULT_TMP_DIR, 755);
    
    ULOG_INFO("%s() - creating pipelines", G_STRFUNC);
    /* start recording */
    /* create related pipelines */
    if (createPipeline(data, PIPELINE_REC))
    {
        ULOG_INFO("%s() - starting recording", G_STRFUNC);
        gchar *basename;
        /* start recording */
        data->rectime = 0;
        gst_element_set_state(GST_ELEMENT(data->recPipeline), 
                           GST_STATE_PLAYING);

        /* update display */
        basename = g_path_get_basename(data->saveFileName);
        gtk_label_set_text(GTK_LABEL(data->mainViewData.fileNameEntry), 
            basename);
        g_free(basename);

        setAppState(data, APPSTATE_RECORDING);
        gtk_widget_set_sensitive(data->buttonSaveAs, TRUE);
        data->file_format = data->filter;
        setFormatString(data, data->file_format);
    }
    else
    {
        ULOG_ERR("Could not create rec pipeline!");
        hildon_banner_show_information(GTK_WIDGET(data->mainView), GTK_STOCK_DIALOG_ERROR, _("Could not create pipeline"));
        setAppState(data, APPSTATE_READY);
    }

    gtk_widget_set_sensitive(data->buttonPlay, FALSE);

    ULOG_DEBUG("%s() - end", G_STRFUNC);     
}

static void cbPlay(GtkWidget* widget, GdkEventButton *event, AppData *data) 
{
    const gchar * file = NULL;
     
    g_assert(NULL != data);

    ULOG_DEBUG("%s() - begin", G_STRFUNC);

    file = getFileName(data);
    if (NULL == data->openFileName || NULL == file || strcmp(file, RECORDER_FILE_UNTITLED) == 0) 
    {
        ULOG_WARN("%s() - nothing to play", G_STRFUNC);
        return;
    }

    if (APPSTATE_PLAYING == getAppState(data))
    {
         if (GST_IS_ELEMENT(data->playPipeline)) 
         {
             gst_element_set_state(GST_ELEMENT(data->playPipeline), GST_STATE_PAUSED);
             setAppState(data, APPSTATE_PAUSED);
         }
         return;
    }

    if (APPSTATE_PAUSED != getAppState(data)) {
        openPlayPipeline(data);
        setAppState(data, APPSTATE_READY);
    }
    
    if (APPSTATE_READY != getAppState(data) && APPSTATE_PAUSED != getAppState(data))
    {
        ULOG_WARN("%s() - state different than PLAYING or READY -> return", G_STRFUNC);
        return;
    }

    ULOG_INFO("filename %s", file);
     
    if (! GST_IS_ELEMENT(data->playPipeline))
    {
        ULOG_WARN("%s() - playPipeline does not exist", G_STRFUNC);
        return;
    }

    gst_element_set_state(GST_ELEMENT(data->playPipeline), 
               GST_STATE_PLAYING);

    setAppState(data, APPSTATE_PLAYING);

    g_timeout_add(PLAY_UPDATE_INTERVAL, (GSourceFunc)cbCheckPosition, data);

    gtk_widget_set_sensitive(data->buttonRec, FALSE);

    ULOG_DEBUG("%s() - end", G_STRFUNC);
}

static void cbStop(GtkWidget* widget, GdkEventButton *event, AppData *data) 
{
    g_assert(NULL != data);

    ULOG_DEBUG("%s() - begin", G_STRFUNC); 

    /* check if we are playing/recording */

    /* stop playing or recording */
    gtk_widget_set_state(data->buttonPlay, GTK_STATE_NORMAL);
    gtk_widget_set_state(data->buttonRec, GTK_STATE_NORMAL);
    gtk_widget_set_sensitive(data->buttonPlay, TRUE);
    gtk_widget_set_sensitive(data->buttonRec, TRUE);

    /* destroy related pipeline */
    switch(getAppState(data))
    {
        case APPSTATE_PLAYING:
            /* don't destroy the playing pipeline. Instead, set the pipeline to PAUSED */
            /* destroyPipeline(data, PIPELINE_PLAY); */
            ULOG_INFO("%s() - Setting playPipeline state to PAUSED", G_STRFUNC);
            gst_element_set_state(GST_ELEMENT(data->playPipeline), 
                     GST_STATE_PAUSED);
            setAppState(data, APPSTATE_READY);
            /* flow through */
        case APPSTATE_PAUSED:
        case APPSTATE_READY:
            /* seek to zero, but not for PCM pipeline */
            /* if (data->playPipelineType == PIPELINE_PLAY || seekToZero(data, GST_ELEMENT(data->playPipeline))) */
            if ( !GST_IS_ELEMENT(data->playPipeline) || seekToZero(data, GST_ELEMENT(data->playPipeline)))
            {
              gtk_adjustment_set_value( 
                   GTK_ADJUSTMENT(data->mainViewData.adjustment), 0);
              gtk_adjustment_value_changed(GTK_ADJUSTMENT(data->mainViewData.adjustment));
              gtk_label_set_text(GTK_LABEL(data->mainViewData.ctime), "00:00");
              gtk_label_set_text(GTK_LABEL(data->mainViewData.stateEntry), 
                         "");  
            }
            break;

        case APPSTATE_RECORDING:
        {
            gdouble len = -1.0;
            gst_element_send_event(GST_ELEMENT(data->recPipeline),
                 gst_event_new_eos());
            gtk_widget_set_sensitive(data->buttonSaveAs, TRUE);
            data->saved = FALSE;

            len = guessMediaLength(data);
            if (len > 0.0)
                setLength(data, len);

            break;
        }
     
        default:
            /* seekToZero(data, GST_ELEMENT(data->playPipeline)); */
            /* should not come here */
            break;
    }

    ULOG_DEBUG("%s() - end", G_STRFUNC); 
}


/* ui construction functions */

static void cbItemGroupChanged( gpointer data )
{
    AppData* app = (AppData* ) data;
    GValue active ={G_TYPE_INVALID};
    gint pcma,ilbc,pcm;
    
    g_value_init(&active, G_TYPE_INT);

    g_object_get_property(G_OBJECT(app->radio_pcma), "active", &active);
    pcma = g_value_get_int(&active);
    g_object_get_property(G_OBJECT(app->radio_ilbc), "active", &active);
    ilbc = g_value_get_int(&active);
    g_object_get_property(G_OBJECT(app->radio_pcm), "active", &active);
    pcm = g_value_get_int(&active);
    
    ULOG_INFO("change type pcma=%d ilbc=%d pcm=%d",pcma, ilbc, pcm);
    if ( pcma == 1 )
        app->filter = FORMAT_PCMA;
    else if ( ilbc == 1 )
        app->filter = FORMAT_ILBC;
    else if ( pcm == 1 )
        app->filter = FORMAT_WAV;
    else
        app->filter = -1;
  
    ULOG_INFO("filter type=%d", app->filter);
}

/* Create the menu items needed for the main view */
static void createMenu( AppData *data )
{
    HildonAppMenu *menu;
    GtkWidget *button_new;
    GtkWidget *button_open;
    GtkWidget *button_save;
    GtkWidget *button_email;

    menu = HILDON_APP_MENU( hildon_app_menu_new() );
    button_new = hildon_gtk_button_new(HILDON_SIZE_AUTO);
    gtk_button_set_label( GTK_BUTTON(button_new), "New");
    button_open = hildon_gtk_button_new(HILDON_SIZE_AUTO);
    gtk_button_set_label( GTK_BUTTON(button_open), "Open");
    button_save = hildon_gtk_button_new(HILDON_SIZE_AUTO);
    data->buttonSaveAs = button_save;
    gtk_widget_set_sensitive(data->buttonSaveAs, FALSE);
    gtk_button_set_label( GTK_BUTTON(button_save), "Save");
    button_email = hildon_gtk_button_new(HILDON_SIZE_AUTO);
    gtk_button_set_label( GTK_BUTTON(button_email), "Send via email");

    hildon_app_menu_append( menu, GTK_BUTTON(button_new));
    hildon_app_menu_append( menu, GTK_BUTTON(button_open));
    hildon_app_menu_append( menu, GTK_BUTTON(button_save));
    hildon_app_menu_append( menu, GTK_BUTTON(button_email));
    g_signal_connect(G_OBJECT(button_new), "clicked",
              G_CALLBACK(cbNew),
              data);
    g_signal_connect(G_OBJECT(button_open), "clicked",
              G_CALLBACK(cbOpen),
              data);
    g_signal_connect(G_OBJECT(button_save), "clicked",
              G_CALLBACK(cbSaveAs),
              data); 
    g_signal_connect( G_OBJECT( button_email ), "clicked",
        GTK_SIGNAL_FUNC (cbEmailing), data);
    
    gtk_widget_show_all( GTK_WIDGET(menu));

    hildon_window_set_app_menu(HILDON_WINDOW(data->mainView), menu);

    data->filter = get_default_filter();

    setFormatString(data, data->filter);

}

static gboolean
cbScaleRelease(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
    AppData* app = (AppData* ) data;
  
    if (getAppState(app) == APPSTATE_RECORDING || NULL == app->playPipeline)
        return FALSE;

    seekToTime(app->playPipeline, gtk_adjustment_get_value(GTK_ADJUSTMENT(app->mainViewData.adjustment)));
  
    return FALSE;

}

static gboolean
evKeypress(GtkWidget *widget, GdkEventKey *ev, AppData *appdata)
{

  switch (ev->keyval)
  {
    case GDK_Return:
      cbRec(widget, NULL, appdata);
      return TRUE;
    case GDK_Right:
      cbPlay(widget, NULL, appdata);
      return TRUE;
    case GDK_Escape:
      cbStop(widget, NULL, appdata);
      return TRUE;
    default:
      break;
  }

  return FALSE;
}


gboolean maemo_recorder_ui_new(AppData *data)
{
    HildonProgram *app = NULL;
    HildonWindow *window = NULL;
    GtkWidget *hbox = NULL;
    GtkWidget *ctime = NULL;
    GtkWidget *etime = NULL;
    GtkWidget *filename = NULL;
    GtkWidget *format = NULL;
    GtkWidget *state = NULL;
    GtkWidget *table = NULL;
    GtkWidget *scale = NULL;
    GtkObject *adjustment = NULL;

    GtkWidget *rec = NULL;
    GtkWidget *play = NULL;
    GtkWidget *stop = NULL;
    GtkWidget *recimage = NULL;
    GtkWidget *playimage = NULL;
    GtkWidget *stopimage = NULL;

    g_assert(NULL != data);

    app = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name(RECORDER_APP_TITLE);
     
    /* main window */
    window = HILDON_WINDOW(hildon_window_new());

    hildon_program_add_window(app, window);

    /* content for main view */

    /* create hbox to divide control area */
    hbox = gtk_hbox_new(FALSE, HILDON_MARGIN_DEFAULT);

    /* create toolbar */

    /* create table for labels */
    table = gtk_table_new (6, 6, TRUE);
    gtk_table_set_homogeneous(GTK_TABLE(table), TRUE);

    gtk_table_set_row_spacings (GTK_TABLE (table), 4);
    gtk_table_set_col_spacings (GTK_TABLE (table), HILDON_MARGIN_TRIPLE);

    filename = gtk_label_new(_(RECORDER_FILE_UNTITLED));
    format = gtk_label_new(RECORDER_FMT_STRING_NONE);
    state = gtk_label_new("");
    adjustment = gtk_adjustment_new (0.00,
                  0.00,
                  100.00,
                  0.01,
                  0.01,
                  0);

    ctime = gtk_label_new("00:00");
    etime = gtk_label_new("00:00");
    scale = gtk_hscale_new(GTK_ADJUSTMENT(adjustment));
    gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), ctime, FALSE, FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_end(GTK_BOX(hbox), etime, FALSE, FALSE, HILDON_MARGIN_DOUBLE);
    gtk_box_pack_end(GTK_BOX(hbox), scale, TRUE, TRUE, HILDON_MARGIN_DOUBLE);
    gtk_box_set_homogeneous(GTK_BOX(hbox), FALSE);
      
    gtk_table_attach_defaults(GTK_TABLE(table), filename,
                    1, 4, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table), format,
                    1, 4, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE(table), state,
                    1, 4, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox,
                    0, 5, 5, 6);


    recimage = gtk_image_new_from_file(REC_ICON);
    rec = gtk_event_box_new();
    data->buttonRec = GTK_WIDGET(rec);
    gtk_container_add(GTK_CONTAINER(rec), recimage);
    playimage = gtk_image_new_from_file(PLAY_ICON);
    play = gtk_event_box_new();
    data->buttonPlay = GTK_WIDGET(play);
    gtk_container_add(GTK_CONTAINER(play), playimage);
    stopimage = gtk_image_new_from_file(STOP_ICON);
    stop = gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(stop), stopimage);

    gtk_table_attach_defaults(GTK_TABLE(table), rec,
                    5, 6, 4, 6);
    gtk_table_attach_defaults(GTK_TABLE(table), play,
                    5, 6, 2, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), stop,
                    5, 6, 0, 2);


    /* connect signals */
    g_signal_connect(G_OBJECT(scale), "button-release-event", G_CALLBACK(cbScaleRelease), data);
    g_signal_connect(G_OBJECT(window), "delete_event", G_CALLBACK(cbDestroy), data);
    g_signal_connect(G_OBJECT(window), "key-press-event",
            G_CALLBACK(evKeypress), data);

     g_signal_connect(G_OBJECT(rec), "button-release-event",
              G_CALLBACK(cbRec),
              data);
     g_signal_connect(G_OBJECT(play), "button-release-event",
              G_CALLBACK(cbPlay),
              data);
     g_signal_connect(G_OBJECT(stop), "button-release-event",
              G_CALLBACK(cbStop),
              data);


    /* packing the view */
    gtk_container_add(GTK_CONTAINER(window), table);
    
    /* store needed widgets */
    data->app = app;
    data->mainView = window;
    data->mainViewData.fileNameEntry = GTK_WIDGET(filename);
    data->mainViewData.formatEntry = GTK_WIDGET(format);
    data->mainViewData.lengthEntry = GTK_WIDGET(etime);
    data->mainViewData.ctime = GTK_WIDGET(ctime);
    data->mainViewData.stateEntry = GTK_WIDGET(state);
    data->mainViewData.adjustment = GTK_OBJECT(adjustment);

    /* show the app */
    gtk_widget_show_all(GTK_WIDGET(window));

    createMenu(data);
    
    return TRUE;
}

void 
maemo_recorder_mime_open(gpointer user_data, gint argc, gchar **argv)
{
    AppData *data;

    ULOG_DEBUG("%s with %d arguments", __FUNCTION__, argc);

    if (argc == 0)
        return;

    g_assert(user_data);
    data = (AppData *) user_data;

    if (argv[0] != NULL)
    {
        ULOG_DEBUG("request to open %s", argv[0]);
        g_free(data->mimeURI);
        data->mimeURI = g_strdup(argv[0]);
        g_idle_add(openURI, (gpointer) data); 
        gtk_window_present(GTK_WINDOW(data->mainView));
    }
}

static void seekToTime(GstElement *pipeline, gdouble secs)
{
    g_assert(NULL != pipeline);
    ULOG_DEBUG("Seeking to: %.2f", secs);

    /* time must be nanoseconds */
    if (!gst_element_seek(pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
            GST_SEEK_TYPE_SET, (gint64) (secs * GST_SECOND),
            GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE))
    {
        ULOG_WARN("seekToTime failed!");
        return;
    }
    ULOG_DEBUG("seekToTime succeeded");
}

static gboolean seekToZero(AppData *data, GstElement *pipeline)
{
    gint plType;
    g_assert(NULL != pipeline);
    ULOG_DEBUG("Seeking to zero");

    /* time must be nanoseconds */
    if (!gst_element_seek(pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH,
            GST_SEEK_TYPE_SET, (gint64) 0,
            GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE))
    {
        ULOG_ERR("seekToZero failed! Trying to destroy and re-create pipeline");
        plType = data->playPipelineType;

        /* gst_element_set_state(pipeline, GST_STATE_READY); */
        destroyPipeline(data, plType);
        return createPipeline(data, plType);
    }

    ULOG_DEBUG("seekToZero succeeded");
    return TRUE;
}

static void
setFormatString(AppData *data, AudioFormat afmt)
{
    gchar *str;
    gchar *format;

    /* these are pretty much always the same */
    gint channels = 1; 
    gint rate = DEFAULT_RATE; /* 8000 */
    gint bits = 8;

    g_assert(data);
    g_assert(GTK_IS_LABEL(data->mainViewData.formatEntry));

    switch (afmt)
    {
        case FORMAT_PCMA:
            format = FORMAT_NAME_PCMA;
            break;
        case FORMAT_PCMU:
            format = FORMAT_NAME_PCMU;
            break;
        case FORMAT_ILBC:
            format = FORMAT_NAME_ILBC;
            rate = ILBC_RATE;
            break;
        /* TODO: we can play wavs with many sampling rates, 2 channels */
        /* we really should migrate to the better format spec */
        case FORMAT_WAV:
            format = FORMAT_NAME_WAV;
            bits = PCM_WIDTH;
            break;
        case FORMAT_PCM:
            format = FORMAT_NAME_PCM;
            bits = PCM_WIDTH;
            break;
        default:
            gtk_label_set_text(GTK_LABEL(data->mainViewData.formatEntry), RECORDER_FMT_STRING_NONE);
            return;
    }

    str = g_strdup_printf("%s, %d %s, %d kHz, %d %s", format, channels, _("ch"), rate/1000, bits, _("bits"));
    gtk_label_set_text(GTK_LABEL(data->mainViewData.formatEntry), str);
    g_free(str);
}

static void setLength(AppData *data, gdouble secs)
{
    guint mins = 0;
    gchar *tmp;

    if (secs < 0.0)
        return;

    if (secs > 0)
    {
        g_object_set(G_OBJECT(data->mainViewData.adjustment), 
            "upper", secs, 
            NULL);    
        gtk_adjustment_changed(GTK_ADJUSTMENT(data->mainViewData.adjustment));
    }

    if (secs >= 60.0)
    {
        mins = secs / 60;
        secs -= mins * 60.0;
    }

    tmp = g_strdup_printf("%02u:%02d", mins, (int)secs);

    /*
    ULOG_INFO("Setting length to %s", tmp);
    */
    gtk_label_set_text(GTK_LABEL(data->mainViewData.lengthEntry), 
                 tmp);
    g_free(tmp);
}

static gdouble guessMediaLength(AppData *data)
{
    GnomeVFSFileSize size = 0;
    gdouble bitrate = 0.0;
    gdouble len = -1.0;

    if (data->openFileName)
        size = getFileLength(data->openFileName);
    else
        return -1.0;
           
    if (size == 0)
        return -1.0;

    ULOG_DEBUG("file size: %llu bytes", size);

    switch (data->file_format)
    {
        case FORMAT_ILBC:
            bitrate = ILBC_BITRATE_30;
            break; 
            
        case FORMAT_PCMA:
        case FORMAT_PCMU:
            bitrate = PCMA_BITRATE;
            break;

        default:
            return -1.0;
    }

    if (bitrate == 0.0)
        return -1.0;

    len = (((gdouble) size) * 8.0) / (bitrate);
    ULOG_DEBUG("guessed media length: %.2f secs", len);

    return len;
}

static GstCaps *createCapsFilter(AudioFormat format)
{
    switch (format)
    {
        case FORMAT_ILBC:
            return gst_caps_new_simple(
                GST_TYPE_ILBC,
                "rate", G_TYPE_INT, ILBC_RATE,
                "channels", G_TYPE_INT, DEFAULT_CHANNELS,
                "mode", G_TYPE_INT, 30, /* 30 ms frames */
                NULL);
        case FORMAT_PCMA:
            return gst_caps_new_simple(
                GST_TYPE_PCMA,
                "rate", G_TYPE_INT, DEFAULT_RATE,
                "channels", G_TYPE_INT, DEFAULT_CHANNELS,
                NULL);
        case FORMAT_PCMU:
            return gst_caps_new_simple(
                GST_TYPE_PCMU,
                "rate", G_TYPE_INT, DEFAULT_RATE,
                "channels", G_TYPE_INT, DEFAULT_CHANNELS,
                NULL);
        case FORMAT_WAV:
        case FORMAT_PCM:
            return gst_caps_new_simple(
                GST_TYPE_PCM,
                "rate", G_TYPE_INT, PCM_RATE,
                "signed", G_TYPE_BOOLEAN, TRUE,
                "channels", G_TYPE_INT, DEFAULT_CHANNELS,
                "endianness", G_TYPE_INT, PCM_ENDIANNESS,
                "width", G_TYPE_INT, PCM_WIDTH,
                "depth", G_TYPE_INT, PCM_DEPTH,
                NULL);
        default:
            ULOG_WARN("%s(): creating ANY caps", G_STRFUNC);
            return gst_caps_new_any();
    }
}

static gboolean cbStopPlayback(AppData *data)
{
    ULOG_INFO("Stopping playback");
   
    g_assert(data != NULL);
    destroyPipelines(data);
    setAppState(data, APPSTATE_READY);
    gtk_widget_set_state(data->buttonPlay, GTK_STATE_NORMAL);

    return FALSE;
}

static gboolean cbUpdateRecLength(AppData *data)
{
    guint mins = 0;
    gdouble secs;
    gchar *tmp;

    data->rectime += REC_UPDATE_INTERVAL/1000.0;

    mins = data->rectime / 60;
    secs = data->rectime - (mins * 60.0);
    tmp = g_strdup_printf("%02u:%02d", mins, (int)secs);

    gtk_label_set_text(GTK_LABEL(data->mainViewData.lengthEntry), 
                 tmp);
    g_free(tmp);

    if (getAppState(data) == APPSTATE_RECORDING)
        return TRUE;

    data->recUpdateId = 0;
    return FALSE;
}


