#ifndef SPEEDMATCH_H
#define SPEEDMATCH_H

#include <QWidget>
#include <QPainter>
#include <QTimer>
#include <QMouseEvent>
#include <QDebug>
#include <QCoreApplication>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>


class SpeedMatch : public QWidget
{
    Q_OBJECT
public:

    SpeedMatch( QWidget* aParent = 0 )
        :QWidget( aParent ),mCurrentShape(0),mPrevShape(0),mPoint(0),mGameCounter(60)
    {        
        mFuncPtrList.append( &SpeedMatch::drawCircle );
        mFuncPtrList.append( &SpeedMatch::drawCircle2 );
        mFuncPtrList.append( &SpeedMatch::drawRect );
        mFuncPtrList.append( &SpeedMatch::drawRect2 );
        mFuncPtrList.append( &SpeedMatch::drawTriangle );

        mNotMatch = new QPushButton("Not Match");
        connect(mNotMatch,SIGNAL(clicked()),this,SLOT(answerNotMatch()));

        mMatch = new QPushButton("Match");
        connect(mMatch,SIGNAL(clicked()),this,SLOT(answerMatch()));

        mStartBtn = new QPushButton("Start");
        connect(mStartBtn,SIGNAL(clicked()),this,SLOT(startGame()));

        mNotMatch->setVisible(false);
        mMatch->setVisible(false);
        mStartBtn->setVisible(true);

        QHBoxLayout* buttonLayout = new QHBoxLayout;
        buttonLayout->addStretch();
        buttonLayout->addWidget(mNotMatch);
        buttonLayout->addStretch();
        buttonLayout->addWidget( mStartBtn );
        buttonLayout->addStretch();
        buttonLayout->addWidget(mMatch);
        buttonLayout->addStretch();
        buttonLayout->addSpacing(100);

        QVBoxLayout* mainLayout = new QVBoxLayout(this);
        mainLayout->addStretch(300);
        mainLayout->addLayout( buttonLayout );

        this->resize(800,480);
    }

public slots:

    void newGame()
    {
        mGameCounter = 60;
        mPrevShape = 0;
        mCurrentShape = 0;
        mDrawAnswer = false;
    }

private slots:

    void startGame()
    {
        mNotMatch->setVisible(true);
        mMatch->setVisible(true);
        mStartBtn->setVisible(false);

        generateRandomShape();
    }

    void answerMatch()
    {
        if( mCurrentShape == mPrevShape ) {
            mPoint++;           
        }
        mAnswer = ( mCurrentShape == mPrevShape );
        mDrawAnswer = true;
        QTimer::singleShot( 500,this,SLOT(nextShape()));
        repaint();
    }

    void answerNotMatch()
    {
        if( mCurrentShape != mPrevShape ) {
            mPoint++;
        }
        mAnswer = ( mCurrentShape != mPrevShape );
        mDrawAnswer = true;
        QTimer::singleShot( 500,this,SLOT(nextShape()));
        repaint();
    }

    void nextShape()
    {
        mDrawAnswer = false;
        mGameCounter --;
        generateRandomShape();
    }

    void paintEvent( QPaintEvent* /*aEvent*/ )
    {
        QPainter painter(this);

        paintGame( &painter );
    }

    void paintGame( QPainter* aPainter )
    {
        if ( mGameCounter < 0 ) {
            mNotMatch->setVisible(false);
            mMatch->setVisible(false);
            mStartBtn->setVisible(false);

            QString str("Your points are \n%1");
            str = str.arg( mPoint );
            paintText( aPainter, str, Qt::green, 80 );
            return;
        }

        FuncPtr funcPtr = mFuncPtrList[mCurrentShape];
        (this->*funcPtr)( aPainter);

        aPainter->drawText( 650 , 160, 100, 40 ,Qt::AlignHCenter ,"Score");
        aPainter->drawText( 650 , 200, 100, 40 ,Qt::AlignHCenter ,QString().setNum( mPoint));

        aPainter->drawText( 650 , 0, 100, 80 ,Qt::AlignCenter ,"Exit");
        aPainter->drawText( 650 , 80, 100, 80 ,Qt::AlignCenter ,"Menu");

        if( mDrawAnswer ) {
            paintAnswer( aPainter );
        }
    }

    void paintAnswer( QPainter* aPainter )
    {
        QPen oldPen = aPainter->pen();
        if( mAnswer ) {
            paintCorrect( aPainter );
        } else {
            paintWrong( aPainter );
        }
        aPainter->setPen( oldPen );
    }

    QPen createPen( QColor aColor )
    {
        QPen pen;  // creates a default pen
        pen.setStyle(Qt::SolidLine);
        pen.setWidth(15);
        pen.setBrush(aColor);
        pen.setCapStyle(Qt::RoundCap);
        pen.setJoinStyle(Qt::RoundJoin);
        return pen;
    }

    void paintCorrect( QPainter* aPainter )
    {                
        aPainter->setPen(createPen(Qt::green));

        aPainter->drawLine( QPoint(275,150),QPoint(300,250));
        aPainter->drawLine( QPoint(300,250),QPoint(425,150));
    }

    void paintWrong( QPainter* aPainter )
    {
        aPainter->setPen(createPen(Qt::red));

        aPainter->drawLine( QPoint(275,125),QPoint(425,275));
        aPainter->drawLine( QPoint(425,125),QPoint(275,275));
    }

    void drawCircle( QPainter* aPainter)
    {
        QRect rect( 250,100,200,200);
        QBrush brush( Qt::SolidPattern);
        brush.setColor(Qt::blue);

        aPainter->setBrush( brush );
        aPainter->drawEllipse( rect );
    }

    void drawCircle2( QPainter* aPainter)
    {        
        QBrush brush( Qt::SolidPattern);

        brush.setColor(Qt::blue);
        aPainter->setBrush( brush );
        aPainter->drawEllipse( QRect( 250,100,200,200));

        brush.setColor(Qt::red);
        aPainter->setBrush( brush );
        aPainter->drawEllipse( QRect( 275,125,150,150) );

        brush.setColor(Qt::white);
        aPainter->setBrush( brush );
        aPainter->drawEllipse( QRect( 300,150,100,100) );
    }

    void drawRect( QPainter* aPainter )
    {
        QRect rect( 250,100,200,200);
        QBrush brush( Qt::SolidPattern);
        brush.setColor(QColor(199,30,210));

        aPainter->setBrush( brush );
        aPainter->drawRoundedRect(rect,5,5);
    }

    void drawRect2( QPainter* aPainter )
    {        
        QBrush brush( Qt::SolidPattern);

        brush.setColor(QColor(199,30,210));
        aPainter->setBrush( brush );
        aPainter->drawRoundedRect(QRect( 250,100,200,200),5,5);

        brush.setColor(Qt::green );
        aPainter->setBrush( brush );
        aPainter->drawRoundedRect(QRect( 300,150,100,100),5,5);
    }

    void drawTriangle( QPainter* aPainter )
    {       
        QBrush brush( Qt::SolidPattern);
        brush.setColor(Qt::gray);
        aPainter->setBrush(brush);

        QPoint points[4] = {
             QPoint(350, 100),
             QPoint(250, 300),
             QPoint(450, 300)
         };

         aPainter->drawPolygon(points, 3);
    }

    void paintText( QPainter* aPainter, const QString& aString,const QColor& aColor = Qt::black ,int aFontSize = 100 )
    {
        aPainter->setPen( aColor );

        QFont font = aPainter->font();
        font.setBold( true );
        font.setPixelSize( aFontSize );
        aPainter->setFont( font );
        aPainter->drawText( 0,0, 800, 480,Qt::AlignCenter, aString );
    }

    void gameTimer()
    {
        repaint();
    }

    void mousePressEvent(QMouseEvent *aEvent)
    {
        if( mGameCounter < 0 ) {
            emit showMenu();
            return;
        }

        int x = aEvent->pos().x() / 100;
        int y = aEvent->pos().y() / 80;
        QPoint mousePoint(x,y);

        if( mousePoint == QPoint( 7,0 )) {
            QCoreApplication::quit();
        }
        else if( mousePoint == QPoint( 7,1 )) {
            emit showMenu();
            return;
        }

        repaint();
    }

private:

    void generateRandomShape()
    {
        int randomness = qrand() % 2;
        if (randomness == 1 ) {
            mPrevShape = mCurrentShape;
            mCurrentShape = qrand() % 5;
        } else {
            mPrevShape = mCurrentShape;
        }
        repaint();
    }

signals:

    void showMenu();

private:

    int mCurrentShape;
    int mPrevShape; 
    int mPoint;
    int mGameCounter;

    bool mDrawAnswer;
    bool mAnswer;

    QPushButton* mMatch;
    QPushButton* mNotMatch;
    QPushButton* mStartBtn;    

    typedef void (SpeedMatch::*FuncPtr)( QPainter*) ;
    QList<FuncPtr> mFuncPtrList;
};

#endif // SPEEDMATCH_H
