/* -*- c-basic-offset: 8 -*-
   rdesktop: A Remote Desktop Protocol client.
   User interface services - X Window System
   Copyright (C) Matthew Chapman 1999-2008
   Copyright 2007 Pierre Ossman <ossman@cendio.se> for Cendio AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <time.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <arpa/inet.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include <jpeglib.h>
#include <jerror.h>

/* for stat(), unlink() */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <libosso.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon.h>

#include "rdesktop.h"
#include "appdata.h"
#include "ui-maemo.h"

extern Window g_wnd;
extern Display *g_display;
extern char g_keymapname[PATH_MAX];
extern char g_username[64];
extern uint32 g_num_devices;
extern RD_BOOL g_loggedin;
extern int g_width;
extern int g_height;
extern int g_server_depth;
extern RD_BOOL g_fullscreen;

/********** private **********/

#define M_TOGGLE_IMAGE_JPG "/usr/share/icons/hicolor/40x40/hildon/rdesktop_toggle.jpg"
#define M_HELP_IMAGE_JPG "/usr/share/icons/hicolor/40x40/hildon/rdesktop_help.jpg"
#define M_TOGGLE_WINDOW_WIDTH 70
#define M_TOGGLE_WINDOW_HEIGHT 70

#define _NET_WM_STATE_REMOVE        0    /* remove/unset property */
#define _NET_WM_STATE_ADD           1    /* add/set property */
#define _NET_WM_STATE_TOGGLE        2    /* toggle property  */

#define M_HELP_FILE_MAIN	"/usr/share/rdesktop_mui/rdesktop__help.pdf"
#define M_HELP_FILE		"/usr/share/rdesktop_mui/rdesktop_help.pdf"
#define M_HELP_VIEWER		"run-standalone.sh dbus-send --print-reply --dest=com.nokia.osso_pdfviewer " \
				"/com/nokia/osso_pdfviewer com.nokia.osso_pdfviewer.mime_open string:'file://%s'"
#define M_INTERACTIVE_MODE	"empty-server-arg"
#define M_GCONF_ROOT		"/apps/rdesktop"
#define M_RESPONSE_NEW		1
#define M_RESPONSE_SELECTED	2
#define M_RESPONSE_DEL		3
#define M_RESPONSE_CONNECT	4

#ifndef u_char
#define u_char unsigned char
#endif

static int m_fsicon_x_location;
static int m_fsicon_y_location;

struct srv_info {
	time_t stamp;
	gchar *server;
	gchar *username;
	gchar *password;
	gchar *domain;
	gchar *kblayout;
	gint bpp;
	gboolean mount_mmc, mount_mydocs, btkbd;
};

struct {
    HildonProgram *app; /* handle to application */
    HildonWindow *main_view; /* handle to app's view */
    osso_context_t *osso; /* handle to osso */
    GConfClient *gconf;
    GSList *servers; /* List of available RDP servers. */
    GtkListStore *list_store; /* Stores the list of servers for the UI. */
    int selected_server;
} data;

static gboolean m_mount_mmc = FALSE;
static gboolean m_mount_mydocs = FALSE;

static XImage *m_fsicon;
static XImage *m_helpicon;

static GtkWidget *m_progressbar;
static GMutex *m_mutex;
static GThread *m_thread;
static int m_progress_done;

static struct {
	const char *label;
	const char *key;
} m_kblayout_options[] = {
	{ "American Dvorak", "en-dv" },
	{ "Czech", "cs" },
	{ "Dansk  (Danish)", "da" },
	{ "English-UK", "en-gb" },
	{ "English-US", "en-us" },
	{ "Estonian", "et" },
	{ "Finnish", "fi" },
	{ "French", "fr" },
	{ "German", "de" },
	{ "Hebrew", "he" },
	{ "Hungarian", "hu" },
	{ "Italiano (Italian)", "it" },
	{ "Japanese", "ja" },
	{ "Korean", "ko" },
	{ "Macedonian", "mk" },
	{ "Nederlands (Dutch)", "nl" },
	{ "Polish", "pl" },
	{ "Portuguese", "pt" },
	{ "Portuguese-Brasil", "pt-br" },
	{ "Russian", "ru" },
	{ "Spanish", "es" },
	{ "Swiz-German", "de-ch" },
	{ "Thai", "th" },
	/*
	{ "", "is" },
	{ "", "hr" },
	{ "", "lt" },
	{ "", "lv" },
	{ "", "nl-be" },
	{ "", "no" },
	{ "", "fo" },
	{ "", "fr-be" },
	{ "", "fr-ca" },
	{ "", "fr-ch" },
	{ "", "sl" },
	{ "", "sv" },
	{ "", "tr" },
	*/
};

static void
m_jpeg_error_exit (j_common_ptr cinfo) {
	        cinfo->err->output_message (cinfo);
		        exit (EXIT_FAILURE);
}

static int
m_get_byte_order (void) {
	union {
		char c[sizeof(short)];
		short s;
	} order;

	order.s = 1;
	if ((1 == order.c[0])) {
		return LSBFirst;
	} else {
		return MSBFirst;
	}
}

static u_char *
m_decode_jpeg (char *filename, int *widthPtr, int *heightPtr) {
	        register JSAMPARRAY lineBuf;
		struct jpeg_decompress_struct cinfo;
		struct jpeg_error_mgr err_mgr;
		int bytesPerPix;
		FILE *inFile;
		u_char *retBuf;

		inFile = fopen (filename, "rb");
		if (NULL == inFile) {
			perror (NULL);
			return NULL;
		}

		cinfo.err = jpeg_std_error (&err_mgr);
		err_mgr.error_exit = m_jpeg_error_exit;

		jpeg_create_decompress (&cinfo);
		jpeg_stdio_src (&cinfo, inFile);
		jpeg_read_header (&cinfo, 1);
		cinfo.do_fancy_upsampling = 0;
		cinfo.do_block_smoothing = 0;
		jpeg_start_decompress (&cinfo);

		*widthPtr = cinfo.output_width;
		*heightPtr = cinfo.output_height;
		bytesPerPix = cinfo.output_components;


		lineBuf = cinfo.mem->alloc_sarray ((j_common_ptr) &cinfo, JPOOL_IMAGE, (*widthPtr * bytesPerPix), 1);
		retBuf = malloc (3 * (*widthPtr * *heightPtr));

		if (NULL == retBuf) {
			perror (NULL);
			return NULL;
		}

		if (3 == bytesPerPix) {
			int lineOffset = (*widthPtr * 3);
			int x;
			int y;

			for (y = 0; y < cinfo.output_height; ++y) {
				jpeg_read_scanlines (&cinfo, lineBuf, 1);

				for (x = 0; x < lineOffset; ++x) {
					retBuf[(lineOffset * y) + x] = lineBuf[0][x];
					++x;
					retBuf[(lineOffset * y) + x] = lineBuf[0][x];
					++x;
					retBuf[(lineOffset * y) + x] = lineBuf[0][x];
				}
			}
		} else if (1 == bytesPerPix) {

			unsigned int col;
			int lineOffset = (*widthPtr * 3);
			int lineBufIndex;
			int x ;
			int y;

			for (y = 0; y < cinfo.output_height; ++y) {
				jpeg_read_scanlines (&cinfo, lineBuf, 1);

				lineBufIndex = 0;
				for (x = 0; x < lineOffset; ++x) {
					col = lineBuf[0][lineBufIndex];

					retBuf[(lineOffset * y) + x] = col;
					++x;
					retBuf[(lineOffset * y) + x] = col;
					++x;
					retBuf[(lineOffset * y) + x] = col;

					++lineBufIndex;
				}
			}
		} else {
			fprintf (stderr, "Error: the number of color channels is %d.  This program only handles 1 or 3\n", bytesPerPix);
			return NULL;
		}
		jpeg_finish_decompress (&cinfo);
		jpeg_destroy_decompress (&cinfo);
		fclose (inFile);

		return retBuf;
}

static XImage *
m_create_image_from_buffer (Display *dis, int screen, u_char *buf, int width, int height)
{
	int depth;
	XImage *img = NULL;
	Visual *vis;
	double rRatio;
	double gRatio;
	double bRatio;
	int outIndex = 0;
	int i;
	int numBufBytes = (3 * (width * height));

	depth = DefaultDepth (dis, screen);
	vis = DefaultVisual (dis, screen);

	rRatio = vis->red_mask / 255.0;
	gRatio = vis->green_mask / 255.0;
	bRatio = vis->blue_mask / 255.0;

	if (depth >= 24) {
		size_t numNewBufBytes = (4 * (width * height));
		u_int32_t *newBuf = malloc (numNewBufBytes);

		for (i = 0; i < numBufBytes; ++i) {
			unsigned int r, g, b;
			r = (buf[i] * rRatio);
			++i;
			g = (buf[i] * gRatio);
			++i;
			b = (buf[i] * bRatio);

			r &= vis->red_mask;
			g &= vis->green_mask;
			b &= vis->blue_mask;

			newBuf[outIndex] = r | g | b;
			++outIndex;
		}
		img = XCreateImage (dis,
				CopyFromParent, depth,
				XYPixmap, 0,
				(char *) newBuf,
				width, height,
				32, 0
				);
	} else if (depth >= 15) {
		size_t numNewBufBytes = (2 * (width * height));
		u_int16_t *newBuf = malloc (numNewBufBytes);

		for (i = 0; i < numBufBytes; ++i) {
			unsigned int r, g, b;

			r = (buf[i] * rRatio);
			++i;
			g = (buf[i] * gRatio);
			++i;
			b = (buf[i] * bRatio);

			r &= vis->red_mask;
			g &= vis->green_mask;
			b &= vis->blue_mask;

			newBuf[outIndex] = r | g | b;
			++outIndex;
		}

		img = XCreateImage (dis,
				CopyFromParent, depth,
				ZPixmap, 0,
				(char *) newBuf,
				width, height,
				16, 0
				);
	} else {
		fprintf (stderr, "This program does not support displays with a depth less than 15.");
		return NULL;
	}

	XInitImage (img);
	/*Set the client's byte order, so that XPutImage knows what to do with the data.*/
	/*The default in a new X image is the server's format, which may not be what we want.*/
	if ((LSBFirst == m_get_byte_order ())) {
		img->byte_order = LSBFirst;
	} else {
		img->byte_order = MSBFirst;
	}

	/*The bitmap_bit_order doesn't matter with ZPixmap images.*/
	img->bitmap_bit_order = MSBFirst;

	return img;
}

static void
m_set_str(char **dest, const char *src)
{
	g_free(*dest);
	*dest = g_strdup(src);
}

static gboolean
m_gconf_client_set_string(GConfClient *client,
		const gchar *key, const gchar *val, GError **err)
{
	if (val) {
		return gconf_client_set_string(client, key, val, err);
	} else {
		return gconf_client_set_string(client, key, "", err);
	}
}

static gboolean m_progress_check(gpointer data)
{
	gboolean ret = TRUE;
	g_mutex_lock(m_mutex);
	if (m_progress_done) {
		gtk_widget_hide_all(m_progressbar);
		gtk_widget_destroy(m_progressbar);
		gtk_main_quit();
		ret = FALSE;
	}
	g_mutex_unlock(m_mutex);
	return ret;
}

static gpointer m_progress_thread(gpointer notused)
{
	gchar *msg;
	msg = g_strdup_printf("RDesktop connecting to %s...", ui_maemo_server);
	m_progressbar = hildon_banner_show_animation(GTK_WIDGET(data.main_view),
			NULL, msg);
	g_free(msg);
	gtk_timeout_add(500, m_progress_check, NULL);
	gtk_main();
	return NULL;
}

/**
 * @return	0 = no
 * @return	1 = yes
 */
static int m_show_confirm_msg(char *msg)
{
	int ret = 0;
	GtkWidget *dialog;

	dialog = hildon_note_new_confirmation(NULL, (gchar*)msg);
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		ret = 1;
	}
	gtk_widget_destroy(dialog);
	return ret;
}

static void m_server_entry_changed(GtkEditable *editable, gpointer data)
{
	GtkDialog *dialog = GTK_DIALOG(data);
	GtkEntry *entry = GTK_ENTRY(editable);
	const gchar *entry_text = gtk_entry_get_text(entry);
	if (entry_text && entry_text[0] != '\0') {
		/* This is the only mandatory field */
		gtk_dialog_set_response_sensitive(dialog, 2, TRUE);
	} else {
		gtk_dialog_set_response_sensitive(dialog, 2, FALSE);
	}
}

//static void m_event(GtkWindow *window,  GdkEvent  *event, void *notused)
//{
//	if (event->type==GDK_EXPOSE) {
//		if (m_iconify) {
//			/* FIXME: hack hack hack! */
//			gtk_window_iconify(GTK_WINDOW(data.main_view));
//			m_iconify=FALSE;
//		}
//	}
//}

static gint m_dbus_callback (const gchar *interface, const gchar *method,
               GArray *arguments, gpointer data,
               osso_rpc_t *retval)
{
	DEBUG_FG("%s, %s", interface, method);

	if (strcmp (method, "top_application") == 0) {
		gtk_window_present (GTK_WINDOW (data));
	}

	retval->type = DBUS_TYPE_INVALID;
	return OSSO_OK;
}

static int
m_mainwin_create(void)
{
	/* NOTE: 2010.02.16: We must show the main window. Otherwise
	 * Maemo will not recognize that we are already running.
	 * This would cause an annoying spinning animation shown in
	 * the title bar and the PDF Viewer would be opened behind
	 * the our dialog windows.
	 */
	gtk_widget_show_all(GTK_WIDGET(data.main_view));

	data.gconf = gconf_client_get_default();

	return 0;
}

static void
m_mainwin_destroy(void)
{
	if (data.main_view) {
		gtk_widget_hide_all(GTK_WIDGET(data.main_view));
	}
	if (data.gconf) {
		g_object_unref(data.gconf);
		data.gconf = NULL;
	}
}

static int
m_help_exists(void)
{
	return (access(M_HELP_FILE_MAIN, R_OK) == 0 ||
		access(M_HELP_FILE, R_OK) == 0);
}

static void
m_show_help(GtkButton *btn, gpointer data)
{
	gchar *cmd = NULL;
	gchar *f = NULL, *ff = NULL;
	struct stat st;

	if (access(M_HELP_FILE_MAIN, R_OK) == 0) {
		f = g_strdup(M_HELP_FILE_MAIN);
	} else {
		f = g_strdup(M_HELP_FILE);
	}
	do {
		/* 2010.02.22: Resolve symlink. osso_pdfviewer cannot
		 * open symlinks.
		 */
		if (!f || lstat(f, &st) != 0) {
			break;
		}
		if (S_ISLNK(st.st_mode)) {
			/* Resolve and prepare for next round. */
			ff = g_file_read_link(f, NULL);
			g_free(f);
			f = ff;
			ff = NULL;
		} else {
			ff = f;
		}
	} while (!ff);
	if (!ff) {
		g_free(f);
		return;
	}
	cmd = g_strdup_printf(M_HELP_VIEWER, ff);
	g_free(ff);
	if (cmd) {
		g_spawn_command_line_async(cmd, NULL);
		g_free(cmd);
	}
}

static void m_servers_free_one(struct srv_info *srv)
{
	free(srv->server);
	free(srv->username);
	free(srv->domain);
	free(srv->password);
	free(srv->kblayout);
	free(srv);
}

static void m_servers_free_all(void)
{
	GSList *l;
	struct srv_info *srv;
	for (l = data.servers; l; l = l->next) {
		srv = (struct srv_info *)l->data;
		m_servers_free_one(srv);
	}
	g_slist_free(data.servers);
	data.servers = NULL;
}

/**
 * @return	0 = success
 * @return	negative = error code
 */
static int m_servers_load(void)
{
	int ret = 0;
	GSList *dirs, *l;
	char *c;
	struct srv_info *srv;
	gchar *gctmp, *path;
	gint gi;

	dirs = gconf_client_all_dirs(data.gconf, M_GCONF_ROOT, NULL);
	for (l = dirs; l; l = l->next) {
		srv = (struct srv_info *)calloc(sizeof(struct srv_info), 1);
		if (!srv) {
			ret = -2;
			goto e;
		}
		/* get the index number */
		c = strrchr(l->data, '/');
		if (!c) {
			DEBUG_FG("Skipping '%s'", (char *)l->data);
			continue;
		}
		srv->stamp = strtoul(c + 1, &gctmp, 10);
		if (*gctmp != '\0') {
			DEBUG_FG("Skipping '%s'", (char *)l->data);
			continue;
		}

		path = g_strdup_printf("%s/server", (char *)l->data);
		gctmp = gconf_client_get_string(data.gconf, path, NULL);
		srv->server = gctmp;
		g_free(path);

		path = g_strdup_printf("%s/username", (char *)l->data);
		gctmp = gconf_client_get_string(data.gconf, path, NULL);
		srv->username = gctmp;
		g_free(path);

		path = g_strdup_printf("%s/domain", (char *)l->data);
		gctmp = gconf_client_get_string(data.gconf, path, NULL);
		srv->domain = gctmp;
		g_free(path);

		path = g_strdup_printf("%s/password", (char *)l->data);
		gctmp = gconf_client_get_string(data.gconf, path, NULL);
		srv->password = gctmp;
		g_free(path);

		path = g_strdup_printf("%s/keyboard_layout", (char *)l->data);
		gctmp = gconf_client_get_string(data.gconf, path, NULL);
		srv->kblayout = gctmp;
		g_free(path);

		path = g_strdup_printf("%s/bit_depth", (char *)l->data);
		gi = gconf_client_get_int(data.gconf, path, NULL);
		srv->bpp = gi;
		if (gi != 8 && gi != 15 && gi != 16 && gi != 24) {
			srv->bpp = 16;
		}
		g_free(path);

		path = g_strdup_printf("%s/share_mmc", (char *)l->data);
		srv->mount_mmc = gconf_client_get_bool(data.gconf, path, NULL);
		g_free(path);

		path = g_strdup_printf("%s/share_mydocs", (char *)l->data);
		srv->mount_mydocs = gconf_client_get_bool(data.gconf, path, NULL);
		g_free(path);

		path = g_strdup_printf("%s/use_bluetooth_keyboard", (char *)l->data);
		srv->btkbd = gconf_client_get_bool(data.gconf, path, NULL);
		g_free(path);

		data.servers = g_slist_prepend(data.servers, srv);
	}

e:	for (l = dirs; l; l = l->next) {
		g_free(l->data);
	}
	g_slist_free(dirs);
	return ret;
}

static int m_servers_save(void)
{
	GSList *l;
	struct srv_info *srv;
	gchar *path;

	for (l = data.servers; l; l = l->next) {
		srv = l->data;

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/server", srv->stamp);
		m_gconf_client_set_string(data.gconf, path, srv->server, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/username", srv->stamp);
		m_gconf_client_set_string(data.gconf, path, srv->username, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/domain", srv->stamp);
		m_gconf_client_set_string(data.gconf, path, srv->domain, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/password", srv->stamp);
		m_gconf_client_set_string(data.gconf, path, srv->password, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/keyboard_layout", srv->stamp);
		m_gconf_client_set_string(data.gconf, path, srv->kblayout, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/bit_depth", srv->stamp);
		gconf_client_set_int(data.gconf, path, srv->bpp, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/share_mmc", srv->stamp);
		gconf_client_set_bool(data.gconf, path, srv->mount_mmc, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/share_mydocs", srv->stamp);
		gconf_client_set_bool(data.gconf, path, srv->mount_mydocs, NULL);
		g_free(path);

		path = g_strdup_printf(M_GCONF_ROOT "/%lu/use_bluetooth_keyboard", srv->stamp);
		gconf_client_set_bool(data.gconf, path, srv->btkbd, NULL);
		g_free(path);
	}
	gconf_client_suggest_sync(data.gconf, NULL);
	return 0;
}

static struct srv_info *
m_servers_get(int n)
{
	return g_slist_nth_data(data.servers, n);
}

static void
m_servers_add(struct srv_info *new_srv)
{
	new_srv->stamp = time(NULL);
	data.servers = g_slist_prepend(data.servers, new_srv);
}

static void
m_servers_del(struct srv_info *srv)
{
	gchar *key;
	gboolean r;
	key = g_strdup_printf(M_GCONF_ROOT "/%lu", srv->stamp);
	if (!key) {
		return;
	}
	r = gconf_client_recursive_unset(data.gconf, key, 0, NULL);
	g_free(key);
	if (r == FALSE) {
		return;
	}
	data.servers = g_slist_remove(data.servers, srv);
}

static void
m_servers_update_list_store(void)
{
	GSList *l;
	GtkTreeIter iter;
	struct srv_info *srv;
	gchar *entry;

	gtk_list_store_clear(data.list_store);
	for (l = data.servers; l; l = l->next) {
		srv = (struct srv_info *)l->data;
		entry = g_strdup_printf("%s at %s", srv->username, srv->server);
		if (entry) {
			gtk_list_store_append(data.list_store, &iter);
			gtk_list_store_set(data.list_store, &iter, 0, entry, -1);
			g_free(entry);
		}
	}
}

static void m_on_row_activated(GtkTreeView *tree_view,
                GtkTreePath *tree_path,
                GtkTreeViewColumn *column,
                gpointer user_data) {
	gint *inds;
	inds = gtk_tree_path_get_indices(tree_path);
	if (inds) {
		data.selected_server = inds[0];
		gtk_dialog_response(GTK_DIALOG(user_data), M_RESPONSE_SELECTED);
	}
	DEBUG_FG("end");
}


/**
 * @return	0 = success, connection can begin
 * @return	1 = error
 */
static int m_display_settings_dialog(int srv_index, GtkWindow *parent)
{
	int retval = 1, i, new_srv = 0;
	int tmp = 0;

	GtkWidget *dialog;
	GtkWidget *server_entry, *user_entry,
	          *password_entry, *domain_entry;
	GtkWidget *kblayout, *kbselector;
	GtkWidget *checkbox_btkbd, *checkbox_bpp,
			*checkbox_mountmmc, *checkbox_mountmydocs;
	GtkWidget *scroller;
	GtkWidget *caption;
	GtkWidget *box;
	GtkSizeGroup *size_group;
	struct srv_info *srv = NULL;

	/* Are there any saved settings? */
	if (srv_index >= 0) {
		srv = m_servers_get(srv_index);
	} else {
		srv = (struct srv_info *)calloc(sizeof(struct srv_info), 1);
		if (!srv)
			goto e;
		new_srv = 1;
	}

	dialog = gtk_dialog_new_with_buttons("Server details",
					     NULL,
					     (GtkDialogFlags) (GTK_DIALOG_MODAL),
					     0);
	if (!new_srv) {
		gtk_dialog_add_button(GTK_DIALOG(dialog), "Delete", M_RESPONSE_DEL);
	}
	gtk_dialog_add_button(GTK_DIALOG(dialog), "Connect", M_RESPONSE_CONNECT);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	size_group = GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));

	scroller = hildon_pannable_area_new();
	g_object_set(G_OBJECT(scroller),
				"size-request-policy", HILDON_SIZE_REQUEST_CHILDREN, NULL);
	gtk_widget_set_size_request(GTK_WIDGET(scroller), -1, 300);

	gtk_box_pack_start(
			GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))),
			scroller, TRUE, TRUE, 0);
	box = gtk_vbox_new(FALSE, 0);
	hildon_pannable_area_add_with_viewport(
			HILDON_PANNABLE_AREA(scroller),
			box);

	/* Create the form. */
	server_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_entry_set_max_length(GTK_ENTRY(server_entry), 256);
	caption = hildon_caption_new(size_group, "Server", server_entry,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	user_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_entry_set_max_length(GTK_ENTRY(user_entry), 64);
	caption = hildon_caption_new(size_group, "User name", user_entry,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	password_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_entry_set_max_length(GTK_ENTRY(password_entry), 256);
	caption = hildon_caption_new(size_group, "Password", password_entry,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(password_entry),
				HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_INVISIBLE);

	domain_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_entry_set_max_length(GTK_ENTRY(domain_entry), 256);
	caption = hildon_caption_new(size_group, "Domain", domain_entry,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	kbselector = hildon_touch_selector_new_text();
	for (i = 0; i < sizeof(m_kblayout_options) / sizeof(m_kblayout_options[0]); i++) {
		hildon_touch_selector_append_text (
				HILDON_TOUCH_SELECTOR (kbselector),
				g_strdup(m_kblayout_options[i].label));
		if (!strcmp(m_kblayout_options[i].key, g_keymapname)) {
			tmp = i;
		}
	}
	kblayout = hildon_picker_button_new (HILDON_SIZE_AUTO,
				HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (kblayout), "Keyboard layout");
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (kblayout),
				HILDON_TOUCH_SELECTOR (kbselector));
	hildon_picker_button_set_active(HILDON_PICKER_BUTTON (kblayout), tmp);
	caption = hildon_caption_new(size_group, NULL, kblayout,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	checkbox_mountmydocs = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label(GTK_BUTTON(checkbox_mountmydocs), "Share MyDocs folder");
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_mountmydocs), TRUE);
	caption = hildon_caption_new(size_group, NULL, checkbox_mountmydocs,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	checkbox_mountmmc = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label(GTK_BUTTON(checkbox_mountmmc), "Share MMC card");
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_mountmmc), TRUE);
	caption = hildon_caption_new(size_group, NULL, checkbox_mountmmc,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	checkbox_bpp = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label(GTK_BUTTON(checkbox_bpp), "Use low color mode (8bpp)");
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_bpp), FALSE);
	caption = hildon_caption_new(size_group, NULL, checkbox_bpp,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	checkbox_btkbd = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label(GTK_BUTTON(checkbox_btkbd), "Use BlueTooth keyboard");
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_btkbd), FALSE);
	caption = hildon_caption_new(size_group, NULL, checkbox_btkbd,
			NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator(HILDON_CAPTION(caption), NULL);
	gtk_box_pack_start(GTK_BOX(box), caption, TRUE, TRUE, 0);

	/* Add callbacks */
	g_signal_connect(G_OBJECT(server_entry), "changed",
			 G_CALLBACK(m_server_entry_changed), dialog);

	/* Set sizes */
	gtk_entry_set_width_chars(GTK_ENTRY(server_entry), 25);
	gtk_entry_set_width_chars(GTK_ENTRY(user_entry), 25);
	gtk_entry_set_width_chars(GTK_ENTRY(password_entry), 25);
	gtk_entry_set_width_chars(GTK_ENTRY(domain_entry), 25);

	/* Do not use autocap. It's annoying! */
	HildonGtkInputMode mode = hildon_gtk_entry_get_input_mode(GTK_ENTRY(server_entry));
	i = HILDON_GTK_INPUT_MODE_DICTIONARY | HILDON_GTK_INPUT_MODE_AUTOCAP;
	hildon_gtk_entry_set_input_mode(
		GTK_ENTRY(server_entry), mode & (~i));
	hildon_gtk_entry_set_input_mode(
		GTK_ENTRY(user_entry), mode & (~i));
	hildon_gtk_entry_set_input_mode(
		GTK_ENTRY(domain_entry), mode & (~i));

	if (!new_srv) {
		/* Fill in default values */
		strncpy(ui_maemo_server, srv->server, sizeof(ui_maemo_server));
		gtk_entry_set_text(GTK_ENTRY(server_entry), ui_maemo_server);
		strncpy(g_username, srv->username, sizeof(g_username));
		gtk_entry_set_text(GTK_ENTRY(user_entry), g_username);
		strncpy(ui_maemo_password, srv->password, sizeof(ui_maemo_password));
		gtk_entry_set_text(GTK_ENTRY(password_entry), ui_maemo_password);
		strncpy(ui_maemo_domain, srv->domain, sizeof(ui_maemo_domain));
		gtk_entry_set_text(GTK_ENTRY(domain_entry), ui_maemo_domain);

		g_server_depth = srv->bpp;
		m_mount_mmc = srv->mount_mmc;
		m_mount_mydocs = srv->mount_mydocs;
		ui_maemo_btkbd = srv->btkbd;

		strncpy(g_keymapname, srv->kblayout, sizeof(g_keymapname));
		for (i = 0;
		     i < sizeof(m_kblayout_options) / sizeof(m_kblayout_options[0]);
		     i++) {
			if(!strcmp(m_kblayout_options[i].key, g_keymapname)) {
				hildon_picker_button_set_active(
						HILDON_PICKER_BUTTON(kblayout),
						i);
				break;
			}
		}
	}

	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_bpp), g_server_depth == 8);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_mountmmc), m_mount_mmc);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_mountmydocs), m_mount_mydocs);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(checkbox_btkbd), ui_maemo_btkbd);

	/* Show */
	gtk_widget_show_all(dialog);
again:	i = gtk_dialog_run(GTK_DIALOG(dialog));
	DEBUG_FG("dialog returned %d", i);
	if (i == M_RESPONSE_CONNECT) {
		retval = 0;
	} else {
		retval = 1;
	}

	if (i == M_RESPONSE_CONNECT) {
		/* Connect and save */
		const char *txt;

		txt = gtk_entry_get_text(GTK_ENTRY(server_entry));
		if (txt) {
			strcpy(ui_maemo_server, txt);
			m_set_str(&srv->server, txt);
		}
		txt = gtk_entry_get_text(GTK_ENTRY(password_entry));
		if (txt) {
			strcpy(ui_maemo_password, txt);
			m_set_str(&srv->password, txt);
		}
		txt = gtk_entry_get_text(GTK_ENTRY(user_entry));
		if (txt) {
			strcpy(g_username, txt);
			m_set_str(&srv->username, txt);
		}
		txt = gtk_entry_get_text(GTK_ENTRY(domain_entry));
		if (txt) {
			strcpy(ui_maemo_domain, txt);
			m_set_str(&srv->domain, txt);
		}

		tmp = hildon_picker_button_get_active(HILDON_PICKER_BUTTON (kblayout));
		STRNCPY(g_keymapname, m_kblayout_options[tmp].key,
				sizeof(g_keymapname));
		m_set_str(&srv->kblayout, m_kblayout_options[tmp].key);

		m_mount_mmc = hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbox_mountmmc));
		srv->mount_mmc = m_mount_mmc;
		m_mount_mydocs = hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbox_mountmydocs));
		srv->mount_mydocs = m_mount_mydocs;
		ui_maemo_btkbd = hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbox_btkbd));
		srv->btkbd = ui_maemo_btkbd;

		srv->bpp = g_server_depth = 16;
		if (hildon_check_button_get_active(HILDON_CHECK_BUTTON(checkbox_bpp))) {
			srv->bpp = g_server_depth = 8;
		}
		if (new_srv) {
			m_servers_add(srv);
		}
		m_servers_save();
	} else if (i == M_RESPONSE_DEL) {
		/* Delete */
		gchar *msg;
		msg = g_strdup_printf("Do you want to delete the \"%s\" entry?", srv->server);
		tmp = m_show_confirm_msg(msg);
		g_free(msg);
		if (tmp) {
			m_servers_del(srv);
			m_servers_free_one(srv);
			m_servers_save();
		} else {
			goto again;
		}
	} else if (new_srv) {
		/* This is a new server but we do not save it. */
		m_servers_free_one(srv);
	}

	/* kill dialog */
	gtk_widget_hide(dialog);
	gtk_widget_destroy(dialog);
	g_object_unref(size_group);
e:	DEBUG_FG("end");
	return retval;
}

/**
 * @return	0 = success, connection can begin
 * @return	1 = error
 */
static int
m_display_servers_dialog(void)
{
	int ret = 0, r;
	GtkWidget *dialog = NULL;
	GtkWidget *button;
	GtkWidget *tree_view;
	GtkWidget *scroller;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkWidget *help_btn;

	r = m_servers_load();
	if (r < 0) {
		ui_maemo_show_msg("Cannot load server list!");
		ret = 1;
		goto e;
	}

	dialog = gtk_dialog_new_with_buttons("Server list",
			GTK_WINDOW(data.main_view), 0, NULL);
	//dialog = gtk_dialog_new_with_buttons("RDesktop servers",
	//		NULL, 0, NULL);
	button = gtk_dialog_add_button(GTK_DIALOG(dialog), "New", M_RESPONSE_NEW);

	if (m_help_exists()) {
		/* Show the help button only if the help file exists. */
		help_btn = hildon_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
						HILDON_BUTTON_ARRANGEMENT_VERTICAL);
		hildon_button_set_text(HILDON_BUTTON (help_btn), "Help",
				"RDesktop User's Guide for N900");
		g_signal_connect(G_OBJECT(help_btn), "clicked",
				 G_CALLBACK(m_show_help), dialog);
		gtk_box_pack_start(
				GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))),
				help_btn, TRUE, TRUE, 0);
	}

	scroller = hildon_pannable_area_new();
	g_object_set(G_OBJECT(scroller),
			"size-request-policy", HILDON_SIZE_REQUEST_CHILDREN, NULL);
	gtk_widget_set_size_request(GTK_WIDGET(scroller), -1, 300);

	gtk_box_pack_start(
			GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))),
			scroller, TRUE, TRUE, 0);

	data.list_store = gtk_list_store_new(1, G_TYPE_STRING);
	m_servers_update_list_store();

	/* Tree view */
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(data.list_store));
	g_signal_connect(tree_view, "row-activated",
					G_CALLBACK (m_on_row_activated), dialog);
	gtk_tree_selection_set_mode(
			gtk_tree_view_get_selection(GTK_TREE_VIEW (tree_view)),
			GTK_SELECTION_BROWSE);
	gtk_container_add (GTK_CONTAINER(scroller), tree_view);

	/* Name column */
	column = gtk_tree_view_column_new();
	renderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(column), "spacing", HILDON_MARGIN_DOUBLE, NULL);
	g_object_set(G_OBJECT(column), "max-width", 320, NULL);
	g_object_set(G_OBJECT(column), "expand", TRUE, NULL);
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(column), renderer, "text", 0);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

	gtk_widget_show_all(dialog);
	ret = 1;
	do {
		r = gtk_dialog_run(GTK_DIALOG(dialog));
		DEBUG_FG("dialog returned %d", r);
		if (r == M_RESPONSE_NEW) {
			ret = m_display_settings_dialog(-1, GTK_WINDOW(dialog));
		} else if (r == M_RESPONSE_SELECTED) {
			ret = m_display_settings_dialog(data.selected_server, GTK_WINDOW(dialog));
		} else {
			break;
		}
		m_servers_update_list_store();
	} while (ret != 0);

e:	if (dialog) {
		gtk_widget_hide_all(dialog);
		gtk_widget_destroy(dialog);
	}
	m_servers_free_all();
	return ret;
}

/********** public **********/

/* location of last mouse act */
uint16 ui_maemo_x_location;
uint16 ui_maemo_y_location;
RD_BOOL ui_maemo_connection_success = True;
RD_BOOL ui_maemo_btkbd = False; /*bluetooth keyboard?*/

char ui_maemo_server[256] = "";
char ui_maemo_password[256] = "";
char ui_maemo_domain[256] = "";

Window ui_maemo_fsicon_win;
Window ui_maemo_helpicon_win;

void ui_maemo_progress_start(void)
{
	if (!m_thread) {
		m_progress_done = 0;
		m_mutex = g_mutex_new();
		m_thread = g_thread_create(m_progress_thread, NULL, TRUE, NULL);
	}
}

void ui_maemo_progress_stop(void)
{
	if (m_thread) {
		g_mutex_lock(m_mutex);
		m_progress_done = 1;
		g_mutex_unlock(m_mutex);
		g_thread_join(m_thread);
	}
}

void ui_maemo_setup_window(Display *display, Window wnd)
{
	{
		Atom atom;
		int one = 1;
		atom = XInternAtom (display, "_HILDON_NON_COMPOSITED_WINDOW", False);
		XChangeProperty (display, wnd, atom,
				XA_INTEGER, 32, PropModeReplace,
				(unsigned char *) &one, 1);
	}

	//fullscreen
	if(g_fullscreen)
	{
		Atom wm_state, state_fs;
		wm_state = XInternAtom (display, "_NET_WM_STATE", False);
		state_fs = XInternAtom (display, "_NET_WM_STATE_FULLSCREEN", False);

		XChangeProperty (display, wnd, wm_state,
				XA_ATOM, 32, PropModeReplace,
				(unsigned char *) &state_fs, 1);
	}
	//set_window_type
	{
		Atom w_type, normal;
		w_type = XInternAtom (display, "_NET_WM_WINDOW_TYPE", False);
		normal = XInternAtom (display, "_NET_WM_WINDOW_TYPE_NORMAL", False);
		XChangeProperty (display, wnd, w_type,
				XA_ATOM, 32, PropModeReplace,
				(unsigned char *) &normal, 1);
	}
}

void ui_maemo_embed_window(Window wnd)
{
//	GtkWidget *m_gsocket;
//
//	m_gsocket = gtk_socket_new();
//	gtk_container_add(GTK_CONTAINER(data.main_view), m_gsocket);
//	gtk_widget_show(m_gsocket);
//	gtk_socket_steal(GTK_SOCKET(m_gsocket), (GdkNativeWindow)wnd);
//	XReparentWindow(g_display, g_wnd,
//			gtk_socket_get_id(GTK_SOCKET(m_gsocket)), 0, 0);
}

static Window
m_create_icon(Display *display, Window wnd, long input_mask,
		int x, int y, unsigned int width, unsigned int height,
		char *icon_path,
		XImage **img)
{
	Window iwin;
	unsigned long background, border;
	int screen_num;
	u_char *buf;
	int imageWidth;
	int imageHeight;

	/* create the pen to draw lines with */
	screen_num = DefaultScreen(display);
	border = BlackPixel(display, screen_num);
	background = WhitePixel(display, screen_num);
	iwin = XCreateSimpleWindow(display, wnd, /* display, parent */
			x, y, /* x, y: the window manager will place the window elsewhere */
			width, height, /* width, height */
			0, border, /* border width & colour, unless you have a window manager */
			background); /* background colour */

	XSelectInput(display, iwin, input_mask);
	XMapWindow(display, iwin);
	/* Calling jpg image */
	buf = m_decode_jpeg (icon_path, &imageWidth, &imageHeight);
	if (NULL == buf) {
		fprintf (stderr, "unable to decode JPEG from %s\n", icon_path);
		/* Do we really have to exit ? wont it show broken image ? */
		exit (EXIT_FAILURE);
	}
	*img = m_create_image_from_buffer(display, DefaultScreen(display),
			buf, imageWidth, imageHeight);

	if (*img == NULL) {
		fprintf (stderr, "image = NULL");
		exit (EXIT_FAILURE);
	}

	/*create_image_from_buffer creates a new buffer after translation, so we can free. */
	free (buf);
	return iwin;
}

void ui_maemo_create_icons(Display *display, Window wnd, long input_mask)
{
	XEvent xevent;

	do {
		XSetInputFocus(display, wnd, RevertToPointerRoot, CurrentTime);
		XMaskEvent(display, FocusChangeMask, &xevent);
	} while (xevent.type != FocusIn);

	m_fsicon_x_location = g_width - M_TOGGLE_WINDOW_WIDTH;
	m_fsicon_y_location = g_height - M_TOGGLE_WINDOW_HEIGHT;
	ui_maemo_fsicon_win = m_create_icon(display, wnd,
			input_mask,
			m_fsicon_x_location, m_fsicon_y_location,
			M_TOGGLE_WINDOW_WIDTH, M_TOGGLE_WINDOW_HEIGHT,
			M_TOGGLE_IMAGE_JPG, &m_fsicon);

//	ui_maemo_helpicon_win = m_create_icon(display, wnd,
//			input_mask,
//			1, 1,
//			M_TOGGLE_WINDOW_WIDTH, M_TOGGLE_WINDOW_HEIGHT,
//			M_HELP_IMAGE_JPG, &m_helpicon);
}

void ui_maemo_show_icons(Display *display, GC gc)
{
	if (ui_maemo_fsicon_win) {
		XPutImage (display, ui_maemo_fsicon_win, gc, m_fsicon, 0, 0, 0, 0,
			M_TOGGLE_WINDOW_WIDTH, M_TOGGLE_WINDOW_HEIGHT);
	}
	if (ui_maemo_helpicon_win) {
		XPutImage (display, ui_maemo_helpicon_win, gc, m_helpicon, 0, 0, 0, 0,
				M_TOGGLE_WINDOW_WIDTH, M_TOGGLE_WINDOW_HEIGHT);
	}
	XFlush (display);
}

void ui_maemo_toggle_fullscreen(void)
{
	XClientMessageEvent xclient;

	if (g_fullscreen)
	{
		m_fsicon_x_location = g_width - M_TOGGLE_WINDOW_WIDTH;
		m_fsicon_y_location = g_height - 120;
	}
	else
	{
		m_fsicon_x_location = g_width - M_TOGGLE_WINDOW_WIDTH;
		m_fsicon_y_location = g_height - M_TOGGLE_WINDOW_HEIGHT;
	}

	g_fullscreen = !g_fullscreen;

	memset (&xclient, 0, sizeof (xclient));
	xclient.type = ClientMessage;
	xclient.send_event = True;
	xclient.window = g_wnd;
	xclient.message_type = XInternAtom (g_display, "_NET_WM_STATE", FALSE);
	xclient.format = 32;
	xclient.data.l[0] =  _NET_WM_STATE_TOGGLE;
	xclient.data.l[1] = XInternAtom (g_display, "_NET_WM_STATE_FULLSCREEN", FALSE);
	xclient.data.l[2] = None;
	xclient.data.l[3] = 0;
	xclient.data.l[4] = 0;

	XSendEvent (g_display, DefaultRootWindow (g_display), False,
			SubstructureRedirectMask | SubstructureNotifyMask,
			(XEvent *)&xclient);

	XMoveWindow(g_display, ui_maemo_fsicon_win, m_fsicon_x_location, m_fsicon_y_location);
}

/**
 * @return	0 = handled
 * @return	negative = not handled
 */
int
ui_maemo_mouse_event(XEvent *xevent, RD_BOOL down)
{
	if ((xevent->xmotion.window == ui_maemo_fsicon_win) && down)
	{
		xwin_toggle_fullscreen();
		return 0;
	}
//	if ((xevent->xmotion.window == ui_maemo_helpicon_win) && down)
//	{
//		if ((xevent->xmotion.state & ShiftMask)) {
//			/* Shift+Click closes the help icon. */
//			XUnmapWindow(g_display, ui_maemo_helpicon_win);
//			XDestroyWindow(g_display, ui_maemo_helpicon_win);
//			ui_maemo_helpicon_win = 0;
//		} else {
//			g_spawn_command_line_async ("/usr/bin/osso_pdfviewer", NULL);
//		}
//		return 0;
//	}
	return -1;
}

/**
 * @return	0 = success
 * @return	1 = error
 */
int ui_maemo_restart_with_args(void)
{
	struct stat s1, s2;
	int r1, r2, i;
	char *bpp;
	char *argv[20];

	if (m_mainwin_create() < 0) {
		return 1;
	}
	/* display dialog */
	r1 = m_display_servers_dialog();
	m_mainwin_destroy();
	if (r1 == 1) {
		return 1;
	}
	ui_maemo_done();

	bzero(argv, sizeof(argv));
	i = 0;
	argv[i++] = "/usr/bin/rdesktop";
	argv[i++] = "-u"; argv[i++] = g_username;
	argv[i++] = "-d"; argv[i++] = ui_maemo_domain;
	argv[i++] = "-p"; argv[i++] = ui_maemo_password;
	argv[i++] = "-k"; argv[i++] = g_keymapname;
	argv[i++] = "-f";
	bpp = g_strdup_printf("%d", g_server_depth);
	if (bpp) {
		argv[i++] = "-a"; argv[i++] = bpp;
		/* No need to free bpp. We will do an exec(). */
	}

	if (m_mount_mmc == TRUE) {
		r1 = stat("/media/mmc1", &s1);
		r2 = stat("/media/mmc2", &s2);

		if (r1 == 0 && r2 == 0 && S_ISDIR(s1.st_mode) && S_ISDIR(s2.st_mode)) {
			argv[i++] = "-r";
			argv[i++] = "disk:MMC_Int=/media/mmc2,MMC_Ext=/media/mmc1";
		} else if (r1 == 0 && S_ISDIR(s1.st_mode)) {
			argv[i++] = "-r";
			argv[i++] = "disk:MMC=/media/mmc1";
		} else if (r2 == 0 && S_ISDIR(s2.st_mode)) {
			argv[i++] = "-r";
			argv[i++] = "disk:MMC=/media/mmc2";
		}
	}

	if (m_mount_mydocs == TRUE) {
		if (stat("/home/user/MyDocs", &s1)==0 && S_ISDIR(s1.st_mode)) {
			argv[i++] = "-r";
			argv[i++] = "disk:Docs=/home/user/MyDocs";
		}
	}

	argv[i++] = ui_maemo_server;
	argv[i] = (char *)NULL;

	DEBUG_FG("Executing...");
	/* Note: 2010.02.15: We run rdesktop again as a child process with
	 * a different D-Bus service name. Then we exit.
	 * This way we can call "rdesktop empty-server-arg" multiple
	 * times and Maemo will let multiple instances of rdesktop running.
	 */
	pid_t pid = fork();
	if (pid == 0) {
		execv("/usr/bin/rdesktop", argv);
	}

	return 1;
}

void ui_maemo_show_msg(char *msg)
{
	GtkDialog *dialog = GTK_DIALOG(hildon_note_new_information(NULL,
				(gchar*)msg));
	gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

int ui_maemo_init(int argc, char **argv, char *server)
{
	hildon_gtk_init(&argc, &argv);

	if (!g_thread_supported()) {
		g_thread_init (NULL);
	}

	data.app = HILDON_PROGRAM ( hildon_program_get_instance () );
	if (!data.app) {
		goto e;
	}
	g_set_application_name ("rdesktop_mui");
	if (strcmp(server, M_INTERACTIVE_MODE) == 0) {
		data.osso = osso_initialize("rdesktop_mui", VERSION, FALSE, 0);
	} else {
		/* NOTE: 2010.02.16: We call this one differently to fool Maemo.
		 * If we called it rdesktop then we could not have
		 * multiple instances running.
		 */
		data.osso = osso_initialize("automatic_rdesktop", VERSION, FALSE, 0);
	}
	if (!data.osso) {
		DEBUG_FG("Cannot initialize osso!");
		goto e;
	}

	int ossores = osso_rpc_set_default_cb_f (data.osso, m_dbus_callback, data.main_view);
	if (ossores != OSSO_OK) {
		DEBUG_FG("osso_rpc_set_default_cb_f failed: %d.", ossores);
		goto e;
	}

	/* Create main window. */
	data.main_view = HILDON_WINDOW(hildon_window_new());
	if (!data.main_view) {
		goto e;
	}
	gtk_window_set_title(GTK_WINDOW(data.main_view), "RDesktop server list");
	hildon_program_add_window(data.app, data.main_view);

	/* We use the server address in xwin.c to
	 * periodically check the network connection.
	 */
	strncpy(ui_maemo_server, server, 64);

	/* Ask parameters from user if needed. */
	if (strcmp(server, M_INTERACTIVE_MODE) == 0) {
		if (ui_maemo_restart_with_args()) {
			goto e;
		}
	}

	return 0;

e:	ui_maemo_done();
	return 1;
}

void ui_maemo_done(void)
{
	if (data.main_view) {
		hildon_program_remove_window(data.app, data.main_view);
		gtk_widget_destroy(GTK_WIDGET(data.main_view));
		data.main_view = NULL;
	}
	if (data.osso) {
		osso_deinitialize(data.osso);
		data.osso = NULL;
	}
}
