/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "resources.h"
#include <QCoreApplication>
#include <QStringList>
#include <iostream>

void printUsage() {
    std::cout << "Usage:\n\tqyoutube-search-list OPTION... " \
    "Search YouTube resources matching optional criteria.\n\n" \
    "Options:\n\t-apiKey\t\t\tThe api key.\n\t" \
    "-accessToken\t\tThe access token.\n\t" \
    "-refreshToken\t\tThe refresh token.\n\t" \
    "-part\t\t\tA comma-separated list of properties to be included in the response.\n\t\t\t\t" \
    "The default is 'snippet'.\n\t" \
    "-forContentOwner\t\tRestricts results to resources owned by the content owner.\n\t" \
    "-forDeveloper\t\tRestricts results to videos uploaded via the developer's\n\t\t\t\t" \
    "application or website.\n\t" \
    "-forMine\t\t\tRestricts the search to only retrieve videos owned by the authenticated user.\n\t" \
    "-relatedVideoId\t\tRetrieves a list of videos that are related to the video.\n\t" \
    "-channelId\t\t\tIndicates that the search should only include resources created by the channel.\n\t" \
    "-channelType\t\tRestricts the search to the specified type of channel.\n\t" \
    "-eventType\t\tRestricts results to broadcast events of the specified type.\n\t" \
    "-location\t\tRestricts results to videos within the specified geographic area.\n\t" \
    "-locationRadius\t\tRestricts results to videos within the specified geographic area.\n\t" \
    "-maxResults\t\tThe maximum number of results to be retrieved (default = 5, max = 50).\n\t" \
    "-onBehalfOfContentOwner\tIndicates that the request is made on behalf of the specified content owner.\n\t" \
    "-order\t\t\tThe order in which results should be returned.\n\t" \
    "-pageToken\t\tSpecifies a particular page in a result set.\n\t" \
    "-publishedAfter\tRestricts results to resources published after the specified datetime.\n\t" \
    "-publishedBefore\tRestricts results to resources published before the specified datetime.\n\t" \
    "-q\t\t\t\tThe query term to search for.\n\t" \
    "-regionCode\t\tRestricts results to the specified region.\n\t" \
    "-relevanceLanguage\tIndicates that results should be most relevant to the specified language.\n\t" \
    "-safeSearch\t\tIndicates whether results should include restricted content.\n\t" \
    "-topicId\t\t\tRestricts results to resources associated with the specified topic.\n\t" \
    "-type\t\t\tRestricts results to resources of the specified type. The default is 'video'.\n\t" \
    "-videoCaption\t\tIndicates whether videos should have captions.\n\t" \
    "-videoCategoryId\tRestricts results to videos belonging to the specified category.\n\t" \
    "-videoDefinition\tRestricts results to videos of the specified definition.\n\t" \
    "-videoDimension\tRestricts results to videos of the specified dimension.\n\t" \
    "-videoDuration\tRestricts results to videos of the specified duration.\n\t" \
    "-videoEmbeddable\tIndicates whether videos should be embeddable.\n\t" \
    "-videoLicense\t\tRestricts results to videos with the specified license.\n\t" \
    "-output\t\t\tThe output format (html, json, tsv (default), xml).";
}

QString getParam(QStringList &args, const QString &name, const QString &defaultValue = QString()) {
    int i = args.indexOf(name);

    if ((i >= 0) && (i < args.size() - 1)) {
        args.removeAt(i);
        return args.takeAt(i);
    }

    return defaultValue;
}

int main(int argc, char *argv[]) {
    QCoreApplication app(argc, argv);
    app.setOrganizationName("QYouTube");
    app.setApplicationName("cli");

    QStringList args = app.arguments();

    if (args.size() < 2) {
	printUsage();
	return 1;
    }

    args.removeFirst();

    const QString key = getParam(args, "-apiKey");

    if (key.isEmpty()) {
        printUsage();
        return 1;
    }
    
    const QString token = getParam(args, "-accessToken");
    const QString refresh = getParam(args, "-refreshToken");
    QString filter = getParam(args, "-forContentOwner");

    if (filter.isEmpty()) {
        filter = getParam(args, "-forDeveloper");

        if (filter.isEmpty()) {
            filter = getParam(args, "-forMine");
        }
    }

    if ((!filter.isEmpty()) && ((token.isEmpty()) || (refresh.isEmpty()))) {
        printUsage();
        return 1;
    }

    const QString contentOwner = getParam(args, "-onBehalfOfContentOwner");

    if ((!contentOwner.isEmpty()) && ((token.isEmpty()) || (refresh.isEmpty()))) {
        printUsage();
        return 1;
    }

    QVariantMap filter;

    const QStringList part = getParam(args, "-part", "snippet").split(",");
    const QString channelId = getParam(args, "-channelId");
    const QString channelType = getParam(args, "-channelType");
    const QString eventType = getParam(args, "-eventType");
    const QString location = getParam(args, "-location");
    const QString locationRadius = getParam(args, "-locationRadius");
    const QString maxResults = getParam(args, "-maxResults");
    const QString order = getParam(args, "-order");
    const QString pageToken = getParam(args, "-pageToken");
    const QString publishedAfter = getParam(args, "-publishedAfter");
    const QString publishedBefore = getParam(args, "-publishedBefore");
    const QString query = getParam(args, "-q");
    const QString regionCode = getParam(args, "-regionCode");
    const QString relevanceLanguage = getParam(args, "-relevanceLanguage");
    const QString safeSearch = getParam(args, "-safeSearch");
    const QString topicId = getParam(args, "-topicId");
    const QString type = getParam(args, "-type");

    QVariantMap params;
    
    const QString videoCaption = getParam(args, "-videoCaption");
    const QString videoCategory = getParam(args, "-videoCategoryId");
    const QString videoDefinition = getParam(args, "-videoDefinition");
    const QString videoDimension = getParam(args, "-videoDimension");
    const QString videoDuration = getParam(args, "-videoDuration");
    const QString videoEmbeddable = getParam(args, "-videoEmbeddable");
    const QString videoLicense = getParam(args, "-videoLicense");
    const QString videoSyndicated = getParam(args, "-videoSyndicated");
    const QString videoType = getParam(args, "-videoType");
    
    QYouTube::Resources resources;
    resources.setApiKey(key);
    resources.setAccessToken(token);
    resources.setRefreshToken(refresh);
    resources.setOutputFormat(output);
    resources.list("/videos", part, filters, params);
    QObject::connect(&resources, SIGNAL(finished()), &app, SLOT(quit()));
    
    return app.exec();
}
