/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "printer.h"
#include "json.h"
#include <QXmlStreamWriter>
#include <iostream>

namespace QYouTube {

static void variantToTsvHeader(const QVariant &var, QByteArray &header, QByteArray &part) {
    if (var.type() == QVariant::Map) {
        QMapIterator<QString, QVariant> iterator(var.toMap());

        while (iterator.hasNext()) {
            iterator.next();

            if (iterator.value().type() == QVariant::Map) {
                part.append(iterator.key() + ".");
                variantToTsvHeader(iterator.value(), header, part);
                part.chop(1);
                part = part.left(part.lastIndexOf(".") + 1);
            }
            else {
                header.append(part + iterator.key());
            }            

            if (iterator.hasNext()) {
                header.append("\t");
            }
        }
    }
}

static void variantToTsv(const QVariant &var, QByteArray &tsv) {
    if (var.type() == QVariant::List) {
        QVariantList list = var.toList();

        while (!list.isEmpty()) {
            const QVariant &v = list.takeFirst();

            if (v.type() == QVariant::List) {
                variantToTsv(v, tsv);
            }
            else if (v.type() == QVariant::Map) {
                variantToTsv(v, tsv);
                tsv.append("\n");
            }
            else {
                tsv.append(v.toString().toUtf8());

                if (!list.isEmpty()) {
                    tsv.append(",");
                }
                else {
                    tsv.append("\t");
                }
            }
        }
    }
    else if (var.type() == QVariant::Map) {
        QMapIterator<QString, QVariant> iterator(var.toMap());

        while (iterator.hasNext()) {
            iterator.next();
            variantToTsv(iterator.value(), tsv);

            if (iterator.hasNext()) {
                tsv.append("\t");
            }
        }
    }
    else {
        tsv.append(var.toString().toUtf8());
    }
}

static void variantToXml(const QVariant &var, QXmlStreamWriter &writer) {
    if (var.type() == QVariant::List) {
        QVariantList list = var.toList();

        while (!list.isEmpty()) {
            const QVariant &v = list.takeFirst();

            if ((v.type() == QVariant::List) || (v.type() == QVariant::Map)) {
                writer.writeStartElement("item");
                variantToXml(v, writer);
                writer.writeEndElement();
            }
            else {
                writer.writeCharacters(v.toString());

                if (!list.isEmpty()) {
                    writer.writeCharacters(",");
                }
            }
        }
    }
    else if (var.type() == QVariant::Map) {
        QMapIterator<QString, QVariant> iterator(var.toMap());

        while (iterator.hasNext()) {
            iterator.next();
            writer.writeStartElement(iterator.key());
            variantToXml(iterator.value(), writer);
            writer.writeEndElement();
        }
    }
    else {
        writer.writeCharacters(var.toString());
    }
}

void Printer::printHtml(const QVariant &var) {
    QByteArray output;
    QXmlStreamWriter writer(&output);
    writer.setAutoFormatting(true);
    writer.writeStartElement("html");
    writer.writeStartElement("body");
    variantToXml(var, writer);
    writer.writeEndElement();
    writer.writeEndElement();
    std::cout << output.constData();
}

void Printer::printJson(const QVariant &var) {
    std::cout << QtJson::Json::serialize(var).constData();
}

void Printer::printTsvHeader(const QVariant &var) {
    QByteArray output;
    QByteArray part;
    variantToTsvHeader(var, output, part);
    output.append("\n");
    std::cout << output.constData();
}

void Printer::printTsv(const QVariant &var) {
    QByteArray output;
    variantToTsv(var, output);
    std::cout << output.constData();
}

void Printer::printXml(const QVariant &var) {
    QByteArray output;
    QXmlStreamWriter writer(&output);
    writer.setAutoFormatting(true);
    writer.writeStartDocument();
    variantToXml(var, writer);
    writer.writeEndDocument();
    std::cout << output.constData();
}

}
