/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "VCard.h"

#include "Attachment.h"
#include "Factory.h"
#include "EventTypes/SMS.h"
#include "NumberToNameLookup.h"
#include "VBody.h"
#include "VEnvelope.h"
#include "VMessage.h"

#include <QDebug>
#include <QDir>
#include <QTextStream>
#include <QTemporaryFile>

#include <glib.h>

#include <rtcom-eventlogger/eventlogger.h>

#include <typeinfo>

using namespace EventParsers::VMGEntities;

VCard::VCard(const Settings &settings, const SMSEntity* parent) :
	SMSEntity(settings, parent), m_Version(2.1), m_Target(VCARD_LOCAL)
{
}

//VCard::VCard(QTextStream& stream)
//{
//}

VCard::VCard(const Settings &settings, const SMSEntity* parent, float version, eTarget target) :
	SMSEntity(settings, parent), m_Version(version), m_Target(target)
{
}

VCard::~VCard()
{
}

void VCard::Write(QTextStream &stream, const EventTypes::SMS &event, const NumberToNameLookup &numberToNameLookup)
{
	stream << "BEGIN:" << getTagName() << "\n";

	stream << "VERSION:" << m_Version << "\n";
	stream << "N:";
	if (m_Target != VCARD_LOCAL)
	{
			qDebug() << event.Tel();
			qDebug() << numberToNameLookup.ContactDetails().count();
		if(numberToNameLookup.ContactDetails().contains(event.Tel()))
			stream << numberToNameLookup.ContactDetails().value(event.Tel()).second;
	}
	stream << "\n";
	stream << "TEL:" << (m_Target == VCARD_LOCAL ? "" : event.Tel()) << "\n";

	stream << "END:" << getTagName() << "\n";
}

bool VCard::Read(const QString &initialLine, QTextStream &stream, EventTypes::SMS &event)
{
	bool hasEnded(false);
	float version(0);
	bool isTopLevel(typeid(*getParent()) == typeid(VMessage));
	bool isSender(typeid(*getParent()) == typeid(VEnvelope));
	Attachment *vCardAttachment;

	if(isAttachment())
	{
		vCardAttachment = new Attachment(
			(QDir::tempPath() + "/attachment-" + QString::number(event.Timestamp().toTime_t()) + "-" + QString::number(event.Attachments().count()) + ".vcard").toUtf8(),
			"text/x-vcard");
	}

	// Stream may or may not have a 'BEGIN' present. Swallow it if it's ours.
	QString lineData(initialLine.length() > 0 ? initialLine : stream.readLine());
	if(lineData.startsWith("BEGIN:"))
	{
		if(lineData != QString("BEGIN:") + getTagName())
		{
			qDebug() << "Invalid stream";
			return false;
		}
		else if(!isAttachment())
		{
			// ...discard this line
			lineData = stream.readLine();
		}
	}

	do
	{
		if(isAttachment())
		{
			vCardAttachment->Stream() << lineData << endl;

			if(lineData.startsWith("END:"))
			{
				if(lineData != QString("END:") + getTagName())
				{
					qDebug() << getTagName() << " parser mismatch error" << lineData;
					break;
				}
				else
				{
					// Save attachment
					event.Attachments().append(vCardAttachment);

					hasEnded = true;
					break;
				}
			}
		}
		else
		{
			if(lineData.startsWith("VERSION:"))
			{
				version = lineData.mid(lineData.indexOf(":")+1).toFloat();
			}
			else if(lineData.startsWith("N:"))
			{
				if((isTopLevel && !event.Destination() == EventTypes::SMS::SENT) || (isSender && event.Destination() == EventTypes::SMS::SENT))
				{
					//QString name = lineData.mid(lineData.indexOf(":") + 1);
					//gchar* g_name = g_strdup(name.toUtf8());
					//event.fld_remote_name = g_name;
				}
				else if(isTopLevel)
				{
					//event.fld_local_name = g_strdup("<SelfHandle>");
				}
			}
			else if(lineData.startsWith("TEL:"))
			{
				if((isTopLevel && !event.Destination() == EventTypes::SMS::SENT) || (isSender && event.Destination() == EventTypes::SMS::SENT))
				{
					QString tel = lineData.mid(lineData.indexOf(":") + 1);

					if(tel.indexOf("0") == 0)
						tel = tel.replace(QRegExp("^0"), "+44");

					gchar* g_tel = g_strdup(tel.toUtf8());
					event.Tel(g_tel);

//					if(tel.length() < 7 || tel.indexOf(QRegExp("[:alpha:]+")) > -1)
//						event.fld_group_uid = g_tel;
//					else
//						event.fld_group_uid = g_strdup(tel.right(7).toUtf8());
				}
				else if(isTopLevel)
				{
					//event.fld_local_uid = g_strdup("ring/tel/ring");
				}
			}
			else if(lineData.startsWith("BEGIN:"))
			{
				iReader* reader = Factory::Instantiate(CurrentSettings(), lineData, this);
				bool valid(NULL != reader && reader->Read(lineData, stream, event));
				delete reader;

				// Quit processing if the nested content is not valid
				if(!valid)
					return valid;
			}
			else if(lineData.startsWith("END:"))
			{
				if(lineData != QString("END:") + getTagName())
				{
					qDebug() << getTagName() << " parser mismatch error" << lineData;
					break;
				}
				else
				{
					hasEnded = true;
					break;
				}
			}
		}

		lineData = stream.readLine();
	}while(!hasEnded && !stream.atEnd());

	if(hasEnded)
	{

	}

	return true;
}
