/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "CSVSymbianEventLogParser.h"

#include <QDebug>

#include <QFile>
#include <QString>
#include <QStringList>

using namespace EventParsers;

class SortByValueDesc
{
public:
	inline bool operator()(const QPair<char, uint> &a, const QPair<char, uint> &b) const
	{
		return b.second < a.second;
	}
};

iEventParser *CSVSymbianEventLogParser::IsValid(QFile &eventFile)
{
	qDebug() << "Checking if a CSV call log file...";

	QByteArray firstLineContent(eventFile.readLine());
	eventFile.seek(0);
	if(firstLineContent.length() > 0)
	{
		// Count the non-alphanumeric characters used
		QHash<char, uint> counts;
		foreach(char c, firstLineContent)
			++counts[c];

		QList<QPair<char, uint> > orderedCounts;
		orderedCounts.reserve(counts.size());
		foreach(char c, counts.keys())
			if(!QChar(c).isLetterOrNumber())
				orderedCounts.append(QPair<char, uint>(c, counts.value(c)));

		qSort(orderedCounts.begin(), orderedCounts.end(), SortByValueDesc());

		// Work around Q_FOREACH macro limitation when dealing with
		// multi-typed templates (comma issue)
		typedef QPair<char, uint> bodge;
		foreach(bodge count, orderedCounts)
			qDebug() << count.first << " = " << count.second;

		char delim;
		// No-one would be mad enough to use quotation marks or apostrophes
		// as their delimiter,but just in case, check the second most
		// frequent character is present thr right number of times for
		// the qutation marks to be present on every column heading (two
		// per heading, less one as they're seperators)
		if((orderedCounts.value(0).first == '"' || orderedCounts.value(0).first == '\'')
			&& ((orderedCounts.value(0).second / 2) - 1 == orderedCounts.value(1).second ))
		{
			// We're good.
			delim = orderedCounts.value(1).first;
		}
		else
			delim = orderedCounts.value(0).first;

		// Check we have the essential fields we need, and grab their
		// column ordering
		QStringList requiredHeadings;
		requiredHeadings << "etype" << "etime" << "remote"
						 << "direction" << "duration" << "number";

		EventParsers::CSVSymbianEventLogParser::ColumnIndicesHash headingPositions;
		headingPositions.reserve(requiredHeadings.count());

		QStringList headings(QString(firstLineContent).split(delim));
		for(QStringList::size_type i(0); i < headings.count(); ++i)
		{
			QRegExp content("^[\"\']?(\\w*)?[\"\']?$");
			content.indexIn(headings.value(i).trimmed());
			QString heading(content.cap(1));
			qDebug() << headings.value(i) << " : " << heading;

			// Check over the required headings
			foreach(QString requiredHeading, requiredHeadings)
			{
				if(heading.toLower() == requiredHeading)
				{
					headingPositions[requiredHeading] = i;
					requiredHeadings.removeOne(requiredHeading);
				}
			}
		}

		// If we found all of the required headings, continue
		if(requiredHeadings.count() == 0)
		{
			return new EventParsers::CSVSymbianEventLogParser(eventFile.fileName(), headingPositions);
		}
	}

	return NULL;
}

CSVSymbianEventLogParser::CSVSymbianEventLogParser(const QString &filename, const ColumnIndicesHash &columns)
{
}

EventTypes::EventFromFileList CSVSymbianEventLogParser::ParseFile(QFile &eventFile, const QList<uint> &recordsToReturn)
{
	qDebug() << "CSV Parsing NYI!";
	return EventTypes::EventFromFileList();
}
