#include "QvernoteView.h"
#include "QvernoteAPI.h"
#include "QNewNotebookDialog.h"
#include "QNotebookListItemWidget.h"
#include "QvernoteTypes.h"
#include "QvernoteSettings.h"
#include "qvernotelogindialog.h"
#include "QNoteSearchDialog.h"
#include "QDeleteNotebookView.h"
#include "QAboutDialog.h"
#include <QGridLayout>
#include <QMessageBox>
#include <QDesktopWidget>
#include <QDebug>
#include "QvernoteStorage.h"
#include "QTrashWindow.h"

#include "QSyncProgressWidget.h"
#include "QKeyGrabber.h"

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#include "QOptionsDialog.h"
#endif




QvernoteView::QvernoteView(QWidget *parent)
    : QvernoteWindow(parent), m_hEvernote(QvernoteAPI::Instance()), isSearchShown(false), isSynchronizing(false)
{
	QvernoteSettings* settings = QvernoteSettings::Instance();
	ui.setupUi(this);

	ui.gridLayout_2->addWidget(&syncWidget, 2, 0, 1, 2, Qt::AlignCenter);
	ui.actionUse_SSL->setChecked(settings->getUseSsl());
	ui.lvNotebookList->setVisible(false);

	if(!m_hEvernote->isOnline() && settings->getWorkOnline())
	{
		displayInformationBox("Network error occurred, working offline");
		settings->setWorkOnline(false);
	}
	else
	{
		ui.pbSync->setEnabled(false);
		displayInformationBox("Initializing note store. It may take a couple of minutes on first sync. Please wait...");
	}

	/*
	ui.actionOnline->setChecked(settings->getWorkOnline());

	QvernoteSettings::DisplayOrientation orientation = settings->getDisplayOrientation();

	QActionGroup* displaySetupGroup = new QActionGroup(this);
	displaySetupGroup->setExclusive(true);

	QAction* actDoPortrait = new QAction(tr("Portrait"), displaySetupGroup);
	actDoPortrait->setCheckable(true);
	actDoPortrait->setChecked(orientation == QvernoteSettings::DO_PORTRAIT);
	QAction* actDoLandscape = new QAction(tr("Landscape"), displaySetupGroup);
	actDoLandscape->setCheckable(true);
	actDoLandscape->setChecked(orientation == QvernoteSettings::DO_LANDSCAPE);
	QAction* actDoAuto = new QAction(tr("Auto"), displaySetupGroup);
	actDoAuto->setCheckable(true);
	actDoAuto->setChecked(orientation == QvernoteSettings::DO_AUTO);

	ui.menubar->addActions(displaySetupGroup->actions());
	*/
	QObject::connect(ui.lvNotebookList, SIGNAL(itemClicked(QListWidgetItem *)),  this, SLOT(loadNotebook(QListWidgetItem* )));
	QObject::connect(ui.pbNewNotebook, SIGNAL(clicked()), this, SLOT(createNewNotebook()));
	QObject::connect(ui.pbSync, SIGNAL(clicked()), this, SLOT(/*synchronizeNoteStore()*/initView()));

	/*
	QObject::connect(ui.actionUse_SSL, SIGNAL(toggled(bool)), this, SLOT(configureSslProtocol(bool)));
	QObject::connect(ui.actionOnline, SIGNAL(toggled(bool)), this, SLOT(configureOnlineMode(bool)));
	QObject::connect(ui.actionAccount, SIGNAL(triggered(bool)), this, SLOT(accountConfig()));
	*/
	QObject::connect(ui.actionSearch, SIGNAL(triggered(bool)), this, SLOT(openSearchDialog()));
	QObject::connect(ui.actionDelete, SIGNAL(triggered(bool)), this, SLOT(openDeleteNotebookWindow()));
	QObject::connect(ui.actionAbout, SIGNAL(triggered(bool)), this, SLOT(openAboutDialog()));
	QObject::connect(ui.actionTrash, SIGNAL(triggered(bool)), this, SLOT(openTrashWindow()));
	QObject::connect(ui.actionOptions, SIGNAL(triggered(bool)), this, SLOT(openOptionsDialog()));

	/*
	QObject::connect(actDoPortrait, SIGNAL(toggled(bool)), this, SLOT(setPortraitOrientation(bool)));
	QObject::connect(actDoLandscape, SIGNAL(toggled(bool)), this, SLOT(setLandscapeOrientation(bool)));
	QObject::connect(actDoAuto, SIGNAL(toggled(bool)), this, SLOT(setAutoOrientation(bool)));
	*/

	QObject::connect(m_hEvernote, SIGNAL(noteUpdated(QString)), this, SLOT(showSyncProgressInfo(QString)));
	QObject::connect(this, SIGNAL(progressDataUpdated(QString)), &syncWidget, SLOT(updateProgressData(QString)));
	setProgressIndicator(true);

}

QvernoteView::~QvernoteView()
{

}

void QvernoteView::initView()
{
	ui.lvNotebookList->setVisible(false);
	syncWidget.setVisible(true);

	synchronizeNoteStore();

	/*
	m_pBackgroundJob->start();
	if(m_hEvernote->isOnline())
		displayInformationBox("Loading notebooks. Please wait...");
	*/
}

void QvernoteView::loadNotebookList(QString notebookFilter)
{
	Notebook notebook;
	bool hasNextNotebook = false;
	int noteCount = 0;
	setProgressIndicator(true);

	//syncDlg.close();
	ui.pbSync->setEnabled(true);
	ui.lvNotebookList->clear();


	for(hasNextNotebook = m_hEvernote->getFirstNotebook(notebook); hasNextNotebook; hasNextNotebook = m_hEvernote->getNextNotebook(notebook))
	{
		if(QString::fromStdString(notebook.name).startsWith(notebookFilter, Qt::CaseInsensitive))
		{
			//Notebook* newNotebook = new Notebook(notebook);
			QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
			iw->setSizeHint(QSize(0, 70));
			noteCount += m_hEvernote->getNotebookContentsCount(notebook);
			ui.lvNotebookList->setItemWidget(iw, new QNotebookListItemWidget(notebook, m_hEvernote, -1, false, ui.lvNotebookList));
		}
	}

	/* Add special "ALL" notebook */
	Notebook specialAllNotebook;
	specialAllNotebook.name = "All notebooks";
	QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
	iw->setSizeHint(QSize(0, 70));
	ui.lvNotebookList->setItemWidget(iw,
			new QNotebookListItemWidget(specialAllNotebook,
					m_hEvernote,
					noteCount,
					false,
					ui.lvNotebookList));

	syncWidget.setVisible(false);
	ui.lvNotebookList->setVisible(true);
	setProgressIndicator(false);
}

void QvernoteView::loadNotebook(QListWidgetItem* p_notebookListItem)
{
	QNotebookListItemWidget* p = (QNotebookListItemWidget*)p_notebookListItem->listWidget()->itemWidget(p_notebookListItem);
	qDebug() << "Loading notebook " << p->getNotebook().name.c_str();

	m_pNotesView = new QNotesView(p->getNotebook(), this);
	m_pNotesView->show();
}

void QvernoteView::createNewNotebook()
{
	QNewNotebookDialog* pDlg = new QNewNotebookDialog(m_hEvernote, this);
	pDlg->show();
}

void QvernoteView::synchronizeNoteStore()
{
	setProgressIndicator(true);
	m_pBackgroundJob->start();

	if(m_hEvernote->isOnline())
	{
		ui.pbSync->setEnabled(false);
		displayInformationBox("Synchronizing with Evernote. Please wait...");
		//syncDlg.exec();

	}
	else
	{
		ui.pbSync->setEnabled(true);
		displayInformationBox("Unable to synchronize in offline mode");
		setProgressIndicator(false);
	}
}

void QvernoteView::reloadNotebookList()
{
	initView();
}

void QvernoteView::deleteSelectedNotebook(QNotebookListItemWidget* notebookWidget)
{
	if(m_hEvernote->deleteNotebook(notebookWidget->getNotebook()) == false)
	{
		displayError("Failed to delete notebook", QString(m_hEvernote->getLastErrorString().c_str()));
	}
}

void QvernoteView::openSearchDialog()
{
	QNoteSearchDialog pDlg(this);
	pDlg.exec();
}

void QvernoteView::searchNote(QString searchString)
{
	qDebug() << "Searching" << searchString;
	setProgressIndicator(true);

	if(m_hEvernote->searchNotes(searchString.toStdString(), 10) == false)
	{
		displayError("Note search failed", QString(m_hEvernote->getLastErrorString().c_str()));
		return;
	}

	setProgressIndicator(false);

	m_pNotesView = new QNotesView("Search results for " + searchString, this);
	m_pNotesView->show();
}

void QvernoteView::orientationChanged()
{
	qDebug() << "Orientation changed";

	 QRect screenGeometry = QApplication::desktop()->screenGeometry();
	 if (screenGeometry.width() > screenGeometry.height())
	 {
		 // Landscape

	 }
	 else
	 {
		 // Portrait
	 }
}

bool QvernoteView::backgroundInit()
{
	isSynchronizing = true;
	if(m_hEvernote->isOnline())
		m_hEvernote->synchronizeLocalStorage();

	m_hEvernote->loadNotebookList();
	isSynchronizing = false;
	return true;
}

void QvernoteView::openDeleteNotebookWindow()
{
	QDeleteNotebookView* pWnd = new QDeleteNotebookView(m_hEvernote, this);
	pWnd->show();
}

void QvernoteView::openAboutDialog()
{
	QAboutDialog pDlg;
	pDlg.exec();
}

void QvernoteView::openTrashWindow()
{
	QTrashWindow* pWnd = new QTrashWindow(m_hEvernote, this);
	pWnd->show();
}

void QvernoteView::showSyncProgressInfo(QString info)
{
	//displayInformationBox(info);
	emit progressDataUpdated(info);
}

void QvernoteView::openOptionsDialog()
{
#ifdef Q_WS_MAEMO_5
	QOptionsDialog pDlg(this);
	pDlg.exec();
#else
	displayError("Cannot open options", "Maemo5 specific code");
#endif
}

void QvernoteView::keyPressEvent(QKeyEvent* event)
{
	if(!isSearchShown && !isSynchronizing)
	{
		QKeyGrabber* dlg = new QKeyGrabber(event->text(), this);
		ui.gridLayout->addWidget(dlg, 2, 0);
		QObject::connect(dlg, SIGNAL(redrawParentContents(QString)), this, SLOT(redrawNotebookList(QString)));
		isSearchShown = true;
	}

	QvernoteWindow::keyPressEvent(event);
}

void QvernoteView::redrawNotebookList(QString searchString)
{
	//qDebug() << __FUNCTION__ << searchString;
	if(searchString.isEmpty())
		isSearchShown = false;

	loadNotebookList(searchString);
}

void QvernoteView::clearNotebookList()
{
	ui.lvNotebookList->clear();
	m_hEvernote->loadNotebookList();
}
