#include "QvernoteView.h"
#include "QvernoteAPI.h"
#include "QNewNotebookDialog.h"
#include "QNotebookListItemWidget.h"
#include "QvernoteTypes.h"
#include "QvernoteSettings.h"
#include "qvernotelogindialog.h"
#include "QNoteSearchDialog.h"
#include "QDeleteNotebookView.h"
#include "QAboutDialog.h"
#include <QGridLayout>
#include <QMessageBox>
#include <QDesktopWidget>
#include <QDebug>
#include "QvernoteStorage.h"
#include "QTrashWindow.h"

#include "QSyncProgressWidget.h"

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

using namespace qvernote;
using namespace qvernote::api;

QvernoteView::QvernoteView(QWidget *parent)
    : QvernoteWindow(parent), m_hEvernote(QvernoteAPI::Instance())
{
	QvernoteSettings* settings = QvernoteSettings::Instance();
	ui.setupUi(this);

	ui.gridLayout_2->addWidget(&syncWidget, 2, 0, 1, 2, Qt::AlignCenter);
	ui.actionUse_SSL->setChecked(settings->getUseSsl());
	ui.lvNotebookList->setVisible(false);

	if(!m_hEvernote->isOnline() && settings->getWorkOnline())
	{
		displayInformationBox("Network error occurred, working offline");
		settings->setWorkOnline(false);
	}
	else
	{
		ui.pbSync->setEnabled(false);
		displayInformationBox("Initializing note store. It may take a couple of minutes on first sync. Please wait...");
	}

	ui.actionOnline->setChecked(settings->getWorkOnline());

	QvernoteSettings::DisplayOrientation orientation = settings->getDisplayOrientation();

	QActionGroup* displaySetupGroup = new QActionGroup(this);
	displaySetupGroup->setExclusive(true);

	QAction* actDoPortrait = new QAction(tr("Portrait"), displaySetupGroup);
	actDoPortrait->setCheckable(true);
	actDoPortrait->setChecked(orientation == QvernoteSettings::DO_PORTRAIT);
	QAction* actDoLandscape = new QAction(tr("Landscape"), displaySetupGroup);
	actDoLandscape->setCheckable(true);
	actDoLandscape->setChecked(orientation == QvernoteSettings::DO_LANDSCAPE);
	QAction* actDoAuto = new QAction(tr("Auto"), displaySetupGroup);
	actDoAuto->setCheckable(true);
	actDoAuto->setChecked(orientation == QvernoteSettings::DO_AUTO);

	ui.menubar->addActions(displaySetupGroup->actions());

	QObject::connect(ui.lvNotebookList, SIGNAL(itemClicked(QListWidgetItem *)),  this, SLOT(loadNotebook(QListWidgetItem* )));
	QObject::connect(ui.pbNewNotebook, SIGNAL(clicked()), this, SLOT(createNewNotebook()));
	QObject::connect(ui.pbSync, SIGNAL(clicked()), this, SLOT(/*synchronizeNoteStore()*/initView()));
	QObject::connect(ui.actionUse_SSL, SIGNAL(toggled(bool)), this, SLOT(configureSslProtocol(bool)));
	QObject::connect(ui.actionOnline, SIGNAL(toggled(bool)), this, SLOT(configureOnlineMode(bool)));
	QObject::connect(ui.actionAccount, SIGNAL(triggered(bool)), this, SLOT(accountConfig()));
	QObject::connect(ui.actionSearch, SIGNAL(triggered(bool)), this, SLOT(openSearchDialog()));
	QObject::connect(ui.actionDelete, SIGNAL(triggered(bool)), this, SLOT(openDeleteNotebookWindow()));
	QObject::connect(ui.actionAbout, SIGNAL(triggered(bool)), this, SLOT(openAboutDialog()));
	QObject::connect(ui.actionTrash, SIGNAL(triggered(bool)), this, SLOT(openTrashWindow()));

	QObject::connect(actDoPortrait, SIGNAL(toggled(bool)), this, SLOT(setPortraitOrientation(bool)));
	QObject::connect(actDoLandscape, SIGNAL(toggled(bool)), this, SLOT(setLandscapeOrientation(bool)));
	QObject::connect(actDoAuto, SIGNAL(toggled(bool)), this, SLOT(setAutoOrientation(bool)));

	QObject::connect(m_hEvernote, SIGNAL(noteUpdated(QString)), this, SLOT(showSyncProgressInfo(QString)));
	QObject::connect(this, SIGNAL(progressDataUpdated(QString)), &syncWidget, SLOT(updateProgressData(QString)));
	setProgressIndicator(true);

}

QvernoteView::~QvernoteView()
{

}

void QvernoteView::initView()
{
	ui.lvNotebookList->setVisible(false);
	syncWidget.setVisible(true);

	synchronizeNoteStore();

	/*
	m_pBackgroundJob->start();
	if(m_hEvernote->isOnline())
		displayInformationBox("Loading notebooks. Please wait...");
	*/
}

void QvernoteView::loadNotebookList()
{
	Notebook notebook;
	bool hasNextNotebook = false;
	int noteCount = 0;
	setProgressIndicator(true);

	//syncDlg.close();
	ui.pbSync->setEnabled(true);
	ui.lvNotebookList->clear();


	for(hasNextNotebook = m_hEvernote->getFirstNotebook(notebook); hasNextNotebook; hasNextNotebook = m_hEvernote->getNextNotebook(notebook))
	{
		//Notebook* newNotebook = new Notebook(notebook);
		QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
		iw->setSizeHint(QSize(0, 70));
		noteCount += m_hEvernote->getNotebookContentsCount(notebook);
		ui.lvNotebookList->setItemWidget(iw, new QNotebookListItemWidget(notebook, m_hEvernote, -1, false, ui.lvNotebookList));

	}

	/* Add special "ALL" notebook */
	Notebook specialAllNotebook;
	specialAllNotebook.name = "All notebooks";
	QListWidgetItem* iw = new QListWidgetItem(ui.lvNotebookList);
	iw->setSizeHint(QSize(0, 70));
	ui.lvNotebookList->setItemWidget(iw,
			new QNotebookListItemWidget(specialAllNotebook,
					m_hEvernote,
					noteCount,
					false,
					ui.lvNotebookList));

	syncWidget.setVisible(false);
	ui.lvNotebookList->setVisible(true);
	setProgressIndicator(false);
}

void QvernoteView::loadNotebook(QListWidgetItem* p_notebookListItem)
{
	QNotebookListItemWidget* p = (QNotebookListItemWidget*)p_notebookListItem->listWidget()->itemWidget(p_notebookListItem);
	qDebug() << "Loading notebook " << p->getNotebook().name.c_str();

	m_pNotesView = new QNotesView(m_hEvernote, p->getNotebook(), this);
	m_pNotesView->show();
}

void QvernoteView::createNewNotebook()
{
	QNewNotebookDialog* pDlg = new QNewNotebookDialog(m_hEvernote, this);
	pDlg->show();
}

void QvernoteView::synchronizeNoteStore()
{
	setProgressIndicator(true);
	m_pBackgroundJob->start();

	if(m_hEvernote->isOnline())
	{
		ui.pbSync->setEnabled(false);
		displayInformationBox("Synchronizing with Evernote. Please wait...");
		//syncDlg.exec();

	}
	else
	{
		ui.pbSync->setEnabled(true);
		displayInformationBox("Unable to synchronize in offline mode");
		setProgressIndicator(false);
	}
}

void QvernoteView::reloadNotebookList()
{
	initView();
}

void QvernoteView::deleteSelectedNotebook(QNotebookListItemWidget* notebookWidget)
{
	if(m_hEvernote->deleteNotebook(notebookWidget->getNotebook()) == false)
	{
		displayError("Failed to delete notebook", QString(m_hEvernote->getLastErrorString().c_str()));
	}
}

void QvernoteView::configureSslProtocol(bool checked)
{
	QvernoteSettings::Instance()->setUseSsl(checked);
	setProgressIndicator(true);
	if(m_hEvernote->reinitNoteStore() == false)
	{
		displayError("Failed to configure SSL", QString::fromStdString(m_hEvernote->getLastErrorString()));
		setProgressIndicator(false);
	}
	else
	{
		reloadNotebookList();
	}
}

void QvernoteView::configureOnlineMode(bool checked)
{
	if(m_hEvernote->setOnline(checked) == false)
	{
		displayError(QString("Failed to go %1").arg((checked)? "online" : "offline"), QString::fromStdString(m_hEvernote->getLastErrorString()));
	}
	else
	{
		displayInformationBox(QString("Working %1").arg((checked)? "online" : "offline"));
		QvernoteSettings::Instance()->setWorkOnline(checked);

		if(m_hEvernote->isOnline())
			//synchronizeNoteStore();
			initView();
	}
}

void QvernoteView::accountConfig()
{
	QvernoteLoginDialog ld;
	QvernoteSettings* settings = QvernoteSettings::Instance();

	if(ld.exec() == QDialog::Rejected)
		return;

	while(true)
	{
		if(m_hEvernote->Authenticate(
							settings->getUsername().toStdString(),
							settings->getPassword().toStdString()))
		{
			break;;
		}

		if(ld.exec() == QDialog::Rejected)
			return;

	}

	if(m_hEvernote->initNoteStore() == true)
	{
		reloadNotebookList();
	}
}

void QvernoteView::openSearchDialog()
{
	QNoteSearchDialog* pDlg = new QNoteSearchDialog(this);
	pDlg->show();
}

void QvernoteView::searchNote(QString searchString)
{
	qDebug() << "Searching" << searchString;
	setProgressIndicator(true);

	if(m_hEvernote->searchNotes(searchString.toStdString(), 10) == false)
	{
		displayError("Note search failed", QString(m_hEvernote->getLastErrorString().c_str()));
		return;
	}

	setProgressIndicator(false);

	m_pNotesView = new QNotesView(m_hEvernote, "Search results for " + searchString, this);
	m_pNotesView->show();
}

void QvernoteView::orientationChanged()
{
	qDebug() << "Orientation changed";

	 QRect screenGeometry = QApplication::desktop()->screenGeometry();
	 if (screenGeometry.width() > screenGeometry.height())
	 {
		 // Landscape

	 }
	 else
	 {
		 // Portrait
	 }
}

bool QvernoteView::backgroundInit()
{
	if(m_hEvernote->isOnline())
		m_hEvernote->synchronizeLocalStorage();

	m_hEvernote->loadNotebookList();

	return true;
}

void QvernoteView::openDeleteNotebookWindow()
{
	QDeleteNotebookView* pWnd = new QDeleteNotebookView(m_hEvernote, this);
	pWnd->show();
}

void QvernoteView::openAboutDialog()
{
	QAboutDialog pDlg;
	pDlg.exec();
}

void QvernoteView::openTrashWindow()
{
	QTrashWindow* pWnd = new QTrashWindow(m_hEvernote, this);
	pWnd->show();
}

void QvernoteView::showSyncProgressInfo(QString info)
{
	//displayInformationBox(info);
	emit progressDataUpdated(info);
}
