/****************************************************************************
 *  menucontroller.cpp
 *
 *  Copyright (c) 2010 by Nigmatullin Ruslan <euroelessar@gmail.com>
 *  Copyright (c) 2010 by Sidorov Aleksey <sauron@citadelspb.com>
 *
 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************
*****************************************************************************/

#include "menucontroller_p.h"
#include "actiongenerator_p.h"
#include <QMap>
#include <QDebug>
#include <QMetaMethod>
#include "debug.h"
#include "settingslayer.h"

namespace qutim_sdk_0_3
{
Q_GLOBAL_STATIC(MenuActionMap, globalActions)
ActionGeneratorMap *actionsCache()
{
	static QScopedPointer<ActionGeneratorMap> actions(new ActionGeneratorMap);
	return actions.data();
}

ActionHandler *handler()
{
	static ActionHandler handler;
	return &handler;
}
Q_GLOBAL_STATIC(QSet<DynamicMenu*>, dynamicMenuSet)

MenuControllerPrivate::MenuControllerPrivate(MenuController *c):
	owner(0), flags(0xffff), q_ptr(c)
{
}

MenuController::MenuController(QObject *parent) :
	QObject(parent), d_ptr(new MenuControllerPrivate(this))
{
}

MenuController::MenuController(MenuControllerPrivate &mup, QObject *parent) :
	QObject(parent), d_ptr(&mup)
{
}

MenuController::~MenuController()
{
	Q_D(MenuController);
	if (d->menu)
		delete d->menu;
}

bool actionGeneratorLessThan(const ActionGenerator *a, const ActionGenerator *b)
{
	if (a->type() == b->type()) {
		if (a->priority() == b->priority())
			return a->text() < b->text();
		else
			return a->priority() > b->priority();
	} else {
		return a->type() < b->type();
	}
}

bool actionLessThan(const ActionInfo &a, const ActionInfo &b)
{
	int cmp = a.menu.size() - b.menu.size();
	if (cmp == 0) {
		for (int i = 0; i < a.menu.size() && !cmp; i++)
			cmp = a.hash.at(i) - b.hash.at(i); //qstrcmp(a.menu.at(i), b.menu.at(i));
		if (cmp != 0)
			return cmp < 0;
	} else {
		return cmp < 0;
	}

	// Items are inside one menu
	return actionGeneratorLessThan(a.gen, b.gen);
}

inline bool isEqualMenu(const QList<uint> &a, const QList<uint> &b)
{
	if (a.size() != b.size())
		return false;
	for (int i = 0; i < a.size(); i++) {
		if (a[i] != b[i])
			return false;
	}
	return true;
}

inline quint64 pack_helper(quint32 a, quint32 b)
{
	return (quint64(a) << 32) | quint64(b);
}

inline void unpack_helper(quint64 h, quint32 *a, quint32 *b)
{
	*b = h & Q_INT64_C(0xffffffff);
	*a = (h - *b) >> 32;
}

ActionEntry *DynamicMenu::findEntry(ActionEntry &entries, const ActionInfo &info)
{
	ActionEntry *current = &entries;
	for (int i = 0; i < info.menu.size(); i++) {
		QMap<uint, ActionEntry>::iterator it = current->entries.find(info.hash.at(i));
		if (it == current->entries.end()) {
			QMenu *menu = current->menu->addMenu(QString::fromUtf8(info.menu.at(i), info.menu.at(i).size()));
			connect(menu, SIGNAL(aboutToShow()), this, SLOT(onAboutToShow()));
			connect(menu, SIGNAL(aboutToHide()), this, SLOT(onAboutToHide()));
			//QAction *action = menu->menuAction();
			//action->setData(qVariantFromValue(const_cast<ActionGenerator*>(info.gen)));
			//action->setData(pack_helper(info.hash.at(i), info.hash.size()));
			it = current->entries.insert(info.hash.at(i), ActionEntry(menu));
		}
		current = &it.value();
	}
	return current;
}

QAction *DynamicMenu::ensureAction(const ActionGenerator *gen)
{
	QObject *controller = m_owners.value(gen);
	Q_ASSERT(controller);
	QAction *action = actionsCache()->value(gen).value(controller);
	if (!action) {
		action = gen->generate<QAction>();
		gen->create(action, controller);
	}
	return action;
}

// Move method somewhere outside the DynamicMenu
QList<ActionInfo> DynamicMenu::allActions() const
{
	const MenuController *owner = m_d->q_func();
	QList<ActionInfo> actions;
	QSet<const QMetaObject *> metaObjects;
	while (owner) {
		int flags = MenuControllerPrivate::get(owner)->flags;
		foreach (const ActionInfo &info, MenuControllerPrivate::get(owner)->actions) {
			m_owners.insert(info.gen, const_cast<MenuController*>(owner));
			actions << info;
		}
		const QMetaObject *meta = owner->metaObject();
		while (meta) {
			if (metaObjects.contains(meta))
				break;
			foreach (const ActionInfo &info, globalActions()->values(meta)) {
				actions << info;
				m_owners.insert(info.gen, const_cast<MenuController*>(owner));
			}
			metaObjects.insert(meta);
			meta = (flags & MenuController::ShowSuperActions) ? meta->superClass() : 0;
		}
		owner = (flags & MenuController::ShowOwnerActions)
				? MenuControllerPrivate::get(owner)->owner : 0;
	}
	qSort(actions.begin(), actions.end(), actionLessThan);
	return actions;
}

typedef QMenu * (*_menu_creator_hook)(const QString &title, QWidget *parent);
LIBQUTIM_EXPORT _menu_creator_hook menu_creator_hook = 0;

inline QMenu *create_menu(MenuController *controller)
{
	QString title = controller->property("title").toString();
	if (title.isEmpty())
		title = controller->property("id").toString();
	QMenu *menu = menu_creator_hook ? menu_creator_hook(title, 0) : new QMenu(title);
	//menu->setProperty("menuController", qVariantFromValue(controller));
	return menu;
}

DynamicMenu::DynamicMenu(const MenuControllerPrivate *d) :
	m_d(d), m_entry(create_menu(d->q_ptr))
{
	dynamicMenuSet()->insert(this);
	m_menu = m_entry.menu;
	connect(m_menu, SIGNAL(aboutToShow()), this, SLOT(onAboutToShow()));
	connect(m_menu, SIGNAL(aboutToHide()), this, SLOT(onAboutToHide()));
	connect(m_menu, SIGNAL(destroyed()), this, SLOT(deleteLater()));
	connect(this, SIGNAL(destroyed()), m_menu, SLOT(deleteLater()));
	QList<ActionInfo> actions = allActions();
	if (actions.isEmpty())
		return;
	addActions(actions);
}

void DynamicMenu::addActions(const QList<ActionInfo> &actions)
{
	// Stable actions, they use new api and are always at memory
	int lastType = actions[0].gen->type();
	QList<uint> lastMenu;
	ActionEntry *currentEntry = &m_entry;
	for (int i = 0; i < actions.size(); i++) {
		const ActionInfo &act = actions[i];

		if (!isEqualMenu(lastMenu, act.hash)) {
			lastType = act.gen->type();
			lastMenu = act.hash;
			currentEntry = findEntry(m_entry, act);
		} else if (lastType != act.gen->type()) {
			lastType = act.gen->type();
			currentEntry->menu->addSeparator();
		}

		QAction *action = ensureAction(act.gen);
		if (action)
			currentEntry->menu->addAction(action);
	}
}

void DynamicMenu::addAction(MenuController *owner, const ActionInfo &info)
{
	m_owners.insert(info.gen, owner);
	onActionAdded(info);
}

void DynamicMenu::removeAction(MenuController *owner, const ActionGenerator *gen)
{
	MenuControllerPrivate *d = MenuControllerPrivate::get(owner);
	const ActionInfo *info = 0;
	for (int i = 0; i < d->actions.size(); i++) {
		if (d->actions.at(i).gen == gen) {
			info = &d->actions.at(i);
			break;
		}
	}
	QAction *action = gen ? actionsCache()->value(gen).value(owner) : 0;
	if (!info || !action)
		return;
	// May be cache menus?
	// Or it will work fine?
	action->deleteLater();
}

void DynamicMenu::onActionAdded(const ActionInfo &info)
{
	QAction *newAction = ensureAction(info.gen);
	if (newAction) {
		QMenu *menu = findEntry(m_entry, info)->menu;
		QList<QAction*> actions = menu->actions();
		int lastType = 0;
		int *lastTypeRef = 0;
		for (int i = 0; i < actions.size(); i++) {
			QAction *action = actions.at(i);
			ActionGenerator *gen = ActionGenerator::get(action);
			if (gen && actionGeneratorLessThan(info.gen, gen)) {
				if (gen->type() != info.gen->type()) {
					if (i > 0 && actions.at(i - 1)->isSeparator()) {
						if (lastTypeRef && lastType == info.gen->type())
							action = actions.at(i - 1);
					} else {
						action = menu->insertSeparator(action);
					}
				}
				menu->insertAction(action, newAction);
				return;
			} else if (gen) {
				lastType = gen->type();
				lastTypeRef = &lastType;
			}
		}
		menu->addAction(newAction);
	}
}

DynamicMenu::~DynamicMenu()
{
	dynamicMenuSet()->remove(this);
	// 		foreach (QAction *action, actions()) {
	// 			if (!actionsCache()->key(action))
	// 				action->deleteLater();
	// 		}
}

void DynamicMenu::onAboutToShow()
{
	QMenu *menu = qobject_cast<QMenu*>(sender());
	if(!menu) //on Symbian it's may be doesn't work ((
		return;
	foreach (QAction *action, menu->actions()) {
		ActionGenerator *gen = ActionGenerator::get(action);
		if (!gen) {
//			if (!action->isSeparator())
//				qWarning() << "DynamicMenu::Invalid ActionGenerator:" << action->text();
			continue;
		}
		QObject *controller = m_owners.value(gen);
		ActionGeneratorPrivate::get(gen)->show(action,controller);
	}
}

void DynamicMenu::onAboutToHide()
{
	QMenu *menu = qobject_cast<QMenu*>(sender());
	Q_ASSERT(menu);
	foreach (QAction *action, menu->actions()) {
		ActionGenerator *gen = ActionGenerator::get(action);
		if (!gen) {
			continue;
		}
		QObject *controller = m_owners.value(gen);
		ActionGeneratorPrivate::get(gen)->hide(action,controller);
	}
}

QMenu *MenuController::menu(bool deleteOnClose) const
{
	Q_UNUSED(deleteOnClose);
	Q_D(const MenuController);
	if (!d->menu)
		d->menu = new DynamicMenu(d);
#ifdef Q_WS_MAEMO_5
	d->menu->menu()->setStyleSheet("QMenu { padding:0px;} QMenu::item { padding:4px; } QMenu::item:selected { background-color: #00a0f8; }");
	//maemo does not use QMenu in main menu. Only the action in menu.
	foreach (QAction *action, d->menu->menu()->actions()) {
		ActionGenerator *gen = ActionGenerator::get(action);
		if (!gen) {
			if (!action->isSeparator())
				qWarning() << "DynamicMenu::Invalid ActionGenerator:" << action->text();
			continue;
		}
		QObject *controller = d_func()->menu->m_owners.value(gen);
		ActionGeneratorPrivate::get(gen)->show(action,controller);
	}
#endif
	return d->menu->menu();
}

void MenuController::showMenu(const QPoint &pos)
{
	menu(true)->popup(pos);
}

void MenuController::addAction(const ActionGenerator *gen, const QList<QByteArray> &menu)
{
	Q_D(MenuController);
	Q_ASSERT(gen);
	ActionInfo info = ActionInfo(gen, gen->d_func(), menu);
	d->actions.append(info);
	foreach (DynamicMenu *menu, *dynamicMenuSet()) {
		MenuController *owner = menu->controller();
		int flags = owner->d_func()->flags;
		while (owner != this && !!(owner = (flags & ShowOwnerActions) ? owner->d_func()->owner : 0))
			flags = owner->d_func()->flags;
		if (owner)
			menu->addAction(this, info);
	}
}

bool MenuController::removeAction(const ActionGenerator *gen)
{
	Q_D(MenuController);
	foreach (DynamicMenu *menu, *dynamicMenuSet()) {
		MenuController *owner = menu->controller();
		if (owner && owner->d_func()->menu)
			menu->removeAction(owner, gen);
	}
	for (int i = 0; i < d->actions.size(); i++) {
		if (d->actions.at(i).gen == gen) {
			d->actions.removeAt(i);
			return true;
		}
	}
	return false;
}

void MenuController::addAction(const ActionGenerator *gen, const QMetaObject *meta,
							   const QList<QByteArray> &menu)
{
	Q_ASSERT(gen && meta);
	MenuActionMap::iterator it;
	it = globalActions()->insertMulti(meta, ActionInfo(gen, gen->d_func(), menu));
	const ActionInfo &info = it.value();
	foreach (DynamicMenu *menu, *dynamicMenuSet()) {
		MenuController *owner = menu->controller();
		int flags = owner->d_ptr->flags;
		const QMetaObject *super = 0;
		do {
			if (!(flags & ShowSelfActions))
				continue;
			super = owner->metaObject();
			while (meta != super && !!(super = (flags & ShowSuperActions) ? super->superClass() : 0)) {}
		} while(!super && !!(owner = (flags & ShowOwnerActions) ? owner->d_func()->owner : 0));
		if (owner)
			menu->addAction(owner, info);
	}
}

void MenuController::setMenuOwner(MenuController *controller)
{
	Q_D(MenuController);
	d->owner = controller;
	if (d->menu)
		d->menu->addActions(d->menu->allActions());
}

void MenuController::setMenuFlags(const MenuFlags &flags)
{
	Q_D(MenuController);
	d->flags = flags;
}

void MenuController::virtual_hook(int id, void *data)
{
	Q_UNUSED(id);
	Q_UNUSED(data);
}

ActionContainer::ActionContainer(MenuController* controller) :
	d_ptr(new ActionContainerPrivate)
{
	Q_D(ActionContainer);
	d->controller = controller;
	d->filterType = TypeMatch;
	d->ensureActions();
}

ActionContainer::ActionContainer(MenuController* controller, ActionContainer::ActionFilter filter, const QVariant& data) :
	d_ptr(new ActionContainerPrivate)
{
	Q_D(ActionContainer);
	d->controller = controller;
	d->filterType = filter;
	d->filterData = data;
	d->ensureActions();
}
QAction* ActionContainer::action(int index) const
{
	Q_D(const ActionContainer);
	QAction *action = d->actions.at(index);
	ActionGenerator *gen = ActionGenerator::get(action);
	if (gen)
		ActionGeneratorPrivate::get(gen)->show(action,d->controller);
	return action;
}

ActionContainer::ActionContainer(const qutim_sdk_0_3::ActionContainer& other) : d_ptr(other.d_ptr)
{

}

int ActionContainer::count() const
{
	return d_func()->actions.count();
}

QList< QByteArray > ActionContainer::menu(int index) const
{
	//TODO implement logic
	Q_UNUSED(index);
	QList<QByteArray> itemlist;
	return itemlist;
}

ActionContainer& ActionContainer::operator=(const qutim_sdk_0_3::ActionContainer &other)
{
	d_ptr = other.d_ptr;
	return *this;
}

ActionContainer::~ActionContainer()
{
	//TODO may be need a delete created actions
}

void ActionContainerPrivate::ensureActions()
{
	actions.clear();

	QList<ActionInfo> actions;
	QSet<const QMetaObject *> metaObjects;
	const MenuController *owner = controller;
	while (owner) {
		foreach (const ActionInfo &info, MenuControllerPrivate::get(owner)->actions) {
			actions << info;
		}
		const QMetaObject *meta = owner->metaObject();
		while (meta) {
			if (metaObjects.contains(meta))
				break;
			foreach (const ActionInfo &info, globalActions()->values(meta)) {
				actions << info;
			}
			metaObjects.insert(meta);
			meta = meta->superClass();
		}
		owner = MenuControllerPrivate::get(owner)->owner;
	}

	qSort(actions.begin(), actions.end(), actionLessThan);

	foreach (ActionInfo info,actions) {
		if (filterData.canConvert(QVariant::Int)) {
			int typeMask = filterData.toInt();
			if (checkTypeMask(info,typeMask)) {
				ensureAction(info);
			}
		}
		else if (filterData.isNull()) {
			ensureAction(info);
		}
	}
}

void ActionContainerPrivate::ensureAction(const ActionInfo &info)
{
	QAction *action = actionsCache()->value(info.gen).value(controller);
	if (!action) {
		action = info.gen->generate<QAction>();
		if (!action)
			return;
		info.gen->create(action, controller);
	}
	actions.append(action);
}

bool ActionContainerPrivate::checkTypeMask(const ActionInfo &info,int typeMask)
{
	switch (filterType) {
	case ActionContainer::TypeMatch: {
		if (info.gen_p->type & typeMask) {
			return true;
		}
		break;
	}
	case ActionContainer::TypeMismatch: {
		if (!(info.gen_p->type & typeMask))
			return true;
		break;
	}
	default:
		break;
	}
	return false;
}

ActionHandler::ActionHandler() : QObject(0)
{

}

void ActionHandler::onActionTriggered(QAction *action)
{
	const ActionGenerator *gen = ActionGenerator::get(action);
	if (!gen) {
//		qWarning("ActionHandler::onActionTriggered: Invalid ActionGenerator");
		return;
	}
	const ActionGeneratorPrivate *d = ActionGeneratorPrivate::get(gen);
	QObject *controller = const_cast<QObject*>(actionsCache()->value(gen).key(action)); //fixme
	QObject *obj = d->receiver ? d->receiver.data() : controller;
	if (!obj)
		return;
	const QMetaObject *meta = obj->metaObject();
	int index = meta->indexOfMethod(d->member.constData() + 1);
	if (index == -1) {
		qWarning("ActionHandler::onActionTriggered: No such method %s::%s",
				 meta->className(), d->member.constData() + 1);
		return;
	}
	QMetaMethod method = meta->method(index);
	qDebug("DynamicMenu::onActionTriggered: Trying %s::%s",
		   meta->className(), d->member.constData() + 1);
	debug() << gen->text() << gen << action << controller;
	switch (d->connectionType) {
	case ActionConnectionObjectOnly:
		method.invoke(obj, Q_ARG(QObject*, controller));
		break;
	case ActionConnectionActionOnly:
		method.invoke(obj, Q_ARG(QAction*, action));
		break;
	case ActionConnectionFull:
		method.invoke(obj, Q_ARG(QAction*, action), Q_ARG(QObject*, controller));
		break;
	case ActionConnectionSimple:
		method.invoke(obj);
		break;
	default:
		break;
	}
}

void ActionHandler::onActionDestoyed(QObject *obj)
{
	QAction *action = reinterpret_cast<QAction*>(obj);
	m_actions.removeAll(action);
	ActionGeneratorMap::iterator it;
	for (it = actionsCache()->begin();it!=actionsCache()->end();it++) {
		if (QObject *key = it->key(action))
			it->remove(key);
	}
}

QAction *ActionHandler::addAction(QAction* action)
{
	connect(action,SIGNAL(destroyed(QObject*)),SLOT(onActionDestoyed(QObject*)));
	connect(action,SIGNAL(triggered()),SLOT(actionTriggered()));
	m_actions.append(action);
	return action;
}

void ActionHandler::actionTriggered()
{
	QAction *action = qobject_cast<QAction*>(sender());
	Q_ASSERT(action);
	onActionTriggered(action);
}

QObject *MenuController::get(QAction *action)
{
	ActionGenerator *gen = ActionGenerator::get(action);
	if (gen)
		return actionsCache()->value(gen).key(action);
	else
		return 0;
}

}
