/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-location: Utility for QBW GPS/Location Management and Operations
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <hildon/hildon.h>
#include <libosso.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-location.h"
#include "queen-beecon-settings-manager.h"

extern gchar *qbwExecReason[];

gchar *qbwLocationAction[] = {
	"QBW_INIT_LOCATION_MONITOR",
	"QBW_ENABLE_LOCATION_MONITOR",
	"QBW_READ_LOCATION_FIX",
	"QBW_DISABLE_LOCATION_MONITOR",
	"QBW_DEINIT_LOCATION_MONITOR"
};

GtkWidget *queen_beecon_location_settings_selector(void)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", NULL, G_STRFUNC);

	GtkWidget *selector;
	GtkListStore *model;
	GtkTreeIter iter;
	HildonTouchSelectorColumn *column = NULL;

	selector = hildon_touch_selector_new ();

	model = gtk_list_store_new (1, G_TYPE_STRING);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "Off", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "On", -1);

	column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR (selector),
													 GTK_TREE_MODEL (model), TRUE);
	g_object_set (G_OBJECT(column), "text-column", 0, NULL);

	model = gtk_list_store_new (1, G_TYPE_STRING);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "USR_SEL", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "CWP", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "ACWP", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "GNSS", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "AGNSS", -1);

	column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR (selector),
													 GTK_TREE_MODEL (model), TRUE);
	g_object_set (G_OBJECT(column), "text-column", 0, NULL);

	model = gtk_list_store_new (1, G_TYPE_STRING);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "DEFAULT", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "1 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "2 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "5 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "10 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "20 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "30 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "60 S", -1);

	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, "120 S", -1);

	column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR (selector),
													 GTK_TREE_MODEL (model), TRUE);
	g_object_set (G_OBJECT(column), "text-column", 0, NULL);

	return selector;
}

static void qbw_location_monitor_on_error(LocationGPSDControl *control, LocationGPSDControlError error, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s(%d)", self, G_STRFUNC, error);
	switch (error) {
		case LOCATION_ERROR_USER_REJECTED_DIALOG:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC, "User didn't enable requested methods");
			break;
		case LOCATION_ERROR_USER_REJECTED_SETTINGS:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC, "User changed settings, which disabled location");
			break;
		case LOCATION_ERROR_BT_GPS_NOT_AVAILABLE:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC, "Problems with BT GPS");
			break;
		case LOCATION_ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC, "Requested method is not allowed in offline mode");
			break;
		case LOCATION_ERROR_SYSTEM:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC, "System error");
			break;
	}
	queen_beecon_location_manager(self, QBW_DISABLE_LOCATION_MONITOR);
	queen_beecon_location_manager(self, QBW_DEINIT_LOCATION_MONITOR);
	self->priv->GPSsettings &= 0xFF;
	write_settings_have_key(self->priv->widgetID, "GPSsettings", (void *) &self->priv->GPSsettings, 'u');
	gchar *x=queen_beecon_op_subsystems_error_message(QBW_SUBSYSTEM_LOCATION);
	hildon_banner_show_informationf (NULL, NULL, "ERROR enabling %s.\nSubsystem(s) disabled in QBW instance [%s]!", x, self->priv->qbwDbusId);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) ERROR enabling %s.\nSubsystem(s) disabled in QBW instance [%s]!", self, x, self->priv->qbwDbusId);
	g_free(x);
}

static void qbw_location_monitor_on_changed(LocationGPSDevice *device, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s device=%p", self, G_STRFUNC, device);
	if (!device) return;

    if (self->priv->GPSfixinfo) {g_free(self->priv->GPSfixinfo);self->priv->GPSfixinfo = NULL;}

	gchar *b = NULL;
	self->priv->GPSfixinfo = g_strdup_printf("%d:", device->status);

	if (device->fix) {
		b = g_strdup_printf("%s%.0f:%.0f:%f:%f:%.0f:%.0f:%.0f:%.3f:%.3f:%.2f:%.2f:%.2f:%.2f:", self->priv->GPSfixinfo,
									device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET ? device->fix->time:atof("NaN"),
									device->fix->ept,
									device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET ? device->fix->latitude:atof("NaN"),
									device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET ? device->fix->longitude:atof("NaN"),
									device->fix->eph,
									device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET ? device->fix->altitude:atof("NaN"),
									device->fix->epv,
									device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET ? device->fix->track:atof("NaN"),
									device->fix->epd,
									device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET ? device->fix->speed:atof("NaN"),
									device->fix->eps,
									device->fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET ? device->fix->climb:atof("NaN"),
									device->fix->epc);
		g_free(self->priv->GPSfixinfo); self->priv->GPSfixinfo = b;
	} else {
		b = g_strdup_printf("%snan:nan:nan:nan:nan:nan:nan:nan:nan:nan:nan:nan:nan:", self->priv->GPSfixinfo);
		g_free(self->priv->GPSfixinfo); self->priv->GPSfixinfo = b;
	}
	if (device && device->cell_info) {
		b = g_strdup_printf("%s%d:%d:", self->priv->GPSfixinfo,
									device->cell_info->flags & LOCATION_CELL_INFO_GSM_CELL_INFO_SET ? device->cell_info->gsm_cell_info.mcc:0,
									device->cell_info->flags & LOCATION_CELL_INFO_WCDMA_CELL_INFO_SET ? device->cell_info->wcdma_cell_info.mcc:0);
									g_free(self->priv->GPSfixinfo); self->priv->GPSfixinfo = b;
	} else {
		b = g_strdup_printf("%s0:0:",
									self->priv->GPSfixinfo);
									g_free(self->priv->GPSfixinfo); self->priv->GPSfixinfo = b;
	}
	b = g_strdup_printf("%s%d:%d", self->priv->GPSfixinfo,
								device->satellites_in_view,
								device->satellites_in_use);
								g_free(self->priv->GPSfixinfo); self->priv->GPSfixinfo = b;
    qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) FIX %s", self, self->priv->GPSfixinfo);

    queen_beecon_update_content (self, qbwExecReason[QBW_LOCATION_MONITOR_FIX]);
}

//static void qbw_location_monitor_on_stop(LocationGPSDControl *control, QueenBeecon *self)
//{
//	QueenBeeconClass *klass = QUEEN_BEECON_GET_CLASS (self);
//	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC);
//   if (self->priv->GPSfixinfo) {g_free(self->priv->GPSfixinfo);self->priv->GPSfixinfo = NULL;}
//}

LocationGPSDControlMethod methods[] = {
	LOCATION_METHOD_USER_SELECTED,
	LOCATION_METHOD_CWP,
	LOCATION_METHOD_ACWP,
	LOCATION_METHOD_GNSS,
	LOCATION_METHOD_AGNSS
};

LocationGPSDControlInterval intervals[] = {
	LOCATION_INTERVAL_DEFAULT,
	LOCATION_INTERVAL_1S,
	LOCATION_INTERVAL_2S,
	LOCATION_INTERVAL_5S,
	LOCATION_INTERVAL_10S,
	LOCATION_INTERVAL_20S,
	LOCATION_INTERVAL_30S,
	LOCATION_INTERVAL_60S,
	LOCATION_INTERVAL_120S
};

//void location_gpsd_control_start(LocationGPSDControl *x){;}
//void location_gpsd_control_stop(LocationGPSDControl *x){;}
//LocationGPSDControl *location_gpsd_control_get_default(){return (LocationGPSDControl*)NULL;}
gboolean queen_beecon_relay_enable_GPS(QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC);
	self->priv->GPSstatusONOFF=TRUE;
	location_gpsd_control_start((LocationGPSDControl *) self->priv->GPScontrol);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) after location_gpsd_control_start", self);
	return FALSE;
}

gboolean queen_beecon_location_manager(QueenBeecon *self, QbwLocationAction action)
{
	QueenBeeconClass *klass = QUEEN_BEECON_GET_CLASS (self);
	LocationGPSDControlInterval method = methods[self->priv->GPSsettings >> 4 & 0xf];
	LocationGPSDControlMethod interval = intervals[self->priv->GPSsettings & 0xf];

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s action=[%d,%s] GPSstatusONOFF=[%d] settings=[%04x]",self, G_STRFUNC, action, qbwLocationAction[action], self->priv->GPSstatusONOFF, self->priv->GPSsettings);

    switch (action) {
	case QBW_INIT_LOCATION_MONITOR:
		if (self->priv->GPSstatusONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR", self);
		self->priv->GPScontrol = location_gpsd_control_get_default();
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR GPScontrol=%p", self, self->priv->GPScontrol);
		self->priv->GPSdevice = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR GPSdevice=%p", self, self->priv->GPSdevice);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR method=%04x interval=%04x", self, method, interval);
		g_object_set(G_OBJECT(self->priv->GPScontrol), "preferred-method", method, "preferred-interval", interval, NULL);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR after set preferred-method preferred-interval", self);
		g_signal_connect(self->priv->GPScontrol, "error-verbose", G_CALLBACK(qbw_location_monitor_on_error), self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR after set signal error-verbose", self);
		g_signal_connect(self->priv->GPSdevice, "changed", G_CALLBACK(qbw_location_monitor_on_changed), self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR after set signal changed", self);
		//g_signal_connect(self->priv->GPScontrol, "gpsd-stopped", G_CALLBACK(qbw_location_monitor_on_stop), self);
		//qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_INIT_LOCATION_MONITOR after set signal gpsd-stopped", self);
		break;
	case QBW_ENABLE_LOCATION_MONITOR:
		if (self->priv->GPSstatusONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_ENABLE_LOCATION_MONITOR", self);
	    if (self->priv->GPSfixinfo) {g_free(self->priv->GPSfixinfo);self->priv->GPSfixinfo = NULL;}
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_ENABLE_LOCATION_MONITOR before location_gpsd_control_start", self);
		klass->GPSrelayEnableTimer = g_timeout_add (3000, (GSourceFunc)queen_beecon_relay_enable_GPS, self);
		break;
	case QBW_READ_LOCATION_FIX:
		if (!self->priv->GPSstatusONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_READ_LOCATION_FIX", self);
		break;
	case QBW_DISABLE_LOCATION_MONITOR:
		if (!self->priv->GPSstatusONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_DISABLE_LOCATION_MONITOR", self);
		location_gpsd_control_stop((LocationGPSDControl *) self->priv->GPScontrol);
	    if (self->priv->GPSfixinfo) {g_free(self->priv->GPSfixinfo);self->priv->GPSfixinfo = NULL;}
		break;
	case QBW_DEINIT_LOCATION_MONITOR:
		if (!self->priv->GPSstatusONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_DEINIT_LOCATION_MONITOR", self);
		self->priv->GPSstatusONOFF=FALSE;
		if (klass->GPSrelayEnableTimer) g_source_remove (klass->GPSrelayEnableTimer); klass->GPSrelayEnableTimer = 0;
		g_object_unref(self->priv->GPScontrol);
		g_object_unref(self->priv->GPSdevice);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->GPSfixinfo = %p", klass, self->priv->GPSfixinfo);
	    if (self->priv->GPSfixinfo) {g_free(self->priv->GPSfixinfo);self->priv->GPSfixinfo = NULL;}
		break;
	default:
		return FALSE; // Unhandled command
	}
	return TRUE;
}
