/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-dbus-service: Utility for QBW DBUS Service Management
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-dbus-monitor.h"
#include "queen-beecon-dbus-service.h"
#include "queen-beecon-settings-manager.h"

extern gchar *qbwExecReason[];

void queen_beecon_register_dbus_service (QueenBeecon *self, gchar *qbwInstId)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s (instance ID=%s)",self, G_STRFUNC, qbwInstId);
	g_return_if_fail (self);

	QueenBeeconClass *klass = QUEEN_BEECON_GET_CLASS (self);

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dc=%p px=%p dbus-reg=%d", self, klass->dbus_conn, klass->bus_proxy, klass->dbus_registered);
	if (klass->dbus_conn && klass->bus_proxy ) {//&& !klass->dbus_registered) {
		guint request_name_ret;
		GError *error = NULL;

		gchar *bpid = g_strconcat(QBW_DBUS_PATH, qbwInstId, NULL);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Registering dbus_g_connection_register_g_object dc=%p ph=%s", self, klass->dbus_conn, bpid);
		dbus_g_connection_register_g_object (klass->dbus_conn, bpid, G_OBJECT (self));
		g_free(bpid);
		if (!org_freedesktop_DBus_request_name (klass->bus_proxy, QBW_DBUS_SERVICE, 0, &request_name_ret, &error)) {
			hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Failed to request DBUS name: %s", self, error->message);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Failed to request name: %s", NULL, error->message);
			g_clear_error (&error);
			return;
		}
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) request_name_ret rqn=%d", self, request_name_ret);

		switch (request_name_ret) {
			case DBUS_REQUEST_NAME_REPLY_EXISTS:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Already running", self);
				break;
			case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER: /* Yay! */
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Yay! Primary owner",self);
				klass->dbus_registered = TRUE;
				break;
			case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Yay! Already owner",self);
				break;
			default:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Uhoh!", self);
				break;
		}
	}
}

//D-Bus CB
void queen_beecon_dbus_cb_need_update_content_layout (QueenBeecon *self, const gchar *update)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (!g_strcmp0(update,"update_layout")) queen_beecon_update_content_layout (self);
	if (!g_strcmp0(update,"update_content")) queen_beecon_update_content (self, qbwExecReason[QBW_DBUS_SET_PARAM_UPDATE]);
}

gboolean queen_beecon_dbus_cb_dbus_method_monitor (QueenBeecon *self, const gchar *newStatus, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	hildon_banner_show_informationf (NULL, NULL, "dbus_method_monitor@qbw(%s) %s", self->priv->qbwDbusId, newStatus);

	if (!newStatus) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Monitor Status specified (on/off)");return FALSE;}

	self->priv->dMM = !g_strcmp0(newStatus,"on")?TRUE:FALSE;
	return TRUE;
}

gboolean queen_beecon_dbus_cb_reset_rearm_timer (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "reset_rearm_timer@qbw(%s)", self->priv->qbwDbusId);

	if (self->priv->updateTimerID) { // New instance loaded, we assume timers need reset
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) removing self->priv->updateTimerID=%d", self, self->priv->updateTimerID);
		g_source_remove (self->priv->updateTimerID);
	}
	if (self->priv->delayIndex) {
		self->priv->delaySeconds = GetSeconds(self->priv->delayIndex);
		if (self->priv->delayIndex==9) self->priv->delaySeconds = self->priv->customIntervalSecs;
	} else {
		self->priv->delaySeconds = 0;
	}
	if (self->priv->delaySeconds > 0 && self->priv->operationalStatus==0) { // New instance loaded, we assume timers need rearm if instance is enabled
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) re-arming execution timer, delaySeconds=%d", self, self->priv->delaySeconds);
		self->priv->updateTimerID = g_timeout_add_seconds (self->priv->delaySeconds, (GSourceFunc)queen_beecon_update_content, self);
	}
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_show (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "show@qbw(%s)", self->priv->qbwDbusId);
	self->priv->widgetVisible = 1; //Show widget
	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_write_settings (self, TRUE);
	//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_update_content_layout (self);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_hide (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "hide@qbw(%s)", self->priv->qbwDbusId);
	self->priv->widgetVisible = 0; //Hide Widget
	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_write_settings (self, TRUE);
	//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_update_content_layout (self);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_wakeup (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "wakeup@qbw(%s)", self->priv->qbwDbusId);
	if (self->priv->operationalStatus==0) return TRUE;
	self->priv->operationalStatus=0;
	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_write_settings (self, TRUE);
	//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	if( (self->priv->updateTimerID == 0) && (self->priv->delaySeconds > 0)) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) re-arming execution timer, delaySeconds=%d", self, self->priv->delaySeconds);
		self->priv->updateTimerID = g_timeout_add_seconds (self->priv->delaySeconds, (GSourceFunc)queen_beecon_update_content, self);
	}
	if (self->priv->updOnDBUS && self->priv->dbus_mon_connection==NULL && self->priv->operationalStatus==0) { //Is DBUS Monitor active?
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) DBUS monitor starting, rearm connection, match and filter", self);
		if (!queen_beecon_dbus_monitor_manager(self, QBW_INIT_DBUS_MONITOR, self->priv->updOnDBUSBus, self->priv->updOnDBUSMatchRule)) { // If error during reinitialization of DBUS monitor ... restore previous settings
			self->priv->updOnDBUS          = 0;
		}
	}
	if (!self->priv->updOnDBUS && self->priv->updOnStartup) queen_beecon_update_content (self, qbwExecReason[QBW_DBUS_WAKEUP]);
	else queen_beecon_update_content_layout (self);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_sleep (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "sleep@qbw@qbw(%s)", self->priv->qbwDbusId);
	if (self->priv->operationalStatus==1) return TRUE;
	self->priv->operationalStatus=1;
	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	queen_beecon_write_settings (self, TRUE);
	//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	if (self->priv->updateTimerID) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) removing self->priv->updateTimerID=%d", self, self->priv->updateTimerID);
		g_source_remove (self->priv->updateTimerID);
	}

	queen_beecon_dbus_monitor_manager(self, QBW_DEINIT_DBUS_MONITOR, self->priv->updOnDBUSBus, self->priv->updOnDBUSMatchRule); // Disable DBUS Monitor

	queen_beecon_update_content_layout (self);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_update_content (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "update_content@qbw(%s)",  self->priv->qbwDbusId);
	queen_beecon_update_content (self, qbwExecReason[QBW_DBUS_UPDATE_CONTENT]);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_update_layout (QueenBeecon *self, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "update_layout@qbw(%s)", self->priv->qbwDbusId);
	queen_beecon_update_content_layout (self);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_exec (QueenBeecon *self, const gchar *cmd, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gchar *backupCmd = self->priv->instanceCmd; // Backup current Instance Command
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "exec@qbw(%s) %s", self->priv->qbwDbusId, cmd);

	if (!cmd) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Command specified");return FALSE;}
	self->priv->instanceCmd = (gchar *)cmd; // Instantiate new one-shot temporary command
	queen_beecon_update_content (self, qbwExecReason[QBW_DBUS_EXEC]); // And run it
	self->priv->instanceCmd = backupCmd; // Revert back to saved command

	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_param_string (QueenBeecon *self, const gchar *param, const gchar *value, const gchar *update, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gboolean RC = FALSE;
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_param_string@qbw(%s) %s %s %s", self->priv->qbwDbusId, param, value, update);

	if (!param || !value) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter or Value specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	RC = write_settings_have_key(self->priv->widgetID, param, (void *) value, 's');
	if (!RC) {
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Error writing parameter, check existence");
		return FALSE;
	} else {
		queen_beecon_read_settings (self, TRUE, FALSE, NULL);
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	}
	queen_beecon_dbus_cb_need_update_content_layout(self, update);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_param_int32 (QueenBeecon *self, const gchar *param, const gint value, const gchar *update, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gboolean RC = FALSE;
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_param_int32@qbw(%s) %s %d %s", self->priv->qbwDbusId, param, value, update);

	if (!param) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	RC = write_settings_have_key(self->priv->widgetID, param, (void *) &value, 'i');
	if (!RC) {
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Error writing parameter, check existence");
		return FALSE;
	} else {
		queen_beecon_read_settings (self, TRUE, FALSE, NULL);
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	}
	queen_beecon_dbus_cb_need_update_content_layout(self, update);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_param_uint32 (QueenBeecon *self, const gchar *param, const guint value, const gchar *update, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gboolean RC = FALSE;
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_param_uint32@qbw(%s) %s %d %s", self->priv->qbwDbusId, param, value, update);

	if (!param) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	RC = write_settings_have_key(self->priv->widgetID, param, (void *) &value, 'u');
	if (!RC) {
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Error writing parameter, check existence");
		return FALSE;
	} else {
		queen_beecon_read_settings (self, TRUE, FALSE, NULL);
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	}
	queen_beecon_dbus_cb_need_update_content_layout(self, update);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_param_double (QueenBeecon *self, const gchar *param, const gdouble value, const gchar *update, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gboolean RC = FALSE;
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_param_double@qbw(%s) %s %f %s", self->priv->qbwDbusId, param, value, update);

	if (!param) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	RC = write_settings_have_key(self->priv->widgetID, param, (void *) &value, 'd');
	if (!RC) {
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Error writing parameter, check existence");
		return FALSE;
	} else {
		queen_beecon_read_settings (self, TRUE, FALSE, NULL);
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	}
	queen_beecon_dbus_cb_need_update_content_layout(self, update);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_param_boolean (QueenBeecon *self, const gchar *param, const gboolean value, const gchar *update, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gboolean RC = FALSE;
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_param_boolean@qbw(%s) %s %d %s", self->priv->qbwDbusId, param, value, update);

	if (!param) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	RC = write_settings_have_key(self->priv->widgetID, param, (void *) &value, 'b');
	if (!RC) {
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Error writing parameter, check existence");
		return FALSE;
	} else {
		queen_beecon_read_settings (self, TRUE, FALSE, NULL);
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	}
	queen_beecon_dbus_cb_need_update_content_layout(self, update);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_get_param (QueenBeecon *self, const gchar *param, gchar **value, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "get_param@qbw(%s) %s", G_STRFUNC, self->priv->qbwDbusId, param);

	if (!param) {g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "No Parameter specified");return FALSE;}

	//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	*value = read_settings_have_key (self->priv->widgetID, param);
	//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_is_on_sight (QueenBeecon *self, gchar **value, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "get_current_results_text@qbw(%s)", G_STRFUNC, self->priv->qbwDbusId);

    gboolean on;
    g_object_get (self, "is-on-current-desktop", &on, NULL);
	*value = g_strdup(on?"true":"false");
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_get_current_results_text (QueenBeecon *self, gchar **value, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "get_current_results_text@qbw(%s)", G_STRFUNC, self->priv->qbwDbusId);

	*value = g_strdup(gtk_label_get_label(GTK_LABEL (self->priv->cmdResult_lb)));
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_set_position (QueenBeecon *self, const gint widXOrigin, const gint widYOrigin, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "set_position@qbw(%s) X=%d Y=%d", self->priv->qbwDbusId, widXOrigin, widYOrigin);

	gtk_window_move (GTK_WINDOW (self), widXOrigin, widYOrigin); // Remember our position ... maybe we need to move a little bit
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_get_position (QueenBeecon *self, gint *widXOrigin, gint *widYOrigin, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "get_position@qbw(%s)", self->priv->qbwDbusId);

	gtk_window_get_position (GTK_WINDOW (self), widXOrigin, widYOrigin); // Remember our position ... maybe we need to move a little bit
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_get_exit_status (QueenBeecon *self, gint *exitStatus, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "get_exit_status@qbw(%s)", self->priv->qbwDbusId);

	*exitStatus = self->priv->cmdExitStatus;
	return TRUE;
}

gboolean queen_beecon_dbus_cb_dbus_logger_action (QueenBeecon *self, const gchar *action, GError **error)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s %s", self, G_STRFUNC, action);
	if (self->priv->dMM) hildon_banner_show_informationf (NULL, NULL, "logger_action@qbw %s", action);
	if (!g_strcmp0(action,"disable")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_VNULL, NULL);return TRUE;}
	if (!g_strcmp0(action,"v0")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_V0, NULL);return TRUE;}
	if (!g_strcmp0(action,"v1")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_V1, NULL);return TRUE;}
	if (!g_strcmp0(action,"v2")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_V2, NULL);return TRUE;}
	if (!g_strcmp0(action,"v3")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_V3, NULL);return TRUE;}
	if (!g_strcmp0(action,"v4")) {qbw_logger(QBW_LOGGER_SET_VERBOSITY, QBW_LOGGER_V4, NULL);return TRUE;}
	if (!g_strcmp0(action,"reset")) {qbw_logger(QBW_LOGGER_RESET_LOGFILE, QBW_LOGGER_VNULL, "", NULL); return TRUE;}
	if (!g_strcmp0(action,"flush")) {qbw_logger(QBW_LOGGER_FLUSH, QBW_LOGGER_VNULL, "", NULL); return TRUE;}
	g_set_error (error, DBUS_GERROR, DBUS_GERROR_REMOTE_EXCEPTION, "Invalid Logger Action specified");
	return FALSE;
}
