#include "imagegridwidget.h"

#include <QPainter>
#include <QPaintEvent>
#include <QResizeEvent>
#include <QApplication>
#include <QFontMetrics>
#include <QMessageBox>

#include <QDebug>
#include <math.h>

#define ROUND(x) ((int)(x-(int)x>=.5?(int)x+1:(int)x))
#define SWAP(x,y) {int t=x;x=y;y=t;}

ImageGridWidget::ImageGridWidget(QWidget* parent) :
        QWidget(parent)
{
    m_mouseOverIndex = -1;
    m_selIndex = -1;
    m_font = ((QApplication *)QApplication::instance())->font();
    m_backgroundColor = palette().background().color();
    m_penColor = QColor(0,0,0);
    m_aspectRatio = 1;
    m_showSelection = true;

    m_selColor = QColor(64,64,64);
    m_selBorderColor = QColor(255,255,255);
    m_clickColor = QColor(128,128,128);

    m_width = 0;
    m_height = 0;

    m_padding = 5;
}

ImageGridWidget::~ImageGridWidget() {
    clear();
}

void ImageGridWidget::clear() {
    while ( !m_list.isEmpty() ) delete m_list.takeLast();
}


void ImageGridWidget::leaveEvent(QEvent *event) {
    if ( m_list.isEmpty()) return;

    if ( m_mouseOverIndex<0 ) return;
    int index = m_mouseOverIndex;
    m_mouseOverIndex = -1;
    repaint(getRect(index));
}

void ImageGridWidget::mousePressEvent(QMouseEvent *event) {
    if ( m_list.isEmpty() || event->button() != Qt::LeftButton ) {
        QWidget::mousePressEvent(event);
        return;
    }

    int index = m_selIndex;
    m_mouseOverIndex = m_selIndex = getIndex(event->x(), event->y());
    if ( index>=0 ) repaint(getRect(index));
    if ( m_selIndex<0 ) return;

    repaint(getRect(m_selIndex));
}

void ImageGridWidget::contextMenuEvent(QContextMenuEvent *pEvent) {
    if ( m_list.isEmpty()) {
    	QWidget::contextMenuEvent(pEvent);
	return;
    }

    // contextmenu use global pos
    QPoint pos = mapFromGlobal(pEvent->globalPos());
    
    int index = m_selIndex;
    m_mouseOverIndex = m_selIndex = getIndex(pos.x(), pos.y());
    if ( index>=0 ) repaint(getRect(index));
    if ( m_selIndex<0 ) {
        return;
    }

    repaint(getRect(m_selIndex));
    onContextMenu(*pEvent, m_list[m_selIndex]);
}

void ImageGridWidget::mouseDoubleClickEvent(QMouseEvent *event) {
    if ( m_list.isEmpty()|| m_selIndex<0 ) {
        QWidget::mouseDoubleClickEvent(event);
        return;
    }
    onDblClick(*event, m_list[m_selIndex]);
}

void ImageGridWidget::mouseReleaseEvent(QMouseEvent *event) {
    if ( m_list.isEmpty()|| event->button() != Qt::LeftButton ) {
        QWidget::mouseReleaseEvent(event);
        return;
    }

    int index = m_mouseOverIndex;
    m_mouseOverIndex = -1;
    repaint(getRect(index));

    if ( m_selIndex<0 ) return;
    onClick(*event, m_list[m_selIndex]);
}

void ImageGridWidget::mouseMoveEvent(QMouseEvent *event) {
    QWidget::mouseMoveEvent(event);

#ifndef Q_WS_MAEMO_5
    if ( m_list.isEmpty()) return;

    int index = m_mouseOverIndex;
    m_mouseOverIndex = getIndex(event->x(), event->y());
    if ( m_mouseOverIndex==index ) return; // nothing has changed

    if ( index>=0 ) update(getRect(index));
    if ( m_mouseOverIndex>=0 ) update(getRect(m_mouseOverIndex));
#endif
}

void ImageGridWidget::setAspectRatio(int x, int y) {
    m_aspectRatio = (double)y/(double)x;

    calculate();
    repaint();
}

void ImageGridWidget::resizeEvent(QResizeEvent *event) {
    QWidget::resizeEvent(event);

    calculate();
    repaint();
}

void ImageGridWidget::calculate() {
    if ( m_list.isEmpty()) {
        m_height = m_width = 0;
        return;
    }

    if (width()<1 || height()<1) {
        m_y = 0;
        return;
    }

    double f = (double)width()/(double)height() * m_aspectRatio;

    int count = m_list.count();
    double x = f;
    m_y = 1;
    while ( (int)x*m_y < count ) {
        x += f;
        if ( (int)x*m_y >= count ) break;
        m_y ++;
    }
    m_x = (int)x;
    if (m_x>count) m_x=count;
    while ((m_x-1)*m_y>=count )m_x--;
    while ((m_y-1)*m_x>=count) m_y--;

    m_width = (width()-2*m_padding) / m_x;
    m_height = (height()-2*m_padding) / m_y;
    m_offX = (width() - m_x*m_width)/2;
    m_offY = (height() - m_y*m_height)/2;
}

void ImageGridWidget::paintEvent(QPaintEvent *event) {
    if ( m_list.isEmpty() || !m_y ) return;

    QPainter painter(this);
    // painter.fillRect( event->rect(), QColor(255,255,0));

    QFontMetrics metrics(m_font, painter.device());
    painter.setFont(m_font);

    // painter.setBrush(QBrush(palette().foreground()));
    // painter.setPen(QColor(255,0,0));

    // test pen
    painter.setPen(m_penColor);
    painter.drawRect(0,0,width()-1, height()-1);

    int index = 0;
    for (int y=0; y<m_y; y++) {
        for ( int x=0; x<m_x && index<m_list.count(); x++, index++ ) {
            QPoint p(m_offX+x*m_width, m_offY+y*m_height);
            QRect r(p, QSize(m_width, m_height));
            //if ( !event->rect().contains(r) ) continue;

            if ( m_showSelection && index==m_selIndex ) {
                // selected
                painter.setPen(m_selBorderColor);
                painter.fillRect(r, m_selColor);
                painter.drawRect(p.x()+2, p.y()+2, m_width-5, m_height-5);

            } else {
                // not selected
                painter.setPen(m_penColor);
                painter.fillRect(r, index==m_mouseOverIndex?m_clickColor : m_backgroundColor);

            }

            Item& item(*m_list[index]);
            QImage image = item.m_image;
            QString title = item.m_name;
            int h = title.isNull() ? image.height() : image.height()+metrics.height();

            p.setX(p.x() + (m_width-image.width())/2);
            p.setY(p.y() + (m_height-h)/2);
            painter.drawImage(p, image);

            if ( title.isNull() ) continue;

            painter.drawText(QRect(r.left(),
                p.y()+image.height(),m_width, metrics.height()),
                item.m_name,
                QTextOption(Qt::AlignCenter)
            );
        }
    }
}

QRect ImageGridWidget::getRect(int index) {
    if ( m_list.isEmpty()) {
        // ...
        return QRect();
    }

    int x = index % m_x;
    int y = index/m_x;

    return QRect(QPoint(m_offX+x*m_width,m_offY+y*m_height), QSize(m_width, m_height));
}

int ImageGridWidget::getIndex(int x, int y) {
    if ( m_list.isEmpty() || m_width==0 || m_height==0 ) return -1;

    int x1 = (x-m_offX)/m_width;
    int y1 = (y-m_offY)/m_height;
    if ( x1<0 || y1<0 ) return -1;
    int index = y1*m_x+x1;
    return index>=m_list.count()?-1:index;
}

void ImageGridWidget::add(Item *pItem, bool bRepaint) {
    m_list.append(pItem);
    if (!bRepaint) return;

    calculate();
    repaint();
}

void ImageGridWidget::del(Item *pItem, bool bRepaint) {
    int index = m_list.indexOf(pItem);
    if ( index<0 ) return;

    m_list.removeAt(index);
    delete pItem;

    if (!bRepaint) return;

    calculate();
    repaint();
}

ImageGridWidget::Item* ImageGridWidget::itemAt(int index) {
    return index<0 || index>=m_list.size() ? NULL : m_list[index];
}

int ImageGridWidget::count() const {
    return m_list.size();
}

ImageGridWidget::Item& ImageGridWidget::operator[](int index) {
    // check???
    // ...
    return *itemAt(index);
}


void ImageGridWidget::setPenColor(const QColor &newColor) {
    m_penColor = newColor;
    repaint();
}

QColor ImageGridWidget::penColor() const {
    return m_penColor;
}

void ImageGridWidget::setBackgroundColor(const QColor &newColor) {
    m_backgroundColor = newColor;
    repaint();
}

QColor ImageGridWidget::backgroundColor() const {
    return m_backgroundColor;
}

void ImageGridWidget::setSelectedColor(const QColor &newColor) {
    m_selColor = newColor;
    repaint();
}

QColor ImageGridWidget::selectedColor() const {
    return m_selColor;
}

void ImageGridWidget::setFont(const QFont &newFont) {
    m_font = newFont;
    repaint();
}

QFont ImageGridWidget::font() const {
    return m_font;
}

int ImageGridWidget::getSelectedIndex() const {
    return m_selIndex;
}

void ImageGridWidget::setSelectedIndex(int index) {
    int prevIndex = m_selIndex;
    m_selIndex = (index<0 || index>=m_list.size())?-1:index;
    if ( prevIndex>=0 ) repaint(getRect(prevIndex));
    if ( m_selIndex>=0 ) repaint(getRect(m_selIndex));
}

bool ImageGridWidget::isShowSelection() const {
    return m_showSelection;
}

void ImageGridWidget::setIsShowSelection(bool show) {
    m_showSelection = show;
    // update ...
}
