#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QFile>
#include <QTemporaryFile>
#include <iostream>
#include <QDesktopServices>
#include "errormessage.h"
#include "stackedwindow.h"
#include "settingsdialog.h"
#include "dialog.h"


using namespace std;

/**
  Constructor for our window.
  @param parent The parent of this window. This specific
  window has no parent as it is the top level window
  */
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
     ui->setupUi(this);

    /*Create the list of day periods that represent the day / night info
      we get from the xml files and add them to the UI*/
    dayPeriods = new QList<DayPeriod*>();
    for(int i = 0; i < 12; i++){
        DayPeriod *dp = new DayPeriod(i);
        dayPeriods->append(dp);
    }

    connect(ui->listWidget, SIGNAL(activated(QModelIndex)), this, SLOT(dayClick(QModelIndex)));

    pg = new ProgressDialog(this);
    needShowList = true;
    iconCache = new QPixmap*[13];
    for(int i = 0; i < 13; i++){
        iconCache[i] = new QPixmap();
    }

    settings = new QSettings("naddeo", "QtWeather", this);
    latlon = new QString(); //graph parameter
    zcode = new QString();  //graph parameter
    wfo = new QString();    //graph parameter
    doc = new QDomDocument("document"); //will be the xml document object
	justStarted = true;

    manager = new QNetworkAccessManager(this);
    mapper = new QSignalMapper(this);
    connect(mapper, SIGNAL(mapped(int)), this, SLOT(setDayIcons(int)));

    /*addapt the application to maemo by providing
      stackable windows and auto rotation*/
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow, true);
    setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif


    /*signal that the UI has finished setting up, and start the
      dialog for getting a zip code from the user*/
	connect(this, SIGNAL(uiReady()), this, SLOT(changeZip()), Qt::QueuedConnection);
	emit uiReady();
}


/**
  Launch the dialog for getting settings from the user.
  This app has settings for default zipcode launching
  at startup time
  */
void MainWindow::setSettings(){
	SettingsDialog sd(&settings->value("home1", ""),
					  &settings->value("home2", ""),
					  &settings->value("default", 0), this);
    sd.exec();
    sd.deleteLater();

	settings->setValue("home1", *sd.getHome1());
	settings->setValue("home2", *sd.getHome2());
	settings->setValue("default", *sd.getDefaultHome());
}

/**
  Deconstructor for the window. When this window closes
  the application ends.
  */
MainWindow::~MainWindow()
{
    delete ui;
}

/**
  Launches the default browser. Loads the current city in the NAOO website
  */
void MainWindow::visitNAOO(){
    QString str("http://forecast.weather.gov/zipcity.php?inputstring=" + zipCode);
    QDesktopServices::openUrl(QUrl(str));
}

/**
  Function for the day buttons. Shows the detailed view of that day
  */
void MainWindow::dayClick(QModelIndex index){
    ui->listWidget->currentItem()->setSelected(false);
    detailWindow(index.row());
}

/**
  Is called when the windows switches from landscape to portrait
  orientation. The function here specifically switches the scroll
  views layout from horizontal to vertical where appropriate
  @param event contains information about the screens new and
  old layout that we can use to check to see what mode it is currently
  in
  */
void MainWindow::resizeEvent(QResizeEvent *event){
    if(event->oldSize().width() > event->size().width()){
        ui->mainLayout->setDirection(QBoxLayout::TopToBottom);
        ui->listWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    }else{
        ui->mainLayout->setDirection(QBoxLayout::LeftToRight);
        ui->listWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::MinimumExpanding);
    }
}


/**
  Sets the QDomDocument 'doc' variable up to be parsed as an XML file.
  @param reply The network reply that contains the reply. It should be a reply
  from the National Weather Service containing an XML file.
  */
void MainWindow::setDomDocument(QString *reply){
    QTemporaryFile file;

    file.open();
    file.write(reply->toStdString().c_str());
    file.close();
    file.open();
    if (!doc->setContent(&file)) {
        cout << "cold not connect set content1" << endl;
        file.close();
        return;
    }
    file.close();
}


/**
  Launches a dialog that prompts the user to enter a zip code
  into a text box and the executes a request to the server for
  an XML file.
  */
void MainWindow::changeZip(){
    zipCache = zipCode;
    zipCode.clear();

	if(!justStarted || settings->value("default") == 0){
		Dialog *d = new Dialog(this);
		d->exec();
		d->deleteLater();
		zipCode = d->getZip();
	}else if(settings->value("default") == 1){
		zipCode = settings->value("home1", "").toString();
	}else if(settings->value("default") == 2){
		zipCode = settings->value("home2", "").toString();
	}
	justStarted = false;


    if(!zipCode.isEmpty()){
        zipCode.truncate(5);
        QString urlString("http://www.weather.gov/forecasts/xml/SOAP_server/ndfdXMLclient.php?whichClient=LatLonListZipCode&listZipCodeList=xxxxx&product=time-series&begin=2004-01-01T00:00:00&end=2015-01-22T00:00:00&maxt=maxt&Submit=Submit");
        urlString.replace(113, 5, zipCode);

        QNetworkReply* zipReply = manager->get(QNetworkRequest(QUrl(urlString)));
        connect(zipReply, SIGNAL(finished()), this, SLOT(gotCoordinates()));

        pg->show();
    }else{
        zipCode = zipCache;
        cout << "Dialog Canceled" << endl;
    }
}


/**
  get the "wfo" (weather forecast office) parameter
  that is required for downlaoding graph images
  */
void MainWindow::parse_for_wfo(QDomElement *docEle){
	wfo->clear();
	wfo->append(QString(docEle->elementsByTagName("wordedForecast").at(0).toElement().attribute("dataSource")));
	wfo->truncate(3);
}


/**
  Takes a DomElement and creates a QList of QStrings containing the temperatures
  */
QList<QString*>* MainWindow::parse_for_temps(QDomElement *docEle){

    QDomNodeList hiList;
    QDomNodeList lowList;
    QDomNodeList tempList = docEle->elementsByTagName("temperature"); //temperature list
    for(int i = 0; i < tempList.size(); i++){
        if(tempList.at(i).toElement().attribute("type") == "maximum"){
            hiList = tempList.at(i).toElement().elementsByTagName("value");
        }else{
            lowList = tempList.at(i).toElement().elementsByTagName("value");
        }
    }

    /*merge temps into one day period ordered list*/
    QList<QString*>* hiloList = new QList<QString*>();

    if(hiList.size() > lowList.size()){
        for(int i = 0; i < lowList.size(); i++){
            hiloList->append(new QString("Hi: " + QString(hiList.at(i).toElement().text()) + " F"));
            hiloList->append(new QString("Lo: " + QString(lowList.at(i).toElement().text()) + " F"));
        }hiloList->append(new QString("Hi: " + QString(hiList.at(lowList.size()).toElement().text()) + " F"));
    }else{
        for(int i = 0; i < hiList.size(); i++){
            hiloList->append(new QString("Lo: " + QString(lowList.at(i).toElement().text()) + " F"));
            hiloList->append(new QString("Hi: " + QString(hiList.at(i).toElement().text()) + " F"));
        }hiloList->append(new QString("Lo: " + QString(lowList.at(hiList.size()).toElement().text()) + " F"));
    }

    return hiloList;
}


/**
  National Weather service does not have a consistent xml feed. Sometimes the days
  of the week are the second 'time-layout' tag, sometimes they're the third, we have
  have to search for them
  */
QList<QString*>* MainWindow::parse_for_days(QDomElement *docEle){
    QDomNodeList weekList = docEle->elementsByTagName("time-layout");
    for(int i = 0; i < weekList.size(); i++){
        if( (weekList.at(i).firstChildElement().text() == "k-p12h-n15-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n14-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n13-1") ){
            weekList = weekList.at(i).toElement().elementsByTagName("start-valid-time");
        }
    }

    QList<QString*>* days = new QList<QString*>();
    for(int i = 0; i < weekList.size(); i++){
        days->append(new QString(weekList.at(i).toElement().attribute("period-name")));
    }

    return days;
}

/**
  Parse the xml file for the city that represents the zipcode. Not all
  xml files will contains the same city zipcode (one of two choices...so far)
  so we need to see which one it has
  */
QString* MainWindow::parse_for_city(QDomElement *docEle){
    QDomNodeList locList(docEle->firstChild().nextSibling().toElement().elementsByTagName("location"));
    QString location;
    if(locList.at(0).toElement().elementsByTagName("city").size() > 0){
        location = QString(locList.at(0).toElement().elementsByTagName("city").at(0).toElement().text());
    }else{
        location = QString(locList.at(0).toElement().elementsByTagName("area-description").at(0).toElement().text());
    }

    QString* city = new QString(location);
    return city;
}


/**
  Parse the dom document for the precipitations and return them
  in a QList of QStrings. Also format the string's to include
  "Precipitation: " so setting label's is easier later
  */
QList<QString*>* MainWindow::parse_for_precip(QDomElement *docEle){
    QDomNodeList* precipList = &docEle->elementsByTagName("probability-of-precipitation").at(0).toElement().elementsByTagName("value");
    QList<QString*>* returnPrecips = new QList<QString*>();

    for(int i = 0 ; i < precipList->size(); i++){
        QString str("Precipitation: " + precipList->at(i).toElement().text() + "%");
        if(precipList->at(i).toElement().text().isEmpty()){
            str = "Precipitation: Effectively 0%";
        }
        returnPrecips->append(new QString(str));
    }
    return returnPrecips;

}


/**
  Slot that handles the changeZip()'s network request, which recieves a latitude
  and longitude conversion. This slot makes another request using that latitude
  and longitude to get the XML file containing the weather information
  */
void MainWindow::gotCoordinates(){
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    QString str(reply->readAll());

    if( !str.startsWith("<error>") ){
        //get the latitude and longitude if we don't already have it
        //empty the file and try to make it represent the new download
        setDomDocument(&str);

        latlon->clear();
        QDomElement docEle = doc->documentElement();
        QDomNode node = docEle.firstChild();
        QString latElement(node.toElement().text());
        latlon->append( QString("lat=" + latElement.replace(",", "&lon="))); //latitude and longitude parameter for the url
        QString urlStr("http://forecast.weather.gov/MapClick.php?xxxxx&FcstType=dwml");
        urlStr.replace(41, 5, *latlon);
        reply->deleteLater();
        QNetworkReply* xmlReply = manager->get(QNetworkRequest(urlStr));
        connect(xmlReply, SIGNAL(finished()), this, SLOT(gotXML()));
    }else{
        pg->hide();
        ErrorMessage em(&zipCode, this);
        em.exec();
        em.deleteLater();
        zipCode = zipCache;
    }
    reply->deleteLater();
}


/**
  Slot for handling the reception of the XML weather file from the NAOO
  server. This slot parses the data and adds them to the interface
  */
void MainWindow::gotXML(){
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    reply->deleteLater();
    QString str(reply->readAll());

    if( !str.startsWith("<error>") ){
        setDomDocument(&str);
        QDomElement docEle = doc->documentElement();
        deleteCache();

        //parse the XML file for the weather data
        QList<QString*>* hiloList = parse_for_temps(&docEle);
        QList<QString*>* weekList = parse_for_days(&docEle);
        QString* location = parse_for_city(&docEle);
        QList<QString*>* precip = parse_for_precip(&docEle);
		parse_for_wfo(&docEle);
		QDomNodeList iconList = docEle.elementsByTagName("conditions-icon").at(0).toElement().elementsByTagName("icon-link");
        QDomNodeList conditionList = docEle.elementsByTagName("weather").at(0).toElement().elementsByTagName("weather-conditions");
        QDomNodeList wordedList = docEle.elementsByTagName("wordedForecast").at(0).toElement().elementsByTagName("text");

        //request the icon for the static forecast pane
        QNetworkReply* iconReply = manager->get(QNetworkRequest(QUrl(iconList.at(0).toElement().text())));
        connect(iconReply, SIGNAL(finished()), this, SLOT(setFIcon()));
        ui->forecast1->setText(wordedList.at(0).toElement().text());

        /*Sets the icons to the 'loading' symbol*/
        ui->forecastIcon1->setPixmap(QPixmap(":/loading.png"));
        for(int i = 0 ;i < 12; i++){
            dayPeriods->at(i)->setLoading();
        }

        //Set the title of this window to the city and
        //pass the weather data to the day periods and request
        //the icons for each day also
        /*we do one extra iteration to download icon number 12
          so that when the user clicks the 11th icon, they can see
          the full day*/
        setWindowTitle(*location);
        ui->title->setText(*weekList->at(0));
        for(int i = 0; i < 13; i++){
            if(i < 12){
                QString title(*weekList->at(i));
                QString condition(conditionList.at(i).toElement().attribute("weather-summary"));
                QString temp(*hiloList->at(i));
                dayPeriods->at(i)->setInfo(title, condition, temp);
            }

            QString icon(iconList.at(i).toElement().text());
            QNetworkReply* dayReply = manager->get(QNetworkRequest(QUrl(icon)));
            connect(dayReply, SIGNAL(finished()), mapper, SLOT(map()));
            mapper->setMapping(dayReply, i);
        }

        //add information to the current day (left pane)
        QString zc("MEZ015");
        QNetworkReply* tempGraphReply = manager->get(QNetworkRequest(QUrl(QString("http://forecast.weather.gov/wtf/meteograms/Plotter.php?" + *latlon + "&wfo=" + *wfo + "&zcode=" + zc + "&gset=20&gdiff=10&unit=0&tinfo=EY5&ahour=0&pcmd=100100000000000000000000000000000000000000000000&lg=en&indu=1!1!1&dd=0&bw=0&hrspan=48&pqpfhr=6"))));
        connect(tempGraphReply, SIGNAL(finished()), this, SLOT(setTempGraph()));
        ui->forecastTemp->setText(*hiloList->at(0));
        ui->forecastPrecip->setText(*precip->at(0));

        if(needShowList){
            needShowList = false;
            for(int i = 0; i < 12; i++){
                ui->listWidget->addItem(dayPeriods->at(i)->getItem());
            }
			ui->weekBanner->setText("Upcoming Days");
			ui->forecastLabel->setText("Hourly Temperatures: ");
			colorize(ui->weekBanner);
			colorize(ui->title);
			colorize(ui->forecastLabel);
        }

        pg->hide();
        delete hiloList;
        delete weekList;
        delete location;
        delete precip;
    }else{
        pg->hide();
        ErrorMessage em(&zipCode, this);
        em.exec();
        em.deleteLater();
        zipCode = zipCache;
    }

}

/**
  Applies the color theme to a label. The color is
  derived from the system theme and implemented as a
  gradient
  */
void MainWindow::colorize(QLabel *label){
	label->setPalette(QApplication::palette().highlight().color());
	QPalette pal = label->palette();

	QLinearGradient linearGrad(QPointF(0, 0), QPointF(0, 35));
	linearGrad.setColorAt(0, pal.color(QPalette::Highlight));
	linearGrad.setColorAt(1, Qt::black);
	QBrush brush(linearGrad);

	pal.setBrush(QPalette::Window, brush);;
	label->setPalette(pal);
}

/**
  Slot for setting the static temperature graph
  in the left pane
  */
void MainWindow::setTempGraph(){
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    QPixmap pix;
    pix.loadFromData(reply->readAll());
    ui->tempGraph->setPixmap(pix);
    reply->deleteLater();
}

/**
  Uses the mapping from the SignalMapper to delegate the proper
  replies to their respective days and passes them the downloaded
  icon
  */
void MainWindow::setDayIcons(int dayIndex){
    QSignalMapper* sm = qobject_cast<QSignalMapper*>(sender());
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sm->mapping(dayIndex));
    reply->deleteLater();

    QPixmap* pix = new QPixmap();
    pix->loadFromData(reply->readAll());

	delete iconCache[dayIndex];
	iconCache[dayIndex] = pix;

	if(dayIndex < 12){//there are 11 days, but 12 pictures
		QIcon icn(*pix);
		dayPeriods->at(dayIndex)->setIcon(&icn);
    }
}




/**
  Sets the first static forecast Icon. This is a reply for a network request
  for the image.
  @param reply The network reply that contains the reply. It should be a reply
  from the National Weather Service containing an XML file.
  */
void MainWindow::setFIcon(){
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    QPixmap pix;
    pix.loadFromData(reply->readAll());
    ui->forecastIcon1->setPixmap(pix);
    reply->deleteLater();
}


/**
  Resets the icon cache to store new icons
  */
void MainWindow::deleteCache(){
    for(int i = 0; i < 13; i++){
		delete iconCache[i];
    }
    for(int i = 0; i < 13; i++){
        iconCache[i] = new QPixmap();
    }
}

/**
  Creates a detailed view of the selected day
  and shows it.
  */
void MainWindow::detailWindow(int day){
    //create lists of all of the relevant information
    QDomElement docEle = doc->documentElement();
    QList<QString*>* hiloList = parse_for_temps(&docEle);
    QList<QString*>* weekList = parse_for_days(&docEle);
    QList<QString*>* precipList = parse_for_precip(&docEle);
    QDomNodeList wordedList = docEle.elementsByTagName("wordedForecast").at(0).toElement().elementsByTagName("text");
    QDomNodeList iconList = docEle.elementsByTagName("conditions-icon").at(0).toElement().elementsByTagName("icon-link");

    int dayVar = day;
    int nightVar = day;
    if(weekList->at(day)->contains(" Night")){
        dayVar = day-1;
    }else{
        nightVar = day+1;
    }

    //get the info from the xml file
    QString offset = QString::number(dayVar*12);
    QString weekDay("Day: " + *weekList->at(dayVar));
    QString hi(*hiloList->at(dayVar));
    QString lo(*hiloList->at(nightVar));
    QString precip(*precipList->at(day));
    QString nightSummary(wordedList.at(nightVar).toElement().text());
    QString daySummary(wordedList.at(dayVar).toElement().text());


    //create a window to show it
    StackedWindow *sw = new StackedWindow(latlon, wfo, zcode,
                                          &offset, iconCache[dayVar], &iconList.at(dayVar).toElement().text(),
                                          iconCache[nightVar], &iconList.at(nightVar).toElement().text(), this);
#ifdef Q_WS_MAEMO_5
    sw->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    sw->setAttribute(Qt::WA_DeleteOnClose);
    sw->setInfo(weekDay, hi, lo, precip, daySummary, nightSummary);
    sw->show();

    delete hiloList;
    delete weekList;
	delete precipList;
}
