#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QFile>
#include <iostream>
#include <QDesktopServices>
#include "errormessage.h"
#include "stackedwindow.h"
#include "dialog.h"


using namespace std;

/**
  Constructor for our window.
  @param parent The parent of this window. This specific
  window has no parent as it is the top level window
  */
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
     ui->setupUi(this);



    /*Create the list of day periods that represent the day / night info
      we get from the xml files and add them to the UI*/
    dayPeriods = new QList<DayPeriod*>();
    for(int i = 0; i < 12; i++){
        DayPeriod *dp = new DayPeriod(i);
        dp->getItem()->setWhatsThis(QString::number(i));
        dayPeriods->append(dp);
    }

    connect(ui->listWidget, SIGNAL(activated(QModelIndex)), this, SLOT(dayClick(QModelIndex)));

    pg = new ProgressDialog(this);
    iconCount = 0;
    needCoords = true;
    needShowList = true;
    latlon = new QString(); //graph parameter
    zcode = new QString();  //graph parameter
    wfo = new QString();    //graph parameter
    doc = new QDomDocument("document"); //will be the xml document object
    iconURLs = new QList<QString*>();

    manager = new QNetworkAccessManager(this);
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(replyFinished(QNetworkReply*)));

    /*addapt the application to maemo by providing
      stackable windows and auto rotation*/
    setAttribute(Qt::WA_Maemo5StackedWindow, true);
    setAttribute(Qt::WA_Maemo5AutoOrientation);


    /*signal that the UI has finished setting up, and start the
      dialog for getting a zip code from the user*/
    connect(this, SIGNAL(uiReady()), this, SLOT(zipEvent()), Qt::QueuedConnection);
    emit uiReady();

}

/**
  Deconstructor for the window. When this window closes
  the application ends.
  */
MainWindow::~MainWindow()
{
    delete ui;
}

/**
  Launches the default browser. Loads the current city in the NAOO website
  */
void MainWindow::visitNAOO(){
    QString str("http://forecast.weather.gov/zipcity.php?inputstring=" + zipCode);
    QDesktopServices::openUrl(QUrl(str));
}

/**
  Function for the day buttons. Shows the detailed view of that day
  */
void MainWindow::dayClick(QModelIndex index){
    ui->listWidget->currentItem()->setSelected(false);
    detailWindow(index.row());
}

/**
  Is called when the windows switches from landscape to portrait
  orientation. The function here specifically switches the scroll
  views layout from horizontal to vertical where appropriate
  @param event contains information about the screens new and
  old layout that we can use to check to see what mode it is currently
  in
  */
void MainWindow::resizeEvent(QResizeEvent *event){
    if(event->oldSize().width() > event->size().width()){
        ui->mainLayout->setDirection(QBoxLayout::TopToBottom);

        ui->listWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    }else{
        ui->mainLayout->setDirection(QBoxLayout::LeftToRight);
        ui->listWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    }
}


/**
  Sets the QDomDocument 'doc' variable up to be parsed as an XML file.
  @param reply The network reply that contains the reply. It should be a reply
  from the National Weather Service containing an XML file.
  */
void MainWindow::setDomDocument(QString *reply){
    QFile file("tmp.xml");

    if (!file.open(QIODevice::WriteOnly)){
        cout << "could not open" << endl;
        return;
    }
    file.write(reply->toStdString().c_str());
    file.close();
    file.open(QIODevice::ReadOnly);
    if (!doc->setContent(&file)) {
        cout << "cold not connect set content1" << endl;
        file.close();
        return;
    }
    file.close();


}

/**
  Called when the network manager finishes a request to the National
  Weather Service's servers. This function is called twice recursively.
  The first time it sends a request to get the coordinates of a zip code.
  The second time it uses the coordinates to request an XML file containing
  weather information for that original zip code. The second section parses
  the XML file as well, getting all the necessary information and setting up
  the UI accordingly.
  @param reply The network reply that contains the reply. It should be a reply
  from the National Weather Service containing an XML file.
  */
void MainWindow::replyFinished(QNetworkReply *reply){

    QString str(reply->readAll());
    if( !str.startsWith("<error>") ){
        //get the latitude and longitude if we don't already have it
        if(needCoords){
            //empty the file and try to make it represent the new download
            setDomDocument(&str);

            latlon->clear();
            QDomElement docEle = doc->documentElement();
            QDomNode node = docEle.firstChild();
            QString latElement(node.toElement().text());
            latlon->append( QString("lat=" + latElement.replace(",", "&lon="))); //latitude and longitude parameter for the url
            QString urlStr("http://forecast.weather.gov/MapClick.php?xxxxx&FcstType=dwml");
            urlStr.replace(41, 5, *latlon);
            needCoords = false;
            manager->get(QNetworkRequest(urlStr));

        /**
          Part two: parse the XML file containing the weather information
          */
        }else{//get the real file for parsing

            //empty the file and try to make it represent the new download
            setDomDocument(&str);

            QDomElement docEle = doc->documentElement();
            QDomNode node = docEle.firstChild();

            QDomNodeList hiList;
            QDomNodeList lowList;
            QDomNodeList tempList = docEle.elementsByTagName("temperature"); //temperature list
            for(int i = 0; i < tempList.size(); i++){
                if(tempList.at(i).toElement().attribute("type") == "maximum"){
                    hiList = tempList.at(i).toElement().elementsByTagName("value");
                }else{
                    lowList = tempList.at(i).toElement().elementsByTagName("value");
                }
            }

            /*merge temps into one day period ordered list*/
            QList<QString> hiloList;
            if(hiList.size() > lowList.size()){
                for(int i = 0; i < lowList.size(); i++){
                    hiloList.append(QString("Hi: " + hiList.at(i).toElement().text()) + " F");
                    hiloList.append(QString("Lo: " + lowList.at(i).toElement().text()) + " F");
                }hiloList.append(QString("Hi: " + hiList.at(lowList.size()).toElement().text()) + " F");
            }else{
                for(int i = 0; i < hiList.size(); i++){
                    hiloList.append(QString("Lo: " + lowList.at(i).toElement().text()) + " F");
                    hiloList.append(QString("Hi: " + hiList.at(i).toElement().text()) + " F");
                }hiloList.append(QString("Lo: " + lowList.at(hiList.size()).toElement().text()) + " F");
            }


            /*National Weather service does not have a consistent xml feed. Sometimes the days
            of the week are the second 'time-layout' tag, sometimes they're the third, we have
            have to search for them*/
            QDomNodeList weekList = docEle.elementsByTagName("time-layout");
            for(int i = 0; i < weekList.size(); i++){
                if( (weekList.at(i).firstChildElement().text() == "k-p12h-n15-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n14-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n13-1") ){
                    weekList = weekList.at(i).toElement().elementsByTagName("start-valid-time");
                }
            }

            QDomNodeList iconList = docEle.elementsByTagName("conditions-icon").at(0).toElement().elementsByTagName("icon-link");
            QDomNodeList conditionList = docEle.elementsByTagName("weather").at(0).toElement().elementsByTagName("weather-conditions");
            QDomNodeList wordedList = docEle.elementsByTagName("wordedForecast").at(0).toElement().elementsByTagName("text");

            //get the "wfo" (weather forecast office) parameter that is required for downlaoding graph images
            wfo->clear();
            try{
                wfo->append(QString(docEle.elementsByTagName("wordedForecast").at(0).toElement().attribute("dataSource")));
                wfo->truncate(3);
            }catch(...){
                cout << "fail"  << endl;
            }



            //get the "zcode" parameter that is required for downlaoding graph images

            if(docEle.elementsByTagName("hazards").size() > 0){
                delete zcode;
                zcode = new QString(docEle.elementsByTagName("hazards").at(0).toElement().firstChild().nextSibling().firstChild().firstChild().toElement().text().split("=").at(1));
                zcode->truncate(6);
            }else{
                cout <<  "fail: no 'hazards' section found in xml file" << endl;
            }



            /*Sets the icons to the 'loading' symbol*/
            for(int i = 0 ;i < 12; i++){
                dayPeriods->at(i)->setLoading();
            }
            ui->forecastIcon1->setPixmap(QPixmap(":/loading.png"));



            /*set the content of the day views*/
            QNetworkAccessManager *icon1 = new QNetworkAccessManager(this);
            connect(icon1, SIGNAL(finished(QNetworkReply*)),
                     this, SLOT(setFIcon(QNetworkReply*)));
            icon1->get(QNetworkRequest(QUrl(iconList.at(0).toElement().text())));
            ui->forecast1->setText(wordedList.at(0).toElement().text());


            /*get the description for the area*/
            QDomNodeList locList(node.nextSibling().toElement().elementsByTagName("location"));
            QDomNode location;
            if(locList.at(0).toElement().elementsByTagName("city").size() > 0){
                QDomNode node(locList.at(0).toElement().elementsByTagName("city").at(0));
                location = node;
            }else{
                QDomNode node(locList.at(0).toElement().elementsByTagName("area-description").at(0));
                location = node;
            }


            /*send the request for the icons*/

            iconURLs->clear();
            for(int i = 0; i < 12; i++){
                QString* str = new QString(iconList.at(i).toElement().text());
                iconURLs->append(str);
            }

            setWindowTitle(location.toElement().text());
            ui->title->setText(weekList.at(0).toElement().attribute("period-name"));
            for(int i = 0; i < 12; i++){
                QString title(weekList.at(i).toElement().attribute("period-name"));
                QString condition(conditionList.at(i).toElement().attribute("weather-summary"));
                QString temp(hiloList.at(i));
                QString icon(iconList.at(i).toElement().text());
                dayPeriods->at(i)->setInfo(title, condition, temp, icon);
            }

            if(needShowList){
                needShowList = false;
                for(int i = 0; i < 12; i++){
                    ui->listWidget->addItem(dayPeriods->at(i)->getItem());
                }
                ui->weekBanner->setText("Upcoming Days");
                ui->weekBanner->setPalette(QApplication::palette().highlight().color());
                ui->title->setPalette(QApplication::palette().highlight().color());
            }

            pg->hide();
            needCoords = true;
        }

    }else{
        pg->hide();
        ErrorMessage em(&zipCode, this);
        em.exec();
        em.deleteLater();
        zipCode = zipCache;
    }
    reply->deleteLater();

}



/**
  Sets the first static forecast Icon. This is a reply for a network request
  for the image.
  @param reply The network reply that contains the reply. It should be a reply
  from the National Weather Service containing an XML file.
  */
void MainWindow::setFIcon(QNetworkReply *reply){
    QPixmap pix;
    pix.loadFromData(reply->readAll());
    ui->forecastIcon1->setPixmap(pix);
    reply->deleteLater();
}


/**
  Launches a dialog that prompts the user to enter a zip code
  into a text box and the executes a request to the server for
  an XML file.
  */
void MainWindow::changeZip(){
    zipCache = zipCode;
    zipCode.clear();
    Dialog *d = new Dialog(this);
    d->exec();
    zipCode = d->getZip();
    if(!zipCode.isEmpty()){
        zipCode.truncate(5);
        QString urlString("http://www.weather.gov/forecasts/xml/SOAP_server/ndfdXMLclient.php?whichClient=LatLonListZipCode&listZipCodeList=xxxxx&product=time-series&begin=2004-01-01T00:00:00&end=2015-01-22T00:00:00&maxt=maxt&Submit=Submit");
        urlString.replace(113, 5, zipCode);
        cout << urlString.toStdString() << endl;
        manager->get(QNetworkRequest(QUrl(urlString)));
        pg->show();
    }else{
        zipCode = zipCache;
        cout << "Dialog Canceled" << endl;
    }
    d->deleteLater();

}


void MainWindow::detailWindow(int day){
    //create lists of all of the relevant information
    QDomElement docEle = doc->documentElement();


    /*Create temperature list*/
    QDomNodeList hiList;
    QDomNodeList lowList;
    QDomNodeList tempList = docEle.elementsByTagName("temperature"); //temperature list
    for(int i = 0; i < tempList.size(); i++){
        if(tempList.at(i).toElement().attribute("type") == "maximum"){
            hiList = tempList.at(i).toElement().elementsByTagName("value");
        }else{
            lowList = tempList.at(i).toElement().elementsByTagName("value");
        }
    }
    /*merge temps into one day period ordered list*/
    QList<QString> hiloList;
    if(hiList.size() > lowList.size()){
        for(int i = 0; i < lowList.size(); i++){
            cout << i << endl;
            hiloList.append(QString("Hi: " + hiList.at(i).toElement().text()) + " F");
            hiloList.append(QString("Lo: " + lowList.at(i).toElement().text()) + " F");
        }hiloList.append(QString("Hi: " + hiList.at(lowList.size()).toElement().text()) + " F");
    }else{
        for(int i = 0; i < hiList.size(); i++){
            hiloList.append(QString("Lo: " + lowList.at(i).toElement().text()) + " F");
            hiloList.append(QString("Hi: " + hiList.at(i).toElement().text()) + " F");
        }hiloList.append(QString("Lo: " + lowList.at(hiList.size()).toElement().text()) + " F");
    }



    QDomNodeList weekList = docEle.elementsByTagName("time-layout");
    /*National Weather service does not have a consistent xml feed. Sometimes the days
    of the week are the second 'time-layout' tag, sometimes they're the third, we have
    have to search for them*/
    for(int i = 0; i < weekList.size(); i++){
        if( (weekList.at(i).firstChildElement().text() == "k-p12h-n15-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n14-1") | (weekList.at(i).firstChildElement().text() == "k-p12h-n13-1") ){
            weekList = weekList.at(i).toElement().elementsByTagName("start-valid-time");
        }
    }
    QDomNodeList wordedList = docEle.elementsByTagName("wordedForecast").at(0).toElement().elementsByTagName("text");
    QDomNodeList iconList = docEle.elementsByTagName("conditions-icon").at(0).toElement().elementsByTagName("icon-link");
    QDomNodeList precipList = docEle.elementsByTagName("probability-of-precipitation").at(0).toElement().elementsByTagName("value");

    int dayVar = day;
    int nightVar = day;
    if(weekList.at(day).toElement().attribute("period-name").contains(" Night")){
        dayVar = day-1;
    }else{
        nightVar = day+1;
    }

    //get the info from the xml file
    QString offset = QString::number(dayVar*12);
    QString dayIcon(iconList.at(dayVar).toElement().text());
    QString nightIcon(iconList.at(nightVar).toElement().text());
    QString weekDay("Day: " + weekList.at(dayVar).toElement().attribute("period-name"));
    QString hi(hiloList.at(dayVar));
    QString lo(hiloList.at(nightVar));
    QString tmpPrecip = precipList.at(day).toElement().text();
    if(tmpPrecip.isEmpty()){
        tmpPrecip = "Effectively 0";
    }
    QString precip("Precipitation: " + tmpPrecip + "%");
    QString nightSummary(wordedList.at(nightVar).toElement().text());
    QString daySummary(wordedList.at(dayVar).toElement().text());

    //create a window to show it
    StackedWindow *sw = new StackedWindow(*latlon, *wfo, *zcode, offset, dayIcon, nightIcon, this);
    sw->setAttribute(Qt::WA_Maemo5StackedWindow);
    sw->setAttribute(Qt::WA_DeleteOnClose);
    sw->setInfo(weekDay, hi, lo, precip, daySummary, nightSummary);
    sw->show();


}
