/*
 * Copyright (c) 2010 David Galindo <nowheremanmail@gmail.com>
 *
 */

#include "utils.h"

#include <Qt>
#include <QtGui>
#include <QFile>
#include <QDesktopServices>
#include <QListView>

#include "eWallet.h"
#include "Constants.h"
#include "DatabaseManager.h"
#include "ListOfCards.h"

class ListOfCards::MyDelegate : public QStyledItemDelegate  {
	ListOfCards & parentWallet;

	public:

	MyDelegate(ListOfCards & w, QObject *parent=0): QStyledItemDelegate (parent), parentWallet (w)
	{ 	}

	void paint ( QPainter * painter, const QStyleOptionViewItem & option, const QModelIndex & index ) const{
		if(option.state & QStyle::State_Selected){
			painter->fillRect(option.rect, option.palette.color(QPalette::Highlight));
		}

		QVariant pointer = index.data(Qt::UserRole + 1);
		QString title = index.data(Qt::DisplayRole).toString();

		//qDebug () << "paiting " << title;

		if (pointer.isNull()) {
			painter->save();
			QFont ft = option.font;
			ft.setBold(true);
			painter->setFont(ft);
			QRect r = option.rect.adjusted(15, 0, 0, 0);
			painter->drawText(r.left(), r.top()-15, r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft|Qt::TextWordWrap, title, &r);
			painter->restore();
		}
		else {
			QVariant val = index.data(Qt::DecorationRole);
			Target * target = (Target *) pointer.value<void *>();
			//parentWallet.list.at(_index.toInt());

			if (target == NULL)  {
				qDebug () << "invalid target";
			}

			QString description = target->getSummary();
			// TODO translate ?? tr(types_string[this->parent->types->at(i)->id - 1])

			QIcon pixmap;
			pixmap = qvariant_cast<QIcon>(val);

			QRect r = option.rect.adjusted(0, 0, option.decorationSize.width(), option.decorationSize.height());
			pixmap.paint(painter, 5, r.top () + 5, SMALL_WIDTH, SMALL_HEIGH, option.decorationAlignment, QIcon::Normal);

			painter->save();
			QFont ft = option.font;
			ft.setBold(true);
			painter->setFont(ft);
			r = option.rect.adjusted(SMALL_WIDTH + 10, 0, 0, -SMALL_HEIGH/2);
			painter->drawText(r.left(), r.top(), r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft|Qt::TextWordWrap, title, &r);

			r = option.rect.adjusted(SMALL_WIDTH + 15, SMALL_HEIGH/2+10, 0, 0);
			ft.setBold(false);
			ft.setItalic(true);
			ft.setPointSize(ft.pointSize()-4);
			painter->setFont(ft);
			painter->drawText(r.left(), r.top(), r.width(), r.height(), Qt::AlignLeft|Qt::TextWordWrap, description, &r);
			painter->restore();
		}
	}

	QSize sizeHint ( const QStyleOptionViewItem & option, const QModelIndex & index ) const {
		QVariant val = index.data(Qt::DecorationRole);
		if (val.isNull()) {
			return QSize(800, 70);
		}
		else {
			return QSize(200, SMALL_HEIGH+10);
		}
	}
};


ListOfCards::ListOfCards(bool isF, QString * group, QWidget *parent) :
	QWidget(parent) {

	isFirst = isF;
	currentGroup = group;

	ascSort = true;
	currentFilter = "";

	isShowGroups = 2;

	showList(true);
	showTargets(true, currentGroup);
}

ListOfCards::ListOfCards(bool isF, QWidget *parent) :
	QWidget(parent) {

	isFirst = isF;
	currentGroup = NULL;
	isShowGroups = 1;

	ascSort = true;
	currentFilter = "";

	showList(true);
	showGroups(true);
}

void ListOfCards::showList(bool interactive) {
	//qDebug () << "going to showList";

	setProperty("FingerScrollable", false);

	if (interactive) {
		setContextMenuPolicy(Qt::DefaultContextMenu);
	} else {
		setContextMenuPolicy(Qt::NoContextMenu);
	}
	//	this->setContextMenuPolicy(Qt::DefaultContextMenu);

	vboxlayout = new QVBoxLayout;
	vboxlayout->setSpacing(0);
	vboxlayout->setMargin(0);
	// Custom list box
	//vboxlayout->addWidget(backButton);

	if(!threadLoad.isNull()) {
		disconnect(threadLoad, SIGNAL(loadImageFor(int)), this, SLOT(loadImageFor(int)));
		qDebug () << "Wait for thread create";

		while (threadLoad->isRunning()) {
			threadLoad->wait(100);
			QCoreApplication::processEvents ();
		}

		delete threadLoad;
		threadLoad = 0;
	}

	fnListWidget = new QListWidget(this);
//	this->originalDelegate = new MyDelegate(*this, this);// TODO fnListWidget->itemDelegate();
//	this->newDelegate = new MyDelegate(*this, this);

	fnListWidget->setItemDelegate(new MyDelegate(*this, this));

	fnListWidget->setSortingEnabled(false);
	fnListWidget->setFocusPolicy(Qt::NoFocus);

	vboxlayout->addWidget(fnListWidget);

	//fnListWidget->grabGesture(Qt::SwipeGesture);

	if (interactive) {
		QObject::connect(fnListWidget, SIGNAL(itemClicked(QListWidgetItem *)),
				this, SLOT(selectItem(QListWidgetItem *)));
		QObject::connect(fnListWidget,
				SIGNAL(itemDoubleClicked(QListWidgetItem *)), this,
				SLOT(selectDItem(QListWidgetItem *)));

		inputWidget = new QHBoxLayout;
		inputWidget->setContentsMargins(2, 2, 2, 0);
		vboxlayout->addLayout(inputWidget);

		cancelSearch = new QPushButton(QIcon(":/resources/icon_clear.png"), "", this);
		searchField = new QLineEdit(this);
		hideSearch = new QPushButton(QIcon(":/resources/icon_hide.png"), "", this);

		inputWidget->addWidget(cancelSearch);
		searchField->setText(currentFilter);
		searchField->setInputMethodHints(Qt::ImhNoAutoUppercase	| Qt::ImhNoPredictiveText);
		inputWidget->addWidget(searchField);
		inputWidget->addWidget(hideSearch);

		//QObject::connect(backButton, SIGNAL(clicked()), this, SLOT(_clickBack()));
		//backButton->hide ();

		QObject::connect(cancelSearch, SIGNAL(clicked()), this,
				SLOT(_cancelSearch()));
		QObject::connect(hideSearch, SIGNAL(clicked()), this,
				SLOT(_hideSearch()));
		QObject::connect(searchField, SIGNAL(textChanged(const QString &)),
				this, SLOT(textChanged(const QString &)));
	} else {
		searchField = NULL;
		cancelSearch = NULL;
		hideSearch = NULL;
	}

	this->setLayout(vboxlayout);
//
//	QWidget * window->setContentsMargins(0, 0, 0, 0);
//	QVBoxLayout * ll = new QVBoxLayout (window);
//	ll->setContentsMargins(0, 0, 0, 0);
//	ll->addWidget(this);

	////qDebug () << "setFocus()";
	this->setFocusPolicy(Qt::StrongFocus);
	this->setFocus();

	//fnListWidget->setFocus ();

	if (interactive) {
		if (currentFilter.length() > 0) {
			cancelSearch->show();
			searchField->show();
			//setFocus ();
			//focusNextChild();
			searchField->setFocus();
			hideSearch->show();
		} else {
			cancelSearch->hide();
			searchField->hide();
			hideSearch->hide();
		}
	}
}

void ListOfCards::_clickBack() {
	isShowGroups = (isShowGroups) % 2 + 1;
	_cancelSearch();
	showInfo(isShowGroups, true);
}
void ListOfCards::changeMode() {
	isShowGroups = (isShowGroups) % 2 + 1;
	_cancelSearch();
	showInfo(isShowGroups, true);
}


void ListOfCards::selectItem(QListWidgetItem * item) {
	if (isShowGroups == 1) {
		if (fnListWidget->currentItem() != item) {
			fnListWidget->setCurrentItem(item);
		}

		QString *_target;

		if (fnListWidget->currentRow() > 0) {
			_target = new QString(grouplist.at(fnListWidget->currentRow() - 1));
		}
		else {
			_target = NULL;
		}

		emit selectGroup (_target);
	}
	else {
		if (fnListWidget->currentItem() != item) {
			fnListWidget->setCurrentItem(item);
		}
		Target * target = list.at(fnListWidget->currentRow());

		emit selectTarget(target);
	}
}

void ListOfCards::selectDItem(QListWidgetItem * item) {
	selectItem(item);
}

void ListOfCards::textChanged(const QString & text) {
	currentFilter = text;
	filterBy(text);
}

void ListOfCards::filterBy(const QString & s) {
	qDebug () << "filter By " << s;
	if (isShowGroups == 1) {
		bool isIncluded = false;

		isIncluded = tr("<All>").contains(s, Qt::CaseInsensitive);
		if (s.length() == 0) {
			fnListWidget->item(0)->setHidden(false);
		} else {
			if (isIncluded) {
				fnListWidget->item(0)->setHidden(false);
			} else {
				fnListWidget->item(0)->setHidden(true);
			}
		}

		for (int i = 0; i < grouplist.size(); i++) {
			if (i > 0 || (i == 0 && grouplist.at(0).length() > 0)) {
				QString target = grouplist.at(i);
				isIncluded = target.contains(s, Qt::CaseInsensitive);
			} else {
				isIncluded = tr("<No Group>").contains(s, Qt::CaseInsensitive);
			}
			//qDebug () << " GROUP 4 " << i;

			////qDebug () << "cc GROUP " << i;
			if (s.length() == 0) {
				fnListWidget->item(i + 1)->setHidden(false);
			} else {
				if (isIncluded) {
					fnListWidget->item(i + 1)->setHidden(false);
				} else {
					fnListWidget->item(i + 1)->setHidden(true);
				}
			}
		}
	} else {
		for (int i = 0; i < list.size(); i++) {
			Target * target = list.at(i);

			if (s.length() == 0) {
				fnListWidget->item(i)->setHidden(false);
			} else {
				if (target->name.contains(s, Qt::CaseInsensitive)) {
					fnListWidget->item(i)->setHidden(false);
				} else {
					fnListWidget->item(i)->setHidden(true);
				}
			}
		}
	}
}

void ListOfCards::_cancelSearch() {
	currentFilter = "";
	searchField->setText("");
	searchField->hide();
	cancelSearch->hide();
	hideSearch->hide();
}

void ListOfCards::_hideSearch() {
	searchField->hide();
	cancelSearch->hide();
	hideSearch->hide();
}

ListOfCards::~ListOfCards() {
	if(!threadLoad.isNull()) {
		disconnect(threadLoad, SIGNAL(loadImageFor(int)), this, SLOT(loadImageFor(int)));
		qDebug () << "Wait for thread destroy";

		while (threadLoad->isRunning()) {
			threadLoad->wait(100);
			QCoreApplication::processEvents ();
		}

		//delete threadLoad;
		//threadLoad = 0;
	}
	qDebug () << "deleting group";
	if (currentGroup != NULL)
		delete currentGroup;
	//delete ::dbManager;
	qDebug () << "end ~ListOfCards";
}


void ListOfCards::keyPressEvent(QKeyEvent* event) {
	qDebug()<<"key pressed " << event->key();

	if (searchField != NULL) {
		if (searchField->isHidden()) {
			//			if (event->key () == Qt::Key_Left || event->key () == Qt::Key_Right ) {
			//				changeMode();
			//				event->accept();
			//			}
			//			else {
			cancelSearch->show();
			searchField->show();
			hideSearch->show();
			searchField->setFocus();

			QCoreApplication::sendEvent(searchField, event);
			event->accept();
			//			}
		} else {
			QWidget::keyPressEvent(event);//event->accept();
		}
	} else {
		QWidget::keyPressEvent(event);//event->accept();
	}
	//timer->stop();
	//timer->start(5000);
}

bool sortTargetsASC(const Target * f, const Target * g) {
	return f->name.toLower() < g->name.toLower();
}
bool sortTargetsDES(const Target * f, const Target * g) {
	return f->name.toLower() > g->name.toLower();
}
bool sortStringASC(const QString f, const QString g) {
	return f.toLower() < g.toLower();
}
bool sortStringDES(const QString f, const QString g) {
	return f.toLower() > g.toLower();
}

void ListOfCards::showGroups(bool reset) {
	qDebug () << "ListOfCards show Groups " << reset;

	if(!threadLoad.isNull()) {
		disconnect(threadLoad, SIGNAL(loadImageFor(int)), this, SLOT(loadImageFor(int)));
		qDebug () << "Wait for thread groups";

		while (threadLoad->isRunning()) {
			threadLoad->wait(100);
			QCoreApplication::processEvents ();
		}

		delete threadLoad;
		threadLoad = 0;
	}

	//fnListWidget->setItemDelegate(originalDelegate);

	int current = fnListWidget->currentRow();

	// Clean previous items from the layout
	fnListWidget->clear();

	if (reset) {
		grouplist.clear();
		grouplist = ::dbManager->getGroups();
	}

	if (ascSort)
		qSort(grouplist.begin(), grouplist.end(), sortStringASC);
	else
		qSort(grouplist.begin(), grouplist.end(), sortStringDES);

	//	fnListWidget->setIconSize (QSize (SMALL_WIDTH, SMALL_HEIGH));
	//	fnListWidget->setContentsMargins(10, 1, 0, 1);

	QListWidgetItem* listitem = new QListWidgetItem(tr("<All>"));

	//listitem->set(target->getSummary());
	fnListWidget->addItem(listitem);

	qDebug () << "groups found " << grouplist.size();

	for (int i = 0; i < grouplist.size(); i++) {
		QString target = grouplist.at(i);

		QListWidgetItem * listitem;
		if (target.length() == 0) {
			listitem = new QListWidgetItem(tr("<No Group>"));
		} else {
			listitem = new QListWidgetItem(target);
		}
		//listitem->set(target->getSummary());
		fnListWidget->addItem(listitem);

		qDebug () << "group found (" << target << ")";
	}

	filterBy(currentFilter);

	if (current >= 0 && current <= grouplist.size())
		fnListWidget->setCurrentRow(current);
	else
		fnListWidget->setCurrentRow(0);

//	update();
}

void ListOfCards::showInfo(int isShowGroups, bool reset) {
	this->isShowGroups = isShowGroups;
	if (isShowGroups == 1) {
		//backButton->hide();
		showGroups(reset);
	} else {
		showTargets(reset, currentGroup);
	}
}


void ListOfCards::loadImageFor(int index) {
	if (index >= 0 && index < list.size()) {
		//qDebug () << "loading image for " << index;

		//QCoreApplication::sendPostedEvents ();

		Target * target = list.at(index);
		QListWidgetItem * listitem = fnListWidget->item(index);
		listitem->setIcon(::dbManager->getImage(target));

		QCoreApplication::processEvents ();
		//QApplication::instance()->processEvents();
	}
}

void ListOfCards::showTargets(bool reset, const QString * filter) {
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
	qDebug () << "ListOfCards show Targets " << reset;

	if(!threadLoad.isNull()) {
		disconnect(threadLoad, SIGNAL(loadImageFor(int)), this, SLOT(loadImageFor(int)));
		qDebug () << "Wait for thread showTargets";

		while (threadLoad->isRunning()) {
			threadLoad->wait(100);
			QCoreApplication::processEvents ();
		}

		delete threadLoad;
		threadLoad = 0;
	}

	//fnListWidget->setItemDelegate(newDelegate);

	int current = fnListWidget->currentRow();

	//#ifdef Q_WS_MAEMO_5
	//	show_banner (this, tr("Loading data, please wait"));
	//#endif

	//QApplication::instance()->processEvents();

	// Clean previous items from the layout
	fnListWidget->clear();

	if (reset) {
		Target * target;
		foreach(target, list)
			{
				delete target;
			}
		//            for (int i = 0; i < list.size (); i++) {
		//                    Target * target = list.at(i);
		//                    delete target;
		//                }
		list.clear();
		list = ::dbManager->getTargets(filter);
	}

	qDebug () << "sorting";

	if (ascSort)
		qSort(list.begin(), list.end(), sortTargetsASC);
	else
		qSort(list.begin(), list.end(), sortTargetsDES);

	fnListWidget->setIconSize(QSize(SMALL_WIDTH, SMALL_HEIGH));
	//fnListWidget->setContentsMargins(10, 1, 0, 1);

	for (int i = 0; i < list.size(); i++) {
		Target * target = list.at(i);

		//QListWidgetItem* listitem = new QListWidgetItem(::dbManager->getImage(target),target->getName());
		QListWidgetItem* listitem = new QListWidgetItem(::dbManager->getImage(NULL), target->getName());
		listitem->setData(Qt::UserRole + 1, qVariantFromValue((void *)target));

		fnListWidget->addItem(listitem);
	}

	filterBy(currentFilter);

	if (current >= 0 && current <= list.size())
		fnListWidget->setCurrentRow(current);
	else
		fnListWidget->setCurrentRow(0);

//	update();

#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif

	threadLoad = new MyLoadImageThread;
	threadLoad->size = list.size(); //parentWallet = this;
	connect(threadLoad, SIGNAL(loadImageFor(int)), this, SLOT(loadImageFor(int)), Qt::BlockingQueuedConnection);
	threadLoad->start();
}

void ListOfCards::sort() {
	ascSort = !ascSort;
	qDebug () << "invert sort " << ascSort;

	showInfo(isShowGroups, false);
}

bool ListOfCards::_insertTarget(Target* target) {
	if (isShowGroups == 2) {
		list.append(target);

		if (ascSort)
			qSort(list.begin(), list.end(), sortTargetsASC);
		else
			qSort(list.begin(), list.end(), sortTargetsDES);

		QListWidgetItem* listitem = new QListWidgetItem(::dbManager->getImage(target), target->name);
		//QListWidgetItem* listitem = new QListWidgetItem(target->name);
		//listitem->setData(Qt::UserRole + 1, *target);

		int pos = list.indexOf(target);
		qDebug() << "position of new " << pos << " OF " << list.size();

		listitem->setData(Qt::UserRole + 1, qVariantFromValue((void *)target));

		if (pos < list.size()) {
			qDebug() << "insert position of new " << pos << " OF "
					<< list.size();
			fnListWidget->insertItem(pos, listitem);
		}
		else {
			qDebug() << "append position of new " << pos << " OF "
					<< list.size();
			fnListWidget->addItem(listitem);
		}
		fnListWidget->setCurrentItem(listitem);
	} else {
		showGroups(true);
	}
	return true;
}

bool ListOfCards::_deleteTarget(Target* target) {
	if (isShowGroups == 2) {
		int i = list.indexOf(target);
		//qDebug () << "DELETE ON " << i;
		//qDebug () << "trying to remove " << i;
		list.removeAt(i);
		//qDebug () << "trying to takeItem " << i;
		fnListWidget->takeItem(i); //oveItem (row);

	}
	//		target->freeFieldList();
	//		target->freeResources();
	//		delete target;
	return true;
}

bool ListOfCards::_updateTarget(Target* target) {
	if (isShowGroups == 2) {

		int i = list.indexOf(target);
		qDebug () << "_updateTarget " << target->id << " on " << i << " " << target->groupName;
		QListWidgetItem * listitem = fnListWidget->item(i);

		// check if the group has been changed
		bool changedGroup = false;
		if (list.length() > 1) {
			if (i >= 1) {
				changedGroup = target->groupName != list.at(i - 1)->groupName;

				qDebug () << "compare groups " << target->groupName << " -- " << list.at(i - 1)->groupName;
			}
			else {
				changedGroup = target->groupName != list.at(i + 1)->groupName;

				qDebug () << "compare groups " << target->groupName << " -- " << list.at(i + 1)->groupName;
			}
		} else {
			// TODO detect that group has been changed and reload all group!!!!

		}

		qDebug() << "group has been changed !!!!" << changedGroup;

		if (changedGroup) {
			// TODO sure!!! delete items?
			list.takeAt(i);
			fnListWidget->takeItem(i);
		} else {
			if (list.length() > 1) {
				if (ascSort)
					qSort(list.begin(), list.end(), sortTargetsASC);
				else
					qSort(list.begin(), list.end(), sortTargetsDES);

				int pos = list.indexOf(target);
				qDebug() << "position of new " << pos << " OF "
						<< list.size();

				if (i != pos) {
					fnListWidget->takeItem(i);
					if (pos < i) {
						fnListWidget->insertItem(pos, listitem);
					} else {
						if (pos < list.size()) {
							qDebug() << "insert position of new " << pos
									<< " OF " << list.size();
							fnListWidget->insertItem(pos, listitem);
						} else {
							qDebug() << "append position of new " << pos
									<< " OF " << list.size();
							fnListWidget->addItem(listitem);
						}
					}
				}
			}
			listitem->setText(target->name);
			fnListWidget->setCurrentItem(listitem);
		}
	} else {
		showGroups(true);
	}
	return true;
}

bool ListOfCards::_updateTargetPicture(Target* target) {
	if (isShowGroups == 2) {
		int i = list.indexOf(target);
		QListWidgetItem* listitem = fnListWidget->item(i);
		listitem->setIcon(::dbManager->getImage(target));
	}
	return true;
}

void ListOfCards::reloadInfo () {
	qDebug () << "reloadInfo";
	if (isShowGroups == 1) {
		showGroups(true);
	} else {
		grouplist.clear();
		grouplist = ::dbManager->getGroups();

		if (grouplist.size() >= 1 && isFirst) {
			if (grouplist.size() > 1 || grouplist.at(0).length() > 0) {
				isShowGroups = 1;
				showGroups(true);
			}
		}
	}
}

QString ListOfCards::getGroup(int i) {
	return grouplist.at(i);
}

void ListOfCards::reloadGroups () {
	grouplist.clear();
	grouplist = ::dbManager->getGroups();

	if (grouplist.size() > 1) {
		showInfo(isShowGroups, false);
	} else {
		if (grouplist.size() == 1 && grouplist.at(0).length() > 0) {
			showInfo(isShowGroups, false);
		} else {
			isShowGroups = 2;
			// TODO //showTargets(true, NULL);
			showInfo(isShowGroups, true);
		}
	}
}

int ListOfCards::currentRow () {
	return fnListWidget->currentRow();
}

void ListOfCards::contextMenuEvent(QContextMenuEvent * event) {
	qDebug () << "Should be show " << this->childAt(event->globalPos());
	if (isShowGroups == 1) {
		if (fnListWidget->currentRow() > 0) {
			QMenu menu(this);
			//menu.addAction(tr("View"), this, SLOT(_viewcTarget()));
			menu.addAction(tr("Rename"), this, SLOT(_renameGroup()));
			menu.addAction(tr("Delete"), this, SLOT(_deleteGroup()));
			menu.exec(event->globalPos());
		}
	}
	else {
		if (fnListWidget->currentRow() >= 0) {
			//Target * target = list.at(fnListWidget->currentRow());

			QMenu menu(this);
			menu.addAction(tr("View"), this, SLOT(_viewTarget()));
			menu.addAction(tr("Edit"), this, SLOT(_editTarget()));
			menu.addAction(tr("New like this"), this, SLOT(_newLikeTarget()));
			menu.addAction(tr("Duplicate"), this, SLOT(_duplicateTarget()));
			menu.addAction(tr("Delete"), this, SLOT(_deleteTarget()));
			menu.exec(event->globalPos());
		}
	}
}

void ListOfCards::_renameGroup() {
	if (fnListWidget->currentRow() > 0) {
		QString target = getGroup (fnListWidget->currentRow() - 1);

		emit renameGroup(target);
	}
}
void ListOfCards::_deleteGroup() {
	if (fnListWidget->currentRow() > 0) {
		QString target = getGroup (fnListWidget->currentRow() - 1);

		emit deleteGroup(target);
	}
}
void ListOfCards::_viewTarget() {
	if (fnListWidget->currentRow() >= 0) {
		if (fnListWidget->isItemHidden(fnListWidget->item(
				fnListWidget->currentRow())))
			return;
		Target * target = list.at(fnListWidget->currentRow());
		emit selectTarget(target);
	}
}
void ListOfCards::_editTarget() {
	if (fnListWidget->currentRow() >= 0) {
		if (fnListWidget->isItemHidden(fnListWidget->item(
				fnListWidget->currentRow())))
			return;
		Target * target = list.at(fnListWidget->currentRow());
		emit editTarget(target);
	}
}
void ListOfCards::_newLikeTarget() {
	if (fnListWidget->currentRow() >= 0) {
		if (fnListWidget->isItemHidden(fnListWidget->item(
				fnListWidget->currentRow())))
			return;
		Target * target = list.at(fnListWidget->currentRow());
		emit newLikeTarget(target);
	}
}
void ListOfCards::_duplicateTarget() {
	if (fnListWidget->currentRow() >= 0) {
		if (fnListWidget->isItemHidden(fnListWidget->item(
				fnListWidget->currentRow())))
			return;
		Target * target = list.at(fnListWidget->currentRow());
		emit duplicateTarget(target);
	}
}
void ListOfCards::_deleteTarget() {
	if (fnListWidget->currentRow() >= 0) {
		if (fnListWidget->isItemHidden(fnListWidget->item(
				fnListWidget->currentRow())))
			return;
		Target * target = list.at(fnListWidget->currentRow());
		emit deleteTarget(target);
	}
}
