/*
 * Copyright (c) 2010 David Galindo <nowheremanmail@gmail.com>

 */

#include "Constants.h"
#include "DatabaseManager.h"
#include "eWallet.h"
#include <QStringList>
#include <QDir>
#include <QVariant>
#include <QBuffer>
#include <QFile>
#include <QDesktopServices>
#include <QDebug>
#include <openssl/evp.h>

#define nBASE64

// ---------------------------------------------------------------------------
// Renter
// ---------------------------------------------------------------------------
Target::Target()
{
	id = 0;
	pictureSmall = NULL;
	picture = NULL;
	pictureBack = NULL;
}
Target::~Target()
{
	if(pictureSmall) delete pictureSmall;
	if(picture) delete picture;
	if(pictureBack) delete pictureBack;
}

void Target::freeResources()
{
	qDebug () << " free resources for " << this->id;
	if(picture) delete picture;
	if(pictureBack) delete pictureBack;
	picture = NULL;
	pictureBack = NULL;
}

void Target::freeFieldList()
{
	TargetField * field;
	foreach(field,this->fields) {
		delete field;
	}
	fields.clear ();
}

bool Target::existField (QString name) {
	TargetField * field;
	foreach(field,this->fields) {
		if (field->name == name) return true;
	}
	return false;
}
QString Target::getSummary () {
	return this->type;
}
QImage * Target::getPictureSmall () {
	//qDebug () << "getPictureSmall " << this->id << " " << pictureSmall;
	if(!pictureSmall) {
		QImage * orig = getPicture();

		if (orig) {
			pictureSmall = new QImage (orig->scaled (SMALL_WIDTH, SMALL_HEIGH));
		}
		else
			pictureSmall = NULL;

	}
	return pictureSmall;
}
QImage * Target::getPicture () {
	if(!picture) {
		picture = ::dbManager->getPicture (this);
	}
	return picture;
}
QImage * Target::getPictureBack () {
	if(!pictureBack) {
		pictureBack = ::dbManager->getPictureBack (this);
	}
	return pictureBack;
}
/*void Target::setPictureSmall (QImage * a) {
	if(pictureSmall) delete pictureSmall;
	pictureSmall = a;
}*/
void Target::setPicture (QImage * a) {
	if(picture) delete picture;
	picture = a;
	if(pictureSmall) delete pictureSmall;
	pictureSmall = NULL;
}
void Target::setPictureBack (QImage * a) {
	if(pictureBack) delete pictureBack;
	pictureBack = a;
}

TargetField::~TargetField()
{
}
// ---------------------------------------------------------------------------
// DatabaseManager
// ---------------------------------------------------------------------------
DatabaseManager::DatabaseManager(QObject *parent) :
		QObject(parent)
{
	opendb = false;
	initdb = false;
}

DatabaseManager::~DatabaseManager()
{
	qDebug () << "CLOSE DB";
	if (db.isOpen()) {
		qDebug () << "CLOSing DB";
		db.close();
	}
}

bool DatabaseManager::isInit()
{
	return db.tables().count() == 3;
}

bool DatabaseManager::initDB()
{
	qDebug () << "INIT DB";
	bool ret = true;

	// Create tables
	if (createIdTable()) {
		createTargetTable();
		createTargetFieldTable();
	}

	// Check that tables exists
	if (db.tables().count() != 3)
		ret = false;

	return ret;
}

bool DatabaseManager::deleteDB()
{
	qDebug () << "REMOVE DB";
	db.close();

	QString path(QDir::home().path());
	path.append(QDir::separator()).append("qtwallet.db.sqlite");
	path = QDir::toNativeSeparators(path);

	return QFile::remove(path);
}

bool DatabaseManager::openDB()
{
	qDebug () << "OPEN DB";
	// Find QSLite driver
	db = QSqlDatabase::addDatabase("QSQLITE");
	// http://doc.trolltech.com/4.5/sql-driver.html#qsqlite

	QString path(QDir::home().path());
	path.append(QDir::separator()).append("qtwallet.db.sqlite");
	path = QDir::toNativeSeparators(path);
	db.setDatabaseName(path);

	// Open databasee
	return db.open();
}

QSqlError DatabaseManager::lastError()
{
	return db.lastError();
}

bool DatabaseManager::createIdTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.exec("create table id (id integer primary key, phrase varchar(20))");
	}
	return ret;
}

int DatabaseManager::nextId()
{
	int ret = -1;
	if (db.isOpen()) {
		QSqlQuery query("select id from id");
		if (query.next()) {
			// Get last used id
			ret = query.value(0).toInt();
			// Increase that
			ret++;
			// Store new value
			query.exec(QString("update id set id=%1 where id=%2").arg(ret).arg(ret - 1));
		} else {
			// Set first id to zero
			if (!query.exec("insert into id values(1, NULL)"))
				return -1;
			else
				ret = 1;
		}
	}

	return ret;
}

void DatabaseManager::lock () {
	initdb = false;
}

bool DatabaseManager::openDB(QString pas) {
	initdb = false;

	qDebug() << "open db with pas";

	if (!opendb)
		opendb = openDB();

	if (!opendb) return false;

	qDebug() << "init db with pas";
	
	if (isInit ()) {
		if (hasSecretPhrase()) {
			initdb = checkPassword(pas);
		}
		else {
			initdb = createPassword(pas);
		}
	}
	else {
		if (initDB())
			initdb = createPassword(pas);
	}
	return initdb;
}

bool DatabaseManager::isReady () {
	return opendb && initdb;
}

bool DatabaseManager::checkPassword (QString _a)
{
	QByteArray t = _a.toLocal8Bit();
	
	keyLen = 32*8;
	for (int i = 0; i < 32; i++) {
		key [i] = i; 
		
		if (i < t.size ()) {
			key[i] = (unsigned char)t[i];
		}
	}
	/*unsigned char key16[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	unsigned char key24[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23};
	unsigned char key32[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};*/
	AES_set_encrypt_key(key, keyLen, &aeskeyE);
	AES_set_decrypt_key(key, keyLen, &aeskeyD);
	
	QByteArray a;
	encrypt(QString("developped by nowhereman"), a);
	QByteArray b = selectSecretPhrase ();
	
	//qDebug()<< "checkPassword " << (a == b);
	
	return a == b;
}

bool DatabaseManager::createPassword (QString _a)
{
	QByteArray t = _a.toLocal8Bit();
	
	keyLen = 32*8;
	for (int i = 0; i < 32; i++) {
		key [i] = i; 
		
		if (i < t.size ()) {
			key[i] = (unsigned char)t[i];
		}
	}
	/*unsigned char key16[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	unsigned char key24[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23};
	unsigned char key32[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};*/
	AES_set_encrypt_key(key, keyLen, &aeskeyE);
	AES_set_decrypt_key(key, keyLen, &aeskeyD);
	
	QByteArray a;
	encrypt(QString("developped by nowhereman"), a);
	
	//qDebug () << "create secret " << a;
	
	return updateSecretPhrase (a);
}

bool DatabaseManager::updateSecretPhrase(QByteArray a) {
	//qDebug () << "update secret " << a;

	bool ret = false;
	if (nextId () > 0) {
		QSqlQuery query;
		ret = query.prepare("update id set phrase = :phrase");
		if (ret) {
			query.bindValue(":phrase", a);
			ret = query.exec();
			//qDebug () << "update secret " << ret;
			return ret;
		}
	}
	return ret;
}

bool DatabaseManager::hasSecretPhrase()
{
	//qDebug () << "has secret?";
	
	QSqlQuery query("select phrase from id");
	if (query.next()) {
		return !query.value(0).isNull();
	}

	return false;
}

QByteArray DatabaseManager::selectSecretPhrase()
{
	//qDebug () << "get secret?";

	QByteArray ret;
	QSqlQuery query("select phrase from id");
	if (query.next()) {
		return query.value(0).toByteArray();
	}

	return ret;
}

bool DatabaseManager::createTargetTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.exec("create table targets "
		                 "(id integer primary key, " // this is autoincrement field http://www.sqlite.org/autoinc.html
		                 "name varchar(30), "
		                 "picture BLOB, "
		                 "pictureBack BLOB, "
		                 "type varchar(5))");

	}
	return ret;
}

bool DatabaseManager::createTargetFieldTable()
{
	// Create table
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.exec("create table fields "
		                 "(id integer primary key, " // this is autoincrement field http://www.sqlite.org/autoinc.html
		                 "targetId integer, "
		                 "orderN integer, "
		                 "encrypted integer, "
		                 "name varchar(30), "
		                 "value varchar(30), "
		                 "type integer)");

	}
	return ret;
}

void DatabaseManager::encrypt (const QString a, QByteArray& b)
{
	unsigned char ivE[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	QByteArray t = a.toLocal8Bit();
	int size = t.size();
	int c = size / 16 + 1;
	short p = 16 - size % 16;

	unsigned char pout [c * 16];
	unsigned char pin [c * 16];
	memset (pin, p, c * 16);
	memset(pout, 0, c * 16);
	memcpy(pin, (unsigned char *)t.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyE, ivE, AES_ENCRYPT);

#ifdef BASE64
	b = QByteArray((char*)pout, c * 16).toBase64();
#else
	b = QByteArray((char*)pout, c * 16);
#endif
}

#ifdef BASE64
void DatabaseManager::decrypt (const QByteArray& aa, QString& b)
{
	QByteArray a = QByteArray::fromBase64(aa);
#else
void DatabaseManager::decrypt (const QByteArray& a, QString& b)
{
#endif
	unsigned char ivD[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	int size = a.size ();
	int c = size / 16;
	unsigned char pout [size];
	unsigned char pin [size];
	memset(pout, 0, size);
	memcpy(pin, (unsigned char *)a.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyD, ivD, AES_DECRYPT);
	pout [size-(short)(pout[size-1])] = 0;

	b = QString ((char*)pout);
}

void DatabaseManager::encrypt (const QByteArray& t, QByteArray& b)
{
	unsigned char ivE[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};

	int size = t.size();
	int c = size / 16 + 1;
	short p = 16 - size % 16;

	unsigned char pout [c * 16];
	unsigned char pin [c * 16];
	memset (pin, p, c * 16);
	memset(pout, 0, c * 16);
	memcpy(pin, (unsigned char *)t.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyE, ivE, AES_ENCRYPT);

#ifdef BASE64
	b = QByteArray((char*)pout, c * 16).toBase64();
#else
	b = QByteArray((char*)pout, c * 16);
#endif
}

#ifdef BASE64 
void DatabaseManager::decrypt (const QByteArray& aa, QByteArray& b)
{
	QByteArray a = QByteArray::fromBase64(aa);
#else
void DatabaseManager::decrypt (const QByteArray& a, QByteArray& b)
{
#endif
	unsigned char ivD[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	int size = a.size ();
	int c = size / 16;
	unsigned char pout [size];
	unsigned char pin [size];
	memset(pout, 0, size);
	memcpy(pin, (unsigned char *)a.constData(), size);

	AES_cbc_encrypt(pin, pout, c*16, &aeskeyD, ivD, AES_DECRYPT);
	int l = size-(short)(pout[size-1]); //pout [l] = 0;

	b = QByteArray ((char*)pout, l);
}


bool DatabaseManager::insertTarget(Target* target)
{
		qDebug() << "step 0" ;

	bool ret = false;
	if (db.isOpen()) {
		//target->id = nextId(); // We demostrates autoincrement in this case

		// http://www.sqlite.org/autoinc.html
		// NULL = is the keyword for the autoincrement to generate next value
		QSqlQuery query;
		ret = query.prepare("INSERT INTO targets (name, type, picture, pictureBack) VALUES (:name, :type, :picture, :pictureBack)");
		QByteArray t1;

		//qDebug() << "step 1" ;
		if (ret) {
		//qDebug() << "step 2" ;
			encrypt(target->name, t1);
		//qDebug() << "step 3" ;
			query.bindValue(":name", t1);
			query.bindValue(":type", target->type);
					//qDebug() << "step 4" ;

			QByteArray im1;
			QBuffer buffer1(&im1);
			QByteArray im_r1;
			QByteArray im2;
			QBuffer buffer2(&im2);
			QByteArray im_r2;
		//qDebug() << "step 5" ;

			if (target->picture != NULL) {
				buffer1.open(QIODevice::WriteOnly);
				target->picture->save(&buffer1, "PNG"); // writes image into ba in PNG format

				encrypt(im1, im_r1);
				query.bindValue(":picture", im_r1);
			} else
				query.bindValue(":picture", QVariant (QVariant::ByteArray));
		//qDebug() << "step 6" ;

			if (target->pictureBack != NULL) {
				buffer2.open(QIODevice::WriteOnly);
				target->pictureBack->save(&buffer2, "PNG"); // writes image into ba in PNG format

				QByteArray im_r2;
				encrypt(im2, im_r2);
				query.bindValue(":pictureBack", im_r2);
			} else
				query.bindValue(":pictureBack", QVariant (QVariant::ByteArray));
						//qDebug() << "step 7" ;

			ret = query.exec();
		}
		// Get database given autoincrement value
		if (ret) {
			// http://www.sqlite.org/c3ref/last_insert_rowid.html
			target->id = query.lastInsertId().toInt();
			qDebug () << "INSERT " << target->id;

		}
	}
	return ret;
}

QList<Target*> DatabaseManager::getTargets(QString filter)
{
	QList<Target*> result;

	QSqlQuery query("select id, name, type from targets order by name");
	//QSqlQuery query("select id, name, type, picture, pictureBack  from targets");
	while (query.next()) {
		Target * item = new Target();
		item->id = query.value(0).toInt();
		decrypt(query.value(1).toByteArray(), item->name);
		item->type = query.value(2).toString();
		item->pictureSmall = NULL;
		item->picture = NULL;
		item->pictureBack = NULL;
/*
		if (query.value(3).isNull()) {
			item->picture = NULL;
		} else {
			item->picture = new QImage ();
			QByteArray im_s;
			decrypt(query.value(3).toByteArray(), im_s);
			if (!item->picture->loadFromData (im_s)) {
				qDebug () << "Error loading image " << item->id;
			} else
				qDebug () << "loaded image " << item->id;
		}
		if (query.value(4).isNull()) {
			item->pictureBack = NULL;
		} else {
			item->pictureBack = new QImage ();
			QByteArray im_s;
			decrypt(query.value(4).toByteArray(), im_s);
			if (!item->pictureBack->loadFromData (im_s)) {
				qDebug () << "Error loading bimage " << item->id;
			} else
				qDebug () << "loaded bimage " << item->id;
		}
*/
		result.append(item);
	}

	return result;
}

bool DatabaseManager::deleteTarget(Target * target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.prepare("delete from targets where id=:code");
		if (ret) {
			qDebug () << "DELETE " << target->id;

			query.bindValue(":code", target->id);
			ret = query.exec();

			delete target;
		}
	}

	return ret;
}

bool DatabaseManager::updateFields(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		for (int i = 0; i < target->fields.count(); i++) {
			TargetField * field = target->fields[i];
	
			if (field->id == -1) {
				field->targetId = target->id;
				field->id = 0;
				ret |= insertTargetField (field);
			}
			else {
				ret |= updateTargetField (field);
			}
		}
	}
	
	return ret;
}

bool DatabaseManager::updateTarget(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.prepare("UPDATE targets set name=:name, type=:type where id=:code");
		QByteArray t1;
		if (ret) {
			encrypt(target->name, t1);
			query.bindValue(":name", t1);
			query.bindValue(":type", target->type);
			query.bindValue(":code", target->id);
			ret = query.exec();

			qDebug () << "updated " << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetPicture(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.prepare("UPDATE targets set picture=:picture where id=:code");
		if (ret) {
			qDebug () << "updating image " << target->id << " " << target->picture;

			QByteArray im1;
			QBuffer buffer1(&im1);
			QByteArray im_r1;

			if (target->picture != NULL) {
				buffer1.open(QIODevice::WriteOnly);
				target->picture->save(&buffer1, "PNG"); // writes image into ba in PNG format

				encrypt(im1, im_r1);
				query.bindValue(":picture", im_r1);
			}
			else {
				qDebug () << "delete image ";

				query.bindValue(":picture", QVariant (QVariant::ByteArray));
			}

			query.bindValue(":code", target->id);
			ret = query.exec();

			qDebug () << "update picturde " << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetPictureBack(Target* target)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.prepare("UPDATE targets set pictureBack=:pictureBack where id=:code");
		if (ret) {
			qDebug () << "UPDATE " << target->id << " " << target->name << " " << target->type;
			QByteArray im2;
			QBuffer buffer2(&im2);
			QByteArray im_r2;

			if (target->pictureBack != NULL) {
				buffer2.open(QIODevice::WriteOnly);
				target->pictureBack->save(&buffer2, "PNG"); // writes image into ba in PNG format

				QByteArray im_r2;
				encrypt(im2, im_r2);
				query.bindValue(":pictureBack", im_r2);
			}
			else {
				query.bindValue(":pictureBack", QVariant (QVariant::ByteArray));
			}

			query.bindValue(":code", target->id);
			ret = query.exec();
			
			qDebug () << "update picture back" << target->id;

		}
	}
	return ret;
}

bool DatabaseManager::updateTargetField(TargetField* field)
{
	bool ret = false;
	if (db.isOpen()) {
		QSqlQuery query;
		ret = query.prepare("UPDATE fields set name=:name, value=:value, type=:type, orderN=:order, "
		                     "encrypted=:encrypted where targetId=:targetId and id=:id");
		if (ret) {
			query.bindValue(":targetId", field->targetId);
			query.bindValue(":id", field->id);

			if (field->encrypted) {
				QByteArray a, b;
				encrypt(field->name, a);
				encrypt(field->value, b);
				query.bindValue(":name", a);
				query.bindValue(":value", b);
			}
			else {
				query.bindValue(":name", field->name);
				query.bindValue(":value", field->value);
			}
			query.bindValue(":type", field->type);
			query.bindValue(":order", field->order);
			query.bindValue(":encrypted", field->encrypted ? 1 : 0);
			ret = query.exec();
		}
	}
	return ret;
}

bool DatabaseManager::insertTargetField(TargetField* field)
{
	bool ret = false;
	if (db.isOpen()) {
		//field->id = nextId(); // We demostrates autoincrement in this case

		// http://www.sqlite.org/autoinc.html
		// NULL = is the keyword for the autoincrement to generate next value

		QSqlQuery query;
		ret = query.prepare("INSERT INTO fields (targetId, name, value, type, orderN, encrypted) "
		                    "VALUES (:targetId, :name, :value, :type, :order, :encrypted)");
		if (ret) {
			query.bindValue(":targetId", field->targetId);

			if (field->encrypted) {
				QByteArray a, b;
				encrypt(field->name, a);
				encrypt(field->value, b);
				query.bindValue(":name", a);
				query.bindValue(":value", b);
			}
			else {
				query.bindValue(":name", field->name);
				query.bindValue(":value", field->value);
			}
			query.bindValue(":type", field->type);
			query.bindValue(":order", field->order);
			query.bindValue(":encrypted", field->encrypted ? 1 : 0);
			ret = query.exec();
		}

		// Get database given autoincrement value
		if (ret) {
			// http://www.sqlite.org/c3ref/last_insert_rowid.html
			field->id = query.lastInsertId().toInt();
		}
	}
	return ret;
}

void DatabaseManager::getFields(Target * target)
{
	
	target->freeFieldList ();
	
	bool ret = false;
	QSqlQuery query;
	ret = query.prepare("select id, name, value, type, targetId, orderN, encrypted from fields where targetId=:targetId order by orderN");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		TargetField* item = new TargetField();
		item->id = query.value(0).toInt();
		item->type= query.value(3).toInt();
		item->targetId = query.value(4).toInt ();
		item->order = query.value(5).toInt ();
		item->encrypted = query.value(6).toInt () != 0;
		if (item->encrypted) {
			decrypt(query.value(1).toByteArray(), item->name);
			decrypt(query.value(2).toByteArray(), item->value);
		}
		else {
			item->name = query.value(1).toString();
			item->value= query.value(2).toString();
		}
		target->fields.append(item);
	}
}

QImage * DatabaseManager::getPicture (Target * target) {
	QImage * result = NULL;
	QSqlQuery query;
	bool ret = query.prepare("select picture from targets where id=:targetId");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		if (query.value(0).isNull()) {
			result = NULL;
		}
		else {
			result = new QImage ();
			QByteArray im_s;
			decrypt(query.value(0).toByteArray(), im_s);
			if (!result->loadFromData (im_s)) {
				qDebug () << "Error loading image " << target->id;
			} else {
				qDebug () << "loaded image " << target->id;
			}
		}
		break;
	}

	return result;
}
QImage * DatabaseManager::getPictureBack (Target * target) {
	QImage * result = NULL;
	QSqlQuery query;
	bool ret = query.prepare("select pictureBack from targets where id=:targetId");
	query.bindValue(":targetId", target->id);
	ret = query.exec ();
	while (query.next()) {
		if (query.value(0).isNull()) {
			result = NULL;
		}
		else {
			result = new QImage ();
			QByteArray im_s;
			decrypt(query.value(0).toByteArray(), im_s);
			if (!result->loadFromData (im_s)) {
				qDebug () << "Error loading back image " << target->id;
			}
			else {
				//qDebug () << "loaded back image " << target->id;
			}
		}
		break;
	}

	return result;
}

