/***************************************************************************
 *   Copyright (C) 2010 by Ixonos Plc   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef PLUGININTERFACE_H
#define PLUGININTERFACE_H

#include <QObject>
#include <QVariant>
//#include <QPluginLoader>

namespace qtrapids
{
	
		// Forward declaration because of co-dependency of classes.
		class PluginInterface;
	

		/** @class PluginHostInterface
		* @brief Defines interface for plugins to access the host application. 
		* A Host is an application that is extended by implementing Plugins, 
		* that implement the additional functionality
		* @note Implementing plugin host should inherit QObject.
	*/
	class PluginHostInterface 
	{
		public:
			
			
			/// @enum PluginWidgetType Allows plugin host to differentiate actions 
			/// when passed as parameter to addWidget(). E.g. Popup a dialog or append tab etc.
			/// @todo add new types
			enum PluginWidgetType {
				BASE_WIDGET,
				TAB_PAGE,
				UNKNOWN_TYPE
			};
			
			/// @enum PluginRequest for requesting host application functionality
			/// E.g. open torrent file etc.
			/// @todo Additional functionality request constants.
			enum PluginRequest {
				OPEN_FILE,
				READ_BUFFER,
				UNKNOWN_REQUEST
			};
			
			/// @brief Sets the plugin GUI element to host application
			/// @note It is up to the host application to decide how to manage
			/// and show the actual widget.
 			virtual bool setGui(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE, PluginInterface* plugin = NULL) = 0;
			
			/// @brief Adds additional plugin wigdets to the host application.
			/// This functio can be called by the plugin recursively, i.e. when GUI events occur
			/// The host application must handle placing the additional widgets.
			/// @todo Could we implement this using in a more manageable way, e.g. signal-slot?
			virtual void addPluginWidget(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE) = 0;
			virtual void addToolbar(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE) = 0;
			virtual void addToolItem(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE) = 0;
			virtual void addMenu(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE) = 0;
			virtual void addMenuItem(QWidget* widget, PluginWidgetType type = UNKNOWN_TYPE) = 0;
			
			/// @brief Plugin can request to host application functionality.
			/// Host application defines the service policy for plugin requests.
			/// @todo Sending events would benefit from Qt signal, how to define
			/// signal in an abstract interface?
			virtual bool eventRequest(QVariant param, PluginRequest req = UNKNOWN_REQUEST) = 0;
	};
  
  
  
	/** @class PluginInterface 
		* @brief Defines interface for a plugin instance.
		* The host application uses PluginInterface interface for calling the plugins
		* that extend the Host functionality
	*/
	class PluginInterface : public QObject 
	{
		public:
			
			/// @struct Info Used to supply information from the host application to the plugin 
			/// for initialization.
			struct Info {
				Info() : directory("") {} //Constructor
				QString directory; ///< directory from which the plugin is loaded
				// ...
			};
			
			/// @brief Initializes the plugin instance.
			virtual void initialize(PluginHostInterface* host, Info info = Info()) = 0;
			
			/// @brief Returns plugin identifier to be used by the host application
			/// @returns Plugin identifier as string representation.
			virtual QString identifier() = 0;
			
			virtual QWidget* getGui() = 0;
	};

} //namespace qtrapids


// Declare the interfaces for the Qt framework.
Q_DECLARE_INTERFACE(qtrapids::PluginInterface,
                    "com.ixonos.qtrapids.PluginInterface/1.0")
Q_DECLARE_INTERFACE(qtrapids::PluginHostInterface,
                    "com.ixonos.qtrapids.PluginHostInterface/1.0")
										
										
//////////////// EXAMPLE PLUGIN DECLARATION /////////////////////////
// A simple plugin example using the PluginInterface
// For more info, see Qt documentation: "How to Create Qt Plugins"
//
// namespace qtrapids
// {
// 
// 	class MyPlugin : public PluginInterface {
// 		Q_OBJECT
//		// NOTE: This macro tells Qt which interfaces the plugin implements (i.e. inherits):
// 		Q_INTERFACES(qtrapids::PluginInterface)
// 		
// 		public:
// 			MyPlugin();
// 			virtual void initialize(PluginHostInterface* host);
// 			virtual QWidget* getGui();
// 		
// // Additional plugin-specific signals and slots.
// 		signals: 
// 			void searchResult(QWidget* resultwidget);
// 			
// 		private slots: 
// 			void on_button_clicked();
// 			void on_result(QWidget* resultWidget);
// 			
// 		private:
// 	};
// 
//
// 	MyPlugin::MyPlugin(): host_(NULL) {}
// 
// 		void SearchPlugin::initialize(AbstractPluginHost* host)
// 		{
// 			host_ = host;
// 			
// 			if (host_ != NULL) {
// 				QWidget *pluginWidget = new QWidget;
// 				QVBoxLayout *vbox = new QVBoxLayout;
// 				QPushButton *searchButton = new QPushButton("Search");
// 				vbox->addWidget(searchButton);
// 				pluginWidget->setLayout(vbox);
// 		
// 				connect(searchButton, SIGNAL(clicked()), this, SLOT(on_searchButton_clicked()));
// 				//connect(this, SIGNAL(searchResult(QWidget*)), this, SLOT(on_searchResult(QWidget*)));
// 				
//				// Call host interface function to set the plugin GUI. Host handles the setup 
//				// to it's own policy
// 				host_->setGui(pluginWidget);
// 			}
// 		}
// } // namespace qtrapids 
//
//// NOTE: Remember to export the actual plugin to be visible for Qt:
//// Q_EXPORT_PLUGIN2(myplugin, qtrapids::MyPlugin)
//
///////////////// END OF EXAMPLE PLUGIN ///////////////////////////////

#endif
