#include <QtGui/QApplication>
#include <QtCore/QTranslator>
#include <QtCore/QLocale>
#include <QtCore/QObject>
#include <QtGui/QMainWindow>
#include <QtGui/QMessageBox>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QStringListIterator>
#include <QtGui/QCloseEvent>
#include <QtCore/QProcess>
#include <QtCore/QMap>
#include <QtCore/QDir>
#include <QtCore/QPluginLoader>
#include <QtGui/QAction>
#include <QtCore/QList>
#include <QtCore/QListIterator>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtCore/QTimer>
#include "apnhelper.h"
#include "mobilehotspotplugin.h"
#include "dbushelper.h"
#include "hotspothelper.h"
#include "pluginscontrolgui.h"
#include "accesspointgui.h"
#include "internetaccessgui.h"
#include "languagegui.h"
#include "mobilehotspotgui.h"


MobileHotspotGUI::MobileHotspotGUI(QWidget *parent)
    : QMainWindow(parent)
{
	qApp->installTranslator(&translator);
	ui.setupUi(this);
	this->setAttribute(Qt::WA_Maemo5StackedWindow);

	dnsmasqProcess = 0;
	loadPlugins();
	sortPlugins();

	configuration.load();
	if(configuration.language == NO_LANGUAGE_SELECTED)
		configureLanguage();
	else
		configureLanguageFinished(false);

	updateDisplayedConfiguration();
}

MobileHotspotGUI::~MobileHotspotGUI()
{

}

void MobileHotspotGUI::closeEvent(QCloseEvent* event){
	if(ui.btnStart->isEnabled())
		event->accept();
	else{
		if(QMessageBox::question(this, tr("Closing the program"), tr("Hotspot is still active ! Do you want to stop it ?"), QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel) == QMessageBox::Yes){
			stop();
			event->accept();
		}
		else
			event->ignore();
	}

}

void MobileHotspotGUI::loadPlugins(){
	plugins.clear();
	sortedPlugins.clear();
	pluginsTranslatorsBaseNames.clear();

	QDir dir("./plugins");
	QStringList list = dir.entryList(QDir::Files | QDir::NoSymLinks);
	QStringListIterator iterator(list);
	while(iterator.hasNext()){
		QString filename = iterator.next();
		QString path = dir.absoluteFilePath(filename);
		QPluginLoader loader(path);
		QObject *loaded = loader.instance();
		if(loaded != 0){
			MobileHotspotPlugin *plugin = qobject_cast<MobileHotspotPlugin*>(loaded);
			if(plugin != 0){
				QString name = plugin->name();
				if(plugins.contains(name))
					qDebug( (QString("WARNING : plugin name ") + name + " is already taken").toAscii().data());
				else{
					pluginsTranslatorsBaseNames.insert(name, QFileInfo(dir, filename).completeBaseName());
					plugins.insert(name, plugin);
					qDebug( (QString("Loaded plugin ") + name).toAscii().data() );
				}
			}
		}
		else
			qDebug(loader.errorString().toAscii().data());
	}
	qDebug( (QString("Loaded ") + QString::number(plugins.size()) + " plugins").toAscii().data() );
}

void MobileHotspotGUI::sortPlugins(){
	sortedPlugins.clear();
	QList<QString> keys = plugins.keys();
	QListIterator<QString> iterator(keys);
	while(iterator.hasNext()){
		QString key = iterator.next();
		int order = plugins.value(key)->order();
		int insertAt = sortedPlugins.size();
		for(int i = 0; i < sortedPlugins.size(); i++){
			QString testedKey = sortedPlugins.at(i);
			int testedOrder = plugins.value(testedKey)->order();
			if(order < testedOrder){
				insertAt = i;
				i = sortedPlugins.size();
			}
		}
		sortedPlugins.insert(insertAt, key);
	}
}

void MobileHotspotGUI::configurePlugins(){
	pluginsControlGUI = new PluginsControlGUI(this);
	pluginsControlGUI->setConfiguration(&configuration, ui.btnStop->isEnabled(), &plugins, &sortedPlugins);
	connect(pluginsControlGUI, SIGNAL(windowClosed()),
		this, SLOT(configurePluginsFinished()));
	pluginsControlGUI->show();
}

void MobileHotspotGUI::configurePluginsFinished(){
	// Some code that is run after the closing of the Configure Plugins window
}

void MobileHotspotGUI::configureLanguage(){
	LanguageGUI dlg;
	dlg.setConfiguration(configuration);
	if(dlg.exec() != dlg.Accepted){
		if(configuration.language != NO_LANGUAGE_SELECTED)
			return;
		else
			configuration.language = DEFAULT_LANGUAGE;
	}
	else
		dlg.configuration(&configuration);
	configureLanguageFinished(true);
}

void MobileHotspotGUI::configureLanguageFinished(bool saveConfiguration){
	if(saveConfiguration)
		configuration.save();

	QString lang = configuration.language;
	if(lang != DEFAULT_LANGUAGE)
		translator.load(lang, "./lang", "_", ".qtmh.qm");
	else
		translator.load("", "", "", "");

	QListIterator<MobileHotspotPlugin*> iterator(plugins.values());
	while(iterator.hasNext()){
		MobileHotspotPlugin *plugin = iterator.next();
		QString name = plugin->name();
		QTranslator *trans = plugin->translator();

		if(lang != DEFAULT_LANGUAGE)
			trans->load( (pluginsTranslatorsBaseNames.value(name) + "-") + lang, "./lang", "_", ".qm");
		else
			trans->load("", "", "", "");
	}

	ui.retranslateUi(this);
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::updateDisplayedConfiguration(){
	ui.lstAccessPoint->clear();
	ui.lstAccessPoint->addItem(configuration.hotspotName);
	ui.lstAccessPoint->addItem(configuration.encryptionType == configuration.ENCRYPTION_WEP ? tr("WEP") : tr("No encryption"));
	if(configuration.cycleWlanDriver)
		ui.lstAccessPoint->addItem(tr("Reset Wifi driver"));

	ui.lstInternetAccess->clear();
	QString s = configuration.internetAPName;
	ui.lstInternetAccess->addItem(s);
	ui.lstInternetAccess->addItem(configuration.internetEnabled ? tr("With internet") : tr("Without internet"));
	ui.lstInternetAccess->addItem(configuration.lanNetwork + "0");

	ui.lstInterfaces->clear();
	ui.lstInterfaces->addItem(tr("Wifi"), configuration.INTERFACE_WLAN);
	ui.lstInterfaces->addItem(tr("USB"), configuration.INTERFACE_USB);
	ui.lstInterfaces->setCurrentIndex(configuration.lanInterface == configuration.INTERFACE_WLAN ? 0 : 1);
}

void MobileHotspotGUI::configureInternetAccess(){
	internetAccessGUI = new InternetAccessGUI(this);
	internetAccessGUI->setConfiguration(configuration);
	connect(internetAccessGUI, SIGNAL(windowClosed()),
		this, SLOT(configureInternetAccessFinished()));
	internetAccessGUI->show();
}

void MobileHotspotGUI::configureInternetAccessFinished(){
	internetAccessGUI->configuration(&configuration);
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::configureAccessPoint(){
	accessPointGUI = new AccessPointGUI(this);
	accessPointGUI->setConfiguration(configuration);
	connect(accessPointGUI, SIGNAL(windowClosed()),
		this, SLOT(configureAccessPointFinished()));
	accessPointGUI->show();
}

void MobileHotspotGUI::configureAccessPointFinished(){
	accessPointGUI->configuration(&configuration);
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::configureInterface(int index){
	configuration.lanInterface = ui.lstInterfaces->itemData(index).toString();
	if(configuration.lanInterface == configuration.INTERFACE_WLAN && configuration.internetAPType != GCONF_APN_TYPE_GPRS){
		configuration.internetAPID = "";
		configuration.internetAPType = "";
		configuration.internetAPName = "";
	}
	configuration.save();
	updateDisplayedConfiguration();
}

void MobileHotspotGUI::start(bool notification){
	int ret;

	/** Find previous connection **/
	previousConnectionID = "";
	previousConnectionType = "";
	waitingForPreviousConnection = true;
	previousConnectionEnableSignal();
	previousConnectionRequest();

	/** Start the hotspot a bit later (hopefully the previous connection will be known) **/
	showNotifications = notification;
	QTimer::singleShot(800, this, SLOT(startProcedure()));
}

void MobileHotspotGUI::startProcedure(){
	/* Credits to Rambo for the procedure */
	bool success = true;
	int ret;
	qDebug("*** STARTING PROCESS ***");

	/** Disable previous connection search **/
	waitingForPreviousConnection = false;
	previousConnectionDisableSignal();

	/** Obvious sanity checks **/
	if(configuration.lanInterface == configuration.INTERFACE_WLAN && configuration.internetInterface() == configuration.INTERFACE_WLAN){
		/** Both LAN and INET are on WLAN, that is not possible **/
		QString fail = tr("Error : LAN and Internet are both using the Wifi interface, please use a LAN on USB");
		QMessageBox::critical(this, tr("Cannot start"), fail);
		return;
	}

	/** Save scan interval **/
	scanInterval = APNHelper::scanInterval();

	/** Notify plugins **/
	QListIterator<QString> pluginsIterator = QListIterator<QString>(sortedPlugins);
	while(pluginsIterator.hasNext()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.next());
		if(plugin->isPluginEnabled()){
			qDebug( (QString("Plugin beforeStarting : ") + plugin->name()).toAscii().data() );
			qApp->installTranslator(plugin->translator());
				plugin->beforeStarting(&configuration);
			qApp->removeTranslator(plugin->translator());
		}
	}

	try{
		/** Load system modules if necessary **/
		alreadyLoadedModules.clear();
		ret = HotspotHelper::loadSystemModules(&alreadyLoadedModules);
		if(ret != 0)
			throw QString(tr("Error while loading system modules, are you running the power kernel ?"));

		/** Internet connection handling **/
		if(configuration.internetEnabled){
			qDebug("Previous internet connection was %s (%s), next one is %s (%s)", previousConnectionID.toAscii().data(), previousConnectionType.toAscii().data(), configuration.internetAPID.toAscii().data(), configuration.internetAPType.toAscii().data());
			if(previousConnectionType == configuration.internetAPType && previousConnectionID == configuration.internetAPID && configuration.internetInterface() != configuration.INTERFACE_WLAN)
				/** Nothing to do to the internet connection **/
				/** In case of USB-Lan and Wifi-Inet we have to disconnect & reconnect to prevent route disappearing problem **/
				qDebug("** Internet connection is already good, skipping this step **");
			else{
				/** Disable current internet connection **/
				ret = HotspotHelper::disableInternetConnection(configuration);
				if(ret != 0)
					throw QString(tr("Error while disconnecting previous internet connection"));

				/** Connect to the chosen APN **/
				ret = HotspotHelper::enableInternetConnection(configuration);
				if(ret != 0)
					throw QString(tr("Error while connecting to the APN, you way need to select it again in the Networking panel"));
			}
		}
		else if(configuration.lanInterface == configuration.INTERFACE_WLAN && previousConnectionType.indexOf("WLAN") >= 0){
			/** WLAN must be disabled in order to get the adhoc WLAN working **/
			ret = HotspotHelper::disableInternetConnection(configuration);
			if(ret != 0)
				throw QString(tr("Error while disconnecting previous internet connection"));
		}
		else
			qDebug("** User requested no internet connection, ignoring it then **");

		/** Disable network automatic search **/
		APNHelper::setScanInterval(0);

		/** USB vs Wifi **/
		if(configuration.lanInterface == configuration.INTERFACE_USB){

			/** Unload all USB modules - May fail for some, but thats normal **/
			HotspotHelper::unloadNokiaUSB();
			HotspotHelper::unloadStandardUSB();

			/** Check USB active module **/
			ret = HotspotHelper::checkUSBMode();
			if(ret != 0)
				throw QString(tr("Error : USB shouldn't be in any mode (no storage mode, no pc-suite mode)"));

			/** Load standard USB module **/
			ret = HotspotHelper::loadStandardUSB();
			if(ret != 0)
				throw QString(tr("Error while loading standard g_ether driver"));

			/** Prepare USB interface **/
			ret = HotspotHelper::prepareUSBInterface(configuration.usbMAC, configuration.lanNetwork);
			if(ret != 0)
				throw QString(tr("Error while preparing the USB interface"));

		}
		else{

			/** Prepare Wifi interface **/
			ret = HotspotHelper::prepareWifiInterface(configuration);
			if(ret != 0)
				throw QString(tr("Error while preparing the Wifi interface"));
		}

		/** Start dnsmasq **/
		dnsmasqProcess = new QProcess();
		ret = HotspotHelper::startDnsMasq(configuration, dnsmasqProcess);
		if(ret != 0)
			throw QString("Error while starting dnsmasq");

		/** Set iptables **/
		ret = HotspotHelper::setIPTables(configuration, &ipforward);
		if(ret != 0)
			throw QString("Error while setting iptables");


		// There :) Finished
	}
	catch(QString error){
		qDebug(error.toAscii().data());
		success= false;
		if(showNotifications)
			QMessageBox::critical(this, tr("Failed to start the hotspot"), error);
	}

	/** Notify plugins **/
	successStarting = success;
	pluginsIterator = QListIterator<QString>(sortedPlugins);
	while(pluginsIterator.hasNext()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.next());
		if(plugin->isPluginEnabled()){
			qDebug( (QString("Plugin afterStarting : ") + plugin->name()).toAscii().data() );
			qApp->installTranslator(plugin->translator());
				plugin->afterStarting(&configuration, successStarting);
			qApp->removeTranslator(plugin->translator());
		}
	}

	/** GUI **/
	if(success){
		if(showNotifications)
			DBUSHelper::notificationSimple(tr("Hotspot started successfully"));
		ui.btnAccessPoint->setEnabled(false);
		ui.btnInternetAccess->setEnabled(false);
		ui.lstInterfaces->setEnabled(false);
		ui.btnStart->setEnabled(false);
		ui.btnStop->setEnabled(true);
		ui.actionUSBAssistant->setEnabled(false);
		qDebug("*** STARTING PROCESS TERMINATED SUCCESSFULLY ***");
	}
	else{
		qDebug("*** STARTING PROCESS FAILED ***");
		stop(false);
	}
}

void MobileHotspotGUI::stop(bool notification){
	/** Nothing to do here, its simply for symetry with start() **/
	showNotifications = notification;
	stopProcedure();
}

void MobileHotspotGUI::stopProcedure(){
	/* Credits to Rambo for the procedure */
	bool success = true;
	int ret;
	qDebug("*** STOPPING PROCESS ***");

	/** Notify plugins **/
	QListIterator<QString> pluginsIterator = QListIterator<QString>(sortedPlugins);
	pluginsIterator.toBack();
	while(pluginsIterator.hasPrevious()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.previous());
		if(plugin->isPluginEnabled()){
			qDebug( (QString("Plugin beforeStopping : ") + plugin->name()).toAscii().data() );
			qApp->installTranslator(plugin->translator());
				plugin->beforeStopping(&configuration, successStarting);
			qApp->removeTranslator(plugin->translator());
		}
	}

	/** Unset iptables **/
	ret = HotspotHelper::unsetIPTables(configuration, ipforward);
	if(ret != 0){
		qDebug("Error while unsetting iptables");
		success = false;
	}

	/** Stop dnsmasq **/
	if(dnsmasqProcess != 0){
		ret = HotspotHelper::stopDnsMasq(configuration, dnsmasqProcess);
		if(ret != 0){
			qDebug("Error while stopping dnsmasq");
			success = false;
		}
		delete dnsmasqProcess;
		dnsmasqProcess = 0;
	}

	/** USB vs Wifi **/
	if(configuration.lanInterface == configuration.INTERFACE_USB){

		/** Unprepare USB interface **/
		ret = HotspotHelper::unprepareUSBInterface();
		if(ret != 0){
			qDebug("Error while unpreparing USB interface");
			success = false;
		}

		/** Unload all USB modules - May fail for some, but thats normal **/
		HotspotHelper::unloadNokiaUSB();
		HotspotHelper::unloadStandardUSB();

		/** Check USB active module **/
		ret = HotspotHelper::checkUSBMode();
		if(ret != 0){
			qDebug("Error while checking usb active modules (should be none, but thats not the case)");
			success = false;
		}

		/** Load Nokias USB module **/
		ret = HotspotHelper::loadNokiaUSB();
		if(ret != 0){
			qDebug("Error while loading Nokia USB module");
			success = false;
		}

	}
	else{

		/** Unprepare Wifi interface **/
		ret = HotspotHelper::unprepareWifiInterface(configuration);
		if(ret != 0){
			qDebug("Error while unpreparing Wifi interface");
			success = false;
		}
	}

	/** Restore scan interval **/
	APNHelper::setScanInterval(scanInterval);

	/** Internet connection handling **/
	if(configuration.internetEnabled){

		if(previousConnectionType == configuration.internetAPType && previousConnectionID == configuration.internetAPID && configuration.internetInterface() != configuration.INTERFACE_WLAN)
			/** Current connection == previous connection **/
			/** In case of USB-Lan and Wifi-Inet we have to disconnect, change powersave & reconnect **/
			qDebug("** Previous connection was the same, nothing to do to the internet connection **");
		else{
			/** Disable current internet connection **/
			ret = HotspotHelper::disableInternetConnection(configuration);
			if(ret != 0){
				qDebug("Error while disconnecting internet connection");
				success = false;
			}

			if(previousConnectionID != ""){
				/** Restore previous internet connection **/
				qDebug("** Restoring previous internet connection **");
				ret = DBUSHelper::internetConnect(previousConnectionID);
				if(ret != 0){
					qDebug("Error while reconnecting previous internet connection");
					success = false;
				}
			}
			else
				qDebug("** There was no previous connection, so no reconnection to do **");
		}
	}
	else if(configuration.lanInterface == configuration.INTERFACE_WLAN && previousConnectionType.indexOf("WLAN") >= 0){
		/** We need to restore disabled WLAN **/
		ret = DBUSHelper::internetConnect(previousConnectionID);
		if(ret != 0){
			qDebug("Error while reconnecting previous internet connection");
			success = false;
		}
	}
	else
		qDebug("** User requested no internet connection, ignoring it then **");

	/** Unload system modules if we loaded them ourselves **/
	ret = HotspotHelper::unloadSystemModules(alreadyLoadedModules);
	if(ret != 0){
		qDebug("Error while unloading system modules");
		// not that critical, no need for a message about this
	}


	// There :) Finished

	/** Notify plugins **/
	pluginsIterator = QListIterator<QString>(sortedPlugins);
	pluginsIterator.toBack();
	while(pluginsIterator.hasPrevious()){
		MobileHotspotPlugin *plugin = plugins.value(pluginsIterator.previous());
		if(plugin->isPluginEnabled()){
			qDebug( (QString("Plugin afterStopping : ") + plugin->name()).toAscii().data() );
			qApp->installTranslator(plugin->translator());
				plugin->afterStopping(&configuration, successStarting, success);
			qApp->removeTranslator(plugin->translator());
		}
	}

	/** GUI **/
	if(showNotifications)
		DBUSHelper::notificationSimple(success ? tr("Hotspot stopped successfully") : tr("Hotspot stopped with errors"));
	ui.btnAccessPoint->setEnabled(true);
	ui.btnInternetAccess->setEnabled(true);
	ui.lstInterfaces->setEnabled(true);
	ui.btnStart->setEnabled(true);
	ui.btnStop->setEnabled(false);
	ui.actionUSBAssistant->setEnabled(true);

	updateDisplayedConfiguration();
	qDebug("*** STOPPING PROCESS FINISHED ***");
}

void MobileHotspotGUI::previousConnectionSignal(const QDBusMessage &message){
	if(! waitingForPreviousConnection)
		return;
	QList<QVariant> args = message.arguments();
	if(args.size() != 8)
		return;
	QString errorMessage = args.at(6).toString();
	if(errorMessage != ""){
		qDebug((QString("Error while testing previous connection : ") + errorMessage).toAscii().data());
		return;
	}
	previousConnectionID = QString(args.at(5).toByteArray());
	previousConnectionType = args.at(3).toString();
}

void MobileHotspotGUI::previousConnectionEnableSignal(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	dbus.connect("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_sig", this, SLOT(previousConnectionSignal(const QDBusMessage&)));
}

void MobileHotspotGUI::previousConnectionDisableSignal(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	dbus.disconnect("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_sig", this, SLOT(previousConnectionSignal(const QDBusMessage&)));
}

void MobileHotspotGUI::previousConnectionRequest(){
	QDBusConnection dbus = QDBusConnection::systemBus();
	QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.icd2", "/com/nokia/icd2", "com.nokia.icd2", "state_req");
	dbus.send(msg);
}

