/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "forecasthungitem.h"
#include "settings.h"
#include "pixmaploader.h"

#include <QPropertyAnimation>

#define LINE_NAME_SUFFIX    "_line"


typedef struct
{
    const char * const prefix;
    const qreal x;
    const qreal y;

    QString name() const { return prefix; }
    QString lineName() const { return name() + LINE_NAME_SUFFIX; }

    qreal scaledX() const { return Settings::scaleWidth(x); }
    qreal scaledY() const { return Settings::scaleHeight(y); }

    QPixmap pic() const { return PixmapLoader::pixmap(name()); }
    QPixmap linePic() const { return PixmapLoader::pixmap(lineName()); }
} HungObjectData;

static HungObjectData HungItemsData[ForecastHungItem::TypeCount] = {
    {"cloud_1",         160.0,  3.0},   // Cloud1
    {"cloud_2",         131.0,  6.0},   // Cloud2
    {"cloud_3",         102.0,  11.0},  // Cloud3
    {"cloud_rain_1",    160.0,  3.0},   // CloudRain1
    {"cloud_rain_2",    131.0,  6.0},   // CloudRain2
    {"cloud_rain_3",    102.0,  11.0},  // CloudRain3
    {"cloud_storm_1",   160.0,  3.0},   // CloudStorm1
    {"cloud_storm_2",   131.0,  6.0},   // CloudStorm2
    {"cloud_storm_3",   102.0,  11.0},  // CloudStorm3
    {"cloud_tstorm_1",  160.0,  3.0},   // CloudTStorm1
    {"cloud_tstorm_2",  131.0,  6.0},   // CloudTStorm2
    {"sun",             198.0,  170.0}, // Sun
    {"cold_sun",        121.5,  90.0},  // ColdSun
    {"moon",            97.5,   18.0},   // Moon
    {"na_icon",         44,   8.0}   // Unknown Climate
};

// ForecastHungItem

ForecastHungItem::ForecastHungItem(ItemType type, QGraphicsItem *parent)
    : QGraphicsItem(parent)
    , m_type(type)
    , m_targetPicTop(0)
{
    setFlag(QGraphicsItem::ItemHasNoContents, true);

    HungObjectData &data = HungItemsData[m_type];

    m_pic = new QGraphicsPixmapItem(data.pic(), this);
    m_line = new QGraphicsPixmapItem(data.linePic(), m_pic);
    m_line->setFlag(QGraphicsItem::ItemStacksBehindParent);

    m_pic->setPos(0.0, 0.0);

    m_line->setPos(data.scaledX(), data.scaledY() - m_line->boundingRect().bottom());
    reset();
}

int ForecastHungItem::loadImages()
{
    int result = 0;
    for (int i = 0; i < ForecastHungItem::TypeCount; ++i) {
        PixmapLoader::load(HungItemsData[i].name());
        PixmapLoader::load(HungItemsData[i].lineName());
        result += 2;
    }
    return result;
}

qreal ForecastHungItem::lineLenght() const
{
    return m_targetPicTop + HungItemsData[m_type].scaledY();
}

void ForecastHungItem::setLineLenght(qreal lenght)
{
    m_targetPicTop = lenght - HungItemsData[m_type].scaledY();
}

void ForecastHungItem::setLinePos(qreal linePos)
{
    setPos(linePos - HungItemsData[m_type].scaledX(), 0.0);
}

QPointF ForecastHungItem::picPos() const
{
    return QPointF(pos().x(), pos().y() + m_targetPicTop);
}

void ForecastHungItem::setPicPos(QPointF picPos)
{
    setPos(picPos.x(), 0.0);
    m_targetPicTop = picPos.y();
}

void ForecastHungItem::reset()
{
    setPicTop(-m_pic->boundingRect().height());
}

QRectF ForecastHungItem::boundingRect () const
{
    return QRectF(0.0, 0.0, m_pic->boundingRect().width(),
                  m_targetPicTop + m_pic->boundingRect().height());
}

void ForecastHungItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *opt,
                             QWidget *widget)
{
    Q_UNUSED(painter);
    Q_UNUSED(opt);
    Q_UNUSED(widget);
}

QAbstractAnimation *ForecastHungItem::getAnimation()
{
    QPropertyAnimation *result = new QPropertyAnimation(this, "picTop");
    result->setStartValue(-m_pic->boundingRect().height());
    result->setEndValue(m_targetPicTop);
    result->setEasingCurve(QEasingCurve::OutBack);
    result->setDuration(500);
    return result;
}
