/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "settings.h"
#include "forecastmodel.h"
#include "forecastresolver.h"
#include "weatherresponse.h"


ForecastModel::ForecastModel()
    : QObject(),
      m_isLoaded(false),
      m_resolver(0)
{

}

ForecastModel *ForecastModel::instance()
{
    static ForecastModel result;
    return &result;
}

void ForecastModel::load(bool force)
{
    if (force) {
        m_requests.clear();
        m_data.clear();
        m_isLoaded = false;
    }

    if (m_isLoaded)
        return;

    m_keys = Settings::storedCities();

    connect(this, SIGNAL(modelResolved(int, ForecastData)),
            this, SLOT(onModelResolved(int, ForecastData)));

    foreach (const QString &city, m_keys) {
        m_requests.append(resolveById(city));
        m_data.append(ForecastData());
    }
}

int ForecastModel::count() const
{
    return m_data.count();
}

ForecastData ForecastModel::itemAt(int index) const
{
    return m_data[index];
}

void ForecastModel::add(const ForecastData &forecast)
{
    m_data.append(forecast);
    emit itemAdded(forecast);
}

void ForecastModel::remove(const ForecastData &forecast)
{
    for (int i = 0; i < m_data.count(); i++) {
        if (m_data[i].key() == forecast.key()) {
            m_data.removeAt(i);
            emit itemRemoved(forecast);
            return;
        }
    }
}

int ForecastModel::indexOf(const ForecastData &forecast) const
{
    for (int i = 0; i < m_data.count(); i++) {
        if (m_data[i].key() == forecast.key())
            return i;
    }

    return -1;
}

void ForecastModel::onModelResolved(int reqId, const ForecastData &forecast)
{
    if (m_requests.removeAll(reqId)) {
        int idx = m_keys.indexOf(forecast.key());
        if (idx >= 0)
            m_data[idx] = forecast;
    }

    if (m_requests.isEmpty()) {
        m_isLoaded = true;

        disconnect(this, SIGNAL(modelResolved(int, ForecastData)),
                   this, SLOT(onModelResolved(int, ForecastData)));

        emit loaded();
    }
}

void ForecastModel::saveToDisk()
{
    QStringList list;
    foreach(const ForecastData &item, m_data)
        list.append(item.key());

    Settings::setStoredCities(list);
}

int ForecastModel::resolveById(const QString &id)
{
    ForecastModel *d = instance();
    return (!d->m_resolver) ? 0 : d->m_resolver->resolve(id);
}

int ForecastModel::resolveByQuery(const QString &query)
{
    ForecastModel *d = instance();
    return (!d->m_resolver) ? 0 : d->m_resolver->resolveByQuery(query);
}

QString ForecastModel::source()
{
    return instance()->m_source;
}

void ForecastModel::setSource(const QString &name)
{
    instance()->m_source = name;
    Settings::setCitySource(name);
}

void ForecastModel::setResolver(ForecastResolver *resolver)
{
    ForecastModel *d = instance();

    if (d->m_resolver) {
        disconnect(d->m_resolver, SIGNAL(forecastResolved(int, WeatherResponse *)),
                   d, SLOT(onForecastResolved(int, WeatherResponse *)));
        disconnect(d->m_resolver, SIGNAL(forecastResponseError(int)),
                   d, SLOT(onForecastResponseError(int)));
        d->m_resolver->deleteLater();
    }

    d->m_resolver = resolver;

    if (d->m_resolver) {
        connect(d->m_resolver, SIGNAL(forecastResolved(int, WeatherResponse *)),
                d, SLOT(onForecastResolved(int, WeatherResponse *)));
        connect(d->m_resolver, SIGNAL(forecastResponseError(int)),
                d, SLOT(onForecastResponseError(int)));
    }
}

void ForecastModel::onForecastResolved(int reqId, WeatherResponse *response)
{
    emit modelResolved(reqId, ForecastData(response));
}

void ForecastModel::onForecastResponseError(int reqId)
{
    emit modelResolved(reqId, ForecastData(0));
}
