/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "pixmaploader.h"
#include "settings.h"
#include <ImtkResource>
#include <QMutexLocker>

#ifdef Q_OS_LINUX
#include <QX11Info>
#endif

#include "settings.h"

// PixmapLoaderThread

PixmapLoaderThread::PixmapLoaderThread(PixmapLoader *loader)
    : QThread(loader)
    , m_loader(loader)
{
}

void PixmapLoaderThread::run()
{
    QString name = m_loader->dequeue();
    while (!name.isEmpty()) {
        QImage image(":images/weather_elements/" + name + ".png");
        emit imageIsReady(name, image);
        name = m_loader->dequeue();
    }
    deleteLater();
}

// PixmapLoader

PixmapLoader::PixmapLoader(QObject *parent)
    : QObject(parent)
    , m_thread(0)
{
}

void PixmapLoader::load(const QString &name)
{
    instance()->enqueue(name);
}

PixmapLoader *PixmapLoader::instance()
{
    static PixmapLoader * const result(new PixmapLoader());
    return result;
}

void PixmapLoader::connectToOnIdleSignal(QObject *receiver, const char *method)
{
    QObject::connect(instance(), SIGNAL(onIdle()), receiver, method, Qt::QueuedConnection);
}

void PixmapLoader::disconnectReceiver(QObject *receiver)
{
    instance()->disconnect(receiver);
}

QPixmap PixmapLoader::pixmap(const QString &name)
{
    PixmapLoader *obj = instance();
    QMutexLocker locker(&obj->m_mutex);

    if (obj->m_store.contains(name))
        return obj->m_store[name];
    else {
#ifdef Q_OS_LINUX
        // apply dithering for 16 bits displays
        if (QX11Info::appDepth() == 16) {
            QImage rawImage(":images/weather_elements/" + name + ".png");
            return QPixmap::fromImage(ImtkResource::applyDithering(rawImage));
        }
#endif
        return QPixmap(":images/weather_elements/" + name + ".png");
    }
}

void PixmapLoader::enqueue(const QString &name)
{
    QMutexLocker locker(&m_mutex);

    if (m_queue.indexOf(name) >= 0 || m_currentImages.indexOf(name) >= 0 || m_store.contains(name))
        return;

    m_queue.append(name);

    if (!m_thread) {
        m_thread = new PixmapLoaderThread(this);
        connect(m_thread, SIGNAL(imageIsReady(QString,QImage)),
                this, SLOT(imageIsReady(QString,QImage)));
        m_thread->start();
    }
    m_condition.wakeOne();
}

QString PixmapLoader::doDequeue()
{
    QString name = m_queue.isEmpty() ? QString() : m_queue.takeAt(0);

    while (!name.isEmpty() && (m_currentImages.indexOf(name) >= 0 || m_store.contains(name)))
        name = m_queue.isEmpty() ? QString() : m_queue.takeAt(0);

    if (!name.isEmpty())
        m_currentImages.append(name);

    return name;
}

QString PixmapLoader::dequeue()
{
    static const int waitTime = 60000;
    QMutexLocker locker(&m_mutex);
    QString result;
    do
        result = doDequeue();
    while (result.isEmpty() && m_condition.wait(&m_mutex, waitTime));
    if (result.isEmpty()) {
        m_thread = 0;
    }
    return result;
}

void PixmapLoader::imageIsReady(const QString &name, QImage image)
{
    QMutexLocker locker(&m_mutex);

    QImage finalImage = image;

#ifdef Q_OS_LINUX
    // apply dithering for 16 bits displays
    if (QX11Info::appDepth() == 16)
        finalImage = ImtkResource::applyDithering(finalImage);
#endif

    QPixmap pixmap = QPixmap::fromImage(finalImage);

    if (!pixmap.isNull())
        m_store[name] = pixmap;

    m_currentImages.removeAll(name);
    if (m_currentImages.isEmpty() && m_queue.isEmpty())
        emit onIdle();
}
