/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "addcitytool.h"
#include "settings.h"
#include "forecastmodel.h"
#include "pixmaploader.h"
#include "addcitytool.h"
#include "forecastmodel.h"


#include <QPainter>
#include <QInputDialog>
#include <QEvent>
#include <QGraphicsScene>
#include <QGraphicsSceneMouseEvent>
#include <QApplication>

#define ADD_BACKGROUND              (PixmapLoader::pixmap("background_add_city"))
#define ADD_ERROR_BACKGROUND        (PixmapLoader::pixmap("background_error_adding"))

#define CLOSE_BUTTON_PIXMAP         (PixmapLoader::pixmap("button_list_delete"))
#define ADD_BUTTON_PIXMAP           (PixmapLoader::pixmap("button_city_send"))
#define BUTTON_LEFT                 (Settings::scaleWidth(408.0))

#define ADD_TEXT_BACKGROUND         (PixmapLoader::pixmap("textfield_add_city"))
#define ADD_TEXT_LEFT               (Settings::scaleWidth(38.0))

#define ADD_SCREEN_FONT_SIZE        (Settings::scaleHeight(40.0))

#define CITY_NAME_FONT_SIZE         Settings::intValue("add-city-font-size")


static inline qreal getCenterVerticalPos(QGraphicsItem *parent, QGraphicsItem *item)
{
    const qreal top = (parent->boundingRect().height() - item->boundingRect().height()) / 2;
    return top - parent->boundingRect().top() - item->boundingRect().top();
}

static inline qreal getCenterVerticalPos(QGraphicsItem *item)
{
    return getCenterVerticalPos(item->parentItem(), item);
}

// AddCityScreen

AddCityScreen::AddCityScreen(const QRectF &boundingRect, QGraphicsItem *parent)
    : QGraphicsItem(parent)
    , m_boundingRect(boundingRect)
{
}

QRectF AddCityScreen::boundingRect () const
{
    return m_boundingRect;
}

void AddCityScreen::paint(QPainter *painter, const QStyleOptionGraphicsItem *opt, QWidget *widget)
{
    Q_UNUSED(painter);
    Q_UNUSED(opt);
    Q_UNUSED(widget);
}

//

#define ADD_A_CITY_TEXT     "Add a city"

AddCityLineEdit::AddCityLineEdit(QWidget *parent)
    : QLineEdit(parent)
    , m_clean(true)
{
    editReset();
    connect(this, SIGNAL(textEdited(QString)), this, SLOT(textEditedSlot(QString)));
}

void AddCityLineEdit::editReset()
{
    m_clean = true;
    setText(ADD_A_CITY_TEXT);
}

void AddCityLineEdit::keyReleaseEvent(QKeyEvent *event)
{
    QLineEdit::keyReleaseEvent(event);
    if (event->key() == Qt::Key_Select) {
        // force return
        emit returnPressed();
    }
}

void AddCityLineEdit::focusOutEvent(QFocusEvent *event)
{
    if (m_clean)
        setText(ADD_A_CITY_TEXT);
    QLineEdit::focusOutEvent(event);
}

void AddCityLineEdit::focusInEvent(QFocusEvent *event)
{
    if (m_clean)
        setText("");
    QLineEdit::focusInEvent(event);
}

void AddCityLineEdit::textEditedSlot(const QString & text)
{
    Q_UNUSED(text);
    static const QString cleanText(ADD_A_CITY_TEXT);
    if (m_clean && text.length() >= cleanText.length() &&
        text.left(cleanText.length()) == cleanText)
        setText(text.right(text.length() - cleanText.length()));
    m_clean = false;
}

// AddCityFirstScreen

AddCityFirstScreen::AddCityFirstScreen(const QRectF &boundingRect, QGraphicsItem *parent)
    : AddCityScreen(boundingRect, parent)
    , m_textBackground(new QGraphicsPixmapItem(ADD_TEXT_BACKGROUND, this))
    , m_button(new ImtkButton(ADD_BUTTON_PIXMAP, this))
    , m_proxy(new QGraphicsProxyWidget(this))
    , m_lineEdit(new AddCityLineEdit())
{
    setFlag(ItemHasNoContents, true);

    m_button->setPos(BUTTON_LEFT, getCenterVerticalPos(m_button));
    m_textBackground->setPos(ADD_TEXT_LEFT, getCenterVerticalPos(m_textBackground));


    QFont font;
    font.setFamily("Nokia Sans");
    font.setPixelSize(CITY_NAME_FONT_SIZE);
    font.setStyleStrategy(QFont::PreferAntialias);
    m_lineEdit->setFont(font);

    m_lineEdit->setFrame(false);
    m_lineEdit->setTextMargins(0, 0, 0, 0);
    m_lineEdit->setAttribute(Qt::WA_NoSystemBackground);
    m_lineEdit->setStyleSheet("background: transparent; color:white");

    QRect rect(0, 0, m_textBackground->pixmap().width(), m_textBackground->pixmap().height());
    rect.adjust(5, 5, -5, -5);
    m_lineEdit->setGeometry(rect);

    m_proxy->setWidget(m_lineEdit);
    m_proxy->setParentItem(m_textBackground);
    m_proxy->setPos(5.0, 5.0);

    m_lineEdit->setFocus();

    connect(m_button, SIGNAL(clicked()), SLOT(submit()));
    connect(m_lineEdit, SIGNAL(returnPressed()), SLOT(submit()));
}

void AddCityFirstScreen::clean()
{
    m_lineEdit->editReset();
}

void AddCityFirstScreen::submit()
{
    if (!m_lineEdit->text().isEmpty())
        emit citySelected(m_lineEdit->text());
}

// AddCitySearchScreen

AddCitySearchScreen::AddCitySearchScreen(const QRectF &boundingRect, QGraphicsItem *parent)
    : AddCityScreen(boundingRect, parent)
{
    QList<QPixmap> images;
    for (int i = 1; i < 26; i++)
        images << PixmapLoader::pixmap(QString("loading_%1").arg(i));

    m_loading = new ImtkThrobber(images, this);
    m_loading->hide();
    m_loading->setPos(BUTTON_LEFT, getCenterVerticalPos(m_loading));

    QFont f("Nokia Sans");
    f.setPixelSize(ADD_SCREEN_FONT_SIZE);

    m_label = new ImtkLabel(this);
    m_label->setFont(f);
    m_label->setMultiline(true);
    m_label->setAlignment(Qt::AlignCenter | Qt::AlignVCenter);

    int maxTextWidth = Settings::scaleWidth(380);
    m_label->setMinimumWidth(maxTextWidth);
    m_label->setMaximumWidth(maxTextWidth);
    m_label->setMinimumHeight(boundingRect.height());
}

void AddCitySearchScreen::forecastResponse(int reqId, const ForecastData &forecast)
{
    if (reqId == m_reqId) {
        reset();
        if (!forecast.isNull() && !forecast.error())
            emit forecastReceived(forecast);
        else
            emit forecastRequestError(m_city);
    }
}

void AddCitySearchScreen::setCityName(const QString &name)
{
    m_city = name;
    m_label->setText(tr("Searching for\n\"%1\"").arg(name));

    ForecastModel *d = ForecastModel::instance();
    connect(d, SIGNAL(modelResolved(int, ForecastData)),
            this, SLOT(forecastResponse(int, ForecastData)));

    m_reqId = d->resolveByQuery(m_city);
    m_loading->start();
    m_loading->show();
    update();
}

void AddCitySearchScreen::reset()
{
    ForecastModel *d = ForecastModel::instance();
    disconnect(d, SIGNAL(modelResolved(int, ForecastData)),
               this, SLOT(forecastResponse(int, ForecastData)));

    m_loading->stop();
    m_loading->hide();
}

void AddCitySearchScreen::paint(QPainter *, const QStyleOptionGraphicsItem *, QWidget *)
{

}

// AddCityErrorScreen

AddCityErrorScreen::AddCityErrorScreen(QGraphicsItem *parent)
    : QGraphicsPixmapItem(ADD_ERROR_BACKGROUND, parent)
    , m_button(new ImtkButton(CLOSE_BUTTON_PIXMAP, this))
{
    m_button->setPos(BUTTON_LEFT, getCenterVerticalPos(m_button));
    connect(m_button, SIGNAL(clicked()), this, SIGNAL(closed()));

    QFont f("Nokia Sans");
    f.setPixelSize(ADD_SCREEN_FONT_SIZE);

    m_label = new ImtkLabel(this);
    m_label->setFont(f);
    m_label->setMultiline(true);
    m_label->setAlignment(Qt::AlignCenter | Qt::AlignVCenter);

    int maxTextWidth = Settings::scaleWidth(380);
    m_label->setMinimumWidth(maxTextWidth);
    m_label->setMaximumWidth(maxTextWidth);
    m_label->setMinimumHeight(boundingRect().height());
}

void AddCityErrorScreen::setCityName(const QString &name, ErrorType type)
{
    if (type == NotFound) {
        m_label->setText(tr("The city \"%1\" was not found").arg(name));
    } else {
        m_label->setText(tr("The city \"%1\" is already on your list").arg(name));
    }
}

// AddCityTool

AddCityTool::AddCityTool(QGraphicsItem *parent)
    : QGraphicsPixmapItem(ADD_BACKGROUND, parent)
    , m_firstScreen(createFirstScreen())
    , m_SearchScreen(createSearchScreen())
    , m_ErrorScreen(createErrorScreen())
{
    setCurrentScreen(m_firstScreen);

    m_timer.setInterval(3000);
    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), SLOT(backToSearch()));
}

AddCityTool::~AddCityTool()
{

}

int AddCityTool::loadImages()
{
    PixmapLoader::load("background_add_city");
    PixmapLoader::load("background_error_adding");
    PixmapLoader::load("button_list_delete");
    PixmapLoader::load("button_city_send");
    PixmapLoader::load("textfield_add_city");
    return 5;
}

void AddCityTool::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void AddCityTool::setInputFocus()
{
    m_firstScreen->setInputFocus();
}

void AddCityTool::cancel()
{
    m_SearchScreen->cancel();
    setCurrentScreen(m_firstScreen);
}

AddCityFirstScreen *AddCityTool::createFirstScreen()
{
    AddCityFirstScreen *result = new AddCityFirstScreen(boundingRect(), this);
    connect(result, SIGNAL(citySelected(QString)), SLOT(citySelected(QString)));
    result->setPos(0.0, 0.0);
    return result;
}

AddCitySearchScreen *AddCityTool::createSearchScreen()
{
    AddCitySearchScreen *result = new AddCitySearchScreen(boundingRect(), this);
    connect(result, SIGNAL(forecastReceived(ForecastData)),
            this, SLOT(forecastReceived(ForecastData)));
    connect(result, SIGNAL(forecastRequestError(QString)),
            this, SLOT(forecastRequestError(QString)));
    result->setPos(0.0, 0.0);
    return result;
}

AddCityErrorScreen *AddCityTool::createErrorScreen()
{
    AddCityErrorScreen *result = new AddCityErrorScreen(this);
    connect(result, SIGNAL(closed()), this, SLOT(errorScreenClosed()));
    result->setPos(0.0, 0.0);
    return result;
}

void AddCityTool::setCurrentScreen(QGraphicsItem *screen)
{
    m_timer.stop();
    m_firstScreen->clean();
    m_firstScreen->setVisible(screen == m_firstScreen);
    m_SearchScreen->setVisible(screen == m_SearchScreen);
    m_ErrorScreen->setVisible(screen == m_ErrorScreen);
}

void AddCityTool::errorScreenClosed()
{
    setCurrentScreen(m_firstScreen);
}

void AddCityTool::forecastReceived(const ForecastData &forecast)
{
    int index = ForecastModel::instance()->indexOf(forecast);

    if (index >= 0) {
        m_ErrorScreen->setCityName(forecast.cityName(), AddCityErrorScreen::AlreadyInList);
        setCurrentScreen(m_ErrorScreen);
        m_timer.start();
        return;
    }

    emit newForecast(forecast);
    setCurrentScreen(m_firstScreen);
}

void AddCityTool::forecastRequestError(const QString &name)
{
    m_ErrorScreen->setCityName(name, AddCityErrorScreen::NotFound);
    setCurrentScreen(m_ErrorScreen);
    m_timer.start();
}

void AddCityTool::backToSearch()
{
    setCurrentScreen(m_firstScreen);
    m_firstScreen->setInputFocus();
}

void AddCityTool::citySelected(const QString &city)
{
    QString text = city.toUpper();

    ForecastModel *model = ForecastModel::instance();

    for (int i = 0; i < model->count(); i++) {
        if (model->itemAt(i).cityName().toUpper() == city) {
            m_ErrorScreen->setCityName(city, AddCityErrorScreen::AlreadyInList);
            setCurrentScreen(m_ErrorScreen);
            return;
        }
    }

    m_SearchScreen->setCityName(city);
    setCurrentScreen(m_SearchScreen);
}
