/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include <math.h>

#include <QPainter>
#include <QGraphicsSceneMouseEvent>

#include "widget_p.h"
#include "listview.h"
#include "kineticscroll.h"


class ImtkListViewItemPrivate
{
public:
    ImtkListViewItemPrivate(ImtkListViewItem *qptr);

    ImtkListViewItem *q;
    int index;
    ImtkListView *view;
    QSizeF size;
    bool isPressed;
    QSizeF preferredSize;

    void setIndex(int value);
    void setPressed(bool selected);
};


ImtkListViewItemPrivate::ImtkListViewItemPrivate(ImtkListViewItem *qptr)
    : q(qptr),
      index(-1),
      view(0),
      isPressed(false),
      preferredSize(QSizeF(50, 50))
{

}

void ImtkListViewItemPrivate::setIndex(int value)
{
    // the index may stay the same when an item is removed and becomes
    // the previous one, but the renderer contents must be changed.
    //if (index != value) {
    q->indexChange(index, value);
    index = value;
    q->update();
    //}
}

void ImtkListViewItemPrivate::setPressed(bool pressed)
{
    if (pressed != isPressed) {
        isPressed = pressed;
        q->selectionChanged();
    }
}


ImtkListViewItem::ImtkListViewItem(QGraphicsItem *parent)
    : QGraphicsItem(parent),
      d(new ImtkListViewItemPrivate(this))
{

}

ImtkListViewItem::~ImtkListViewItem()
{
    delete d;
}

int ImtkListViewItem::index() const
{
    return d->index;
}

ImtkListView *ImtkListViewItem::listView() const
{
    return d->view;
}

QSizeF ImtkListViewItem::preferredSize() const
{
    return d->preferredSize;
}

void ImtkListViewItem::setPreferredSize(const QSizeF &size)
{
    d->preferredSize = size;
}

bool ImtkListViewItem::isPressed() const
{
    return d->isPressed;
}

void ImtkListViewItem::contentsChanged()
{
    update();
}

void ImtkListViewItem::selectionChanged()
{
    update();
}

void ImtkListViewItem::indexChange(int oldIndex, int newIndex)
{
    Q_UNUSED(oldIndex);
    Q_UNUSED(newIndex);

    update();
}

QSizeF ImtkListViewItem::size() const
{
    return d->size;
}

void ImtkListViewItem::resize(const QSizeF &value)
{
    if (value != d->size) {
        prepareGeometryChange();
        d->size = value;
        update();
    }
}

QRectF ImtkListViewItem::boundingRect() const
{
    return QRectF(QPointF(), size());
}



class ImtkListViewPrivate : public ImtkWidgetPrivate
{
public:
    ImtkListViewPrivate(ImtkListView *qptr);

    void init();
    void refillRenderers();
    void reconfigureRenderers();
    void repositionRenderers();
    bool setPixelPosition(int value);
    int visibleItemCount();
    void setMaximumOffset(int value);
    void updateMaximumOffset();

    int totalCount() const { return !model ? 0 : model->count(); }

    int itemSpace;
    int currentItem;

    int offset;
    int maximumOffset;
    int spareRenderers;

    ImtkAbstractModel *model;
    QList<ImtkListViewItem *> children;
    ImtkAbstractListViewCreator *creator;

private:
    IMTK_DECLARE_PUBLIC(ImtkListView);
};


ImtkListViewPrivate::ImtkListViewPrivate(ImtkListView *qptr)
    : ImtkWidgetPrivate(qptr),
      itemSpace(50),
      currentItem(0),
      maximumOffset(0),
      spareRenderers(2),
      model(0),
      creator(0)
{
    offset = -itemSpace;
}

void ImtkListViewPrivate::init()
{
    Q_Q(ImtkListView);
    q->setHandlesChildEvents(true);
    q->setFlag(QGraphicsItem::ItemHasNoContents);
    q->setFlag(QGraphicsItem::ItemClipsChildrenToShape);
}

bool ImtkListViewPrivate::setPixelPosition(int value)
{
    Q_Q(ImtkListView);
    value = qBound<int>(0, value, q->maximumOffset());

    if (value == q->offset())
        return false;

    const int idx = value / itemSpace;
    const int overlap = idx - currentItem;

    if (overlap == 0) {
        offset = -itemSpace - (value % itemSpace);
    } else if (overlap == -1 || overlap == 1) {
        currentItem += overlap;
        offset = (currentItem - 1) * itemSpace - value;

        int newIdx;
        ImtkListViewItem *renderer;

        if (overlap == 1) {
            renderer = children.takeFirst();
            children.append(renderer);
            newIdx = currentItem + children.count() - spareRenderers;
        } else {
            renderer = children.takeLast();
            children.prepend(renderer);
            newIdx = currentItem - 1;
        }

        if (newIdx >= 0 && newIdx < totalCount()) {
            renderer->d->setIndex(newIdx);
            renderer->show();
        } else
            renderer->hide();
    } else {
        currentItem += overlap;
        offset = (currentItem - 1) * itemSpace - value;
        refillRenderers();
    }

    repositionRenderers();
    return true;
}

void ImtkListViewPrivate::reconfigureRenderers()
{
    if (!model)
        return;

    Q_Q(ImtkListView);

    const int w = q->size().width();
    const int h = q->size().height();

    int nitems = qRound(ceil(h / (qreal)itemSpace));

    if (currentItem >= totalCount() || nitems >= totalCount()) {
        currentItem = 0;
        offset = -itemSpace;
    }

    nitems += spareRenderers;

    if (!children.isEmpty()) {
        while (nitems < children.count())
            delete children.takeLast();
    }

    foreach (ImtkListViewItem *item, children)
        item->resize(QSizeF(w, itemSpace));

    if (creator && nitems > children.count()) {
        while (nitems > children.count()) {
            ImtkListViewItem *item = creator->create();

            if (item) {
                children.append(item);
                item->d->view = q;
                item->setParentItem(q);
                item->resize(QSizeF(w, itemSpace));
            }
        }
    }

    repositionRenderers();
    refillRenderers();
}

void ImtkListViewPrivate::refillRenderers()
{
    if (children.isEmpty())
        return;

    if (totalCount() == 0) {
        foreach (ImtkListViewItem *item, children)
            item->hide();
        return;
    }

    ImtkListViewItem *first = children[0];
    if (currentItem > 0) {
        first->d->setIndex(currentItem - 1);
        first->show();
    } else
        first->hide();

    const int nc = children.count();
    const int end = qMin(currentItem + nc - 1, totalCount());

    for (int i = currentItem; i < end; i++) {
        ImtkListViewItem *item = children[i - currentItem + 1];
        item->d->setIndex(i);
        item->show();
    }

    for (int i = end - currentItem + 1; i < nc; i++)
        children[i]->hide();
}

void ImtkListViewPrivate::repositionRenderers()
{
    int x = 0;
    int y = offset;

    foreach (ImtkListViewItem *item, children) {
        item->setPos(x, y);
        y += itemSpace;
    }
}

int ImtkListViewPrivate::visibleItemCount()
{
    int r = children.count();
    int e = totalCount();

    if (r == e)
        return e - 1;
    else if (r > e)
        return e;
    else
        return qMax(r - spareRenderers, 0);
}

void ImtkListViewPrivate::setMaximumOffset(int value)
{
    Q_Q(ImtkListView);
    value = qMax(value, 0);
    if (maximumOffset != value) {
        maximumOffset = value;
        emit q->maximumOffsetChanged();
    }
}

void ImtkListViewPrivate::updateMaximumOffset()
{
    Q_Q(ImtkListView);
    if (totalCount() == 0)
        setMaximumOffset(0);
    else
        setMaximumOffset(qRound(totalCount() * itemSpace - q->size().height()));
}



ImtkListView::ImtkListView(QGraphicsItem *parent)
    : ImtkWidget(*new ImtkListViewPrivate(this), parent)
{
    Q_D(ImtkListView);
    d->init();
}

ImtkListView::ImtkListView(ImtkListViewPrivate &dptr, QGraphicsItem *parent)
    : ImtkWidget(dptr, parent)
{
    Q_D(ImtkListView);
    d->init();
}

ImtkListView::~ImtkListView()
{
    Q_D(ImtkListView);
    if (d->creator)
        delete d->creator;
}

ImtkAbstractModel *ImtkListView::model() const
{
    Q_D(const ImtkListView);
    return d->model;
}

void ImtkListView::setModel(ImtkAbstractModel *model)
{
    Q_D(ImtkListView);
    if (d->model == model)
        return;

    if (d->model) {
        disconnect(d->model, SIGNAL(updated()), this, SLOT(reconfigureRenderers()));

        disconnect(d->model, SIGNAL(itemAdded(int)), this, SLOT(itemAdded(int)));
        disconnect(d->model, SIGNAL(itemChanged(int)), this, SLOT(itemChanged(int)));
        disconnect(d->model, SIGNAL(itemRemoved(int)), this, SLOT(itemRemoved(int)));
        disconnect(d->model, SIGNAL(itemMoved(int, int)), this, SLOT(itemMoved(int, int)));
    }

    d->model = model;

    if (d->model) {
        connect(d->model, SIGNAL(updated()), this, SLOT(reconfigureRenderers()));

        connect(d->model, SIGNAL(itemAdded(int)), this, SLOT(itemAdded(int)));
        connect(d->model, SIGNAL(itemChanged(int)), this, SLOT(itemChanged(int)));
        connect(d->model, SIGNAL(itemRemoved(int)), this, SLOT(itemRemoved(int)));
        connect(d->model, SIGNAL(itemMoved(int, int)), this, SLOT(itemMoved(int, int)));
    }

    modelChanged();
}

void ImtkListView::setCreator(ImtkAbstractListViewCreator *creator)
{
    Q_D(ImtkListView);
    if (d->creator == creator)
        return;

    if (d->creator)
        delete d->creator;

    d->creator = creator;

    if (creator) {
        // get item preferred size just once
        ImtkListViewItem *item = creator->create();
        if (item) {
            d->itemSpace = qMax<qreal>(item->preferredSize().height(), 1);
            delete item;
            d->updateMaximumOffset();
        }
    }
}

int ImtkListView::offset() const
{
    Q_D(const ImtkListView);
    return (d->currentItem - 1) * d->itemSpace - d->offset;
}

void ImtkListView::setOffset(int position)
{
    Q_D(ImtkListView);
    int oldOffset = offset();
    d->setPixelPosition(qBound(0, position, maximumOffset()));

    if (oldOffset != offset())
        emit offsetChanged();
}

int ImtkListView::maximumOffset() const
{
    Q_D(const ImtkListView);
    return d->maximumOffset;
}

void ImtkListView::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    QGraphicsWidget::resizeEvent(event);

    Q_D(ImtkListView);
    d->updateMaximumOffset();
    d->reconfigureRenderers();
}

void ImtkListView::reconfigureRenderers()
{
    Q_D(ImtkListView);
    d->reconfigureRenderers();
}

int ImtkListView::indexAtOffset(int value)
{
    if (value < 0 || value > size().height())
        return -1;

    Q_D(ImtkListView);
    int idx = (value + offset()) / d->itemSpace;
    return (idx < d->totalCount()) ? idx : -1;
}

ImtkListViewItem *ImtkListView::itemFromIndex(int index)
{
    Q_D(ImtkListView);
    if (index < 0 || index >= d->totalCount() || d->currentItem < 0)
        return 0;

    int base = d->currentItem;
    int top = base + d->visibleItemCount();
    int cindex = index - base + 1;

    if (cindex < d->children.count() && base <= index && index <= top)
        return d->children[cindex];
    else
        return 0;
}

void ImtkListView::modelChanged()
{
    Q_D(ImtkListView);
    d->currentItem = 0;
    d->updateMaximumOffset();
    setOffset(0);
    reconfigureRenderers();
}

void ImtkListView::itemChanged(int index)
{
    ImtkListViewItem *item = itemFromIndex(index);

    if (item)
        item->contentsChanged();
}

void ImtkListView::itemAdded(int index)
{
    Q_D(ImtkListView);
    Q_UNUSED(index);
    // XXX: optimize (check if it's in visible area)
    d->updateMaximumOffset();
    reconfigureRenderers();
}

void ImtkListView::itemRemoved(int index)
{
    Q_D(ImtkListView);
    Q_UNUSED(index);
    // XXX: optimize (check if it's in visible area)
    d->updateMaximumOffset();
    reconfigureRenderers();
}

void ImtkListView::itemMoved(int oldIndex, int newIndex)
{
    Q_UNUSED(oldIndex);
    Q_UNUSED(newIndex);
    // XXX: optimize (check if it's in visible area)
    reconfigureRenderers();
}

void ImtkListView::setItemPressed(ImtkListViewItem *item, bool pressed)
{
    item->d->setPressed(pressed);
}




class ImtkKineticListViewPrivate : public ImtkListViewPrivate
{
public:
    ImtkKineticListViewPrivate(ImtkKineticListView *qptr);

    bool isClickPossible(int y);
    int getPosSmooth(int y);
    bool moveOffset(int offset);

    void showSelection();
    void hideSelection();

    bool dragging;
    bool mouseDown;
    int actualPosY;
    int mouseDownPos;
    int moveConstant;
    int clickConstant;
    double clickInitTime;
    double clickBlockTime;
    ImtkKineticScroll *kinetic;
    ImtkListViewItem *lastSelectedRenderer;

private:
    IMTK_DECLARE_PUBLIC(ImtkKineticListView);
};


ImtkKineticListViewPrivate::ImtkKineticListViewPrivate(ImtkKineticListView *qptr)
    : ImtkListViewPrivate(qptr),
      kinetic(0)
{
    dragging = false;
    moveConstant = 15;
    clickConstant = 20;
    clickInitTime = 0.4;
    clickBlockTime = 0.5;
    actualPosY = -1;
    mouseDownPos = -1;
    lastSelectedRenderer = 0;
}

void ImtkKineticListViewPrivate::showSelection()
{
    if (!isClickPossible(actualPosY))
        return;

    Q_Q(ImtkKineticListView);
    int idx = q->indexAtOffset(actualPosY);
    if (idx >= 0) {
        if (lastSelectedRenderer)
            q->setItemPressed(lastSelectedRenderer, false);

        lastSelectedRenderer = q->itemFromIndex(idx);

        if (lastSelectedRenderer)
            q->setItemPressed(lastSelectedRenderer, true);
    }
}

void ImtkKineticListViewPrivate::hideSelection()
{
    Q_Q(ImtkKineticListView);
    if (lastSelectedRenderer)
        q->setItemPressed(lastSelectedRenderer, false);

    lastSelectedRenderer = 0;
}

bool ImtkKineticListViewPrivate::isClickPossible(int y)
{
    if (dragging || mouseDownPos < 0)
        return false;
    else
        return abs(y - mouseDownPos) <= clickConstant;
}

int ImtkKineticListViewPrivate::getPosSmooth(int y)
{
    if (abs(mouseDownPos - y) <= moveConstant)
        return y;
    else if (mouseDownPos - y < 0)
        return y - moveConstant;
    else
        return y + moveConstant;
}

bool ImtkKineticListViewPrivate::moveOffset(int value)
{
    Q_Q(ImtkKineticListView);
    int finalOffset = q->offset() - value;

    q->setOffset(finalOffset);

    if (value == 0 || finalOffset != q->offset()) {
        kinetic->kineticStop();
        return false;
    }

    return true;
}



ImtkKineticListView::ImtkKineticListView(QGraphicsItem *parent)
    : ImtkListView(*new ImtkKineticListViewPrivate(this), parent)
{
    Q_D(ImtkKineticListView);

    d->kinetic = new ImtkKineticScroll(this);
    connect(d->kinetic, SIGNAL(signalMoveOffset(int)), SLOT(kineticMove(int)));
}

ImtkKineticListView::ImtkKineticListView(ImtkKineticListViewPrivate &dptr,
                                         QGraphicsItem *parent)
    : ImtkListView(dptr, parent)
{
    Q_D(ImtkKineticListView);

    d->kinetic = new ImtkKineticScroll(this);
    connect(d->kinetic, SIGNAL(signalMoveOffset(int)), SLOT(kineticMove(int)));
}

ImtkKineticListView::~ImtkKineticListView()
{

}

void ImtkKineticListView::kineticMove(int value)
{
    Q_D(ImtkKineticListView);
    d->moveOffset(value);
}

void ImtkKineticListView::mousePressEvent(QGraphicsSceneMouseEvent *e)
{
    Q_D(ImtkKineticListView);
    int y = e->pos().y();
    d->mouseDownPos = y;
    d->dragging = !d->kinetic->mouseDown(y);
    d->actualPosY = y;
    d->mouseDown = true;

    d->showSelection();
}

void ImtkKineticListView::mouseReleaseEvent(QGraphicsSceneMouseEvent *e)
{
    Q_D(ImtkKineticListView);
    d->mouseDown = false;
    if (d->mouseDownPos >= 0) {
        int y = e->pos().y();
        if (d->isClickPossible(y)) {
            int idx = indexAtOffset(y);

            if (idx >= 0 && idx < d->totalCount())
                emit itemClicked(idx);

            d->kinetic->mouseCancel();
        } else {
            d->kinetic->mouseUp(d->getPosSmooth(y));
        }
    }

    d->hideSelection();
}

void ImtkKineticListView::mouseMoveEvent(QGraphicsSceneMouseEvent *e)
{
    Q_D(ImtkKineticListView);
    if (d->mouseDownPos >= 0) {
        int y = e->pos().y();
        d->actualPosY = y;

        if (!d->isClickPossible(y)) {
            d->dragging = true;
            d->hideSelection();
        }

        if (abs(d->mouseDownPos - y) > d->moveConstant)
            d->kinetic->mouseMove(d->getPosSmooth(y));
    }
}

void ImtkKineticListView::modelChanged()
{
    Q_D(ImtkKineticListView);
    d->kinetic->kineticStop();
    ImtkListView::modelChanged();
}
