//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lsbattery.h"
#ifdef Q_WS_MAEMO_5
    #include <mce/dbus-names.h>
    #include <mce/mode-names.h>
#endif
#include "math.h"
#include <QProcess>

LsBattery::LsBattery(QString name) :
        LsWidget(name.isEmpty() ? "Battery": name)
{
    m_IsCharging = false;
    m_Percentage = 100;
    applySettings();
}

void LsBattery::paint(QPainter *painter, const QStyleOptionGraphicsItem*,
                      QWidget*)
{
    if (m_IsCharging && !m_ChargingImage.isNull()){
        painter->drawPixmap(0, 0, m_ChargingImage);
    }else{
        if (m_Images.count() > 0){
            int idx = ceil(m_Percentage / (100.0 / (float)m_Images.count()) - 1.0);
            painter->drawPixmap(0, 0, m_Images.at(idx));
        }
    }

    if (!m_IsCharging){
        QRectF rect = boundingRect();
        painter->setFont(m_PercFont);
        painter->setPen(m_PercColor);
        painter->drawText(rect, Qt::AlignCenter | Qt::TextSingleLine, QString("%1%").arg(QString::number(m_Percentage, 'f', 1)));
    }
}

void LsBattery::stopUpdate()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().disconnect(QString(), "/org/freedesktop/Hal/devices/bme", "org.freedesktop.Hal.Device",
                                            "PropertyModified", this, SLOT(bmePropertyModifiedSlot(QDBusMessage)));
#endif
}

void LsBattery::startUpdate()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().connect(QString(), "/org/freedesktop/Hal/devices/bme", "org.freedesktop.Hal.Device",
                                         "PropertyModified", this, SLOT(bmePropertyModifiedSlot(QDBusMessage)));
    bmePropertyModifiedSlot(QDBusMessage());
#endif
}

void LsBattery::applySettings()
{
    setSizeAndPosition(QSize(64, 40),QPoint(0, 0));
    QString images = getSetting("Images",
                                ":/images/battery_0.png;:/images/battery_1.png;:/images/battery_2.png;:/images/battery_3.png;:/images/battery_4.png;:/images/battery_5.png;:/images/battery_6.png").toString();

    m_Images.clear();
    foreach (QString img, images.split(';')){
        QPixmap i(img);
        if (!i.isNull())
            m_Images.append(i);
    }

    m_ChargingImage = QPixmap(getSetting("ChargingImage", ":/images/battery_charging.png").toString());
    m_PercColor = getColorSetting("PercentageColor", QColor(255,255,255));
    m_PercFont = getFontSetting("PercentageFont", QFont("", 15, QFont::Normal, false));
}

#ifdef Q_WS_MAEMO_5
void LsBattery::bmePropertyModifiedSlot(QDBusMessage)
{
    qDebug() << "Battery state changed";
    QProcess* proc = new QProcess(this);
    proc->start("hal-device bme", QIODevice::ReadOnly);
    proc->waitForFinished();
    if (proc->exitCode() == 0){
        QByteArray ba = proc->readAll();
        QString output(ba);
        QStringList lines = output.split("\n", QString::SkipEmptyParts);

        float vcurrent = 0;
        float current = 0;
        float design = 0;
        float vdesign = 0;
        bool charging = false;

        foreach (QString line, lines){
            line = line.trimmed();

            QStringList e = line.split("=", QString::SkipEmptyParts);
            if (e.count() > 1){
                QString name = e.at(0).trimmed();
                QString value = e.at(1).trimmed();
                int pos = value.indexOf(" ");
                if (pos != -1)
                    value = value.left(pos);

                if (name == "battery.reporting.current"){
                   current = value.toFloat();
                }else if (name == "battery.reporting.design"){
                   design = value.toFloat();
                }else if (name == "battery.voltage.current" && vcurrent == 0){
                   vcurrent = value.toFloat();
                }else if (name == "battery.voltage.design"){
                   vdesign = value.toFloat();
                }else if (name == "battery.rechargeable.is_charging"){
                   charging = value == "true";
                }
            }
        }

        m_IsCharging = charging;
        if (design > 0)
            m_Percentage = current / design * 100.0;
        else
            m_Percentage = 0;
    }
    delete proc;

    qDebug() << "Battery percentage" << m_Percentage;
    update();
}
#endif
