//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lscalendar.h"
#include "lockscreen.h"


LsCalendar::LsCalendar(QString name) :
        LsWidget(name.isEmpty() ? "Calendar": name)
{
    applySettings();
}

void LsCalendar::applySettings()
{
    m_NoEventsText = getSetting("NoEventsText", "No events").toString();
    m_NumberOfEntries = getSetting("Entries", 3).toInt();

    m_DateFormat = getSetting("DateFormat", "dd/MM").toString();
    m_DateColor = getColorSetting("DateColor", QColor(255,255,255));
    m_DateFont = getFontSetting("DateFont", QFont("", 30, QFont::Normal, false));

    m_TextColor = getColorSetting("TextColor", QColor(255,255,255));
    m_TextFont = getFontSetting("TextFont", QFont("", 30, QFont::Normal, false));

    setSizeAndPosition(QSize(100, 50),QPoint(0, 0));
}

void LsCalendar::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
           QWidget*)
{
    painter->setClipRect(option->exposedRect);

    QRectF rect = boundingRect();
    if (m_Entries.count() == 0){
        painter->setFont(m_TextFont);
        painter->setPen(m_TextColor);
        painter->drawText(rect, Qt::AlignCenter, m_NoEventsText);
    }else{
        int maxDateWidth = 0;
        QFontMetrics dfm(m_DateFont);
        foreach(CalendarEntry* entry, m_Entries){
            QString date = entry->DateStart.toString(m_DateFormat);
            if (dfm.width(date) > maxDateWidth)
                maxDateWidth = dfm.width(date);
        }

        QFontMetrics tfm(m_TextFont);
        foreach(CalendarEntry* entry, m_Entries){
            rect.setLeft(0);

            QString date = entry->DateStart.toString(m_DateFormat);
            painter->setFont(m_DateFont);
            painter->setPen(m_DateColor);
            painter->drawText(rect, Qt::AlignLeft, date);

            rect.setLeft(maxDateWidth + 5);

            if (!entry->Icon.isNull()){
                painter->drawPixmap(rect.left(), rect.top(), entry->Icon);
                rect.setLeft(rect.left() + entry->Icon.width() + 5);
            }else{
                rect.setLeft(rect.left() + 16 + 5);
            }

            QString text = tfm.elidedText(entry->Text, Qt::ElideRight, boundingRect().width() - rect.left());
            painter->setFont(m_TextFont);
            painter->setPen(m_TextColor);
            painter->drawText(rect, Qt::AlignLeft, text);

            rect.setTop(rect.top() + dfm.height());
        }
    }
}

void LsCalendar::startUpdate()
{
    if (LockScreen::ScreenshotMode)
        updateCalendarEntries();
    else
        QTimer::singleShot(250, this, SLOT(updateCalendarEntries()));
}

void LsCalendar::stopUpdate()
{

}

void LsCalendar::updateCalendarEntries()
{
    getEntries();
    update();
}

#ifdef Q_WS_MAEMO_5
bool toAssending( CalendarEntry* s1 , CalendarEntry* s2 ) {
    QDateTime d1 = s1->DateStart;
    QDateTime d2 = s2->DateStart;
    if (s1->Type == E_BDAY){
        d1 = d1.addYears(QDateTime::currentDateTime().date().year() - d1.date().year());
        if (d1.date() < QDate::currentDate())
            d1 = d1.addYears(1);
    }

    if (s2->Type == E_BDAY){
        d2 = d2.addYears(QDateTime::currentDateTime().date().year() - d2.date().year());
        if (d2.date() < QDate::currentDate())
            d2 = d2.addYears(1);
    }
    return d1 < d2;
}
#endif

void LsCalendar::getEntries()
{
    foreach(CalendarEntry* entry, m_Entries)
        delete entry;
    m_Entries.clear();

#ifdef Q_WS_MAEMO_5
    QList<CalendarEntry*> entries;

    CCalendarDB *pDb = CCalendarDB::Instance();
    if (pDb == NULL){
        qDebug() << "Error accessing calendar database.";
        return;
    }

    struct tm today_midnight;
    time_get_local(&today_midnight);
    today_midnight.tm_hour = today_midnight.tm_min = today_midnight.tm_sec = 0;
    time_t today_midnight_time = mktime(&today_midnight);
    int iStDate = today_midnight_time;

    today_midnight.tm_year += 1;
    time_t end_time = mktime(&today_midnight);
    int iEndDate = end_time;

    int eventType = E_EVENT;
    int bdayType = E_BDAY;

    int iNewStDate = 0;
    int iNewEndDate = 0;
    CMulticalendar *mc = CMulticalendar::MCInstance();
    int time_diff = time_get_time_diff(iStDate, mc->getSystemTimeZone().c_str(), "UTC");

    iNewStDate = iStDate - time_diff;
    iNewEndDate = iEndDate - time_diff;

    char *sqlQuery =
      sqlite3_mprintf(
              "select * from %s where "
              "%s in (%d, %d) AND (%s - %s) < %d AND (%s - %s) > %d AND (%s != %d AND %s != %d) "
              "union select * from %s where "
              "%s in (%d, %d) AND (%s = 2  OR allday = 1) AND (%s - %s)>= %d AND (%s - %s)< %d AND (%s != %d AND %s != %d) "
              "union select * from %s where "
              "%s in (%d, %d) AND allday = 0 AND (%s = %d OR %s = %d) AND (%d < %s OR %d = %s) "
              "union select * from %s where "
              "%s in (%d, %d) AND allday = 1 AND (%s = %d OR %s = %d) AND (%d < (%s + %s) OR %d = %s) "
              "LIMIT %d OFFSET %d",
              COMPONENTS_TABLE,
              ENTRY_FIELD_TYPE, eventType, bdayType, ENTRY_FIELD_DTSTART, ENTRY_FIELD_TZOFFSET, iNewEndDate,ENTRY_FIELD_DTEND,
              ENTRY_FIELD_TZOFFSET, iNewStDate,
              ENTRY_FIELD_FLAGS, HAS_RECURRENCE,ENTRY_FIELD_FLAGS, HAS_RECURRENCE_ALARM,

              COMPONENTS_TABLE,
              ENTRY_FIELD_TYPE, eventType, bdayType, ENTRY_FIELD_TYPE,
              ENTRY_FIELD_DTSTART, ENTRY_FIELD_TZOFFSET, iNewStDate,
              ENTRY_FIELD_DTSTART, ENTRY_FIELD_TZOFFSET, iNewEndDate,
              ENTRY_FIELD_FLAGS, HAS_RECURRENCE,ENTRY_FIELD_FLAGS, HAS_RECURRENCE_ALARM,

              COMPONENTS_TABLE,
              ENTRY_FIELD_TYPE, eventType, bdayType,
              ENTRY_FIELD_FLAGS, HAS_RECURRENCE,ENTRY_FIELD_FLAGS, HAS_RECURRENCE_ALARM,iStDate,
              ENTRY_FIELD_UNTIL,NULLID, ENTRY_FIELD_UNTIL,

              COMPONENTS_TABLE,
              ENTRY_FIELD_TYPE, eventType, bdayType,
              ENTRY_FIELD_FLAGS, HAS_RECURRENCE,ENTRY_FIELD_FLAGS, HAS_RECURRENCE_ALARM,
              iNewStDate, ENTRY_FIELD_UNTIL, ENTRY_FIELD_TZOFFSET,NULLID, ENTRY_FIELD_UNTIL,
              100, 0);

    //qDebug() << sqlQuery;
    int error = 0;
    int sqliteError = 0;
    QueryResult *queryResult = pDb->getRecords(sqlQuery,sqliteError);
    pDb->sqliteErrorMapper(sqliteError, error);
    sqlite3_free(sqlQuery);

    if(queryResult == 0){                
        if(error == CALENDAR_OPERATION_SUCCESSFUL)
            error = CALENDAR_FETCH_NOITEMS;
    }else{
        for(int i = 0; i < queryResult->iRow; i++){
            CComponent *componentEntry = new CComponent();

            for(int j = 0; j < queryResult->iColumn; j++){
                int k = queryResult->iColumn + (i * queryResult->iColumn);

                if ((queryResult->pResult[k + j]) == 0)
                    continue;

                switch (j)
                {
                case DB_COLUMN_ID2:
                  componentEntry->setCalendarId(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID1:
                  componentEntry->setId(queryResult->pResult[k + j]);
                  break;
                case DB_COLUMN_ID3:
                  componentEntry->setType(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID4:
                  componentEntry->setFlags(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID5:
                  componentEntry->setDateStart(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID6:
                  componentEntry-> setDateEnd(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID7:
                  if(queryResult->pResult[k + j])
                  {
                    componentEntry->setSummary((string) queryResult-> pResult[k + j]);
                  }
                  break;
                case DB_COLUMN_ID8:
                  if(queryResult->pResult[k + j])
                  {
                    componentEntry->setLocation((string) queryResult->pResult[k + j]);
                  }
                  break;
                case DB_COLUMN_ID9:
                  if(queryResult->pResult[k + j])
                  {
                    componentEntry->setDescription((string) queryResult->pResult[k + j]);
                  }
                  break;
                case DB_COLUMN_ID10:
                  componentEntry->setStatus(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID11:
                  if(queryResult->pResult[k + j])
                  {
                    componentEntry->setGUid(queryResult->pResult[k + j]);
                  }
                  break;
                case DB_COLUMN_ID12:
                  componentEntry-> setUntil(atoi (queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID13:
                  componentEntry-> setAllDay(atoi (queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID14:
                  componentEntry->setCreatedTime(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID15:
                  componentEntry->setLastModified(atoi(queryResult->pResult[k + j]));
                  break;
                case DB_COLUMN_ID16:
                  componentEntry->setTzid(queryResult->pResult[k + j]);
                  break;
                default:
                  break;
                }
            }

            if(componentEntry->getFlags() == HAS_ALARM || componentEntry->getFlags() == HAS_RECURRENCE_ALARM)
                componentEntry->getAlarmProperties();

            if(componentEntry->getFlags() == HAS_RECURRENCE || componentEntry->getFlags() == HAS_RECURRENCE_ALARM){
                componentEntry->getRecurrenceProperties();
                if(componentEntry->getInstanceNumber(iStDate, iEndDate))
                    addEntry(&entries, componentEntry);
            }else{
                addEntry(&entries, componentEntry);
            }
            delete componentEntry;
            componentEntry = 0;
        }

        if(queryResult){
            sqlite3_free_table(queryResult->pResult);
            delete queryResult;
        }
    }

    qSort( entries.begin() , entries.end(), toAssending );
    foreach(CalendarEntry* entry, entries){
        if (m_Entries.count() < m_NumberOfEntries){
            if (entry->Type != E_BDAY && entry->DateStart >= QDateTime::currentDateTime())
                m_Entries.append(entry);
            else if (entry->Type == E_BDAY)
                m_Entries.append(entry);
            else
                delete entry;
        }else
            delete entry;
    }

#endif
}

#ifdef Q_WS_MAEMO_5
void LsCalendar::addEntry(QList<CalendarEntry*>* entries, CComponent *componentEntry)
{
    CalendarEntry* entry = new CalendarEntry();
    entry->CalendarId = componentEntry->getCalendarId();
    entry->Type = componentEntry->getType();
    entry->DateStart = QDateTime::fromTime_t(componentEntry->getDateStart());
    entry->DateEnd = QDateTime::fromTime_t(componentEntry->getDateEnd());
    entry->Text = QString(componentEntry->getSummary().c_str());
    entry->Icon = QPixmap(GetIconPath(componentEntry));
    if (!entry->Icon.isNull())
        entry->Icon = entry->Icon.scaledToWidth(16);
    entries->append(entry);

    //qDebug() << "Calendar:" << entry->DateStart << entry->DateEnd << entry->Text;
}

QString LsCalendar::GetIconPath(CComponent* component)
{
  QString res;
  int error = 0;
  int eventType = component->getType();
  CCalendar* cal = CMulticalendar::MCInstance()->getCalendarById(component->getCalendarId(), error);

  switch(cal->getCalendarColor())
  {
  case COLOUR_DARKBLUE:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_darkblue.png";
    break;
  case COLOUR_DARKGREEN:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_darkgreen.png";
    break;
  case COLOUR_DARKRED:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_darkred.png";
    break;
  case COLOUR_ORANGE:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_orange.png";
    break;
  case COLOUR_VIOLET:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_violet.png";
    break;
  case COLOUR_YELLOW:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_yellow.png";
    break;
  case COLOUR_BLUE:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_blue.png";
    break;
  case COLOUR_RED:
    return "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_red.png";
    break;
  case COLOUR_GREEN:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_green.png";
    break;
  default:
    res = "/usr/share/icons/hicolor/48x48/hildon/calendar_colors_white.png";
  }
  delete cal;

  if(eventType == E_TODO || eventType == E_BDAY || component->getAlarm())
  {
    if(eventType == E_BDAY)
      res = "/usr/share/icons/hicolor/48x48/hildon/calendar_birthday.png";
    else if(eventType == E_TODO)
      res = "/usr/share/icons/hicolor/48x48/hildon/calendar_todo.png";
    else
      res = "/usr/share/icons/hicolor/48x48/hildon/calendar_alarm.png";
  }
  return res;
}
#endif
