//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lockscreen.h"
#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <X11/Xutil.h>
    #include <mce/dbus-names.h>
    #include <mce/mode-names.h>
#endif

bool LockScreen::Debug;
bool LockScreen::ScreenshotMode;
QSettings* LockScreen::Settings;

LockScreen::LockScreen() :
        QObject(NULL)
{
    m_LabelBackground = NULL;
    m_Movie = NULL;
    m_Proxy = NULL;

    m_Player = new SystemPlayer(this);

    m_Scene.setSceneRect(0, 0, 800, 480);
    m_Scene.setItemIndexMethod(QGraphicsScene::NoIndex);

    m_View = new QGraphicsView(&m_Scene);
    m_View->setWindowFlags(Qt::Dialog);
    m_View->setRenderHint(QPainter::Antialiasing, LockScreen::Settings->value("Main/Antialiasing", 1).toInt() == 1);    
    m_View->setCacheMode(QGraphicsView::CacheBackground);
    m_View->setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    m_View->setFrameStyle(QFrame::NoFrame);
    m_View->setContentsMargins(0,0,0,0);
    m_View->setOptimizationFlag(QGraphicsView::DontSavePainterState, true);

    if (LockScreen::Settings->value("Main/TransparentBackground", 0).toInt() == 1){
        m_View->setAttribute(Qt::WA_TranslucentBackground, true);
        QPalette p = m_View->viewport()->palette();
        p.setColor(QPalette::Base, Qt::transparent);
        m_View->viewport()->setPalette(p);
    }else{
        m_View->setAttribute(Qt::WA_TranslucentBackground, false);
    }

    m_WasPortrait = LsWidget::isPortrait();
    m_isDialogActive = false;

    LsWidget::setOrientation(LsWidget::Auto);
#ifdef Q_WS_MAEMO_5
    QString orientation = LockScreen::Settings->value("Main/Orientation", "Auto").toString().toLower();
    if (orientation == "landscape"){
        LsWidget::setOrientation(LsWidget::Landscape);
        m_View->setAttribute(Qt::WA_Maemo5LandscapeOrientation);
        m_WasPortrait = false;
    } else if (orientation == "portrait"){
        LsWidget::setOrientation(LsWidget::Portrait);
        m_View->setAttribute(Qt::WA_Maemo5PortraitOrientation);
        m_WasPortrait = true;
    }else{
        m_View->setAttribute(Qt::WA_Maemo5AutoOrientation);
        connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onOrientationChanged(int)));
        m_WasPortrait = !LsWidget::isPortrait();
    }
#else
    QString orientation = LockScreen::Settings->value("Main/Orientation", "auto").toString().toLower();
    if (orientation == "landscape"){
        LsWidget::setOrientation(LsWidget::Landscape);
        m_WasPortrait = false;
    }else if (orientation == "portrait"){
        LsWidget::setOrientation(LsWidget::Portrait);
        m_WasPortrait = true;
    }else{
        m_WasPortrait = false;
    }
#endif
    setBackground();
    if (LsWidget::isPortrait()){
        m_Scene.setSceneRect(0, 0, 480, 800);
        m_View->setFixedSize(480, 800);
    }else{
        m_Scene.setSceneRect(0, 0, 800, 480);
        m_View->setFixedSize(800, 480);
    }
    createWidgets();

#ifdef Q_WS_MAEMO_5
    QDBusConnection conn = QDBusConnection::systemBus();
    conn.connect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                 MCE_DISPLAY_SIG, this, SLOT(onDisplayStateChangedSlot(QString)));

    conn.connect("",
                 MCE_SIGNAL_PATH,
                 MCE_SIGNAL_IF,
                 MCE_TKLOCK_MODE_SIG, this, SLOT(onTkLockChanged(QString)));

    conn.connect("",
                 MCE_SIGNAL_PATH,
                 MCE_SIGNAL_IF,
                 MCE_CALL_STATE_SIG, this, SLOT(onCallStateChanged(QString)));

    conn.connect("com.nokia.csd.Call",
                 "/com/nokia/csd/call",
                 "com.nokia.csd.Call",
                 "Coming", this, SLOT(onIncomingCallSlot(QDBusMessage)));

    conn.connect("com.nokia.system_ui",
                 "/com/nokia/system_ui/signal",
                 "com.nokia.system_ui.signal",
                 "alarm_dialog_status", this, SLOT(onAlarmDialogStatusChanged(QDBusMessage)));

/*    conn.connect("",
                 "/org/freedesktop/Hal/devices/platform_kb_lock",
                 "org.freedesktop.Hal.Device",
                 "PropertyModified", this, SLOT(onUnlockButton(QDBusMessage)));

    conn.connect("",
                 "/org/freedesktop/Hal/devices/platform_slide",
                 "org.freedesktop.Hal.Device",
                 "PropertyModified", this, SLOT(onUnlockButton(QDBusMessage)));
*/

    m_View->setAttribute(Qt::WA_Maemo5NonComposited);
#endif    

    setHildonFlags();

    /*if (LockScreen::Debug){
        QFontDatabase fDb;
        qDebug() << "Font families available:";
        foreach (const QString family, fDb.families()){
            qDebug() << " " << family;
        }
    }*/
}

void LockScreen::setHildonFlags()
{
#ifdef Q_WS_MAEMO_5
    quint32 enable = {10};
    Atom winStackingAtom = XInternAtom(QX11Info::display(), "_HILDON_STACKING_LAYER", false);
    XChangeProperty(QX11Info::display(), m_View->winId(), winStackingAtom, XA_CARDINAL, 32, PropModeReplace, (uchar*) &enable, 1);

    enable = 1;
    Atom winDNDAtom = XInternAtom(QX11Info::display(), "_HILDON_DO_NOT_DISTURB", false);
    XChangeProperty(QX11Info::display(), m_View->winId(), winDNDAtom, XA_INTEGER, 32, PropModeReplace, (uchar*) &enable, 1);

    enable = 1;
    Atom winNTAtom = XInternAtom(QX11Info::display(), "_HILDON_WM_ACTION_NO_TRANSITIONS", false);
    XChangeProperty(QX11Info::display(), m_View->winId(), winNTAtom, XA_CARDINAL, 32, PropModeReplace, (uchar*) &enable, 1);
#endif
}

bool LockScreen::hide()
{
    if (!m_View->isVisible())
        return false;

    m_View->hide();
    QTimer::singleShot(250, this, SLOT(stopUpdate()));
    return true;
}

bool LockScreen::saveScreenshot(QString fileName)
{
    qDebug() << "Saving screenshot to:" << fileName;
    if (LsWidget::getOrientation() == LsWidget::Auto)
        setSizeAndPosition();
    m_View->update();
    QPixmap pixMap = QPixmap::grabWidget(m_View);
    return pixMap.save(fileName);
}

bool LockScreen::show()
{
    if (m_View->isVisible())
        return false;

#ifdef Q_WS_MAEMO_5
    m_View->showFullScreen();    
#else
    m_View->show();
#endif    
    if (LsWidget::getOrientation() == LsWidget::Auto)
        setSizeAndPosition();
    m_View->update();
    return true;
}

LsWidget* LockScreen::createWidget(QString type, QString name)
{
    qDebug() << "Creating widget" << name << "of type" << type;
    if (type == "Rect")
        return new LsRect(name);
    else if (type == "DateTime")
        return new LsDateTime(name);
    else if (type == "Battery")
        return new LsBattery(name);
    else if (type == "UnlockButton")
        return new LsUnlockButton(name);
    else if (type == "SlideToUnlockButton")
        return new LsSlideToUnlock(name);
    else if (type == "MediaPlayPauseButton")
        return new LsMediaPlayPauseButton(name, m_Player);
    else if (type == "MediaNextButton")
        return new LsMediaNextButton(name, m_Player);
    else if (type == "MediaBackButton")
        return new LsMediaPreviousButton(name, m_Player);
    else if (type == "MediaMetadata")
        return new LsMediaMetadata(name, m_Player);
    else if (type == "MediaAlbumArt")
        return new LsMediaAlbumArt(name, m_Player);
    else if (type == "MediaTimeLabel")
        return new LsMediaTimeLabel(name, m_Player);
    else if (type == "Image")
        return new LsImage(name);
    else if (type == "Notification")
        return new LsNotification(name);
    else if (type == "Text")
        return new LsText(name);
    else if (type == "Weather")
        return new LsWeather(name);
    else if (type == "Wifi" || type == "Internet")
        return new LsWifi(name);
    else if (type == "OperatorName")
        return new LsOperatorName(name);
    else if (type == "SignalStrength")
        return new LsPhoneSignal(name);
    else if (type == "RadioMode")
        return new LsRadioMode(name);
    else if (type == "Bluetooth")
        return new LsBluetooth(name);
    else if (type == "Profile")
        return new LsProfile(name);
    else if (type == "TextCommand")
        return new LsTextCommand(name);
    return NULL;
}

void LockScreen::deleteWidgets()
{
    foreach(const QGraphicsItem* gi, m_Scene.items())
        delete gi;
    m_Scene.clear();
}

void LockScreen::setBackground()
{
    bool isPortrait = LsWidget::isPortrait();

    QString bkg;
    if (isPortrait)
        bkg = LockScreen::Settings->value("Main/BackGroundImageP", "/opt/usr/share/themes/alpha/backgrounds/lockslider.png").toString();
    else
        bkg = LockScreen::Settings->value("Main/BackGroundImageL", "/opt/usr/share/themes/alpha/backgrounds/lockslider.png").toString();

    if (m_BkgPath != bkg){
        qDebug() << "Setting background" << bkg;
        if (!bkg.isEmpty()){
            if (bkg.endsWith(".gif", Qt::CaseInsensitive)){
                if (m_LabelBackground == NULL){
                    m_LabelBackground = new QLabel();
                    m_LabelBackground->setFrameStyle(QFrame::NoFrame);
                    m_LabelBackground->setContentsMargins(0,0,0,0);
                    m_LabelBackground->setAlignment(Qt::AlignCenter);
                    m_LabelBackground->setScaledContents(false);
                    m_Proxy = m_Scene.addWidget(m_LabelBackground);
                    m_Proxy->setPos(0,0);
                }
                if (m_Movie != NULL){
                    m_LabelBackground->setMovie(NULL);
                    m_Movie->stop();
                    delete m_Movie;
                    m_Movie = NULL;
                }
                m_Movie = new QMovie(bkg);
                m_Movie->start();
                m_LabelBackground->setMovie(m_Movie);                
                m_Scene.setBackgroundBrush(QBrush(QColor(0,0,0,0)));
            }else{
                if (m_LabelBackground && m_Movie){
                    m_LabelBackground->setMovie(NULL);
                    delete m_Movie;
                    m_Movie = NULL;
                }
                m_Scene.setBackgroundBrush(QPixmap(bkg));
            }
        }else
            m_Scene.setBackgroundBrush(QBrush(QColor(0,0,0,0)));

        m_BkgPath = bkg;
    }
}

void LockScreen::createWidgets()
{
    QSettings settings(LockScreen::Settings->fileName(), QSettings::IniFormat);

    int size = settings.beginReadArray("Main/Widgets");
    for (int i=0; i<size; i++){
        settings.setArrayIndex(i);

        QString w = settings.value("Name").toString();
        QStringList elem = w.split('.');
        if (elem.count() == 2){
            LsWidget* lw = createWidget(elem.at(0), elem.at(1));
            if (lw != NULL)
                m_Scene.addItem(lw);
        }
    }
    settings.endArray();
    startUpdate();
}

void LockScreen::onDisplayStateChangedSlot(QString displayState)
{
#ifdef Q_WS_MAEMO_5
    qDebug() << "Display state changed:" << displayState;
    m_displayState = displayState;
    if (displayState == MCE_DISPLAY_OFF_STRING) {
        if (m_tklockState == MCE_TK_LOCKED && !m_isDialogActive)
            show();
        stopUpdate();
    }else if (displayState == MCE_DISPLAY_ON_STRING) {
        if (m_tklockState == MCE_TK_LOCKED && !m_isDialogActive){            
            if (LsWidget::getOrientation() == LsWidget::Auto)
                onOrientationChanged(0);
            startUpdate();
        }
    }
#endif
}

void LockScreen::onTkLockChanged(QString locked)
{
    if (m_tklockState == locked)
        return;

    qDebug() << "Lock state changed:" << locked;
    m_tklockState = locked;
#ifdef Q_WS_MAEMO_5
    if (locked == MCE_TK_UNLOCKED) {
        hide();
    } else if (locked == MCE_TK_LOCKED && !m_isDialogActive) {
        if (show())
            startUpdate();
    }
#endif
}

void LockScreen::onCallStateChanged(QString state)
{
    qDebug() << "Call state changed:" << state;
#ifdef Q_WS_MAEMO_5
    if (state != MCE_CALL_STATE_NONE)
        m_isDialogActive = true;
    else{
        m_isDialogActive = false;
        if (m_tklockState == MCE_TK_LOCKED && m_displayState == MCE_DISPLAY_OFF_STRING)
            show();
    }
#else
    Q_UNUSED(state)
#endif
}

#ifdef Q_WS_MAEMO_5
/*void LockScreen::onUnlockButton(QDBusMessage)
{
    if (m_tklockState == MCE_TK_LOCKED)
        hide();
}*/

void LockScreen::onAlarmDialogStatusChanged(QDBusMessage msg)
{
    int status = 0;
    QVariant val = msg.arguments()[0];
    status = val.toInt();

    qDebug() << "Alarm dialog status changed" << status;
    m_isDialogActive = status == 5;
    if (m_isDialogActive){
        hide();
    }else if (m_tklockState == MCE_TK_LOCKED && m_displayState == MCE_DISPLAY_OFF_STRING)
        show();
}

void LockScreen::onIncomingCallSlot(QDBusMessage)
{
    qDebug() << "Incoming call";
    hide();
}
#endif

void LockScreen::startUpdate()
{
    qDebug() << "Start update";
    if (m_Movie)
        m_Movie->start();

    foreach (QGraphicsItem* item, m_Scene.items()){
        LsWidget* w = dynamic_cast<LsWidget*>(item);
        if (w != NULL)
            w->startUpdate();
    }
}

void LockScreen::stopUpdate()
{
    qDebug() << "Stop update";
    if (m_Movie)
        m_Movie->stop();

    foreach (QGraphicsItem* item, m_Scene.items()){
        LsWidget* w = dynamic_cast<LsWidget*>(item);
        if (w != NULL)
            w->stopUpdate();
    }
}

#ifdef Q_WS_MAEMO_5
void LockScreen::onOrientationChanged(int)
{
    if (!m_View->isVisible())
        return;

    setSizeAndPosition();
}
#endif

void LockScreen::setSizeAndPosition()
{
    bool isPortrait = LsWidget::isPortrait();
    if (m_WasPortrait != isPortrait){
        m_WasPortrait = isPortrait;

        qDebug() << "Orientation changed portrait=" << isPortrait;
        setBackground();
        if (isPortrait){
            m_Scene.setSceneRect(0, 0, 480, 800);        
            m_View->setFixedSize(480, 800);
            if (m_LabelBackground){
                m_LabelBackground->setFixedSize(480, 800);
                m_Proxy->setPos(0,0);
            }
        }else{
            m_Scene.setSceneRect(0, 0, 800, 480);
            m_View->setFixedSize(800, 480);
            if (m_LabelBackground){
                m_LabelBackground->setFixedSize(800, 480);
                m_Proxy->setPos(0,0);
            }
        }

        foreach(QGraphicsItem* item, m_Scene.items()){
            LsWidget* w = dynamic_cast<LsWidget*>(item);
            if (w != NULL){
                w->onOrientationChanged();
                w->setSizeAndPosition(QSize(w->size().width(), w->size().height()),
                                      QPoint(w->pos().x(), w->pos().y()));
            }
        }
        m_View->update();
    }
}

bool LockScreen::isLocked()
{
    bool res = false;
#ifdef Q_WS_MAEMO_5
    QDBusConnection conn = QDBusConnection::systemBus();
    QDBusInterface device("com.nokia.mce",
                          "/com/nokia/mce/request", "com.nokia.mce.request", conn);
    QDBusMessage msg = device.call("get_tklock_mode");

    if (msg.arguments().count() > 0 ){
        QString state = msg.arguments()[0].toString();
        qDebug() << "State:" << state;
        res = state == "locked";
    }
#else
    res = true;
#endif
    return res;
}
