//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lsnotification.h"
#include <QUuid>
#include "lockscreen.h"

LsNotification::LsNotification(QString name) :
        LsWidget(name.isEmpty() ? "Notification": name)
{
    applySettings();
    hide();

#ifdef Q_WS_MAEMO_5
    QString dbName = QString("/home/user/.config/hildon-desktop/notifications.db");
    QString connName = QUuid::createUuid();
    m_Db = QSqlDatabase::addDatabase("QSQLITE", connName);
    m_Db.setConnectOptions("QSQLITE_OPEN_READONLY");
    m_Db.setDatabaseName(dbName);
    if (!m_Db.open()){
        qWarning() << QString("WARNING: error opening notifications database");
    }
#endif
}

void LsNotification::paint(QPainter *painter, const QStyleOptionGraphicsItem*,
                           QWidget*)
{
    QRectF rect(boundingRect());
    QFontMetrics fm(m_Font);

    switch (m_Alignment){
    case Qt::AlignBottom:
        {
            if (!m_Icon.isNull()){
                painter->drawPixmap(size().width() / 2 - m_Icon.width() / 2, 0, m_Icon);
                rect.setTop(rect.top() + m_Icon.height() + MINIMUM_TEXT_MARGIN);
            }

            painter->setFont(m_Font);
            painter->setPen(m_Color);
            painter->drawText(rect, Qt::AlignTop | Qt::AlignHCenter | Qt::TextSingleLine, m_Text);
        }
        break;
    case Qt::AlignTop:
        {
            int y = 0;

            painter->setFont(m_Font);
            painter->setPen(m_Color);
            painter->drawText(rect, Qt::AlignTop | Qt::AlignHCenter | Qt::TextSingleLine, m_Text);

            QSize tSize = fm.size(Qt::TextSingleLine, m_Text);
            y += tSize.height() + MINIMUM_TEXT_MARGIN;
            if (!m_Icon.isNull()){
                painter->drawPixmap(size().width() / 2 - m_Icon.width() / 2, y, m_Icon);
            }
        }
        break;
    case Qt::AlignLeft:
        {
            int x = 0;

            painter->setFont(m_Font);
            painter->setPen(m_Color);
            painter->drawText(rect, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, m_Text);

            QSize tSize = fm.size(Qt::TextSingleLine, m_Text);
            x += tSize.width() + MINIMUM_TEXT_MARGIN;
            if (!m_Icon.isNull()){
                painter->drawPixmap(x, size().height() / 2 - m_Icon.height() / 2, m_Icon);
            }
        }
        break;
    case Qt::AlignRight:
        {
            if (!m_Icon.isNull()){
                painter->drawPixmap(0, size().height() / 2 - m_Icon.height() / 2, m_Icon);
                rect.setLeft(rect.left() + m_Icon.width() + MINIMUM_TEXT_MARGIN);
            }

            painter->setFont(m_Font);
            painter->setPen(m_Color);
            painter->drawText(rect, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, m_Text);
        }
        break;
    default:
        break;
    }

}

void LsNotification::applySettings()
{
    setSizeAndPosition(QSize(100, 100), QPoint(0, 0));

    QString type = getSetting("Type", "call").toString();
    if (type == "call"){
        m_Type = NotificationMissedCall;
        m_BaseText = getSetting("Text", tr("%1 missed calls")).toString();
        m_Icon = QPixmap(getSetting("Icon", "/opt/qtlockscreen/default/call.png").toString());
    }else if (type == "sms"){
        m_Type = NotificationNewSMS;
        m_BaseText = getSetting("Text", tr("%1 unread messages")).toString();
        m_Icon = QPixmap(getSetting("Icon", "/opt/qtlockscreen/default/message.png").toString());
    }else if (type == "email"){
        m_Type = NotificationNewEmail;
        m_BaseText = getSetting("Text", tr("%1 new emails")).toString();
        m_Icon = QPixmap(getSetting("Icon", "/opt/qtlockscreen/default/email.png").toString());
    }else if (type == "im"){
        m_Type = NotificationNewChat;
        m_BaseText = getSetting("Text", tr("%1 new IM")).toString();
        m_Icon = QPixmap(getSetting("Icon", "/opt/qtlockscreen/default/message.png").toString());
    }

    m_Alignment = getAlignmentSetting("TextAlignment", Qt::AlignBottom);
    m_Color = getColorSetting("Color", QColor(255,255,255));
    m_Font = getFontSetting("Font", QFont("", 14, QFont::Normal, false));
}

void LsNotification::stopUpdate()
{

}

void LsNotification::startUpdate()
{
    if (LockScreen::ScreenshotMode)
        updateNotification();
    else
        QTimer::singleShot(250, this, SLOT(updateNotification()));
}

void LsNotification::updateNotification()
{
    int sms = 0;
    int email = 0;
    int call = 0;
    int chat = 0;

#ifdef Q_WS_MAEMO_5
    if (m_Db.isOpen()){
        QSqlQuery qry(m_Db);
        QString sql = "SELECT H.value, H2.value, COUNT(*)   FROM notifications N, hints H, hints \
                       H2 WHERE N.id=H.nid AND H.id='category' and H2.id = 'time' and H2.nid = H.nid \
                       GROUP BY  H.value ORDER BY H2.value;";
        if (!qry.prepare(sql)){
            qWarning() << "Error preparing query:" << qry.lastError().text();
        }else{
            if (!qry.exec()){
                qWarning() << "Error executing query:" << qry.lastError().text();
            }else{
                QSqlRecord rec;
                while(qry.next()){
                    rec = qry.record();
                    QString name = rec.value(0).toString();
                    qDebug() << name << rec.value(2).toInt();
                    //sms-message
                    //auth-request
                    //chat-invitation
                    //chat-message
                    //missed-call
                    //email-message
                    //voice-mail
                    if (name == "email-message")
                        email = rec.value(2).toInt();
                    else if (name == "sms-message")
                        sms = rec.value(2).toInt();
                    else if (name == "missed-call")
                        call = rec.value(2).toInt();
                    else if (name == "chat-message")
                        chat = rec.value(2).toInt();
                }
            }
        }
    }

    if (LockScreen::Debug){
        sms = 1;
        email = 1;
        call = 1;
        chat = 1;
    }
#else
    sms = 1;
    email = 1;
    call = 1;
    chat = 1;
#endif

    qDebug() << "Emails:" << email;
    qDebug() << "Calls:" << call;
    qDebug() << "SMS:" << sms;
    qDebug() << "IM:" << chat;

    int value = 0;
    switch (m_Type){
    case NotificationMissedCall:
        value = call;        
        break;
    case NotificationNewSMS:
        value = sms;
        break;
    case NotificationNewEmail:
        value = email;
        break;
    case NotificationNewChat:
        value = chat;
        break;
    default:
        break;
    }

    m_Text = m_BaseText.arg(value);
    if (value == 0)
        hide();
    else
        show();
    update();
}
