#include "thememanager.h"
#include "ui_thememanager.h"

ThemeManager::ThemeManager(QWidget *parent) :
    QDialog(parent),
    m_ui(new Ui::ThemeManager)
{
    m_ui->setupUi(this);
    m_ui->commentTextEdit->setProperty("FingerScrollable", true);
    this->setWindowTitle("Theme manager");
    m_ui->downloadPushButton->setDisabled(true);
    m_ui->listWidget->addItem("Device");
    m_ui->listWidget->addItem("Database");
    m_ui->previousPushButton->setDisabled(true);
    m_ui->deletePushButton->setDisabled(true);
    connect(m_ui->listWidget,SIGNAL(itemClicked(QListWidgetItem*)),this,SLOT(updateListWidget(QListWidgetItem*)));
    deviceThemes = false;
    QUrl url;
    url.setHost("mercury.wipsl.com");
    url.setPath("/qtirreco/webdb/");
    url.setPort(80);

    client = new QxtXmlRpcClient(this, url);
    connect(client,SIGNAL(finish(int,QVariant)),this,SLOT(processReturnValue(int,QVariant)));
    connect(client,SIGNAL(networkError(int,QString)),this,SLOT(error(int,QString)));
    connect(client,SIGNAL(fault(int,int,QString)),this,SLOT(fault(int,int,QString)));
    connect(m_ui->versionComboBox,SIGNAL(activated(int)),this,SLOT(themeVersionChanged(int)));
    connect(m_ui->versionComboBox,SIGNAL(activated(QString)),this,SLOT(themeVersionChanged(QString)));
    connect(m_ui->previousPushButton,SIGNAL(clicked()),this,SLOT(clearAllData()));

    themeNameIndex = 0;
    themeVersionIndex = 0;
    currentVersionRow = 0;
    themeButtonIndex = 0;
    themeBgIndex = 0;
    downloadId = 0;
    progressValue = 0;
    versionFound = false;
    ownTheme = false;
    uploadButtonIndex = 0;
    uploadBgIndex = 0;
    progressItemCount = 0;
    uploadItemsCount = 0;
    deviceThemeCount = 0;
}

ThemeManager::~ThemeManager()
{
    delete m_ui;
}

void ThemeManager::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        m_ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
void ThemeManager::themeVersionChanged(int clickedRow)
{
    currentVersionRow = clickedRow;
}
void ThemeManager::themeVersionChanged(QString version)
{
    int versionId = themeVersions.value(version);
    QList<QVariant> params;
    params.clear();
    params.append(versionId);
    themeVersions.clear();
    dbThemeVersionIdList.clear();
    dbThemeDateList.clear();
    themeDatas =  client->call("getThemeById",params);
}
void ThemeManager::on_newPushButton_clicked()
{
    ThemeCreatorDlg *themeCreatorDlg = new ThemeCreatorDlg(this);
    connect(themeCreatorDlg,SIGNAL(themeCreated()),this,SLOT(clearAllData()));
    themeCreatorDlg->exec();
    delete themeCreatorDlg;
}
void ThemeManager::updateListWidget(QListWidgetItem *item)
{
    if(deviceThemes)
    {
        m_ui->nameLabel->clear();
        m_ui->authorLabel->clear();
        m_ui->commentTextEdit->clear();
        m_ui->pictureLabel->clear();
        m_ui->versionComboBox->clear();
        for(int i=0;i<deviceThemeCount;i++)
        {
            if(item->text().contains(themeDirList.at(i),Qt::CaseInsensitive))
            {
                QFile file(themesPath+themeDirList.at(i)+"/theme.conf");
                tName = themeDirList.at(i);
                if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
                {
                    QMessageBox::warning(this,"Error","Couldn't load theme info!",
                    QMessageBox::NoButton,QMessageBox::NoButton);
                    return;
                }
                ownTheme = false;
                QTextStream in(&file);
                while (!in.atEnd())
                {
                    QString line = in.readLine();
                    readThemeConf(line);
                }
                m_ui->deletePushButton->setEnabled(true);
                file.close();
            }
        }
    }
    if(item->text() == "Device")
    {
        this->setWindowTitle("Themes on device");
        deviceThemes = true;
        m_ui->previousPushButton->setEnabled(true);
        m_ui->downloadPushButton->setText("Upload");
        m_ui->listWidget->clear();
        m_ui->versionComboBox->setDisabled(true);
        this->loadThemes();
        return;
    }
    if(item->text() == "Database")
    {
        this->setDisabled(true);
        this->setWindowTitle("Loading themes on database...");
        m_ui->previousPushButton->setEnabled(true);
        m_ui->downloadPushButton->setText("Download");
        this->getThemeIdsFromDb();
    }

    if(!deviceThemes && item->text() != "Database" && item->text() != "Device")
    {
        this->setDisabled(true);
        this->setWindowTitle("Loading theme data...");
        currentVersionRow = 0;
        themeButtonIndex = 0;
        themeBgIndex = 0;
        themeVersions.clear();
        dbThemeVersionIdList.clear();
        dbThemeNamesList.clear();
        dbThemeDateList.clear();
        dbButtonIdsList.clear();
        dbBackgroundIdsList.clear();
        QList<QVariant> params;
        params.clear();
        int selectedItemId = themeMap.value(item->text());
        params.append(selectedItemId);
        themeDatas = client->call("getThemeById",params);
    }


}
void ThemeManager::getThemeIdsFromDb()
{
    deviceThemes = false;
    QList<QVariant> params;
    params.clear();
    themeIds = client->call("getThemes",params);
}
void ThemeManager::processReturnValue(int Id,QVariant data)
{
   QStringList sList;
   sList.clear();
   sList = data.toStringList();
   if (Id == themeIds)
    {
        themeMap.clear();
        themeNameIndex = 0;
        dbThemeNamesList.clear();
        dbThemeIdList.clear();
        dbThemeIdList = sList;
        m_ui->listWidget->clear();

        QList<QVariant> params;
        dbThemeNamesList.clear();
        params.clear();
        QString idString = dbThemeIdList.at(0);
        bool ok;
        int idInt = idString.toInt(&ok,10);
        params.append(idInt);
        themeNames = client->call("getThemeNameById",params);

    }
    if (Id == themeNames)
    {
        dbThemeNamesList.append(sList);
        if(themeNameIndex == 0)
        {
            dbThemeNamesList.clear();
        }

        this->getThemeNamesByIds();
        themeNameIndex++;
    }
    if (Id == themeDatas)
    {
        m_ui->versionComboBox->clear();
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        m_ui->nameLabel->setText(name.toString());
        themeNameFromConf = name.toString();
        QVariant user = array.value("user");
        m_ui->authorLabel->setText(user.toString());
        themeAuthor = user.toString();
        QVariant comment = array.value("comment");
        m_ui->commentTextEdit->setText(comment.toString());
        themeComments = comment.toString();

        QVariant preview_button = array.value("preview_button");
        themePreviewButton = preview_button.toString();

        QVariant folder = array.value("folder");
        QVariant uploaded = array.value("uploaded");

        QVariant modified = array.value("modified");
        QVariant downloaded = array.value("downloaded");
        QVariant download_count = array.value("download_count");
        m_ui->downloadsLabel->setText(download_count.toString());

        QList<QVariant> nameParam;
        nameParam.clear();
        nameParam.append(m_ui->nameLabel->text());
        themeVersionIds = client->call("getThemeVersionsByName",nameParam);

    }
    if (Id == themeVersionIds)
    {
        themeVersionIndex = 0;
        dbThemeVersionIdList = sList;
        QList<QVariant> params;
        params.clear();
        QString idString = dbThemeVersionIdList.at(0);
        bool ok;
        int idInt = idString.toInt(&ok,10);
        params.append(idInt);
        themeDates = client->call("getThemeDateById",params);

    }
    if (Id == themeDates)
    {
        dbThemeDateList.append(sList);
        if(themeVersionIndex == 0)
        {
            dbThemeDateList.clear();
        }

        this->getThemeVersionIds();
        themeVersionIndex++;
    }
    if (Id == themePreviewImage)
    {
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        QPalette palette;
        palette.setBrush(QPalette::Window, previewPix.scaled(128,128,Qt::IgnoreAspectRatio,Qt::FastTransformation));
        m_ui->pictureLabel->setPalette(palette);
        m_ui->pictureLabel->setAutoFillBackground(true);
        this->setEnabled(true);
        this->setWindowTitle("Theme: " + m_ui->nameLabel->text());
        m_ui->downloadPushButton->setEnabled(true);

    }
    if (Id == themeButtonIds)
    {
        dbButtonIdsList = sList;
        this->getThemeButtonsByIds();

    }
    if (Id == buttonData) // COPY BUTTONS AND MAKE FOLDERS
    {
        themeButtonIndex++;
        progressValue += 100 / (dbButtonIdsList.count()+2);
        emit progressState(progressValue);
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        QVariant allow_text = array.value("allow_text");
        QVariant text_format_up = array.value("text_format_up");
        QVariant text_format_down = array.value("text_format_down");
        QVariant text_padding = array.value("text_padding");
        QVariant h_align = array.value("h_align");
        QVariant v_align = array.value("v_align");
        QVariant image_up_hash = array.value("image_up_hash");
        unpressedHash = image_up_hash.toString();
        QVariant image_up_name = array.value("image_up_name");
        QVariant image_down_hash = array.value("image_down_hash");
        pressedHash = image_down_hash.toString();
        QVariant image_down_name = array.value("image_down_name");
        QVariant folder = array.value("folder");

        QDir dir;
        dir.mkpath(themesPath+m_ui->nameLabel->text().toLower()+"/buttons/"+folder.toString().toLower());
        buttonPath = QString(themesPath+m_ui->nameLabel->text().toLower()+"/buttons/"+folder.toString().toLower());
        if(themeButtonIndex == 1)
        {
            QFile themeConf(themesPath+m_ui->nameLabel->text().toLower()+"/theme.conf");
            if (!themeConf.open(QIODevice::ReadWrite | QIODevice::Text))
            return;
            QTextStream in(&themeConf);
            in << "[theme]\n";
            in << "name="+ themeNameFromConf +"\n";
            in << "source=database\n";
            in << "author="+ themeAuthor +"\n";
            in << "comments="+ themeComments +"\n";
            in << "version="+ themeVersion +"\n";
            in << "preview_button=" + themePreviewButton;
            themeConf.close();
        }


        
        QFile buttonConf(buttonPath+"/button.conf");
        if (!buttonConf.open(QIODevice::ReadWrite | QIODevice::Text))
        return;
        QTextStream in2(&buttonConf);
        in2 << "[theme-button]\n";
        in2 << "name="+name.toString()+"\n";
        in2 << "allow-text="+allow_text.toString()+"\n";
        in2 << "up=unpressed.png\n";
        in2 << "down=pressed.png\n";
        if(allow_text.toBool() == true)
        {
            in2 << "text-format-up="+text_format_up.toString()+"\n";
            in2 << "text-format-down="+text_format_down.toString()+"\n";
        }

        buttonConf.close();
        // Get button imagedatas and make .png files
        this->getUnpressedImage();
    }
    if (Id == unpressedToFile) // Name always unpressed.png
    {
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(buttonPath+"/unpressed.png","PNG");
        this->getPressedImage();
    }
    if (Id == pressedToFile) // Name always pressed.png
    {
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(buttonPath+"/pressed.png","PNG");
        this->getThemeButtonsByIds();
    }
    if (Id == themeBgIds)
    {
        dbBackgroundIdsList = sList;
        if(dbBackgroundIdsList.count() > 0 && !dbBackgroundIdsList.contains("0"))
        {
            this->setWindowTitle("Downloading theme backgrounds...");
            this->getThemeBackgroundsByIds();
        }
        else
        {
             progressValue = 0;
             emit themeLoaded();
             QMessageBox::information(this,"Theme manager","Theme downloaded succesfully.",
                                     QMessageBox::NoButton,QMessageBox::NoButton);
             this->setEnabled(true);
             this->setWindowTitle("Themes available on database");
             dbBackgroundIdsList.clear();
             m_ui->downloadPushButton->setDisabled(true);
             m_ui->authorLabel->clear();
             m_ui->commentTextEdit->clear();
             m_ui->downloadsLabel->clear();
             m_ui->nameLabel->clear();
             m_ui->pictureLabel->setAutoFillBackground(false);
             m_ui->versionComboBox->clear();
        }
    }
    if (Id == bgData)
    {
        themeBgIndex++;
        progressValue += 100 / (dbButtonIdsList.count()+2);
        emit progressState(progressValue);
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        QVariant image_hash = array.value("image_hash");
        bgHash = image_hash.toString();
        QVariant image_name = array.value("image_name");
        QVariant folder = array.value("folder");

        QDir dir;
        dir.mkpath(themesPath + m_ui->nameLabel->text().toLower()+"/bg/"+folder.toString());
        bgPath = QString(themesPath + m_ui->nameLabel->text().toLower()+"/bg/"+folder.toString());

        QFile bgConf(bgPath+"/bg.conf");
        if (!bgConf.open(QIODevice::ReadWrite | QIODevice::Text))
        return;
        QTextStream in(&bgConf);
        in << "[theme-bg]\n";
        in << "name="+name.toString()+"\n";
        in << "image=image.png";
        bgConf.close();
        // Get background imagedatas and make .png files
        this->getBgImage();
    }
    if (Id == bgToFile)
    {
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(bgPath+"/image.png","PNG");
        this->getThemeBackgroundsByIds();
    }
    if (Id == uploadTheme)
    {
        bool ok;
        themeIdToUpload = data.toInt(&ok);
        this->uploadThemeButton();
    }
    if (Id == addButtonToTheme)
    {
        progressItemCount += 100 / uploadItemsCount;
        emit progressState(progressItemCount);
        uploadButtonIndex++;
        this->uploadThemeButton();
    }
    if (Id == addBgToTheme)
    {
        progressItemCount += 100 / uploadItemsCount;
        emit progressState(progressItemCount);
        uploadBgIndex++;
        this->uploadThemeBg();
    }
    if (Id == setThemeDownloadable)
    {
        bool upload = data.toBool();
        if(upload)
        {
            QFile confFile(themesPath + themeToUploadDir + "/theme.conf");
            if (!confFile.open(QIODevice::ReadWrite | QIODevice::Text))
            {
                QMessageBox::warning(this,"Error","Couldn't edit theme config!",
                QMessageBox::NoButton,QMessageBox::NoButton);
                return;
            }
            QTextStream in(&confFile);

            QString fileData = in.readAll();
            confFile.remove();

            fileData.replace("source=own","source=database",Qt::CaseInsensitive);

            QFile newConfFile(themesPath + themeToUploadDir + "/theme.conf");
            if (!newConfFile.open(QIODevice::ReadWrite | QIODevice::Text))
            {
                QMessageBox::warning(this,"Error","Couldn't edit theme config!",
                QMessageBox::NoButton,QMessageBox::NoButton);
                return;
            }
            QTextStream in2(&newConfFile);
            in2 << fileData;

            newConfFile.close();

            emit themeUploaded();
            this->setEnabled(true);
            QMessageBox::information(this,"Theme manager","Theme uploaded succesfully.",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
            this->setWindowTitle("Themes on device");
            progressItemCount = 0;
            uploadItemsCount = 0;
            m_ui->downloadPushButton->setDisabled(true);

        }
        if(!upload)
        {
            QMessageBox::warning(this,"Error","Theme upload failed!",
                QMessageBox::NoButton,QMessageBox::NoButton);
            return;
        }
    }
}
void ThemeManager::getUnpressedImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(unpressedHash);
    QString unpressedName = "unpressed.png";
    params.append(unpressedName);
    unpressedToFile = client->call("getImageData",params);
}
void ThemeManager::getPressedImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(pressedHash);
    QString pressedName = "pressed.png";
    params.append(pressedName);
    pressedToFile = client->call("getImageData",params);
}
void ThemeManager::getBgImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(bgHash);
    QString pressedName = "image.png";
    params.append(pressedName);
    bgToFile = client->call("getImageData",params);
}
void ThemeManager::getThemeButtonsByIds() // Padding in button config not used ?
{
     if(themeButtonIndex < dbButtonIdsList.count())
        {
            QList<QVariant> params;
            params.clear();
            QString idString = dbButtonIdsList.at(themeButtonIndex);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            params.append(idInt);
            buttonData = client->call("getButtonById",params);
        }
    else
    {
        dbBackgroundIdsList.clear();
        QList<QVariant> params;
        params.clear();
        params.append(downloadId);
        themeBgIds = client->call("getBackgrounds",params);
    }
}
void ThemeManager::getThemeBackgroundsByIds()
{
    if(themeBgIndex < dbBackgroundIdsList.count())
    {
         QList<QVariant> params;
         params.clear();
         QString idString = dbBackgroundIdsList.at(themeBgIndex);
         bool ok;
         int idInt = idString.toInt(&ok,10);
         params.append(idInt);
         bgData = client->call("getBgById",params);
    }
    else
    {
         progressValue = 0;

         QMessageBox::information(this,"Theme manager","Theme downloaded succesfully.",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
         this->setEnabled(true);
         this->setWindowTitle("Themes available on database");
         themeBgIndex = 0;
         dbBackgroundIdsList.clear();
         m_ui->downloadPushButton->setDisabled(true);
         m_ui->authorLabel->clear();
         m_ui->commentTextEdit->clear();
         m_ui->downloadsLabel->clear();
         m_ui->nameLabel->clear();
         m_ui->pictureLabel->setAutoFillBackground(false);
         m_ui->versionComboBox->clear();
         emit themeLoaded();

    }
}
void ThemeManager::on_downloadPushButton_clicked()
{
    if(m_ui->downloadPushButton->text() == "Download")
    {
        QString version;
        versionFound = false;
        QString themeName = m_ui->listWidget->currentItem()->text().toLower();
        QDir dir(themesPath + themeName);
        if(dir.exists())
        {
            // Read version from conf file
            QFile file(themesPath+themeName+"/theme.conf");
            if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
            {
                QMessageBox::warning(this,"Error with theme config","Couldn't load " + themeName + " theme config file!" +
                             " Theme will be deleted, please download theme again from theme manager.",
                             QMessageBox::NoButton,QMessageBox::NoButton);
                this->deleteTheme(themeName);
                this->clearAllData();
                return;
            }
            QTextStream in(&file);
            while (!in.atEnd() && !versionFound)
            {
                QString line = in.readLine();
                version = readVersion(line);
            }
            file.close();
            if(version == "Version not found")
            {
                 QMessageBox::warning(this,"Error","Version not found!",
                 QMessageBox::NoButton,QMessageBox::NoButton);
                 return;
            }
            themeToDelete = themeName;
            QDialog *replace = new QDialog(this);
            replace->setWindowTitle("Replace theme");
            replace->resize(800,150);
            QLabel *label = new QLabel(replace);
            label->setAlignment(Qt::AlignHCenter);
            label->setGeometry(0,20,800,45);
            label->setFont(QFont("sans-serif",14));
            label->setText("Are you sure you want to replace " + m_ui->listWidget->currentItem()->text() + " version: "
                            + version + "\nwith version: " + m_ui->versionComboBox->currentText() + "?");
            QPushButton *replaceButton = new QPushButton(replace);
            replaceButton->setGeometry(280,75,225,50);
            replaceButton->setText("Replace");
            replaceButton->show();
            connect(replaceButton,SIGNAL(clicked()),replace,SLOT(deleteLater()));
            connect(replaceButton,SIGNAL(clicked()),this,SLOT(replaceTheme()));
            replace->exec();
        }
        else
        {
            progress = new QProgressBar(this);
            progress->setGeometry(650,10,128,64);
            progress->setRange(0,100);
            connect(this,SIGNAL(progressState(int)),progress,SLOT(setValue(int)));
            connect(this,SIGNAL(themeLoaded()),progress,SLOT(deleteLater()));
            progress->show();
            this->setWindowTitle("Loading theme buttons...");
            this->setDisabled(true);
            dbButtonIdsList.clear();
            downloadId = themeVersions.value(m_ui->versionComboBox->currentText());
            QList<QVariant> params;
            params.clear();
            params.append(downloadId);
            themeButtonIds = client->call("getButtons",params);
        }
    }
    if(m_ui->downloadPushButton->text() == "Upload")
    {
        uploadButtonIndex = 0;
        uploadBgIndex = 0;
        // First login to DB
        LoginDbDlg *loginDlg = new LoginDbDlg(this);
        connect(loginDlg,SIGNAL(loginReady(QString,QString)),this,SLOT(startUpload(QString,QString)));
        loginDlg->exec();
        delete loginDlg;
    }
}
QString ThemeManager::readVersion(QString line)
{
        if(line.contains("version=", Qt::CaseInsensitive))
        {
            QString version = line.remove(0,8);
            versionFound = true;
            return version;
        }
        else
        {
            return QString("Version not found");
        }
}
void ThemeManager::getThemeNamesByIds()
{
    if(themeNameIndex < dbThemeIdList.count())
        {
            QList<QVariant> params;

            params.clear();
            QString idString = dbThemeIdList.at(themeNameIndex);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            params.append(idInt);
            themeNames = client->call("getThemeNameById",params);
        }
    else
    {
        m_ui->listWidget->addItems(dbThemeNamesList);
        for(int i=0;i<dbThemeNamesList.count();i++)
        {
            QString idString = dbThemeIdList.at(i);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            themeMap.insert(dbThemeNamesList.at(i),idInt);
        }
        this->setEnabled(true);
        this->setWindowTitle("Themes available on database");
    }
}
void ThemeManager::getThemeVersionIds()
{
     if(themeVersionIndex < dbThemeVersionIdList.count())
        {
            QList<QVariant> params;

            params.clear();
            QString idString = dbThemeVersionIdList.at(themeVersionIndex);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            params.append(idInt);
            themeDates = client->call("getThemeDateById",params);
        }
    else
    {
        m_ui->versionComboBox->addItems(dbThemeDateList);
        if(m_ui->versionComboBox->count() <= 1)
        {
            m_ui->versionComboBox->setDisabled(true);
        }
        else m_ui->versionComboBox->setEnabled(true);

        for(int i=0;i<dbThemeDateList.count();i++)
        {
            QString idString = dbThemeVersionIdList.at(i);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            themeVersions.insert(dbThemeDateList.at(i),idInt);
        }
        m_ui->versionComboBox->setCurrentIndex(currentVersionRow);
        QList<QVariant> params;
        params.clear();
        int id = themeVersions.value(m_ui->versionComboBox->currentText());
        themeVersion = m_ui->versionComboBox->currentText();
        params.append(id);
        themePreviewImage = client->call("getPreviewButton",params);
    }
}
void ThemeManager::error(int i ,QString string)
{
    QMessageBox::warning(this,"Error",""+ string,
                     QMessageBox::NoButton,QMessageBox::NoButton);
}
void ThemeManager::fault(int i,int j,QString string)
{
    QMessageBox::warning(this,"Fault",""+ string,
                     QMessageBox::NoButton,QMessageBox::NoButton);
}
void ThemeManager::loadThemes()
{

    QDir *themeDirs = new QDir(themesPath);

    strList = themeDirs->entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    themeDirList = strList;
    deviceThemeCount = strList.count();

    for(int i=0;i<deviceThemeCount;i++)
    {
        QListWidgetItem *item = new QListWidgetItem();

        QString themeConf = themesPath + strList.at(i) + "/theme.conf";
        QFile file(themeConf);
        if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                 QMessageBox::warning(this,"Error with theme config","Couldn't load " + strList.at(i) + " theme config file!" +
                                 " Theme will be deleted, please download theme again from theme manager.",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
                this->deleteTheme(themeDirList.at(i));
                this->clearAllData();
                return;
            }
        else {
            QTextStream in(&file);
            while (!in.atEnd())
            {
                QString line = in.readLine();
                process_line(line);
            }

            item->setText(themeName);

            m_ui->listWidget->insertItem(i, item);
            m_ui->listWidget->show();
            file.close();
        }
    }
}
void ThemeManager::process_line(QString line)
{
        if(line.contains("name=", Qt::CaseInsensitive))
        {
            QString x = line;
            int count;
            int chars;
            count = line.length();
            chars = count - 5;
            themeName = x.right(chars);
        }
}
void ThemeManager::readThemeConf(QString line)
{
        if(line.contains("name=", Qt::CaseInsensitive))
            {
                m_ui->nameLabel->setText(line.remove(0,5));
                uploadThemeName = line;
            }
        if(line.contains("author=", Qt::CaseInsensitive))
            {
                m_ui->authorLabel->setText(line.remove(0,7));
            }
        if(line.contains("comments=", Qt::CaseInsensitive))
            {
                m_ui->commentTextEdit->setText(line.remove(0,9));
                uploadComments = line;
            }
        if(line.contains("version=", Qt::CaseInsensitive))
            {
                m_ui->versionComboBox->insertItem(0,line.remove(0,8));
            }
        if(line.contains("preview_button=", Qt::CaseInsensitive))
            {
                QString button = line.remove(0,15);
                button.replace(QChar(' '),QString("_"),Qt::CaseInsensitive); // If line contains space replace with _
                uploadPreviewName = button;
                QPalette palette;
                palette.setBrush(QPalette::Window, QPixmap(themesPath+tName+"/buttons/"+button.toLower()+"/unpressed.png").scaled(128,128,Qt::IgnoreAspectRatio,Qt::FastTransformation));
                m_ui->pictureLabel->setPalette(palette);
                m_ui->pictureLabel->setAutoFillBackground(true);
            }
        if(line.contains("source=own", Qt::CaseInsensitive)) // If theme not yet uploaded
            {
                m_ui->downloadPushButton->setEnabled(true);
                ownTheme = true;
            }
        if(!ownTheme)
            {
                m_ui->downloadPushButton->setDisabled(true);
            }
}
void ThemeManager::clearAllData()
{
    themeButtonIndex = 0;
    themeBgIndex = 0;
    this->setWindowTitle("Theme manager");
    m_ui->listWidget->clear();
    m_ui->listWidget->addItem("Device");
    m_ui->listWidget->addItem("Database");
    m_ui->previousPushButton->setDisabled(true);
    m_ui->downloadPushButton->setDisabled(true);
    m_ui->deletePushButton->setDisabled(true);
    deviceThemes = false;
    m_ui->nameLabel->clear();
    m_ui->authorLabel->clear();
    m_ui->commentTextEdit->clear();
    m_ui->pictureLabel->setAutoFillBackground(false);
    m_ui->downloadsLabel->clear();
    m_ui->versionComboBox->clear();
    strList.clear();
    themeDirList.clear();

    dbThemeIdList.clear();
    dbThemeNamesList.clear();
    themeMap.clear();
    themeVersions.clear();
    dbThemeVersionIdList.clear();
    dbThemeDateList.clear();
    currentVersionRow = 0;

             dbButtonIdsList.clear();
             dbBackgroundIdsList.clear();

}
void ThemeManager::on_deletePushButton_clicked()
{
    QStringList remoteNamesUsingThemeToBeDeleted;
    remoteNamesUsingThemeToBeDeleted.clear();
    usedThemes.clear();
    themeToDelete = m_ui->listWidget->currentItem()->text().toLower();
    QString themeToDeleteName = m_ui->listWidget->currentItem()->text();
    bool themeUsedInRemote = false;
    QStringList themesUsedInRemotes;
    QDir dir(remotesPath);
    QStringList remoteFiles = dir.entryList(QDir::Files);

    for(int i=0;i<remoteFiles.count();i++) {
       QString fileName = remoteFiles.at(i);
       if(fileName.contains("remotes.conf",Qt::CaseInsensitive) || fileName.contains("~",Qt::CaseInsensitive))
           remoteFiles.removeAt(i);
    }

    // Fill usedThemes
    for(int i=0;i<remoteFiles.count();i++) {
        QFile file(remotesPath + remoteFiles.at(i));
        file.open(QIODevice::ReadOnly | QIODevice::Text);
           QTextStream in(&file);
           while (!in.atEnd()) {
               QString line = in.readLine();
               this->readUsedThemes(line);
           }
        if(usedThemesByRemote.contains(themeToDelete,Qt::CaseInsensitive))
            remoteNamesUsingThemeToBeDeleted.append(remoteFiles.at(i));
        usedThemesByRemote.clear();
        file.close();
    }


    if(usedThemes.contains(themeToDelete,Qt::CaseInsensitive)) {

        themeUsedInRemote = true;
        QString remoteNames;
        remoteNamesUsingThemeToBeDeleted.removeDuplicates();
        for(int i=0;i<remoteNamesUsingThemeToBeDeleted.count();i++) {
            QString remName =  remoteNamesUsingThemeToBeDeleted.at(i);
            remName.chop(5);
            remoteNames.append(remName + ",");
        }
        remoteNames.chop(1);
        QMessageBox::information(this,"Theme manager","Unable to delete theme: " + themeToDeleteName + "\nTheme used in remote: " +
                                  remoteNames, QMessageBox::NoButton,QMessageBox::NoButton);
        remoteNames.clear();


    }

    if(!themeUsedInRemote) {

        QDialog *del = new QDialog(this);
        del->setWindowTitle("Delete theme");
        del->resize(800,150);
        QLabel *label = new QLabel(del);
        label->move(195,25);
        label->setFont(QFont("sans-serif",14));
        label->setText("Are you sure you want to delete " + m_ui->listWidget->currentItem()->text() + " theme? ");
        QPushButton *delButton = new QPushButton(del);
        delButton->setGeometry(280,75,225,50);
        delButton->setText("Delete");
        delButton->show();
        connect(delButton,SIGNAL(clicked()),del,SLOT(deleteLater()));
        connect(delButton,SIGNAL(clicked()),this,SLOT(deleteTheme()));
        del->exec();
    }
}
void ThemeManager::readUsedThemes(QString line)
{
    QString themeTemp;
    if(line.contains("Themes: ",Qt::CaseInsensitive)) {
        line.remove(0,8);
        //line.append(' ');
        for(int i=0;i<line.length();i++) {
            if(!line.at(i).isSpace())
                themeTemp.append(line.at(i));

            else {
                usedThemesByRemote.append(themeTemp);
                if(!usedThemes.contains(themeTemp,Qt::CaseInsensitive)) {
                    usedThemes.append(themeTemp);

                }
                themeTemp.clear();
            }
        }
    }
}
void ThemeManager::deleteTheme(QString delThemeName)
{
    QDir dir;
    QString homePath = dir.currentPath();
    dir.setPath(themesPath + delThemeName + "/buttons/");
    QStringList buttonDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<buttonDirs.count();i++)
    {
        dir.setPath(themesPath + delThemeName + "/buttons/" + buttonDirs.at(i));
        QStringList buttonFiles = dir.entryList(QDir::Files);
        for(int j=0;j<buttonFiles.count();j++)
        {
            QFile file(themesPath + delThemeName + "/buttons/" + buttonDirs.at(i) + "/" + buttonFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + delThemeName + "/bg/");
    QStringList bgDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<bgDirs.count();i++)
    {
        dir.setPath(themesPath + delThemeName + "/bg/" + bgDirs.at(i));
        QStringList bgFiles = dir.entryList(QDir::Files);
        for(int j=0;j<bgFiles.count();j++)
        {
            QFile file(themesPath + delThemeName + "/bg/" + bgDirs.at(i) + "/" + bgFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + delThemeName + "/");
    QStringList extraFiles = dir.entryList(QDir::Files);
    for(int i=0;i<extraFiles.count();i++)
    {
        QFile file(themesPath + delThemeName + "/" + extraFiles.at(i));
        file.remove();
    }
    dir.setPath(homePath);
    QString delCommand("rm -rf /home/user/MyDocs/qtirreco/.themes/" + delThemeName);
    system(delCommand.toAscii().data());
}
void ThemeManager::deleteTheme()
{
    QDir dir;
    QString homePath = dir.currentPath();
    dir.setPath(themesPath + themeToDelete + "/buttons/");
    QStringList buttonDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<buttonDirs.count();i++)
    {
        dir.setPath(themesPath + themeToDelete + "/buttons/" + buttonDirs.at(i));
        QStringList buttonFiles = dir.entryList(QDir::Files);
        for(int j=0;j<buttonFiles.count();j++)
        {
            QFile file(themesPath + themeToDelete + "/buttons/" + buttonDirs.at(i) + "/" + buttonFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + themeToDelete + "/bg/");
    QStringList bgDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<bgDirs.count();i++)
    {
        dir.setPath(themesPath + themeToDelete + "/bg/" + bgDirs.at(i));
        QStringList bgFiles = dir.entryList(QDir::Files);
        for(int j=0;j<bgFiles.count();j++)
        {
            QFile file(themesPath + themeToDelete + "/bg/" + bgDirs.at(i) + "/" + bgFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + themeToDelete + "/");
    QStringList extraFiles = dir.entryList(QDir::Files);
    for(int i=0;i<extraFiles.count();i++)
    {
        QFile file(themesPath + themeToDelete + "/" + extraFiles.at(i));
        file.remove();
    }
    dir.setPath(homePath);
    QString delCommand("rm -rf /home/user/MyDocs/qtirreco/.themes/" + themeToDelete);
    system(delCommand.toAscii().data());
    QDir themeDir(themesPath + themeToDelete);
    if(!themeDir.exists())
    {
        QMessageBox::information(this,"Theme manager","Theme deleted succesfully.",
                                     QMessageBox::NoButton,QMessageBox::NoButton);
        themeToDelete.clear();
        m_ui->listWidget->clear();
        QListWidgetItem *item = new QListWidgetItem();
        item->setText("Device");
        this->updateListWidget(item);
        delete item;
        m_ui->pictureLabel->setAutoFillBackground(false);

        // Set active theme and if no themes, clearAllData()
        if(m_ui->listWidget->count() > 0)
        {
            m_ui->listWidget->setCurrentRow(0);
            this->updateListWidget(m_ui->listWidget->currentItem());
        }
        else
        {
            this->clearAllData();
        }
    }
}
void ThemeManager::replaceTheme()
{
    QDir dir;
    QString homePath = dir.currentPath();
    dir.setPath(themesPath + themeToDelete + "/buttons/");
    QStringList buttonDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<buttonDirs.count();i++)
    {
        dir.setPath(themesPath + themeToDelete + "/buttons/" + buttonDirs.at(i));
        QStringList buttonFiles = dir.entryList(QDir::Files);
        for(int j=0;j<buttonFiles.count();j++)
        {
            QFile file(themesPath + themeToDelete + "/buttons/" + buttonDirs.at(i) + "/" + buttonFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + themeToDelete + "/bg/");
    QStringList bgDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    for(int i=0;i<bgDirs.count();i++)
    {
        dir.setPath(themesPath + themeToDelete + "/bg/" + bgDirs.at(i));
        QStringList bgFiles = dir.entryList(QDir::Files);
        for(int j=0;j<bgFiles.count();j++)
        {
            QFile file(themesPath + themeToDelete + "/bg/" + bgDirs.at(i) + "/" + bgFiles.at(j));
            file.remove();
        }
    }
    dir.setPath(themesPath + themeToDelete + "/");
    QStringList extraFiles = dir.entryList(QDir::Files);
    for(int i=0;i<extraFiles.count();i++)
    {
        QFile file(themesPath + themeToDelete + "/" + extraFiles.at(i));
        file.remove();
    }
    dir.setPath(homePath);
    QString delCommand("rm -rf /home/user/MyDocs/qtirreco/.themes/" + themeToDelete);
    system(delCommand.toAscii().data());

    QDir themeDir(themesPath + themeToDelete);
    if(!themeDir.exists())
    {
        progress = new QProgressBar(this);
        progress->setGeometry(650,10,128,64);
        progress->setRange(0,100);
        connect(this,SIGNAL(progressState(int)),progress,SLOT(setValue(int)));
        connect(this,SIGNAL(themeLoaded()),progress,SLOT(deleteLater()));
        progress->show();
        this->setWindowTitle("Loading theme buttons...");
        this->setDisabled(true);
        dbButtonIdsList.clear();
        downloadId = themeVersions.value(m_ui->versionComboBox->currentText());
        QList<QVariant> params;
        params.clear();
        params.append(downloadId);
        themeButtonIds = client->call("getButtons",params);

    }
}
void ThemeManager::startUpload(QString user,QString pw)
{
    uploadUserName = user;
    uploadPwHash = pw;
    this->setWindowTitle("Uploading theme...");
    this->createNewTheme();
}
void ThemeManager::createNewTheme()
{
    QList<QVariant> params;
    params.clear();
    params.append(uploadThemeName);
    if(uploadComments.isEmpty()) uploadComments = " ";
    params.append(uploadComments);
    params.append(uploadPreviewName);
    params.append(uploadThemeName.toLower());
    params.append(uploadUserName);
    params.append(uploadPwHash);
    themeToUploadDir = uploadThemeName.toLower();
    uploadTheme = client->call("createNewTheme",params);
}
void ThemeManager::uploadThemeButton()
{
   if(progressItemCount == 0)
   {
        progress = new QProgressBar(this);
        progress->setGeometry(650,10,128,64);
        progress->setRange(0,100);
        connect(this,SIGNAL(progressState(int)),progress,SLOT(setValue(int)));
        connect(this,SIGNAL(themeUploaded()),progress,SLOT(deleteLater()));
        progress->show();
        this->setWindowTitle("Uploading theme buttons...");
        this->setDisabled(true);
    }


    QString uploadImageUpBase64;
    QString uploadImageDownBase64;
    QString uploadImageUpHash;
    QString uploadImageDownHash;

    QDir dir;
    QString homePath = dir.currentPath();
    QDir d;
    d.setPath(themesPath + themeToUploadDir + "/bg/");
    QStringList bgs = d.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    dir.setPath(themesPath + themeToUploadDir + "/buttons/");
    QStringList buttonDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    uploadItemsCount = buttonDirs.count() + bgs.count();;
    if(uploadButtonIndex < buttonDirs.count())
    {
        // Read button conf
        QFile file(themesPath + themeToUploadDir + "/buttons/" + buttonDirs.at(uploadButtonIndex)+"/button.conf");
        if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
                {
                    QMessageBox::warning(this,"Error","Couldn't load button config!",
                    QMessageBox::NoButton,QMessageBox::NoButton);
                    return;
                }
                QTextStream in(&file);
                while (!in.atEnd())
                {
                    QString line = in.readLine();
                    readButtonConf(line);
                }
        file.close();
        // Button image base64 strings and hashs
        QImage imageUp(themesPath + themeToUploadDir + "/buttons/" + buttonDirs.at(uploadButtonIndex)+"/unpressed.png");
        QBuffer buffer;
        buffer.open(QIODevice::WriteOnly);
        imageUp.save(&buffer,"PNG");
        QByteArray arr = buffer.data();
        QByteArray hasharray(QCryptographicHash::hash(arr,QCryptographicHash::Sha1));
        hasharray = hasharray.toHex();
        uploadImageUpHash.append(hasharray);
        uploadImageUpBase64 = QString(arr.toBase64());
        buffer.close();

        QImage imageDown(themesPath + themeToUploadDir + "/buttons/" + buttonDirs.at(uploadButtonIndex)+"/pressed.png");
        QBuffer buffer2;
        buffer2.open(QIODevice::WriteOnly);
        imageDown.save(&buffer2,"PNG");
        QByteArray arr2= buffer2.data();
        QByteArray hasharray2(QCryptographicHash::hash(arr2,QCryptographicHash::Sha1));
        hasharray2 = hasharray2.toHex();
        uploadImageDownHash.append(hasharray2);
        uploadImageDownBase64 = QString(arr2.toBase64());
        buffer2.close();


        // We are using default padding value 5 (int) and aligns 0.5 (double), image up name unpressed.png and down pressed.png
        QString folder = buttonDirs.at(uploadButtonIndex);

        QList<QVariant> params;
        params.clear();
        params.append(uploadButtonName);
        params.append(uploadAllowText);
        params.append(uploadTextFormatUp);
        params.append(uploadTextFormatDown);
        int padding = 5;
        double hAlign = 0.5;
        double vAlign = 0.5;
        params.append(padding);
        params.append(hAlign);
        params.append(vAlign);
        params.append(uploadImageUpHash);
        QString upName = "unpressed.png";
        params.append(upName);
        params.append(uploadImageUpBase64);
        params.append(uploadImageDownHash);
        QString downName = "pressed.png";
        params.append(downName);
        params.append(uploadImageDownBase64);
        params.append(folder);
        params.append(themeIdToUpload);
        params.append(uploadUserName);
        params.append(uploadPwHash);

        dir.setPath(homePath);

        addButtonToTheme = client->call("addButtonToTheme",params);
    }
    else
    {
        dir.setPath(homePath);
        this->uploadThemeBg();
    }

}
void ThemeManager::uploadThemeBg()
{
    this->setWindowTitle("Uploading theme backgrounds...");
    QString uploadBg64;
    QString uploadBgHash;


    QDir dir;
    QString homePath = dir.currentPath();
    dir.setPath(themesPath + themeToUploadDir + "/bg/");
    QStringList bgDirs = dir.entryList(QDir::NoDotAndDotDot | QDir::Dirs, QDir::Name);
    if(uploadBgIndex < bgDirs.count())
    {
        // Read background conf
        QFile file(themesPath + themeToUploadDir + "/bg/" + bgDirs.at(uploadBgIndex)+"/bg.conf");
        if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
                {
                    QMessageBox::warning(this,"Error","Couldn't load background config!",
                    QMessageBox::NoButton,QMessageBox::NoButton);
                    return;
                }
                QTextStream in(&file);
                while (!in.atEnd())
                {
                    QString line = in.readLine();
                    readBgConf(line);
                }
        file.close();
        // bg image base64 string and hash
        QImage bg(themesPath + themeToUploadDir + "/bg/" + bgDirs.at(uploadBgIndex)+"/image.png");
        QBuffer buffer;
        buffer.open(QIODevice::WriteOnly);
        bg.save(&buffer,"PNG");
        QByteArray arr = buffer.data();
        buffer.close();
        QByteArray hasharray(QCryptographicHash::hash(arr,QCryptographicHash::Sha1));
        uploadBgHash = hasharray.toHex();

        uploadBg64 = QString(arr.toBase64());
        buffer.close();
        // We are using default padding value 5 (int) and aligns 0.5 (double), image up name unpressed.png and down pressed.png
        QString folder = bgDirs.at(uploadBgIndex);

        QList<QVariant> params;
        params.clear();
        params.append(uploadBgName);
        params.append(uploadBgHash);
        QString bgName = "image.png";
        params.append(bgName);
        params.append(uploadBg64);
        params.append(folder);
        params.append(themeIdToUpload);
        params.append(uploadUserName);
        params.append(uploadPwHash);

        addBgToTheme = client->call("addBgToTheme",params);
    }
    else
    {
       QList<QVariant> params;
       params.clear();
       params.append(themeIdToUpload);
       bool downloadable = true;
       params.append(downloadable);
       params.append(uploadUserName);
       params.append(uploadPwHash);
       setThemeDownloadable = client->call("setThemeDownloadable",params);
    }

}
void ThemeManager::readButtonConf(QString line)
{
      if(line.contains("name=", Qt::CaseInsensitive))
         {
             uploadButtonName = line.remove(0,5);
         }
      if(line.contains("allow-text=", Qt::CaseInsensitive))
         {
             line = line.remove(0,11);
             if(line == "true")
             {
                 uploadAllowText = true;
             }
             if(line == "false")
             {
                 uploadAllowText = false;
             }
         }
      if(line.contains("text-format-up=", Qt::CaseInsensitive))
         {
             uploadTextFormatUp = line.remove(0,15);
         }
      if(line.contains("text-format-down=", Qt::CaseInsensitive))
         {
              uploadTextFormatDown = line.remove(0,17);
         }
}
void ThemeManager::readBgConf(QString line)
{
      if(line.contains("name=", Qt::CaseInsensitive))
         {
             uploadBgName = line.remove(0,5);
         }
}
