#ifndef REVERSIGAME_H
#define REVERSIGAME_H

#include <QObject>
#include <QList>
#include <QPoint>

#include "reversidefs.h"

class Engine;

class ReversiGame: public QObject
{
    Q_OBJECT
public:
    ReversiGame();
    ~ReversiGame();

    void init(int mode, int strength);
    ChipColor chipColor(unsigned x, unsigned y) const { return cells_[x][y]; }
    ChipColor currentPlayer() const {return currentPlayer_; }
    ChipColor opponentPlayer() const {return opponentPlayer_; }
    void makeStep(unsigned x, unsigned y);
    const QList<QPoint> &changedCells() const {return changedCells_;}
    unsigned score(ChipColor player) const { return score_[player];}
    void switchPlayer();
    /** 
     * @brief Checks availability step for current player
     * 
     * @return 
     */
    bool isStepAvailable() const;
    void startNextTurn();
    bool isComputersTurn() const { return currentPlayer_ == computerColor_; }

signals:
    void stepComplete();
    void stepImpossible();
    void updateInfo();
    void newGame();
    void gameOver();
    void stepUnavailable();

private:
    
    QList<QPoint> checkVertical(unsigned x, unsigned y, unsigned endY, unsigned stepY);
    QList<QPoint> checkHorizontal(unsigned x, unsigned endX, unsigned stepX, unsigned y);
    QList<QPoint> checkDiagonal(unsigned x, unsigned endX, unsigned stepX, unsigned y, unsigned endY, unsigned stepY);

    /** 
     * @brief check availability of move by flipping chips in vertical
     * 
     * @param x
     * @param y
     * @param endY
     * @param stepY
     * 
     * @return 
     */
    bool isVerticalAvailable(unsigned x, unsigned startY, unsigned endY, unsigned stepY) const;
    /** 
     * @brief check availability of move by flipping chips in horizontal
     *  
     * @param x
     * @param y
     * @param endY
     * @param stepY
     * 
     * @return 
     */
    bool isHorizontalAvailable(unsigned startX, unsigned endX, unsigned stepX, unsigned y) const;
    /** 
     * @brief check availability of move by flipping chips in diagonal
     * 
     * @param x
     * @param endX
     * @param stepX
     * @param y
     * @param endY
     * @param stepY
     * 
     * @return 
     */
    bool isDiagonalAvailable(unsigned startX, unsigned endX, unsigned stepX, unsigned startY, unsigned endY, unsigned stepY) const;

    void computerMove();
    bool isGameOver();

private:
    unsigned score_[2];
    int mode_;
    ChipColor cells_[FIELDSIZE][FIELDSIZE];
    ChipColor currentPlayer_;
    ChipColor opponentPlayer_;
    ChipColor computerColor_;
    QList<QPoint> changedCells_;
    Engine  *engine_;
};

#endif // REVERSIGAME_H
