#include "todomodel.h"

#include <QSqlDatabase>
#include <QDir>
#include <QSqlQuery>
#include <QVariant>
#include <QDebug>

TodoModel* TodoModel::sm_instance = NULL;

namespace
{
    const char* QTD_DATABASE = ".qtdone.db.sqlite";


    void doexec(QSqlQuery& q) {
        bool r = q.exec();
        if (!r)
            qDebug() << "Db error :" << q.lastError();
    }
}



TodoModel::TodoModel(QObject *parent) :
    QObject(parent)
{
}


bool TodoModel::openDB()
{
    // Find QSLite driver
    db = QSqlDatabase::addDatabase("QSQLITE");

    #ifdef Q_OS_LINUX
    // NOTE: We have to store database file into user home folder in Linux
    QString path(QDir::home().path());
    path.append(QDir::separator()).append(QTD_DATABASE);
    path = QDir::toNativeSeparators(path);
    QFileInfo finfo(path);
    bool didexist = finfo.exists();

    db.setDatabaseName(path);
    #else
    // NOTE: File exists in the application private folder, in Symbian Qt implementation
    db.setDatabaseName(QTD_DATABASE);
    #endif

    bool r = db.open();
    if (!didexist)
        createTables();
    // Open database
    m_catmapping = new SqlMapping(&db, "CATEGORY");
    return r;
}

QSqlError TodoModel::lastError()
{
    // If opening database has failed user can ask
    // error description by QSqlError::text()
    return db.lastError();
}

bool TodoModel::deleteDB()
{
    // Close database
    db.close();

    #ifdef Q_OS_LINUX
    // NOTE: We have to store database file into user home folder in Linux
    QString path(QDir::home().path());
    path.append(QDir::separator()).append(QTD_DATABASE);
    path = QDir::toNativeSeparators(path);
    return QFile::remove(path);
    #else

    // Remove created database binary file
    return QFile::remove(QTD_DATABASE);
    #endif
}

void TodoModel::getTasksForFolder(const QString& foldername, QList<TaskEntry>& tasks)
{
    QSqlQuery q;

    if (currentCategory().length() == 0) {
        q.prepare("select id, txt from TASKS where folder = :f order by id desc");
    }
    else {
        q.prepare("select id, txt from TASKS where folder = :f and category = :cat order by id desc");
        int cat = currentCategoryId();
        qDebug() << "Cat:" << cat;
        q.bindValue(":cat", QVariant(cat));
    }



    q.bindValue(":f", QVariant(foldername));
    q.exec();    
    while (q.next()) {
        int id = q.value(0).toInt();

        QString desc = q.value(1).toString();
        //QString f = q.value(2).toString();
        TaskEntry ent;
        ent.desc = desc;
        ent.folder = foldername;
        ent.id = id;
        qDebug() << "task:" << desc;
        tasks.append(ent);
    }
    emit folderSizeReport(foldername, tasks.length());
}

bool TodoModel::createTables()
{

    //bool ret = false;

    // todo: extract as nice utility class
    m_tabledesc.clear();
    m_tabledesc <<
        "TASKS" <<
            "id integer primary key autoincrement,"
            "txt varchar(100), "
            "folder varchar(20),"
            "category integer" <<
        "CATEGORY" <<
            "id integer primary key autoincrement,"
            "txt varchar(100)";


    QList<QString>::iterator it = m_tabledesc.begin();

    for (; it != m_tabledesc.end(); ++it)
    {
        const QString& tabname = *it;
        ++it;
        const QString& desc = *it;
        QString drop = QString("drop table %1;").arg(tabname);
        QString create = QString("create table %1 (%2);").arg(tabname).arg(desc);

        {
            QSqlQuery q(drop);
            bool r = q.exec();
            if (!r)
                qDebug() << "cannot drop : " << drop;

        }
        {
            QSqlQuery q(create);

            bool r = q.exec();
            if (!r) {
                qDebug() << "Cannot create - " << create << " : " << q.lastError();
            }

        }


    }
    db.commit();
    /*
    if (db.isOpen()) {
        QSqlQuery query;
        query.exec("drop table TASKS");
        ret = query.exec("create table TASKS "
                  "(id integer primary key autoincrement,"
                  "desc varchar(100), "
                  "folder varchar(20),"
                  "category integer"
                  ");\n"
                  "create table CATEGORY "
                  "(id integer primary key autoincrement,"
                  "desc varchar(20));"

                  );



        if (!ret)
            qDebug() << query.lastError();

    }
    return ret;
    */
    return true;
}

void TodoModel::createTask(const QString &task, const QString &folder)
{

    QSqlQuery q("insert into TASKS (txt, folder, category) values (:d,:f, :cat)");
    QVariant t = task;
    QVariant f = folder;    
    QVariant cat = currentCategoryId();

    q.bindValue(":d", t);
    q.bindValue(":f", f);
    if (cat.toInt())
        q.bindValue(":cat", cat);

    doexec(q);
    db.commit();
}

void TodoModel::commitTask(TaskEntry &tent)
{
    QSqlQuery q("update TASKS set txt = ?, folder = ? where id = ?");
    q.bindValue(0, tent.desc);
    q.bindValue(1, tent.folder);
    q.bindValue(2, tent.id);

    q.exec();
    db.commit();
    emit updatedFolder(tent.folder);

}

TaskEntry TodoModel::findById(int id)
{
    QSqlQuery q("select txt, folder from TASKS where id = ?");
    q.bindValue(0,QVariant(id));
    q.exec();

    bool r = q.first();
    Q_ASSERT(r);
    TaskEntry t;
    t.desc = q.value(0).toString();
    t.folder = q.value(1).toString();
    t.id = id;
    return t;

}

TodoModel* TodoModel::instance()
{
    if (!sm_instance)
        sm_instance = new TodoModel();

    return sm_instance;
}

void TodoModel::doEmitUpdated(const QString& folder)
{
    emit updatedFolder(folder);

}


int TodoModel::countItems(const QString &folder)
{
    QSqlQuery q("select count(id) from tasks where folder=:f");
    q.bindValue(":f", QVariant(folder) );
    q.exec();
    q.first();
    int c = q.value(0).toInt();
    emit folderSizeReport(folder, c);
    return c;
}

QStringList TodoModel::allFolders()
{
    QStringList folders;
    folders << "Now" << "Later" <<"Done" << "Defer";
    return folders;
}

void TodoModel::deleteTask(TaskEntry& tent)
{
    QSqlQuery q("delete from tasks where id = :id");
    q.bindValue("id", QVariant(tent.id));
    q.exec();
    db.commit();
    emit updatedFolder(tent.folder);
}

void TodoModel::listCategories(QStringList &cats)
{
    SqlMapping::MapT m = m_catmapping->getAll();
    SqlMapping::MapT::iterator it = m.begin();

    for (; it != m.end(); ++it) {
        cats.append(it.key());
    }
}

void TodoModel::createCategory(const QString &cat)
{
    m_catmapping->addDesc(cat);
    db.commit();
}


SqlMapping::SqlMapping(QSqlDatabase *db, const QString &tablename) : m_db(db), m_tablename(tablename)
{
}

SqlMapping::MapT SqlMapping::getAll()
{
    QSqlQuery q(QString("select id, txt from %1 order by txt").arg(m_tablename));
    //q.bindValue(":tab", m_tablename);
    doexec(q);

    while (q.next()) {
        m_cache.insert(q.value(1).toString(), q.value(0).toInt());
    }

    return m_cache;

}

int SqlMapping::addDesc(const QString &desc)
{

    QString qq = QString("insert into %1(txt) values (:txt)").arg(m_tablename);
    QSqlQuery q(qq);

    q.bindValue(":txt", QVariant(desc));
    doexec(q);

    QVariant lid = q.lastInsertId();

    return lid.toInt();

}

int SqlMapping::findId(const QString &desc)
{

    SqlMapping::MapT::const_iterator it = m_cache.find(desc);
    if (it == m_cache.end()) {
        getAll();
        it = m_cache.find(desc);

    }
    return it.value();

}


void TodoModel::setCurrentCategory(const QString &category)
{
    qDebug() << "current cat" << category;
    if (category == "None") {
        m_currentCategory = "";
    } else {
        m_currentCategory = category;
    }
    countAllFolders();    
    // other folders are updated on entry - and inbox is
    // the only place to change category
    emit updatedFolder("inbox");
}

QString TodoModel::currentCategory()
{
    return m_currentCategory;
}

int TodoModel::currentCategoryId()
{
    return m_catmapping->findId(currentCategory());


}

void TodoModel::countAllFolders()
{
    QSqlQuery q;
    if (m_currentCategory.length() == 0) {
        q.prepare("select folder, count(folder) from TASKS group by FOLDER");
    } else {
        q.prepare("select folder, count(folder) from TASKS where CATEGORY = (:cat) group by FOLDER");
        q.bindValue(0, QVariant(currentCategoryId()));
    }

    doexec(q);

    QStringList fds = allFolders();
    while (q.next()) {
        QString folder = q.value(0).toString();
        int count  = q.value(1).toInt();;
        emit folderSizeReport(folder, count);
        fds.removeOne(folder);

    }

    foreach(const QString& s, fds) {
        emit folderSizeReport(s, 0);
    }
}

QString TodoModel::windowTitle(const QString &prefix)
{
    QString cat = currentCategory();
    QString res = prefix;
    if (cat.length()) {
        res.append(QString(" [%1]").arg(cat));
    }
    return res;

}
