/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Graphics Dojo project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include "widget.h"

#include "context2d.h"
//#include <qmaemo-ext/system/qorientationinfo.h>
#include <math.h>

static float brand()
{
    return (::rand() / (float) RAND_MAX);
}

struct Vector
{
 public:
    Vector()
        :x(0), y(0)
    {}
    Vector(float x_, float y_)
        : x(x_), y(y_)
    {}

    bool equal(const Vector &v) const {
        return this->x == v.getX() && this->y == v.getY();
    }
    
    float getX() const {
        return this->x;
    }

    float getY() const {
        return this->y;
    }
    
    void setX(float x) {
        this->x = x;
    }
    
    void setY(float y) {
        this->y = y;
    }
    
    void addX(float x) {
        this->x += x;
    }
    
    void addY(float y) {
        this->y += y;
    }
    
    void set(const Vector &v) {
        this->x = v.getX();
        this->y = v.getY();
    }
    
    void add(const Vector &v) {
        this->x += v.getX();
        this->y += v.getY();
    }
    void sub(const Vector &v) {
        this->x -= v.getX();
        this->y -= v.getY();
    }
    
    float dotProd(const Vector &v) {
        return this->x * v.getX() + this->y * v.getY();
    }
    
    float length() const {
        return sqrt(this->x * this->x + this->y * this->y);
    }
    void scale(float scaleFactor) {
        this->x *= scaleFactor;
        this->y *= scaleFactor;
    }
    
    QString toString() const {
        return QString(" X: %1 Y: %2").arg(x).arg(y);
    }
    float x, y;
};

struct Environment {
public:
    float left, right, top, bottom;
    Vector r;
public:
    Environment(float x, float y, float w, float h) {
        this->left = x;
        this->right = x + w;
        this->top = y;
        this->bottom = y + h;
        this->r = Vector(0.0, 0.0);
    }

    bool collision(Vector *curPos,
                   Vector * /*prevPos*/) {
	    bool collide = false;

	    if(curPos->getX() < this->left)
	    {
		curPos->setX(this->left);
		collide = true;
	    }
	    else if(curPos->getX() > this->right)
	    {
		curPos->setX(this->right);
		collide = true;
	    }
	    if(curPos->getY() < this->top)
	    {
		curPos->setY(this->top);
		collide = true;
	    }
	    else if(curPos->getY() > this->bottom)
	    {
		curPos->setY(this->bottom);
		collide = true;
	    }
	    return collide;
	}
    void draw(Context2D *, float)
    {
    }
};

struct PointMass  {
public:
    Vector cur, prev;
    float mass;
    Vector force, result;
    float friction;
public:
    PointMass()
        : mass(0), friction(0)
    {}
    PointMass(float cx, float cy, float mass)
    {
        this->cur = Vector(cx, cy);
        this->prev = Vector(cx, cy);
        this->mass = mass;
        this->force = Vector(0.0, 0.0);
        this->result = Vector(0.0, 0.0);
        this->friction = 0.01;
    }
    
    float getXPos() const {
        return this->cur.getX();
    }
    float getYPos() const {
        return this->cur.getY();
    }
    Vector *getPos()  {
        return &this->cur;
    }
    float getXPrevPos() const {
        return this->prev.getX();
    }
    float getYPrevPos() const {
        return this->prev.getY();
    }
    Vector *getPrevPos()  {
        return &this->prev;
    }

    void addXPos(float dx) {
        this->cur.addX(dx);
    }
    void addYPos(float dy) {
        this->cur.addY(dy);
    }
    void setForce(const Vector &force) {
        this->force.set(force);
    }
    void addForce(const Vector &force) {
        this->force.add(force);
    }
    float getMass() const
    {
        return this->mass;
    }
    void setMass(float mass)
    {
        this->mass = mass;
    }
    void move(float dt)
    {
        float t, a, c, dtdt;

        dtdt = dt * dt;

        a = this->force.getX() / this->mass;
        c = this->cur.getX();
        t = (2.0 - this->friction) * c - (1.0 - this->friction) * this->prev.getX() + a * dtdt;
        this->prev.setX(c);
        this->cur.setX(t);

        a = this->force.getY() / this->mass;
        c = this->cur.getY();
        t = (2.0 - this->friction) * c - (1.0 - this->friction) * this->prev.getY() + a * dtdt;
        this->prev.setY(c);
        this->cur.setY(t);
    }
    void setFriction(float friction)
    {
        this->friction = friction;
    }
    float getVelocity() const
    {
        float cXpX, cYpY;

        cXpX = this->cur.getX() - this->prev.getX();
        cYpY = this->cur.getY() - this->prev.getY();

        return cXpX * cXpX + cYpY * cYpY;
    }
    void draw(Context2D *ctx, float scaleFactor)
    {
        ctx->setStrokeStyle("#000000");
        ctx->setLineWidth(2);
        ctx->setFillStyle("#000000");
        ctx->beginPath();
        ctx->arc(this->cur.getX() * scaleFactor,
                 this->cur.getY() * scaleFactor,
                 4.0, 0.0, M_PI * 2.0, true);
        ctx->fill();
    }
};

struct ConstraintY {
    PointMass pointMass;
    float y, shortConst, longConst;
    Vector delta;
    
    ConstraintY(const PointMass &pointMass, float y,
                float shortConst, float longConst)
    {
        this->pointMass = pointMass;
        this->y = y;
        this->delta = Vector(0.0, 0.0);
        this->shortConst = shortConst;
        this->longConst = longConst;
    }

    void sc()
    {
        float dist;

        dist = fabs(this->pointMass.getYPos() - this->y);
        this->delta.setY(-dist);

        if(this->shortConst != 0.0 && dist < this->shortConst)
        {
            float scaleFactor;

            scaleFactor = this->shortConst / dist;
            this->delta.scale(scaleFactor);
            pointMass.getPos()->sub(this->delta);
        }
        else if(this->longConst != 0.0 && dist > this->longConst)
        {
            float scaleFactor;

            scaleFactor = this->longConst / dist;
            this->delta.scale(scaleFactor);
            pointMass.getPos()->sub(this->delta);
        }
    }
};

struct Joint {
    PointMass *pointMassA, *pointMassB;
    Vector delta;
    Vector *pointMassAPos, *pointMassBPos;
    float shortConst, longConst, scSquared, lcSquared;
    
    Joint(PointMass *pointMassA,
          PointMass *pointMassB,
          float shortConst, float longConst)
    {
        this->pointMassA = pointMassA;
        this->pointMassB = pointMassB;
        this->delta = Vector(0.0, 0.0);
        this->pointMassAPos = pointMassA->getPos();
        this->pointMassBPos = pointMassB->getPos();

        this->delta.set(*this->pointMassBPos);
        this->delta.sub(*this->pointMassAPos);

        this->shortConst = this->delta.length() * shortConst;
        this->longConst = this->delta.length() * longConst;
        this->scSquared = this->shortConst * this->shortConst;
        this->lcSquared = this->longConst * this->longConst;
    }
    void setDist(float shortConst, float longConst)
    {
        this->shortConst = shortConst;
        this->longConst = longConst;
        this->scSquared = this->shortConst * this->shortConst;
        this->lcSquared = this->longConst * this->longConst;
    }
    void scale(float scaleFactor)
    {
        this->shortConst = this->shortConst * scaleFactor;
        this->longConst = this->longConst * scaleFactor;
        this->scSquared = this->shortConst * this->shortConst;
        this->lcSquared = this->longConst * this->longConst;
    }
    void sc()
    {
        this->delta.set(*this->pointMassBPos);
        this->delta.sub(*this->pointMassAPos);

        float dp = this->delta.dotProd(this->delta);

        if (this->shortConst != 0.0 && dp < this->scSquared) {
            float scaleFactor;

            scaleFactor = this->scSquared / (dp + this->scSquared) - 0.5;

            this->delta.scale(scaleFactor);

            this->pointMassAPos->sub(this->delta);
            this->pointMassBPos->add(this->delta);
        } else if(this->longConst != 0.0 && dp > this->lcSquared) {
            float scaleFactor;

            scaleFactor = this->lcSquared / (dp + this->lcSquared) - 0.5;

            this->delta.scale(scaleFactor);

            this->pointMassAPos->sub(this->delta);
            this->pointMassBPos->add(this->delta);
        }
    }
};

float pointMassDist(const PointMass &pointMassA,
                    const PointMass &pointMassB)
{
    float aXbX, aYbY;

    aXbX = pointMassA.getXPos() - pointMassB.getXPos();
    aYbY = pointMassA.getYPos() - pointMassB.getYPos();

    return sqrt(aXbX * aXbX + aYbY * aYbY);
}

struct Stick {
public:
    float length, lengthSquared;
    PointMass *pointMassA, *pointMassB;
    Vector delta;
public:
    Stick(PointMass *pointMassA,
          PointMass *pointMassB)
    {
        this->length = pointMassDist(*pointMassA, *pointMassB);
        this->lengthSquared = this->length * this->length;
        this->pointMassA = pointMassA;
        this->pointMassB = pointMassB;
        this->delta = Vector(0.0, 0.0);
    }
    PointMass &getPointMassA() {
        return *this->pointMassA;
    }
    PointMass &getPointMassB() {
        return *this->pointMassB;
    }
    void scale(float scaleFactor)
    {
        this->length *= scaleFactor;
        this->lengthSquared = this->length * this->length;
    }
    void sc(Environment *)
    {
        float dotProd, scaleFactor;
        Vector *pointMassAPos, *pointMassBPos;

        pointMassAPos = this->pointMassA->getPos();
        pointMassBPos = this->pointMassB->getPos();

        this->delta.set(*pointMassBPos);
        this->delta.sub(*pointMassAPos);

        dotProd = this->delta.dotProd(this->delta);

        scaleFactor = this->lengthSquared / (dotProd + this->lengthSquared) - 0.5;
        this->delta.scale(scaleFactor);

        pointMassAPos->sub(this->delta);
        pointMassBPos->add(this->delta);
    }
    void setForce(const Vector &force)
    {
        this->pointMassA->setForce(force);
        this->pointMassB->setForce(force);
    }
    void addForce(const Vector &force)
    {
        this->pointMassA->addForce(force);
        this->pointMassB->addForce(force);
    }
    void move(float dt)
    {
        this->pointMassA->move(dt);
        this->pointMassB->move(dt);
    }
    void draw(Context2D *ctx, float scaleFactor)
    {
        this->pointMassA->draw(ctx, scaleFactor);
        this->pointMassB->draw(ctx, scaleFactor);

        ctx->setLineWidth(3);
        ctx->setFillStyle("#000000");
        ctx->setStrokeStyle("#000000");
        ctx->beginPath();
        ctx->moveTo(this->pointMassA->getXPos() * scaleFactor,
                   this->pointMassA->getYPos() * scaleFactor);
        ctx->lineTo(this->pointMassB->getXPos() * scaleFactor,
                   this->pointMassB->getYPos() * scaleFactor);
        ctx->stroke();
    }
};

int clampIndex(int index, int maxIndex)
{
    index += maxIndex;
    return index % maxIndex;
}

struct Blob {
public:
    float x, y;
    float radius;
    int drawFaceStyle;
    int drawEyeStyle;
    bool selected;
    int numPointMasses;
    PointMass *middlePointMass;
    QList<PointMass*> pointMasses;
    QList<Stick*> sticks;
    QList<Joint*> joints;
public:
    Blob(float x, float y, float radius, int numPointMasses)
    {
        this->x = x;
        this->y = y;
        this->middlePointMass = 0;
        this->radius = radius;
        this->drawFaceStyle = 1;
        this->drawEyeStyle = 1;
        this->selected = false;

        this->numPointMasses = 8;

        //float f = 0.1;
        float low = 0.95, high = 1.05;
        int  i, p;
        float t;

        for(i = 0, t = 0.0; i < numPointMasses; i++)
	{
	    this->pointMasses.append(new PointMass(cos(t) * radius + x,
                                                   sin(t) * radius + y, 1.0));
	    t += 2.0 * M_PI / numPointMasses;
	}

        this->middlePointMass = new PointMass(x, y, 1.0);

        this->pointMasses[0]->setMass(4.0);
        this->pointMasses[1]->setMass(4.0);

        for(i = 0; i < numPointMasses; i++)
	{
	    this->sticks.append(new Stick(
                                    this->pointMasses[i],
                                    this->pointMasses[clampIndex(
                                                          i + 1,
                                                          numPointMasses)]));
	}

        for(i = 0, p = 0; i < numPointMasses; i++) {
	    this->joints.append(
                new Joint(this->pointMasses[i],
                          this->pointMasses[
                              clampIndex(i + numPointMasses / 2 + 1,
                                         numPointMasses)], low, high));
	    this->joints.append(
                new Joint(this->pointMasses[i],
                          this->middlePointMass,
                          high * 0.9, low * 1.1));
            // 0.8, 1.2 works
	}
    }
    void addBlob(const Blob &blob)
    {
        float dist;

        Joint *joint =  new Joint(this->middlePointMass,
                                  blob.getMiddlePointMass(), 0.0, 0.0);
        
        dist = this->radius + blob.getRadius();
        joint->setDist(dist * 0.95, 0.0);
        this->joints.append(joint);
    }
    PointMass *getMiddlePointMass() const
    {
        return this->middlePointMass;
    }
    float getRadius() const
    {
        return this->radius;
    }
    float getXPos() const
    {
        return this->middlePointMass->getXPos();
    }
    float getYPos() const
    {
        return this->middlePointMass->getYPos();
    }
    void scale(float scaleFactor)
    {
        int i;

        for(i = 0; i < this->joints.count(); i++)
        {
            this->joints[i]->scale(scaleFactor);
        }
        for(i = 0; i < this->sticks.count(); i++)
        {
            this->sticks[i]->scale(scaleFactor);
        }
        this->radius *= scaleFactor;
    }
    void move(float dt)
    {
        int i;

        for(i = 0; i < this->pointMasses.count(); i++)
        {
            this->pointMasses[i]->move(dt);
        }
        this->middlePointMass->move(dt);
    }
    void sc(Environment *env) {
        int i, j;

        for(j = 0; j < 4; j++)
        {
            for(i = 0; i < this->pointMasses.count(); i++)
            {
                if(env->collision(this->pointMasses[i]->getPos(),
                                  this->pointMasses[i]->getPrevPos()) == true)
                {
                    this->pointMasses[i]->setFriction(0.75);
                }
                else
                {
                    this->pointMasses[i]->setFriction(0.01);
                }
            }
            for(i = 0; i < this->sticks.count(); i++)
            {
                this->sticks[i]->sc(env);
            }
            for(i = 0; i < this->joints.count(); i++)
            {
                this->joints[i]->sc();
            }
        }
    }
    void setForce(const Vector &force)
    {
        int i;

        for(i = 0; i < this->pointMasses.count(); i++)
        {
            this->pointMasses[i]->setForce(force);
        }
        this->middlePointMass->setForce(force);
    }
    void addForce(const Vector &force)
    {
        int i;

        for(i = 0; i < this->pointMasses.count(); i++)
        {
            this->pointMasses[i]->addForce(force);
        }
        this->middlePointMass->addForce(force);
        this->pointMasses[0]->addForce(force);
        this->pointMasses[0]->addForce(force);
        this->pointMasses[0]->addForce(force);
        this->pointMasses[0]->addForce(force);
    }
    void  moveTo(float x, float y)
    {
        int i;
        Vector *blobPos;

        blobPos = this->middlePointMass->getPos();
        x -= blobPos->getX();
        y -= blobPos->getY();

        for(i = 0; i < this->pointMasses.count(); i++)
        {
            blobPos = this->pointMasses[i]->getPos();
            blobPos->addX(x);
            blobPos->addY(y);
        }
        blobPos = this->middlePointMass->getPos();
        blobPos->addX(x);
        blobPos->addY(y);
    }
    void setSelected(bool selected)
    {
        this->selected = selected;
    }
    void drawEars(Context2D *ctx, float scaleFactor)
    {
        ctx->setStrokeStyle("#000000");
        ctx->setFillStyle("#FFFFFF");
        ctx->setLineWidth(2);

        ctx->beginPath();
        ctx->moveTo((-0.55 * this->radius) * scaleFactor,
                    (-0.35 * this->radius) * scaleFactor);
        ctx->lineTo((-0.52 * this->radius) * scaleFactor,
                    (-0.55 * this->radius) * scaleFactor);
        ctx->lineTo((-0.45 * this->radius) * scaleFactor,
                    (-0.40 * this->radius) * scaleFactor);
        ctx->fill();
        ctx->stroke();

        ctx->beginPath();
        ctx->moveTo((0.55 * this->radius) * scaleFactor,
                    (-0.35 * this->radius) * scaleFactor);
        ctx->lineTo((0.52 * this->radius) * scaleFactor,
                    (-0.55 * this->radius) * scaleFactor);
        ctx->lineTo((0.45 * this->radius) * scaleFactor,
                    (-0.40 * this->radius) * scaleFactor);
        ctx->fill();
        ctx->stroke();
    }
    void drawHappyEyes1(Context2D *ctx, float scaleFactor)
    {
        ctx->setLineWidth(1);
        ctx->setFillStyle("#FFFFFF");
        ctx->beginPath();
        ctx->arc((-0.15 * this->radius) * scaleFactor,
                 (-0.20 * this->radius) * scaleFactor,
                 this->radius * 0.12 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->fill();
        ctx->stroke();

        ctx->beginPath();
        ctx->arc(( 0.15 * this->radius) * scaleFactor,
                 (-0.20 * this->radius) * scaleFactor,
                 this->radius * 0.12 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->fill();
        ctx->stroke();

        ctx->setFillStyle("#000000");
        ctx->beginPath();
        ctx->arc((-0.15 * this->radius) * scaleFactor,
                 (-0.17 * this->radius) * scaleFactor,
                 this->radius * 0.06 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->fill();

        ctx->beginPath();
        ctx->arc(( 0.15 * this->radius) * scaleFactor,
                 (-0.17 * this->radius) * scaleFactor,
                 this->radius * 0.06 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->fill();
    }
    void drawHappyEyes2(Context2D *ctx, float scaleFactor)
    {
        ctx->setLineWidth(1);
        ctx->setFillStyle("#FFFFFF");
        ctx->beginPath();
        ctx->arc((-0.15 * this->radius) * scaleFactor,
                 (-0.20 * this->radius) * scaleFactor,
                 this->radius * 0.12 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->stroke();

        ctx->beginPath();
        ctx->arc(( 0.15 * this->radius) * scaleFactor,
                (-0.20 * this->radius) * scaleFactor,
                this->radius * 0.12 * scaleFactor, 0, 2.0 * M_PI, false);
        ctx->stroke();

        ctx->setLineWidth(1);
        ctx->beginPath();
        ctx->moveTo((-0.25   * this->radius) * scaleFactor,
                    (-0.20 * this->radius) * scaleFactor);
        ctx->lineTo((-0.05 * this->radius) * scaleFactor,
                   (-0.20 * this->radius) * scaleFactor);
        ctx->stroke();

        ctx->beginPath();
        ctx->moveTo(( 0.25   * this->radius) * scaleFactor,
                   (-0.20 * this->radius) * scaleFactor);
        ctx->lineTo(( 0.05 * this->radius) * scaleFactor,
                   (-0.20 * this->radius) * scaleFactor);
        ctx->stroke();
    }
    void drawHappyFace1(Context2D *ctx, float scaleFactor)
    {
        ctx->setLineWidth(2);
        ctx->setStrokeStyle("#000000");
        ctx->setFillStyle("#000000");
        ctx->beginPath();
        ctx->arc(0.0, 0.0,
                this->radius * 0.25 * scaleFactor, 0, M_PI, false);
        ctx->stroke();
    }
    void drawHappyFace2(Context2D *ctx, float scaleFactor)
    {
        ctx->setLineWidth(2);
        ctx->setStrokeStyle("#000000");
        ctx->setFillStyle("#000000");
        ctx->beginPath();
        ctx->arc(0.0, 0.0,
                this->radius * 0.25 * scaleFactor, 0, M_PI, false);
        ctx->fill();
    }
    void drawOohFace(Context2D *ctx, float scaleFactor)
    {
        ctx->setLineWidth(2);
        ctx->setStrokeStyle("#000000");
        ctx->setFillStyle("#000000");
        ctx->beginPath();
        ctx->arc(0.0, (0.1 * this->radius) * scaleFactor,
                this->radius * 0.25 * scaleFactor, 0, M_PI, false);
        ctx->fill();

        ctx->beginPath();

        ctx->moveTo((-0.25 * this->radius) * scaleFactor,
                    (-0.3 * this->radius) * scaleFactor);
        ctx->lineTo((-0.05 * this->radius) * scaleFactor,
                    (-0.2 * this->radius) * scaleFactor);
        ctx->lineTo((-0.25 * this->radius) * scaleFactor,
                    (-0.1 * this->radius) * scaleFactor);

        ctx->moveTo((0.25 * this->radius) * scaleFactor,
                    (-0.3 * this->radius) * scaleFactor);
        ctx->lineTo((0.05 * this->radius) * scaleFactor,
                    (-0.2 * this->radius) * scaleFactor);
        ctx->lineTo((0.25 * this->radius) * scaleFactor,
                    (-0.1 * this->radius) * scaleFactor);

        ctx->stroke();
    }
    void drawFace(Context2D *ctx, float scaleFactor)
    {
        if(this->drawFaceStyle == 1 && brand() < 0.05)
        {
            this->drawFaceStyle = 2;
        }
        else if(this->drawFaceStyle == 2 && brand() < 0.1)
        {
            this->drawFaceStyle = 1;
        }

        if(this->drawEyeStyle == 1 && brand() < 0.025)
        {
            this->drawEyeStyle = 2;
        }
        else if(this->drawEyeStyle == 2 && brand() < 0.3)
        {
            this->drawEyeStyle = 1;
        }

        if(this->middlePointMass->getVelocity() > 0.004)
        {
            this->drawOohFace(ctx, scaleFactor);
        }
        else
        {
            if(this->drawFaceStyle == 1)
            {
                this->drawHappyFace1(ctx, scaleFactor);
            }
            else
            {
                this->drawHappyFace2(ctx, scaleFactor);
            }

            if(this->drawEyeStyle == 1)
            {
                this->drawHappyEyes1(ctx, scaleFactor);
            }
            else
            {
                this->drawHappyEyes2(ctx, scaleFactor);
            }
        }
    }
    PointMass *getPointMass(int index) const
    {
        index += this->pointMasses.count();
        index = index % this->pointMasses.count();
        return this->pointMasses[index];
    }
    void drawBody(Context2D *ctx, float scaleFactor)
    {
        int i;

        ctx->setStrokeStyle("#000000");
        if(this->selected == true)
        {
            ctx->setFillStyle("#C11B17");
        }
        else
        {
            ctx->setFillStyle("#FDD017");
        }
        ctx->setLineWidth(5);
        ctx->beginPath();
        ctx->moveTo(this->pointMasses[0]->getXPos() * scaleFactor,
                    this->pointMasses[0]->getYPos() * scaleFactor);

        for(i = 0; i < this->pointMasses.count(); i++)
        {
            float px, py, nx, ny, tx, ty, cx, cy;
            PointMass *prevPointMass, *currentPointMass,
                *nextPointMass, *nextNextPointMass;

            prevPointMass = this->getPointMass(i - 1);
            currentPointMass = this->pointMasses[i];
            nextPointMass = this->getPointMass(i + 1);
            nextNextPointMass = this->getPointMass(i + 2);

            tx = nextPointMass->getXPos();
            ty = nextPointMass->getYPos();

            cx = currentPointMass->getXPos();
            cy = currentPointMass->getYPos();

            px = cx * 0.5 + tx * 0.5;
            py = cy * 0.5 + ty * 0.5;

            nx = cx - prevPointMass->getXPos() + tx -
                 nextNextPointMass->getXPos();
            ny = cy - prevPointMass->getYPos() + ty -
                 nextNextPointMass->getYPos();

            px += nx * 0.16;
            py += ny * 0.16;

            px = px * scaleFactor;
            py = py * scaleFactor;

            tx = tx * scaleFactor;
            ty = ty * scaleFactor;

            ctx->bezierCurveTo(px, py, tx, ty, tx, ty);
        }

        ctx->closePath();
        ctx->stroke();
        ctx->fill();
    }
    void drawSimpleBody(Context2D *ctx, float scaleFactor)
    {
        for(int i = 0; i < this->sticks.count(); i++)
        {
            this->sticks[i]->draw(ctx, scaleFactor);
        }
    }
    void draw(Context2D *ctx, float scaleFactor)
    {
        Vector up, ori;
        float ang;

        this->drawBody(ctx, scaleFactor);

        ctx->setStrokeStyle("#000000");
        ctx->setFillStyle("#000000");

        ctx->save();
        ctx->translate(this->middlePointMass->getXPos() * scaleFactor,
                      (this->middlePointMass->getYPos() - 0.35 * this->radius) *
                       scaleFactor);

        up = Vector(0.0, -1.0);
        ori = Vector(0.0, 0.0);
        ori.set(*this->pointMasses[0]->getPos());
        ori.sub(*this->middlePointMass->getPos());
        ang = acos(ori.dotProd(up) / ori.length());
        if(ori.getX() < 0.0) {
            ctx->rotate(-ang);
        } else {
            ctx->rotate(ang);
        }

        // this->drawEars(ctx, scaleFactor);
        this->drawFace(ctx, scaleFactor);

        ctx->restore();
    }
};

struct BlobCollective {
    int maxNum;
    int numActive;
    QVector<Blob*> blobs;
    Vector tmpForce;
    Blob *selectedBlob;

    BlobCollective(float x, float y, int startNum, int maxNum)
        : blobs(maxNum)
    {
        Q_UNUSED(startNum);
        this->maxNum = maxNum;
        this->numActive = 1;
        this->tmpForce = Vector(0.0, 0.0);
        this->selectedBlob = 0;

        this->blobs[0] = new Blob(x, y, 0.4, 8);
    }
    void split()
    {
        int i;
        float maxRadius = 0.0;
        int emptySlot;
        Blob *motherBlob, *newBlob;

        if(this->numActive == this->maxNum)
        {
            return;
        }

        emptySlot = this->blobs.count() - 1;
        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] != 0 &&
               this->blobs[i]->getRadius() > maxRadius)
            {
                maxRadius = this->blobs[i]->getRadius();
                motherBlob = this->blobs[i];
            }
            else if(this->blobs[i] == 0)
            {
                emptySlot = i;
            }
        }

        motherBlob->scale(0.75);
        newBlob = new Blob(motherBlob->getXPos(),
                           motherBlob->getYPos(),
                           motherBlob->getRadius(), 8);

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            this->blobs[i]->addBlob(*newBlob);
            newBlob->addBlob(*this->blobs[i]);
        }
        this->blobs[emptySlot] = newBlob;

        this->numActive++;
    }
    int findSmallest(int exclude)
    {
        float minRadius = 1000.0;
        int i, minIndex;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(i == exclude || this->blobs[i] == 0)
            {
                continue;
            }
            if(this->blobs[i]->getRadius() < minRadius)
            {
                minIndex = i;
                minRadius = this->blobs[i]->getRadius();
            }
        }
        return minIndex;
    }
    int findClosest(int exclude)
    {
        int foundIndex =0;
        float minDist = 1000.0, dist, aXbX, aYbY;
        int i;
        PointMass *myPointMass, *otherPointMass;

        myPointMass = this->blobs[exclude]->getMiddlePointMass();
        for(i = 0; i < this->blobs.count(); i++)
        {
            if(i == exclude || this->blobs[i] == 0)
            {
                continue;
            }

            otherPointMass = this->blobs[i]->getMiddlePointMass();
            aXbX = myPointMass->getXPos() - otherPointMass->getXPos();
            aYbY = myPointMass->getYPos() - otherPointMass->getYPos();
            dist = aXbX * aXbX + aYbY * aYbY;
            if(dist < minDist)
            {
                minDist = dist;
                foundIndex = i;
            }
        }
        return foundIndex;
    }
    void join()
    {
        int blob1Index, blob2Index;
        //Blob *blob1, *blob2;
        float r1, r2, r3;

        if(this->numActive == 1)
        {
            return;
        }

        blob1Index = this->findSmallest(-1);
        blob2Index = this->findClosest(blob1Index);

        r1 = this->blobs[blob1Index]->getRadius();
        r2 = this->blobs[blob2Index]->getRadius();
        r3 = sqrt(r1 * r1 + r2 * r2);

        this->blobs[blob1Index] = 0;
        this->blobs[blob2Index]->scale(0.945 * r3 / r2);

        this->numActive--;
    }
    Vector selectBlob(float x, float y)
    {
        int i;
        float minDist = 10000.0;
        PointMass *otherPointMass;
        //Blob *selectedBlob;
        Vector selectOffset;

        if(this->selectedBlob != 0)
        {
            return Vector();
        }

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }

            otherPointMass = this->blobs[i]->getMiddlePointMass();
            float aXbX = x - otherPointMass->getXPos();
            float aYbY = y - otherPointMass->getYPos();
            float dist = aXbX * aXbX + aYbY * aYbY;
            if(dist < minDist)
            {
                minDist = dist;
                if(dist < this->blobs[i]->getRadius() * 0.5)
                {
                    this->selectedBlob = this->blobs[i];
                    selectOffset = Vector(aXbX, aYbY);
                }
            }
        }

        if(this->selectedBlob != 0)
        {
            this->selectedBlob->setSelected(true);
        }
        return selectOffset;
    }
    void unselectBlob()
    {
        if(this->selectedBlob == 0)
        {
            return;
        }
        this->selectedBlob->setSelected(false);
        this->selectedBlob = 0;
    }
    void selectedBlobMoveTo(float x, float y)
    {
        if(this->selectedBlob == 0)
        {
            return;
        }
        this->selectedBlob->moveTo(x, y);
    }
    void move(float dt)
    {
        int i;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            this->blobs[i]->move(dt);
        }
    }
    void sc(Environment *env)
    {
        int i;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            this->blobs[i]->sc(env);
        }
    }
    void setForce(const Vector &force)
    {
        int i;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            if(this->blobs[i] == this->selectedBlob)
            {
                this->blobs[i]->setForce(Vector(0.0, 0.0));
                continue;
            }
            this->blobs[i]->setForce(force);
        }
    }
    void addForce(const Vector &force)
    {
        int i;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            if(this->blobs[i] == this->selectedBlob)
            {
                continue;
            }
            this->tmpForce.setX(force.getX() * (brand() * 0.75 + 0.25));
            this->tmpForce.setY(force.getY() * (brand() * 0.75 + 0.25));
            this->blobs[i]->addForce(this->tmpForce);
        }
    }
    void draw(Context2D *ctx, float scaleFactor)
    {
        int i;

        for(i = 0; i < this->blobs.count(); i++)
        {
            if(this->blobs[i] == 0)
            {
                continue;
            }
            this->blobs[i]->draw(ctx, scaleFactor);
        }
    }
};

Environment *env = 0;
float width = 800.0;
float height = 400.0;
float scaleFactor = 200.0;
BlobCollective *blobColl = 0;
Vector  gravity;
QPointF savedMouseCoords;
QPointF selectOffset;

void updateBlobs()
{
    float dt = 0.05;

    if (savedMouseCoords != QPointF() &&
        selectOffset != QPointF()) {
        blobColl->selectedBlobMoveTo(savedMouseCoords.x() - selectOffset.x(),
                                     savedMouseCoords.y() - selectOffset.y());
    }

    blobColl->move(dt);
    blobColl->sc(env);
    blobColl->setForce(gravity);
}

void draw(Context2D *ctx)
{
    //Qt extension
    ctx->clearRect(0, 0, width, height);

    env->draw(ctx, scaleFactor);
    blobColl->draw(ctx, scaleFactor);
}

void timeout()
{
   
}


QPointF getMouseCoords(QMouseEvent *event) {
    if(event == 0) {
        return QPointF();
    }
    return QPointF(event->x()/scaleFactor,
                   event->y()/scaleFactor);
}

void init()
{
    env = new Environment(0.2, 0.2, 2.6, 1.6);
    blobColl = new BlobCollective(1.0, 1.0, 1, 200);
    gravity = Vector(0.0, 10.0);

    timeout();
}

void toggleGravity()
{
    if(gravity.getY() > 0.0) {
        gravity.setY(0.0);
    } else {
        gravity.setY(10.0);
    }
}

Widget::Widget()
{
    init();
/*
    Maemo::QOrientationInfo *of = Maemo::QOrientationInfo::instance();
    connect(of, SIGNAL(orientationChanged(qreal,qreal)), SLOT(setGravity(qreal,qreal)));
    of->startListening();
*/
    m_timer.setInterval(1000/30);
    connect(&m_timer, SIGNAL(timeout()),
            this, SLOT(update()));
    m_timer.start();
}

Widget::~Widget(){
/*   Maemo::QOrientationInfo *of = Maemo::QOrientationInfo::instance();
   of->stopListening();*/
}

QSize Widget::sizeHint() const
{
    return QSize(800, 480);
}

void Widget::paintEvent(QPaintEvent *)
{
    //qDebug()<<"updating";
    Context2D ctx(this);
    draw(&ctx);
    updateBlobs();
    QPainter p(this);
    p.drawPixmap(0, 0, ctx.end());
}

void Widget::keyPressEvent(QKeyEvent *e)
{

    switch(e->key())
    {
        // left
    case Qt::Key_Left:
	gravity = Vector(-10.0, 0.0);
        break;

        // up
    case Qt::Key_Up:
	gravity = Vector(0.0, -10.0);
        break;

        // right
    case Qt::Key_Right:
	gravity = Vector(10.0, 0.0);
        break;

        // down
    case Qt::Key_Down:
	gravity = Vector(0.0, 10.0);
        break;

        // join 'j'
    case Qt::Key_J:
        blobColl->join();
        break;

        // split 'h'
    case Qt::Key_H:
        blobColl->split();
        break;

        // toggle gravity 'g'
    case Qt::Key_G:
        toggleGravity();
        break;

    default:
        break;
    }
}

void Widget::mousePressEvent(QMouseEvent *e)
{
    QPointF mouseCoords;

    mouseCoords = getMouseCoords(e);
    if(mouseCoords == QPointF())
    {
        return;
    }
    Vector vec = blobColl->selectBlob(mouseCoords.x(), mouseCoords.y());
    selectOffset = QPointF(vec.x, vec.y);
}

void Widget::mouseReleaseEvent(QMouseEvent *)
{
    blobColl->unselectBlob();
    savedMouseCoords = QPointF();
    selectOffset = QPointF();
}

void Widget::mouseMoveEvent(QMouseEvent *e)
{
    QPointF mouseCoords;

    if(selectOffset == QPointF())
    {
        return;
    }
    mouseCoords = getMouseCoords(e);
    if(mouseCoords == QPointF())
    {
        return;
    }
    blobColl->selectedBlobMoveTo(mouseCoords.x() - selectOffset.x(),
                                 mouseCoords.y() - selectOffset.y());

    savedMouseCoords = mouseCoords;
}

#include <QDebug>
void Widget::setGravity(qreal pitch,qreal roll){
    qDebug() << "XX" << pitch << roll;
}
