/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "webview_p.h"
#include "webview_p_p.h"
#include "webhistory_p.h"
#include "websettings_p.h"
#include <QKeyEvent>
#include <QPaintEvent>
#include <QPainter>

WebView::WebView(QWidget *parent) :
    Item(*new WebViewPrivate(this), parent)
{
    Q_D(WebView);

    d->page = new WebPage(this);
    d->page->setView(this);
    d->suppressor = new WebViewSelectionSuppressor(this);

    this->connect(d->page, SIGNAL(urlChanged()), this, SIGNAL(urlChanged()));
    this->connect(d->page, SIGNAL(titleChanged()), this, SIGNAL(titleChanged()));
    this->connect(d->page, SIGNAL(iconChanged()), this, SIGNAL(iconChanged()));
    this->connect(d->page, SIGNAL(contentSizeChanged()), this, SIGNAL(contentSizeChanged()));
    this->connect(d->page, SIGNAL(preferredWidthChanged()), this, SIGNAL(preferredWidthChanged()));
    this->connect(d->page, SIGNAL(preferredHeightChanged()), this, SIGNAL(preferredHeightChanged()));
    this->connect(d->page, SIGNAL(zoomFactorChanged()), this, SIGNAL(zoomFactorChanged()));
    this->connect(d->page, SIGNAL(progressChanged()), this, SIGNAL(progressChanged()));
    this->connect(d->page, SIGNAL(statusChanged()), this, SIGNAL(statusChanged()));
    this->connect(d->page, SIGNAL(statusTextChanged()), this, SIGNAL(statusTextChanged()));
    this->connect(d->page, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));
    this->connect(d->page, SIGNAL(newWindowComponentChanged()), this, SIGNAL(newWindowComponentChanged()));
    this->connect(d->page, SIGNAL(newWindowParentChanged()), this, SIGNAL(newWindowParentChanged()));
    this->connect(d->page, SIGNAL(linkClicked(QUrl)), this, SIGNAL(linkClicked(QUrl)));
    this->connect(d->page, SIGNAL(repaintRequested(QRect)), this, SLOT(update()));
}

WebView::WebView(WebViewPrivate &dd, QWidget *parent) :
    Item(dd, parent)
{
    Q_D(WebView);

    d->page = new WebPage(this);
    d->page->setView(this);
    d->suppressor = new WebViewSelectionSuppressor(this);

    this->connect(d->page, SIGNAL(urlChanged()), this, SIGNAL(urlChanged()));
    this->connect(d->page, SIGNAL(titleChanged()), this, SIGNAL(titleChanged()));
    this->connect(d->page, SIGNAL(iconChanged()), this, SIGNAL(iconChanged()));
    this->connect(d->page, SIGNAL(contentSizeChanged()), this, SIGNAL(contentSizeChanged()));
    this->connect(d->page, SIGNAL(preferredWidthChanged()), this, SIGNAL(preferredWidthChanged()));
    this->connect(d->page, SIGNAL(preferredHeightChanged()), this, SIGNAL(preferredHeightChanged()));
    this->connect(d->page, SIGNAL(zoomFactorChanged()), this, SIGNAL(zoomFactorChanged()));
    this->connect(d->page, SIGNAL(progressChanged()), this, SIGNAL(progressChanged()));
    this->connect(d->page, SIGNAL(statusChanged()), this, SIGNAL(statusChanged()));
    this->connect(d->page, SIGNAL(statusTextChanged()), this, SIGNAL(statusTextChanged()));
    this->connect(d->page, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));
    this->connect(d->page, SIGNAL(newWindowComponentChanged()), this, SIGNAL(newWindowComponentChanged()));
    this->connect(d->page, SIGNAL(newWindowParentChanged()), this, SIGNAL(newWindowParentChanged()));
    this->connect(d->page, SIGNAL(linkClicked(QUrl)), this, SIGNAL(linkClicked(QUrl)));
    this->connect(d->page, SIGNAL(repaintRequested(QRect)), this, SLOT(update()));
}

WebView::~WebView() {}

QUrl WebView::url() const {
    Q_D(const WebView);

    return d->page->url();
}

void WebView::setUrl(const QUrl &url) {
    Q_D(WebView);

    d->page->setUrl(url);
}

QUrl WebView::baseUrl() const {
    Q_D(const WebView);

    return d->page->baseUrl();
}

QUrl WebView::requestedUrl() const {
    Q_D(const WebView);

    return d->page->requestedUrl();
}

QString WebView::icon() const {
    Q_D(const WebView);

    return d->page->icon();
}

QString WebView::title() const {
    Q_D(const WebView);

    return d->page->title();
}

int WebView::contentWidth() const {
    Q_D(const WebView);

    return d->page->contentWidth();
}

int WebView::contentHeight() const {
    Q_D(const WebView);

    return d->page->contentHeight();
}

int WebView::preferredWidth() const {
    Q_D(const WebView);

    return d->page->preferredWidth();
}

void WebView::setPreferredWidth(int width) {
    Q_D(WebView);

    d->page->setPreferredWidth(width);
}

int WebView::preferredHeight() const {
    Q_D(const WebView);

    return d->page->preferredHeight();
}

void WebView::setPreferredHeight(int height) {
    Q_D(const WebView);

    d->page->setPreferredHeight(height);
}

qreal WebView::zoomFactor() const {
    Q_D(const WebView);

    return d->page->zoomFactor();
}

void WebView::setZoomFactor(qreal zoom) {
    Q_D(WebView);

    d->page->setZoomFactor(zoom);
}

bool WebView::textSelectionEnabled() const {
    Q_D(const WebView);

    return !d->suppressor->isEnabled();
}

void WebView::setTextSelectionEnabled(bool enabled) {
    Q_D(WebView);

    if (enabled == this->textSelectionEnabled()) {
        return;
    }

    if (enabled) {
        d->suppressor->disable();
    }
    else {
        d->suppressor->enable();
    }

    emit textSelectionEnabledChanged();
}

WebPage::LinkDelegationPolicy WebView::linkDelegationPolicy() const {
    Q_D(const WebView);

    return d->page->linkDelegationPolicy();
}

void WebView::setLinkDelegationPolicy(WebPage::LinkDelegationPolicy policy) {
    Q_D(WebView);

    d->page->setLinkDelegationPolicy(policy);
}

int WebView::progress() const {
    Q_D(const WebView);

    return d->page->progress();
}

WebPage::Status WebView::status() const {
    Q_D(const WebView);

    return d->page->status();
}

QString WebView::statusText() const {
    Q_D(const WebView);

    return d->page->statusText();
}

QString WebView::selectedText() const {
    Q_D(const WebView);

    return d->page->selectedText();
}

QDeclarativeComponent* WebView::newWindowComponent() const {
    Q_D(const WebView);

    return d->page->newWindowComponent();
}

void WebView::setNewWindowComponent(QDeclarativeComponent *component) {
    Q_D(WebView);

    d->page->setNewWindowComponent(component);
}

QWidget* WebView::newWindowParent() const {
    Q_D(const WebView);

    return qobject_cast<QWidget*>(d->page->newWindowParent());
}

void WebView::setNewWindowParent(QWidget *parent) {
    Q_D(WebView);

    d->page->setNewWindowParent(parent);
}

bool WebView::findText(const QString &text) {
    Q_D(WebView);

    return d->page->findText(text);
}

bool WebView::findAllText(const QString &text) {
    Q_D(WebView);

    return d->page->findAllText(text);
}

void WebView::load(const QUrl &url) {
    Q_D(WebView);

    d->page->load(url);
}

void WebView::reload() {
    Q_D(WebView);

    d->page->reload();
}

void WebView::stop() {
    Q_D(WebView);

    d->page->stop();
}

void WebView::back() {
    Q_D(WebView);

    d->page->back();
}

void WebView::forward() {
    Q_D(WebView);

    d->page->forward();
}

void WebView::paintEvent(QPaintEvent *event) {
    Q_D(WebView);

    QPainter painter(this);
    d->page->mainFrame()->render(&painter);
    event->accept();
}

void WebView::resizeEvent(QResizeEvent *event) {
    Q_D(WebView);

    d->page->setViewportSize(event->size());
    Item::resizeEvent(event);
}

void WebView::keyPressEvent(QKeyEvent *event) {
    Q_D(WebView);

    switch (event->key()) {
    case Qt::Key_C:
        if (event->modifiers() & Qt::ControlModifier) {
            d->page->copy();
            event->accept();
            return;
        }

        break;
    case Qt::Key_X:
        if (event->modifiers() & Qt::ControlModifier) {
            d->page->cut();
            event->accept();
            return;
        }

        break;
    case Qt::Key_V:
        if (event->modifiers() & Qt::ControlModifier) {
            d->page->paste();
            event->accept();
            return;
        }

        break;
    default:
        break;
    }

    Item::keyPressEvent(event);
}

QDeclarativeListProperty<QObject> WebViewPrivate::jsObjects() {
    return qvariant_cast< QDeclarativeListProperty<QObject> >(page->property("javaScriptWindowObjects"));
}

WebHistory* WebViewPrivate::history() {
    return qvariant_cast<WebHistory*>(page->property("history"));
}

WebSettings* WebViewPrivate::settings() {
    return qvariant_cast<WebSettings*>(page->property("settings"));
}

#include "moc_webview_p.cpp"
